﻿/**
* Copyright 2011 Microsoft Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// Module dependencies.
var xmlbuilder = require('xmlbuilder');

var Constants = require('../../../util/constants');
var ISO8061Date = require('../../../util/iso8061date');

// Expose 'ContainerAclResult'.
exports = module.exports = ContainerAclResult;

function ContainerAclResult(signedIdentifiers) {
  if (signedIdentifiers) {
    this.signedIdentifiers = signedIdentifiers;
  }
}

/**
* Builds an XML representation for container acl permissions.
*
* @param  {array}  entity The signed identifiers.
* @return {string} The XML container acl permissions.
*/
ContainerAclResult.serialize = function (signedIdentifiersJs) {
  var doc = xmlbuilder.create();
  doc = doc.begin(Constants.SIGNED_IDENTIFIERS_ELEMENT, { version: '1.0', encoding: 'utf-8' });

  if (Array.isArray(signedIdentifiersJs) && signedIdentifiersJs.length > 0) {
    signedIdentifiersJs.forEach(function (signedIdentifier) {
      doc = doc
        .ele(Constants.SIGNED_IDENTIFIER_ELEMENT)
          .ele(Constants.ID)
            .txt(signedIdentifier.Id)
          .up()
          .ele(Constants.ACCESS_POLICY);

      if (signedIdentifier.AccessPolicy.Start) {
        var startIsoString = signedIdentifier.AccessPolicy.Start;
        if (startIsoString instanceof Date) {
          // Convert to expected ISO 8061 date format
          startIsoString = ISO8061Date.format(startIsoString);
        }

        doc = doc
            .ele(Constants.START)
              .txt(startIsoString)
            .up();
      }

      if (signedIdentifier.AccessPolicy.Expiry) {
        var expiryIsoString = signedIdentifier.AccessPolicy.Expiry;
        if (expiryIsoString instanceof Date) {
          // Convert to expected ISO 8061 date format
          expiryIsoString = ISO8061Date.format(expiryIsoString);
        }

        doc = doc
            .ele(Constants.EXPIRY)
              .txt(expiryIsoString)
            .up();
      }

      if (signedIdentifier.AccessPolicy.Permissions) {
        doc = doc
            .ele(Constants.PERMISSION)
              .txt(signedIdentifier.AccessPolicy.Permissions)
            .up();
      }

      doc = doc.up().up();
    });
  }

  return doc.doc().toString();
};

ContainerAclResult.parse = function (containerAclXml) {
  return containerAclXml;
};