//
// database.js
//

var mongodb = require('mongodb');
var assert = require('assert');
var database = exports;

/**
 * データベースアクセス用基本クラス。
 * コレクションオブジェクトの作成など、基本的なデータベースアクセスメソッドを
 * 提供する
 */
var Database = function () {};

/**
 * データベースオブジェクトを取得する
 */
Database.prototype._open = function (database, callback) {
  var self = this;

  // すでにデータベースオブジェクトが作成されていた場合、再利用する
  if (this._db && !this._db.close) {
    callback(null, this._db);
    return;
  }
  var server = new mongodb.Server('127.0.0.1', 27017, {});
  var db_connector = new mongodb.Db(database, server, {safe:true});

  // データベースを開く
  db_connector.open(function (err, db) {
    if (err) {
      callback(err);
      return;
    }
    self._db = db;
    callback(null, db);
  });
}

/**
 * データベース接続をすべてクローズする
 */
Database.prototype.close = function() {
  if (this._db) {
    this._db.close();
    delete this._db;
  }
}

/**
 * コレクションにアクセスするためのオブジェクトを取得する
 */
Database.prototype._getCollection = function (collectionName, callback) {
  this._open('mobbs', function(err, db) {
    if (err) {
      return callback(err);
    }
    db.createCollection(collectionName, callback);
  });
};

/**
 * トピックデータにアクセスするためのクラス
 */
var Topics = function () {};
Topics.prototype = new Database();

/**
 * Topicsクラスのインスタンスを生成する
 */
database.getTopics = function () {
  return new Topics();
}

/**
 * トピックを新しい順にソートし、start番目からend番目までのトピックを取得する
 */
Topics.prototype.getLatest = function (start, end, callback) {
  // topicsコレクションを取得する
  this._getCollection('topics', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    // topicsデータの全データにアクセスするカーソルを作成する
    var cursor = collection.find({});
    // start、endの値に応じてカーソル位置を移動させる
    cursor.sort([['date', -1]]).limit(end - start).skip(start);
    // カーソルの対象となる値を配列として返す
    cursor.toArray(callback);
  });
};

/**
 * トピックIDで指定されたトピックを返す
 */
Topics.prototype.findById = function (topicId, callback) {
  this._getCollection('topics', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    // topicIdフィールドが指定したトピックIDであるドキュメントを取得
    collection.findOne({topicId: topicId}, callback);
  });
};

/**
 * トピックを新規作成する
 */
Topics.prototype.insert = function (topic, callback) {
  var self = this;

  // countersコレクションからカウンタの値を取得する
  this._getCollection('counters', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    collection.findAndModify({name:'topics'}, {}, {$inc: {count:1}}, createTopic);
  });

  // 取得したカウンタの値を作成するトピックのトピックIDとしてトピックを作成
  function createTopic(err, counter) {
    var nextId = counter.count;
    var newTopic = {
      topicId: nextId,
      title: topic.title,
      text: topic.text,
      date: new Date(),
      postBy: topic.postBy || '',
      relatedUrl: topic.relatedUrl || '',
      replies: []
    };
    self._getCollection('topics', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      collection.insert(newTopic, function (err, obj) {
        if (err) {
          callback(err);
          return;
        }
        callback(err, obj[0]);
      });
    });
  }
};

/**
 * 指定したトピックのreplies要素に指定したコメントIDを追加する
 */
Topics.prototype.appendReply = function (topicId, commentId, callback) { 
  this._getCollection('topics', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    var cursor = collection.findAndModify(
      {topicId:topicId},
      {},
      {$push: {replies:commentId}},
      callback
    );
  });
};

/**
 * コメントデータにアクセスするためのオブジェクト
 */
var Comments = function () {
  this._db = null;
  this._client = null;
};
Comments.prototype = new Database();

/**
 * Commentsクラスのインスタンスを生成する
 */
database.getComments = function () {
  return new Comments();
}

/**
 * コメントIDで指定したコメントを取得
 */
Comments.prototype.findById = function (commentId, callback) {
  this._getCollection('comments', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    // commentIdフィールドが指定したコメントIDであるドキュメントを取得
    var cursor = collection.findOne({commentId: commentId}, callback);
  });
};

/**
 * 指定したコメントのreplies要素に指定したコメントIDを追加する
 */
Comments.prototype.appendReply = function (parentId, childId, callback) { 
  this._getCollection('comments', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    var cursor = collection.findAndModify(
      {commentId:parentId},
      {},
      {$push: {replies:childId}},
      callback
    );
  });
};

/**
 * コメントを新規作成してデータベースに挿入する
 */
Comments.prototype.insert = function (comment, callback) {
  var self = this;
  // countersコレクションからカウンタの値を取得する
  this._getCollection('counters', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    collection.findAndModify({name:'comment'}, {}, {$inc: {count:1}}, createComment);
  });

  // 取得したカウンタの値をコメントIDとするコメントを作成
  function createComment(err, counter) {
    var nextId = counter.count;
    var newComment = {
      commentId: nextId,
      topicId: comment.topicId,
      title: comment.title,
      text: comment.text,
      date: new Date(),
      postBy: comment.postBy || '',
      relatedUrl: comment.relatedUrl || '',
      replies: [],
      parentCommentId: comment.parentCommentId || null
    };

    // 親コメントが指定されていない場合、トピックのrepliesにIDを追加
    if (comment.parentCommentId === null) {
      var topics = database.getTopics();
      topics.appendReply(comment.topicId, nextId, function (err, topic) {
        topics.close();
        if (err) {
          callback(err);
          return;
        }
        insertComment(newComment, callback);
        });
    } else {
      self.appendReply(comment.parentCommentId, nextId, function (err, parent) {
        if (err) {
          callback(err);
          return;
        }
        insertComment(newComment, callback);
      });
    }
  }

  // コメントをデータベースに挿入する
  function inserteComment(newComment) {
    this._getCollection('comments', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      collection.insert(newComment, {}, function (err, obj) {
        if (err) {
          callback(err);
          return;
        }
        callback(err, obj[0]);
      });
    });
  }
};

/**
 * コメントに付けられたコメントをツリー状のオブジェクトで取得する
 */
Comments.prototype.getCommentTree = function (topicId, callback) {
  var self = this;
  var topics = database.getTopics();

  topics.findById(topicId, function (err, topic) {
    topics.close();
    if (err) {
      callback(err);
      return;
    }
    self._getCollection('comments', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      var items = {};
      var cursor = collection.find({topicId:topicId});
      cursor.sort([['date', -1]]);
      cursor.each(function (err, comment) {
        if (err) {
          callback(err);
          return;
        }
        if (comment !== null) {
          items[comment.commentId] =  comment;
        } else {
          // ループ終了時の処理
          comments.close();
          var tree = buildCommentTree(topic.replies, items, []);
          callback(null, tree);
        }
      });
    });
  });

  function buildCommentTree(parents, comments, tree) {
    var comment;
    for (var i = 0; i < parents.length; i++) {
      comment = comments[parents[i]];
      comment.children = [];
      buildCommentTree(comment.replies, comments, comment.children);
      tree.push(comment);
    }
    return tree;
  }
};
