// export.js
/*
 * usage:
 * node export.js <target directory> <output directory>
 */

var config = require('../../config');
var path = require('path');
var fs = require('fs');

var HTML_HEADER = 'header.txt';
var HTML_FOOTER = 'footer.txt';
var IMAGE_DIR = 'figure';

var headerFullPath = path.join(path.dirname(module.filename), HTML_HEADER);
var footerFullPath = path.join(path.dirname(module.filename), HTML_FOOTER);

var HTML_HEAD = fs.readFileSync(headerFullPath, {encoding: 'utf8'});
var HTML_FOOT = fs.readFileSync(footerFullPath, {encoding: 'utf8'});

var usage = 'node export.js <target directory> <output directory>';


var figureList = [];

function exportSingle(target, output) {
  fs.readFile(target, 'utf8', function (err, data) {
    if (err) {
      console.log('Cannot read file: ' + target);
      return;
    }

    var result = parseText(data);
    result.html = makeCaptionHtml(result.figure);

    fs.writeFileSync(path.join(output, 'honmon.txt'),
                     result.text.join('\n'));
    fs.writeFileSync(path.join(output, 'caption.txt'),
                     result.caption.join('\n'));
    fs.writeFileSync(path.join(output, 'filelist.txt'),
                     figureList.join('\n'));

    var out = fs.createWriteStream(path.join(output, 'figures.html'), {flags: 'w', encoding: 'utf8', mode: 0666});
    out.write(HTML_HEAD);
    out.write('\n');
    out.write(result.html.join('\n'));
    out.write('\n');
    out.write(HTML_FOOT);
    out.end();

  });
}

function findImageFile(name, targetPath) {
  var dirname = path.dirname(targetPath);
  var figDirs = config.figureDir;

  for (var k = 0; k < figDirs.length; k++) {
    var targetDir = path.join(dirname, figDirs[k]);
    var files = fs.readdirSync(targetDir);
    var exts = config.figureFormat;
    for (var i = 0; i < files.length; i++) {
      for (var j = 0; j < exts.length; j++) {
        if (files[i].indexOf(name + exts[j]) != 0) {
          return name + exts[j];
        }
      }
    }
  }
  return '';
}

function makeCaptionHtml(text) {
  var results = [];
  for (var i = 0; i < text.length; i++) {

    // 図
    if (text[i].match(/^☆図/)) {
      var figId = text[i].match(/^☆図([0-9]+-[0-9]+)/);
      var figName = 'fig' + figId[1] + '.fw.png';
      results.push('<span class="caption">' + text[i] + '</span>');
      i++;
      results.push('<img src="' + figName + '">');
      figureList.push(figName);
      i++;
    }
    
    // end
    results.push(text[i]);
  }
  return results;
}

// replace image file name to full pathname
function replaceImageToPathname(text, targetPath) {
  var results = [];
  for (var i = 0; i < text.length; i++) {

    // 図
    if (text[i].match(/^☆図/)) {
      results.push(text[i]);
      i++;
      var file = findImageFile(text[i], targetPath);
      figureList.push(file);
      text[i] = file;
    }
    
    // end
    results.push(text[i]);
  }
  return results;
}

// parse main text to separate body text and captions/tables
function parseText(data) {
  var lines = data.split('\n');
  var counter = 0;
  var texts = [];
  var captions = [];
  var figures = [];
  var isCaption = false;

  for (counter = 0; counter < lines.length; counter++) {

    // figure box
    if (lines[counter].match(/^＞＞＞＞/)) {
      isCaption = false;
      counter++;
      while (!lines[counter].match(/^＞＞＞＞/)) {
        if (lines[counter].match(/^＝＝＝＝/)) {
          isCaption = true;
          counter++;
          continue;
        }
        captions.push(lines[counter]);
        if (!isCaption) {
          figures.push(lines[counter]);
        }
        counter++;
      }
      counter++;
    }

    // cleanup blank line
    if (lines[counter].match(/^\s+$/)) {
      lines[counter] = '';
    }

    // end
    if ( lines[counter] === ''
         && texts.length > 0
         && texts[texts.length-1] === '' ) {
      // do nothing
    } else {
      texts.push(lines[counter]);
    }
  }
  var result = {};
  result.text = texts;
  result.caption = replaceImageToPathname(captions, target);
  result.figure = figures;
  return result;
}


// main action
if (require.main == module) {
  if (process.argv.length < 4) {
    process.stdout.write(usage + '\n');
    process.exit(1);
  }
  var target = process.argv[2];
  var output = process.argv[3];
  exportSingle(target, output);
}
