package com.tryjava.net;

import java.io.InputStream;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.PasswordAuthentication;
import java.net.URL;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;

import com.tryjava.AppConfig;
import com.tryjava.util.AppUtil;

/**
 * HTTPクライアントを作成してみる。（Java標準APIのみ）
 *
 * プロキシを使ってみる。Basic認証あり。
 *
 * TODO ProxySelectorを使うとプロキシ要否を使い分けられるらしい。<br>
 * JavaのHTTP通信でプロキシを使う http://www.mwsoft.jp/programming/java/http_proxy.html<br>
 *
 * 参考<br>
 * JavaでのHttpClientサンプル - CLOVER<br>
 * http://d.hatena.ne.jp/Kazuhira/20131026/1382796711<br>
 * Javaで認証が必要なProxyサーバー経由でHTTP接続する - Qiita
 * http://qiita.com/tag1216/items/24a78d0b4bf2ca641d97<br>
 *
 * @since 2016/5/26
 */
public class JavaHttpClient02Proxy {
	private static final String PROXY_HOST = AppConfig.getString("proxy.host");
	private static final int PROXY_PORT = AppConfig.getInt("proxy.port");
	private static final String PROXY_USER = AppConfig.getString("proxy.user");
	private static final String PROXY_PASSWORD = AppConfig.getString("proxy.password");

	public void run() {
		doGet();
	}

	void doGet() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		// プロキシ設定
		System.setProperty("proxySet", "true");
		System.setProperty("proxyHost", PROXY_HOST);
		System.setProperty("proxyPort", String.valueOf(PROXY_PORT));
		Authenticator.setDefault(new Authenticator() {
			@Override
			protected PasswordAuthentication getPasswordAuthentication() {
				return new PasswordAuthentication(PROXY_USER, PROXY_PASSWORD.toCharArray());
			}
		});

		HttpURLConnection con = null;
		try {
			// リクエスト作成
			URL url = new URL("http://www.google.co.jp/");
			con = (HttpURLConnection) url.openConnection();
			con.setRequestMethod("GET");

			// レスポンスコード取得
			int statusCode = con.getResponseCode();
			System.out.println(statusCode + " " + con.getResponseMessage());
			if (statusCode != HttpURLConnection.HTTP_OK) {
				throw new RuntimeException("エラーレスポンスを受信しました。statusCode=" + statusCode);
			}

			// レスポンスボディ取得
			System.out.println(con.getContentType());
			InputStream is = con.getInputStream();
			String encoding = getEncoding(con);
			String content = IOUtils.toString(is, encoding);
			System.out.println(content);

		} catch (Exception e) {
			throw new RuntimeException("HTTP通信に失敗しました。", e);
		} finally {
			if (con != null) {
				con.disconnect();
			}
		}
	}

	/**
	 * 簡易的な文字コード判定処理
	 */
	String getEncoding(HttpURLConnection con) {
		String encoding = null;
		Matcher matcher = Pattern.compile("charset=(.*)", Pattern.CASE_INSENSITIVE).matcher(con.getContentType());
		if (matcher.find()) {
			encoding = matcher.group(1);
		}
		System.out.println("encoding=" + encoding);
		return encoding;
	}
}
