//
//  HMDocument.m
//  testAVFoundation
//
//  Created by 堀 昌樹 on 12/06/07.
//  Copyright (c) 2012年 __MyCompanyName__. All rights reserved.
//

#import "HMDocument.h"

#import <AVFoundation/AVFoundation.h>

@interface HMDocument ()
{
	CGFloat accessoryHeight;
}
@property (retain, nonatomic) AVURLAsset *asset;
@property (retain, nonatomic) AVPlayerItem *item;
@property (retain, nonatomic) AVPlayer *player;
@property (retain, nonatomic) id timeObserver;

@property (retain, nonatomic) CATextLayer *captionLayer;
@property (copy, nonatomic) NSString *caption;

@end

@implementation HMDocument
@synthesize view = _view;
@synthesize playButton = _playButton;
@synthesize currentTime = _currentTime;
@synthesize canPlay = _canPlay;

@synthesize asset = _asset;
@synthesize item = _item;
@synthesize player = _player;
@synthesize timeObserver = _timeObserver;
@synthesize captionLayer = _captionLayer;
@synthesize caption = _caption;

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self
													name:AVPlayerItemDidPlayToEndTimeNotification
												  object:_item];
	
	[_player release];
	[_item release];
	[_asset release];
	[_captionLayer release];
	
	[super dealloc];
}

- (NSString *)windowNibName
{
	return @"HMDocument";
}

- (void)windowControllerDidLoadNib:(NSWindowController *)aController
{
	[super windowControllerDidLoadNib:aController];
	
	NSRect wFrame = [[self windowForSheet] frame];
	NSRect vFrame = [_view frame];
	accessoryHeight = wFrame.size.height - vFrame.size.height;
	
	[self syncUI];
}

- (BOOL)readFromURL:(NSURL *)url ofType:(NSString *)typeName error:(NSError **)outError
{
	self.asset = [AVURLAsset URLAssetWithURL:url options:nil];
	
	return _asset ? YES : NO;
}


- (NSSize)fitSizeToSize:(NSSize)toSize
{
	if(!_asset) return toSize;
	
	CGSize movieSize = _asset.naturalSize;
	if(movieSize.width == 0) return toSize;
	
	CGFloat targetViewWidth = toSize.width;
	CGFloat targetViewHeight = targetViewWidth * (movieSize.height / movieSize.width);
	
	targetViewHeight += accessoryHeight;
	
	return NSMakeSize((long)targetViewWidth, (long)targetViewHeight);
}
- (void)sizeTofitWidnow
{
	NSWindow *window = [self windowForSheet];
	NSRect frame = [window frame];
	NSSize newSize = [self fitSizeToSize:frame.size];
	frame.origin.y += frame.size.height - newSize.height;
	frame.size = newSize;
	[window setFrame:frame display:YES animate:NO];
}
- (void)syncUI
{
	if((_player.currentItem != nil) &&
	   ([_player.currentItem status] == AVPlayerItemStatusReadyToPlay)) {
		self.canPlay = YES;
	}
	else {
		self.canPlay = NO;
	}
	
	CGFloat rate = _player.rate;
	if(_player.rate == 0) {
		_playButton.title = @">";
	} else {
		_playButton.title = @"||";
	}
	
	if(rate == 0.5) {
		self.caption = @"x 1/2";
	} else if(rate == 0.25) {
		self.caption = @"x 1/4";
	} else if(rate == 0.125) {
		self.caption = @"x 1/8";
	} else if(rate > 1) {
		self.caption = [NSString stringWithFormat:@"x %.0f", rate];
	} else if(rate < 0) {
		self.caption = @"<<";
	} else {
		self.caption = @"";
	}
}

#pragma mark -
#pragma mark KVC & KVO
- (void)setAsset:(AVURLAsset *)asset
{
	if(_asset == asset) return;
	
	[_asset autorelease];
	_asset = [asset retain];
	
	NSString *tracksKey = @"tracks";
	[_asset loadValuesAsynchronouslyForKeys:[NSArray arrayWithObject:tracksKey]
						  completionHandler:
	 ^ {
		 dispatch_async(dispatch_get_main_queue(),
						^{
							NSError *error = nil;
							AVKeyValueStatus status = [_asset statusOfValueForKey:tracksKey error:&error];
							if(status == AVKeyValueStatusLoaded) {
								self.item = [AVPlayerItem playerItemWithAsset:_asset];
							} else {
								NSLog(@"Asset's tracks were not loaded.\n%@", [error localizedDescription]);
								if(![self presentError:error]) {
									[self close];
								}
							}
						});
	 }];
}
- (void)setItem:(AVPlayerItem *)item
{
	if(_item == item) return;
	
	NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
	if(_item) {
		[_item removeObserver:self forKeyPath:@"status"];
		[nc removeObserver:self name:AVPlayerItemDidPlayToEndTimeNotification object:_item];
	}
	[_item autorelease];
	_item = [item retain];
	
	if(!_item) return;
	
	[_item addObserver:self
			forKeyPath:@"status"
			   options:0
			   context:_item];
	[nc addObserver:self
		   selector:@selector(playerItemDidReachEnd:)
			   name:AVPlayerItemDidPlayToEndTimeNotification
			 object:_item];
	
	self.player = [AVPlayer playerWithPlayerItem:_item];
}
- (void)setPlayer:(AVPlayer *)player
{
	if(_player == player) return;
	
	if(_player) {
		[_player removeObserver:self forKeyPath:@"rate"];
		[_player pause];
		[_player removeTimeObserver:self.timeObserver];
		self.timeObserver = nil;
	}
	[_player autorelease];
	_player = [player retain];
	
	if(!_player) return;
	
	[_player addObserver:self
			  forKeyPath:@"rate"
				 options:0
				 context:_player];
	self.timeObserver = [_player addPeriodicTimeObserverForInterval:CMTimeMake(24, 600)
																queue:dispatch_get_main_queue()
														   usingBlock:
						   ^(CMTime time) {
							   Float64 seconds = CMTimeGetSeconds(time);
							   self.currentTime = seconds;
						   }];
	
	AVPlayerLayer *layer = [AVPlayerLayer playerLayerWithPlayer:_player];
	[_view setWantsLayer:YES];
	[_view setLayer:layer];
	self.captionLayer = nil;
}
- (void)setCaption:(NSString *)caption
{
	if(![_view layer]) return;
	
	[_caption autorelease];
	_caption = [caption copy];
	
	if(!_captionLayer) {
		self.captionLayer = [[[CATextLayer alloc] init] autorelease];
		_captionLayer.fontSize = 40;
		_captionLayer.frame = CGRectMake([_view frame].size.width - 120 - 3, 3, 120, 35);
		_captionLayer.autoresizingMask = kCALayerMinXMargin | kCALayerMaxYMargin;
		_captionLayer.cornerRadius = 5;
		_captionLayer.truncationMode = kCATruncationMiddle;
		_captionLayer.alignmentMode = kCAAlignmentCenter;
		_captionLayer.foregroundColor = CGColorCreateGenericGray(1, 0.8);
		_captionLayer.backgroundColor = CGColorCreateGenericGray(0, 0.3);
		[[_view layer] addSublayer:_captionLayer];
	}
	
	_captionLayer.string = caption;
	if(!caption || [caption length] == 0) {
		_captionLayer.hidden = YES;
	} else {
		_captionLayer.hidden = NO;
	}
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context
{
	if(context == _item) {
		dispatch_async(dispatch_get_main_queue(),
					   ^{
						   [self sizeTofitWidnow];
						   [self syncUI];
					   });
		return;
	}
	if(context == _player) {
		dispatch_async(dispatch_get_main_queue(),
					   ^{
						   [self syncUI];
					   });
		return;
	}
	[super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
}

#pragma mark -
#pragma mark NSWindowDelegate
- (NSSize)windowWillResize:(NSWindow *)sender toSize:(NSSize)frameSize
{
	return [self fitSizeToSize:frameSize];
}
- (void)windowWillClose:(NSNotification *)notification
{
	[_player removeObserver:self forKeyPath:@"rate"];
	[_item removeObserver:self forKeyPath:@"status"];
	[_player pause];
	[_player removeTimeObserver:self.timeObserver];
	self.timeObserver = nil;
}

#pragma mark -
#pragma mark Notifications
- (void)playerItemDidReachEnd:(NSNotification *)notification
{
	[_item seekToTime:kCMTimeZero];
}

#pragma mark -
#pragma mark Actions
- (IBAction)togglePlayAndPause:(id)sender
{
	if(_player.rate == 0) {
		[_player play];
	} else {
		[_player pause];
	}
}
- (IBAction)changeCurrentTime:(id)sender
{
	CMTime halfSeconds = CMTimeMake(300, 600);
	CGFloat timeValue = [sender doubleValue] * 600;
	CMTime time = CMTimeMake(timeValue, 600);
	[_item seekToTime:time toleranceBefore:halfSeconds toleranceAfter:halfSeconds];
}
- (IBAction)volumeUp:(id)sender
{
	_player.volume += 0.1;
}
- (IBAction)volumeDown:(id)sender
{
	if(_player.volume < 0.1) return;
	_player.volume -= 0.1;
}

- (IBAction)fastForward:(id)sender
{
	if(_player.rate > 8) return;
	_player.rate *= 2;
}
- (IBAction)backward:(id)sender
{
	_player.rate = -1.0;
}
- (IBAction)slow:(id)sender
{
	if(_player.rate < 0.13) return;
	_player.rate *= 0.5;
}

- (BOOL)validateMenuItem:(NSMenuItem *)menuItem
{
	SEL action = [menuItem action];
	if(action == @selector(togglePlayAndPause:)) {
		if(_player.rate == 0) {
			[menuItem setTitle:@"Play"];
		} else {
			[menuItem setTitle:@"Pause"];
		}
	}
	return YES;
}

@end
