#
# Copyright (c) 2021, 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#

module Luce
  # Maintains the context needed to parse a Markdown document
  class Document
    getter link_references = Hash(String, LinkReference).new
    getter link_resolver : Resolver?
    getter image_link_resolver : Resolver?

    @[Deprecated("Use `#encode_html?` instead. Will be removed in version 1.0.")]
    def encode_html : Bool
      encode_html?
    end

    getter? encode_html : Bool

    # Whether to use default block syntaxes
    @[Deprecated("Use `#with_default_block_syntaxes?` instead. Will be removed in version 1.0.")]
    def with_default_block_syntaxes : Bool
      with_default_block_syntaxes?
    end

    # Whether to use default block syntaxes
    getter? with_default_block_syntaxes : Bool

    # Whether to use default inline syntaxes.
    #
    # Need to set both `with_default_block_syntaxes?` and `encode_html?` to
    # `false` to disable all inline syntaxes including HTML encoding syntaxes.
    @[Deprecated("Use `#with_default_inline_syntaxes?` instead. Will be removed in version 1.0")]
    def with_default_inline_syntaxes : Bool
      with_default_inline_syntaxes?
    end

    # Whether to use default inline syntaxes.
    #
    # Need to set both `#with_default_block_syntaxes?` and `#encode_html?` to
    # `false` to disable all inline syntaxes including HTML encoding syntaxes.
    getter? with_default_inline_syntaxes : Bool

    getter block_syntaxes : Array(BlockSyntax) = Array(BlockSyntax).new
    getter inline_syntaxes : Array(InlineSyntax) = Array(InlineSyntax).new

    @[Deprecated("Use `#has_custom_inline_syntaxes?` instead. Will be removed in version 1.0")]
    def has_custom_inline_syntaxes : Bool
      has_custom_inline_syntaxes?
    end

    getter? has_custom_inline_syntaxes : Bool

    def initialize(
      block_syntaxes : Array(BlockSyntax)?,
      inline_syntaxes : Array(InlineSyntax)?,
      extension_set : ExtensionSet?,
      @link_resolver : Resolver? = nil,
      @image_link_resolver : Resolver? = nil,
      @encode_html : Bool = true,
      @with_default_block_syntaxes = true,
      @with_default_inline_syntaxes = true
    )
      @has_custom_inline_syntaxes = ((!inline_syntaxes.nil? && !inline_syntaxes.empty?) || false) ||
                                    ((!extension_set.nil? && !extension_set.inline_syntaxes.empty?) || false)

      @block_syntaxes.concat(block_syntaxes) unless block_syntaxes.nil?
      @inline_syntaxes.concat(inline_syntaxes) unless inline_syntaxes.nil?

      if extension_set.nil?
        if with_default_block_syntaxes
          @block_syntaxes.concat(ExtensionSet::COMMON_MARK.block_syntaxes)
        end

        if with_default_inline_syntaxes
          @inline_syntaxes.concat(ExtensionSet::COMMON_MARK.inline_syntaxes)
        end
      else
        @block_syntaxes.concat(extension_set.block_syntaxes)
        @inline_syntaxes.concat(extension_set.inline_syntaxes)
      end
    end

    # Parses the given *lines* of Markdown to a series of AST nodes.
    def parse_lines(lines : Array(String)) : Array(Node)
      nodes = BlockParser.new(lines, self).parse_lines
      parse_inline_content nodes
      nodes
    end

    # Parses the given inline Markdown *text* to a series of AST nodes.
    def parse_inline(text : String) : Array(Node)
      InlineParser.new(text, self).parse
    end

    private def parse_inline_content(nodes : Array(Node)) : Nil
      i = 0
      while i < nodes.size
        node = nodes[i]
        if node.is_a? UnparsedContent
          inline_nodes = parse_inline(node.text_content)
          nodes.delete_at(i)
          nodes.insert_all(i, inline_nodes)
          i += inline_nodes.size - 1
        elsif node.is_a?(Element) && node.children.nil? == false
          parse_inline_content(node.children.not_nil!)
        end
        i += 1
      end
    end
  end

  # A [link reference
  # definition](https://spec.commonmark.org/0.28/#link-reference-definitions).
  class LinkReference
    # The [link label](https://spec.commonmark.org/0.28/#link-label).
    #
    # Temporarily, this class is also being used to represent the link
    # data for an inline link (the destination and title), but this
    # should change before the shard is released.
    getter label : String

    # The [link destination](https://spec.commonmark.org/0.28/#link-destination).
    getter destination : String

    # The [link title](https://spec.commonmark.org/0.28/#link-title).
    getter title : String?

    # Construct a new `LinkReference`, with all necessary fields.
    #
    # If the parsed link reference definition does not include a title,
    # use `nil` for the *title* parameter.
    def initialize(@label : String, @destination : String, @title : String?)
    end
  end
end
