//
//  MSPager.m
//  Manuscript
//
//  Created by 二鏡 on 12/02/11.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSPager.h"
#import "MSPageFormat.h"
#import "MSTextView.h"
#import "MSLayoutBlock.h"
#import "MSLayoutBlockManager.h"
#import "MSTypesetter.h"

static NSString *sFontChangeContext = @"fontChange";
static NSString *sBoxChangeContext = @"boxChange";
static NSString *sGridChangeContext = @"gridChange";

inline static NSRange
lineRangeForRect(const NSRect &bounds, const NSRect &visibleRect, const NSSize &padding, const CGFloat &lineWidth)
{
    CGFloat rightHidden = NSMaxX(bounds)-NSMaxX(visibleRect)-padding.width;
    NSUInteger start = (NSUInteger)(rightHidden/lineWidth); // 右端のインデックス
    NSUInteger count = NSWidth(visibleRect)/lineWidth +2;
    return NSMakeRange(start,count);
}

inline static NSRect
boundingRectForLineRange(const NSRect &bounds,
                         const NSSize &padding, 
                         const CGFloat &lineWidth,
                         const CGFloat &lineHeight,
                         NSRange lineRange)
{
    CGFloat x = NSMaxX(bounds)-padding.width-lineWidth*NSMaxRange(lineRange);
    CGFloat y = (NSHeight(bounds)-lineHeight)/2.0;
    CGFloat w = lineWidth*lineRange.length;
    CGFloat h = lineHeight;
    return NSMakeRect(x,y,w,h);
}

@interface MSPager ()
- (NSPoint)convertPoint:(NSPoint)viewPoint
            toLineIndex:(NSUInteger)index;
@end

@implementation MSPager

@synthesize textView = textView_, format = format_, content = content_;
- (id)init
{
    lines_ = [NSMutableArray array];
    typesetter_ = [[MSTypesetter alloc] init];
    return self;
}

- (void)updateViewSize
{
    // ここでboxCount計算
    NSSize padding = format_.padding;
    
    // 可能行高
    CGFloat boxSize = format_.boxSize;
    // ぶら下げが入るので、配列可能数は一つ下げる
    
    // 行数を数えないとだめなのだ。でもそれは不可能でしょ？
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat minHeight = format_.boxCount*boxSize+padding.height*2;
    CGFloat minWidth = ([lines_ count]+1)*boundingWidth+padding.width;
    textView_.minSize = NSMakeSize(minWidth, minHeight); // 適当なサイズ
    [textView_ sizeToFit];
    [textView_ setNeedsDisplay: YES];
}

- (void)observeValueForKeyPath:(NSString *)keyPath
                      ofObject:(id)object
                        change:(NSDictionary *)change 
                       context:(void *)context
{
    if(context == sFontChangeContext)
    {
        id font = [format_ font];
        [content_ changeFont: font];
        [self updateViewSize];
        return;
    }
    
    if(context == sBoxChangeContext)
    {
        [content_ invalidateLineMapping];
        [self _replaceLinesInRange: NSMakeRange(0,[lines_ count])
                        withBlocks: NSMakeRange(0,[content_ blockCount])];
        [self updateViewSize];
        return;
    }
    
    if(context == sGridChangeContext)
    {
        [textView_ setNeedsDisplay: YES];
        return;
    }
    
    [super observeValueForKeyPath: keyPath ofObject:object change:change context:context];
}

- (void)setFormat:(MSPageFormat*)aFormat
{
    format_ = aFormat;
    [format_ addObserver: self
              forKeyPath: @"fontSize"
                 options: 0
                 context: sFontChangeContext];
    [format_ addObserver: self
              forKeyPath: @"fontName"
                 options: 0
                 context: sFontChangeContext];
    [format_ addObserver: self
              forKeyPath: @"boxCount"
                 options: 0
                 context: sBoxChangeContext];
    [format_ addObserver: self
              forKeyPath: @"showsGrid"
                 options: 0
                 context: sGridChangeContext];
    [self updateViewSize];
}

- (void)setTextView:(MSTextView*)aView
{
    if(textView_ == aView)
        return;
    
    textView_ = aView;
    textView_.pager = self;
    
    [self updateViewSize];
}

- (void)setContent:(MSLayoutBlockManager*)aContent
{
    id center = [NSNotificationCenter defaultCenter];
    [center removeObserver: self
                      name: nil
                    object: content_];
    
    content_ = aContent;
    [center addObserver: self
               selector: @selector(layoutBlockWillChange:)
                   name: MSLayoutBlockManagerWillChangeBlocksNotification
                 object: content_];
    [center addObserver: self
               selector: @selector(layoutBlockDidChange:)
                   name: MSLayoutBlockManagerDidChangeBlocksNotification
                 object: content_];
}

- (NSUInteger)lineIndexAtPoint:(NSPoint)aPoint
{
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSSize padding = format_.padding;
    
    // 右ベタの距離
    CGFloat rightOver = NSMaxX(bounds) - padding.width - aPoint.x;
    
    // 基準index
    NSUInteger index = (NSUInteger)(rightOver/boundingWidth);
    if(index > [lines_ count])
        return NSNotFound;
    
    // box右端
    CGFloat boxRight = NSMaxX(bounds)-padding.width-(index+1)*boundingWidth+boxSize;
    if(aPoint.x > boxRight)
        return NSNotFound;
    else
        return index;
}

- (NSUInteger)indexOfCharacterOfNearestBorderAtPoint:(NSPoint)aPoint
                                              isTail:(BOOL*)isTail
{
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSSize padding = format_.padding;
    CGFloat interline = format_.interline;
    
    // 右ベタの距離
    CGFloat rightOver = NSMaxX(bounds) - padding.width - aPoint.x;
    
    // 基準index
    NSUInteger lIndex = (NSUInteger)(rightOver/boundingWidth);

    aPoint = [self convertPoint: aPoint toLineIndex: lIndex];

    // 行間を考慮して前後に補正
    if(aPoint.x > boxSize+interline/2.0 && lIndex != 0)
        lIndex--;

    if(lIndex >= [lines_ count]) // 左の限界を超えたものは全て末尾と見なされる
        return content_.textStorage.length;
    
    // 行内の計算
    
    VTLogicalLineContainer *container = [lines_ objectAtIndex: lIndex];
    const LogicalLine *line = [container line];
    bool tail;
    CFIndex location = line->query_nearest_border_index(aPoint.y, tail);
    *isTail = tail;
    NSRange blockRange = [content_ characterRangeOfBlock: container.container];
    return blockRange.location + line->get_character_range().location + location;
}

- (NSUInteger)indexOfCharacterOfPreviousLineFromPoint:(NSPoint)aPoint
{
    NSUInteger lineIndex = [self lineIndexAtPoint: aPoint];
    if( lineIndex == 0 )
        return NSNotFound;
    VTLogicalLineContainer *container = [lines_ objectAtIndex: lineIndex-1];
    const LogicalLine *line = container.line;
    
    aPoint = [self convertPoint: aPoint toLineIndex: lineIndex];
    bool isTail;
    CFIndex index = line->query_nearest_border_index(aPoint.y, isTail);
    if(isTail)
        index--;
    
    // グローバル変換
    NSRange blockRange = [content_ characterRangeOfBlock: container.container];
    CFRange lineRange = line->get_character_range();
    return blockRange.location + lineRange.location + index;
}

- (NSUInteger)indexOfCharacterOfNextLineFromPoint:(NSPoint)aPoint
{
    NSUInteger lineIndex = [self lineIndexAtPoint: aPoint];
    if( lineIndex +1 >= [lines_ count] ) // 0行判定があるのでこんな感じ
        return NSNotFound;
    // あとはここをなんとかすればいい
    VTLogicalLineContainer *container = [lines_ objectAtIndex: lineIndex+1];
    const LogicalLine *line = container.line;
    aPoint = [self convertPoint: aPoint toLineIndex: lineIndex];
    bool isTail;
    CFIndex index = line->query_nearest_border_index(aPoint.y, isTail);
    if(isTail)
        index--;
    
    // グローバル変換
    NSRange blockRange = [content_ characterRangeOfBlock: container.container];
    CFRange lineRange = line->get_character_range();
    return blockRange.location + lineRange.location + index;
}

- (NSPoint)convertPoint:(NSPoint)viewPoint
            toLineIndex:(NSUInteger)index
{
    // index行目の基準原点相対に変換する
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSUInteger boxCount = format_.boxCount;
    NSSize padding = format_.padding;

    CGFloat left = NSMaxX(bounds)-padding.width-boundingWidth*(index+1);
    CGFloat top = (NSHeight(bounds)-(boxSize*boxCount))/2.0;
    
    return NSMakePoint(viewPoint.x-left, viewPoint.y-top);
}

- (NSUInteger)indexOfCharacterAtPoint:(NSPoint)aPoint
{
    // 行があわない
    NSUInteger lineIndex = [self lineIndexAtPoint: aPoint];
    if(lineIndex == NSNotFound || lineIndex == [lines_ count])
        return NSNotFound;
    
    // ローカルに変換してチェック
    VTLogicalLineContainer *container = [lines_ objectAtIndex: lineIndex];
    LogicalLine *line = [container line];
    NSPoint localPoint = [self convertPoint: aPoint 
                                toLineIndex: lineIndex];
    CFIndex localIndex = line->query_box_index(localPoint.y);
    if(localIndex == kCFNotFound)
        return NSNotFound;
    
    // 大域に変換するためにブロックを参照
    MSLayoutBlock *block = container.container;
    NSRange range = [content_ characterRangeOfBlock: block];
    if(range.location == NSNotFound)
        return NSNotFound;
    
    CFRange lineRange = line->get_character_range();
    NSUInteger ret = range.location + lineRange.location + localIndex;
    NSAssert( ret < content_.textStorage.length, @"invalid index");
    return ret;
}

- (NSPoint)lineOriginAtIndex:(NSUInteger)lineIndex
{   
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSSize padding = format_.padding;
    NSUInteger boxCount = format_.boxCount;

    CGFloat dy = floor((NSHeight(bounds)-boxSize*boxCount)/2);
    CGFloat dx = NSMaxX(bounds)-(lineIndex+1)*boundingWidth-padding.width+boxSize/2.0;
    return NSMakePoint(dx,dy);
}

- (NSUInteger)lineIndexOfCharacterIndex:(NSUInteger)charIndex
{
    MSLayoutBlock *block = [content_ blockAtCharacterIndex: charIndex];
    if(block == nil)
        return NSNotFound; // 末尾にも対応
    
    NSUInteger offset = block.characterLocation; // blockのゲタ
    NSUInteger bLineIndex = [block lineIndexAtCharacterIndex: charIndex-offset];
    NSAssert( bLineIndex != NSNotFound, @"invalid character range in block");
    
    NSRange lineRange = [content_ lineRangeOfBlock: block];
    return bLineIndex + lineRange.location; // 基準とする行番号を得た
}

- (NSRect)firstRectForCharacterRange:(NSRange)aRange
                         actualRange:(NSRangePointer)actualRange
{
    MSLayoutBlock *block = [content_ blockAtCharacterIndex: aRange.location];
    if(block == nil)
        return NSZeroRect;
    
    NSUInteger offset = block.characterLocation; // blockのゲタ
    // blockローカルに変換
    NSRange localRange = aRange; localRange.location -= offset;
    VTLogicalLineContainer *container = [block lineAtCharacterIndex: localRange.location];
    if(container == nil)
        return NSZeroRect;
    
    const LogicalLine *line = container.line;
    CFRange lineRange = line->get_character_range();
    
    NSUInteger lineIndex = [lines_ indexOfObject: container];
    NSPoint origin = [self lineOriginAtIndex: lineIndex];

    // lineローカルに変換
    localRange.location -= lineRange.location;    
    NSRect ret = line->make_bounding_rect(localRange, actualRange, origin);

    // actualRangeをグローバルに逆変換
    if(actualRange != NULL)
        (*actualRange).location += lineRange.location + offset;
    return ret;
}

- (NSUInteger)beginningOfLineAtCharacterIndex:(NSUInteger)charIndex
{
    MSLayoutBlock *block = [content_ blockAtCharacterIndex: charIndex];
    if(block == nil)
    {
        block = [content_ lastBlock];
        if(block == nil)
            return 0; // ゼロの時
        
        // 末尾のみ特殊処理
        const LogicalLine *line = [[lines_ lastObject] line];
        if(line->get_is_closed())
            return charIndex; // extra fragmentの扱いは不動で
        
        NSRange range = [content_ characterRangeOfBlock: block];
        CFRange localRange = line->get_character_range();
        NSUInteger ret = localRange.location;
        return ret + range.location;
    }
    
    NSUInteger lIndex = ({ // 行数の決定
        NSUInteger bLineIndex = [block lineIndexAtCharacterIndex: [block convertCharacterIndexToLocal:charIndex]];
        NSAssert( bLineIndex != NSNotFound, @"invalid character range in block");
        
        NSRange lineRange = [content_ lineRangeOfBlock: block];
        bLineIndex + lineRange.location; // ここまで行番号
    });
    
    const LogicalLine *line = [[lines_ objectAtIndex: lIndex] line];
    CFRange range = line->get_character_range();
    NSUInteger ret = range.location; // 行頭
    
    return [block convertCharacterIndexToGlobal: ret];
}

- (NSUInteger)endOfLineAtCharacterIndex:(NSUInteger)charIndex
{
    MSLayoutBlock *block = [content_ blockAtCharacterIndex: charIndex];
    if(block == nil)
        return charIndex; // 末尾にも対応
    
    NSUInteger lIndex = ({ // 行数の決定
        NSUInteger bLineIndex = [block lineIndexAtCharacterIndex: [block convertCharacterIndexToLocal:charIndex]];
        NSAssert( bLineIndex != NSNotFound, @"invalid character range in block");
        
        NSRange lineRange = [content_ lineRangeOfBlock: block];
        bLineIndex + lineRange.location; // ここまで行番号
    });
    
    const LogicalLine *line = [[lines_ objectAtIndex: lIndex] line];
    CFRange range = line->get_character_range();
    NSUInteger ret = range.location+range.length; // 末端＝次の行の先頭
    
    // 改行文字は戻す
    if(line->get_is_closed())
        ret--; 
    
    // グローバル変換
    return [block convertCharacterIndexToGlobal: ret];
}

- (NSRect)topCaretRectForLineIndex:(NSUInteger)index
{
    CGFloat boxSize = format_.boxSize;    
    NSPoint origin = [self lineOriginAtIndex: index];
    CGFloat x = origin.x - boxSize/2;
    CGFloat y = origin.y;
    CGFloat w = boxSize;
    CGFloat h = 1.0;
    return NSMakeRect(x,y,w,h);
}

- (NSRect)caretRectForCharacterIndex:(NSUInteger)index
{
    MSLayoutBlock *block = [content_ blockAtCharacterIndex: index];

    if(block == nil)
    {
        // 特殊ケース
        // blockなしで行頭のというのはemptyしかありえない
        if(index == 0)
            return [self topCaretRectForLineIndex: 0];
        
        // 最後までindexが来ない時は末端と見なすべき
        VTLogicalLineContainer *container = [lines_ lastObject];
        const LogicalLine *line = [container line];
        if(line->get_is_closed())
        {
            return [self topCaretRectForLineIndex: [lines_ count]];
        }
        CGFloat border = line->get_border_tail();
        NSPoint origin = [self lineOriginAtIndex: [lines_ count]-1];
        return line->make_caret_rect(border, origin);
    }
    
    NSRange blockRange = [content_ lineRangeOfBlock: block]; // 原点計算に行が必要
    NSUInteger localIndex = [block convertCharacterIndexToLocal: index];
    
    id lines = [block lines];
    for(NSUInteger i=0;i<blockRange.length;i++)
    {
        VTLogicalLineContainer *container = [lines objectAtIndex: i];
        const LogicalLine *line = [container line];
        CFRange lineRange = line->get_character_range(); // これは我慢するしかない？
        if(_CFLocationInRange(localIndex, lineRange))
        {
            CFIndex lineLocalIndex = line->convert_index_to_local(localIndex);
            CGFloat border = line->get_border(lineLocalIndex);
            NSPoint origin = [self lineOriginAtIndex: blockRange.location+i];
            return line->make_caret_rect(border, origin);
        }
    }
    
    return NSZeroRect; // きたらバグ
}

- (NSRect)caretRectForCharacterIndexWithExtend:(NSUInteger)index
{
    // 検索方式が違うぜぇ？
    MSLayoutBlock *block = [content_ blockAtCharacterIndex: index];
    
    if(block == nil)
    {
        // 特殊ケース
        // blockなしで行頭のというのはemptyしかありえない
        if(index == 0)
            return [self topCaretRectForLineIndex: 0];
        
        // 最後までindexが来ない時は末端と見なすべき
        VTLogicalLineContainer *container = [lines_ lastObject];
        const LogicalLine *line = [container line];
        if(line->get_is_closed())
        {
            return [self topCaretRectForLineIndex: [lines_ count]];
        }
        CGFloat border = line->get_border_tail();
        NSPoint origin = [self lineOriginAtIndex: [lines_ count]-1];
        return line->make_caret_rect(border, origin);
    }
    
    NSRange blockRange = [content_ lineRangeOfBlock: block]; // 原点計算に行が必要
    NSUInteger localIndex = [block convertCharacterIndexToLocal: index];
    
    id lines = [block lines];
    for(NSUInteger i=0;i<blockRange.length;i++)
    {
        VTLogicalLineContainer *container = [lines objectAtIndex: i];
        const LogicalLine *line = [container line];
        CFRange lineRange = line->get_character_range(); // これは我慢するしかない？
        if(_CFLocationInRangeEQ(localIndex, lineRange))
        {
            CFIndex lineLocalIndex = line->convert_index_to_local(localIndex);
            CGFloat border = line->get_border(lineLocalIndex);
            NSPoint origin = [self lineOriginAtIndex: blockRange.location+i];
            return line->make_caret_rect(border, origin);
        }
    }
    
    return NSZeroRect; // きたらバグ
}

- (NSRange)lineRangeForCharacterRange:(NSRange)charRange
                              inRange:(NSRange)lineRangeLimit
{
    VTLogicalLineContainer *topContainer = [lines_ objectAtIndex: lineRangeLimit.location];
    const LogicalLine *line = topContainer.line;
    MSLayoutBlock *block = topContainer.container;
    NSRange blockRange = [content_ characterRangeOfBlock: block];
    NSUInteger rangeLimit = NSMaxRange(lineRangeLimit);
    
    // 直前の末尾=対象行の先頭インデックス
    NSUInteger lineLimit = blockRange.location + line->get_character_range().location;
    
    // lineLimitから始めて行のcountを増分としてレンジを回し、
    // その内部でcharRangeの先頭と行末の範囲を決定する
    
    NSUInteger i, start = NSNotFound;
    NSUInteger headCharacter = charRange.location;
    NSUInteger tailCharacter = NSMaxRange(charRange);

    // 1. 先頭文字が出現するまで行を読み飛ばす
    for(i=lineRangeLimit.location; i<rangeLimit; i++)
    {
        line = [[lines_ objectAtIndex: i] line];
        CFIndex length = line->get_character_count();
        lineLimit += length;
        if(lineLimit > headCharacter)
        {
            start = i;
            // 一行にcharRangeが含まれる
            if(lineLimit >= tailCharacter)
                return NSMakeRange(start,1);
            break;
        }
    }

    // 先頭文字が含まれなかった
    if(start == NSNotFound)
    {
        return NSMakeRange(NSNotFound,0);
    }
    
    // 2. 末尾文字が出現するまで進める
    i++;
    for(;i<rangeLimit;i++)
    {
        line = [[lines_ objectAtIndex: i] line];
        CFIndex length = line->get_character_count();
        lineLimit += length;
        if(lineLimit >= tailCharacter)
        {
            return NSMakeRange(start,i-start+1);
        }
    }
    
    // 末尾文字が含まれない->後ろは全ての行が対象
    return NSMakeRange(start, rangeLimit-start);
}

- (NSUInteger)localCharacterIndexFromGlobalIndex:(NSUInteger)charIndex
                                     inLineIndex:(NSUInteger)lineIndex
{
    VTLogicalLineContainer *container = [lines_ objectAtIndex: lineIndex];
    MSLayoutBlock *block = container.container;
    const LogicalLine *line = container.line;
    NSRange blockRange = [content_ characterRangeOfBlock: block];
    CFRange lineRange = line->get_character_range();
    NSUInteger lineHead = blockRange.location + lineRange.location;
    if( charIndex < lineHead ||
        charIndex >= lineHead + lineRange.length )
        return NSNotFound;
    
    return charIndex - lineHead;
}

- (CGPathRef)_createClippingForImageOfRange:(NSRange)characterRange
                                       line:(NSUInteger)index
{
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSUInteger boxCount = format_.boxCount;
    NSSize padding = format_.padding;
    CGFloat lineHeight = boxSize*boxCount;
    CGFloat interline = format_.interline;
    
    // 一行モード
    VTLogicalLineContainer *container = [lines_ objectAtIndex: index];
    MSLayoutBlock *block = container.container;
    const LogicalLine *line = container.line;
    NSRange blockRange = [content_ characterRangeOfBlock: block];
    CFRange lineRange = line->get_character_range();
    NSUInteger lineHead = blockRange.location + lineRange.location;
    
    // 先頭文字は行内に含まれること
    NSAssert(characterRange.location < lineHead+lineRange.length, @"invalid range");
    
    CGFloat top,bottom;
    if(characterRange.location < lineHead)
        top = line->get_border(0);
    else
        top = line->get_border(characterRange.location-lineHead);
    
    if(NSMaxRange(characterRange) < lineHead+lineRange.length)
        bottom = line->get_border(NSMaxRange(characterRange)-lineHead);
    else
        bottom = line->get_border_tail();
    
    CGFloat x = NSMaxX(bounds)-padding.width-boundingWidth*(index+1)-interline;
    CGFloat y = (NSHeight(bounds)-lineHeight)/2.0 + top;
    CGFloat w = boxSize+interline*2;
    CGFloat h = bottom-top;
    
    // clipを構成
    CGRect clipRect = CGRectMake(x,y,w,h);
    CGMutablePathRef ret = CGPathCreateMutable();
    CGPathAddRect(ret, NULL, clipRect);
    return ret;
}

- (CGPathRef)_createClippingPathForImageOfRange:(NSRange)characterRange
                                      lineRange:(NSRange)lineRange
{
    // 一行モード
    if(lineRange.length == 1)
        return [self _createClippingForImageOfRange: characterRange line: lineRange.location];
    
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSUInteger boxCount = format_.boxCount;
    NSSize padding = format_.padding;
    CGFloat lineHeight = boxSize*boxCount;
    CGFloat interline = format_.interline;

    // 一般
    // 矩形から右上と左上を欠いた形状
    CGPoint points[8];
    
    // 1. 右端のtopと左端のbottomが必要
    CGFloat rightTop;
    {
        NSUInteger index = lineRange.location;
        NSUInteger first = [self localCharacterIndexFromGlobalIndex: characterRange.location
                                                        inLineIndex: index];
        if(first == NSNotFound)
        {
            // 見えていない領域に開始文字があるのでクリップしない
            rightTop = 0.0;
        }
        else
        {
            const LogicalLine *line = [[lines_ objectAtIndex: index] line];
            rightTop = line->get_border(first);
        }
    }
    
    CGFloat leftBottom;
    {
        NSUInteger index = NSMaxRange(lineRange)-1;
        NSUInteger last  = [self localCharacterIndexFromGlobalIndex: NSMaxRange(characterRange)
                                                        inLineIndex: index];
        
        const LogicalLine *line = [[lines_ objectAtIndex: index] line];
        if( last == NSNotFound)
            leftBottom = line->get_border_tail();
        else
            leftBottom = line->get_border(last);
    }
    
    // 点構成
    CGFloat leftX = NSMaxX(bounds)-padding.width-boundingWidth*NSMaxRange(lineRange)-interline;
    CGFloat rightX = NSMaxX(bounds)-padding.width-boundingWidth*lineRange.location;

    CGFloat top = (NSHeight(bounds)-lineHeight)/2.0;
    CGFloat bottom = top+lineHeight; 
    
    // 左上から反時計回りで構成
    // 0+------+7
    //  |      |
    // 1+-+2   +-+5
    //    |    6  |
    //   3+------+4
    points[0] = CGPointMake(leftX, top);
    points[1] = CGPointMake(leftX, top+leftBottom);
    points[2] = CGPointMake(leftX+boundingWidth, top+leftBottom);
    points[3] = CGPointMake(leftX+boundingWidth, bottom);
    points[4] = CGPointMake(rightX, bottom);
    points[5] = CGPointMake(rightX, top+rightTop);
    points[6] = CGPointMake(rightX-boundingWidth-interline, top+rightTop);
    points[7] = CGPointMake(rightX-boundingWidth-interline, top);
    
    CGMutablePathRef ret = CGPathCreateMutable();
    CGPathAddLines(ret, NULL, points, 8);
    CGPathCloseSubpath(ret);
    
    return ret;
}

- (NSImage*)generateTextImageForDraggingOfRange:(NSRange)characterRange
                                         inRect:(NSRect*)rect
{
    NSRect bounds = [textView_ bounds];
    NSRect visibleRect = [textView_ visibleRect];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSUInteger boxCount = format_.boxCount;
    NSSize padding = format_.padding;
    CGFloat interline = format_.interline;
    
    NSRange lineRange = lineRangeForRect(bounds, visibleRect, padding, boundingWidth);
    NSUInteger limit = [lines_ count];
    if( lineRange.location >= limit )
        return nil;
    
    // 可視で有意なレンジ
    lineRange.length = MIN(lineRange.length, limit-lineRange.location);
    
    // 描画に用いるべきレンジ
    NSRange targetRange = [self lineRangeForCharacterRange: characterRange
                                                   inRange: lineRange];
    if(targetRange.location == NSNotFound)
        return nil;

    // 1. lineのboudingの生成
    CGFloat lineHeight = boxSize*boxCount;
    NSRect imageBounds = boundingRectForLineRange(bounds, padding, boundingWidth, lineHeight, targetRange);
    imageBounds.origin.x -= interline;
    imageBounds.size.width += interline*2; // 前後の行間分までは描画範囲とする
    *rect = NSIntersectionRect(imageBounds, visibleRect);
    
    // 仮想原点
    NSPoint origin = rect->origin;
    origin.x *= -1.0;
    origin.y *= -1.0;

    // 2. 描画対象を生成
    NSImage *ret = [[NSImage alloc] initWithSize: rect->size];
    [ret setFlipped: YES];
    [ret lockFocus];
    
    CGContextRef context = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
    CGAffineTransform trans = CGAffineTransformMakeTranslation(origin.x, origin.y); // 仮想原点に補正
    CGContextConcatCTM(context, trans);
    
    // 3. clippingの設定
    CGPathRef clip = [self _createClippingPathForImageOfRange: characterRange
                                                    lineRange: targetRange];
    CGContextAddPath(context, clip);
    CGContextClip(context);
    CFRelease(clip);
    
    // 4. 行を描画
    CGFloat h = boxSize*format_.boxCount; // 行高
    CGFloat dy = floor((NSHeight(bounds)-h)/2)+1; // 枠線の分だけシフト

    CGAffineTransform invert = CGAffineTransformMakeScale(1,-1);
    CGContextSetTextMatrix(context, invert);
        
    // (i+1)*bounding-boxSize/2がセンター
    for(NSUInteger i=targetRange.location;i<NSMaxRange(targetRange);i++)
    {
        VTLogicalLineContainer *line = [lines_ objectAtIndex: i];
        CGFloat x = NSMaxX(bounds)-(i+1)*boundingWidth-padding.width+boxSize/2.0;
        CGPoint origin = CGPointMake(x ,dy);
        [line line]->draw(context, origin);        
    }

    [ret unlockFocus];
    
    return ret;
}

- (void)drawLinesInRect:(NSRect)dirtyRect
{
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSSize padding = format_.padding;
    CGFloat border = format_.lineMarkBorder;
  
    NSRange lineRange = lineRangeForRect(bounds, dirtyRect, padding, boundingWidth);
    NSUInteger limit = [lines_ count];
    if( lineRange.location >= limit ) // 描画すべき行はない
        return;
    
    // setup context
    CGContextRef context = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
    CGAffineTransform invert = CGAffineTransformMakeScale(1,-1);
    CGContextSetTextMatrix(context, invert);

    // 基本行を構成
    CGFloat h = boxSize*format_.boxCount; // 行高
    CGFloat dy = floor((NSHeight(bounds)-h)/2)+1; // 枠線の分だけシフト
    
    // この範囲で、意味のあるindexについててきめん描画
    lineRange.length = MIN(lineRange.length, limit-lineRange.location);
    
    // (i+1)*bounding-boxSize/2がセンター
    for(NSUInteger i=lineRange.location;i<NSMaxRange(lineRange);i++)
    {
        VTLogicalLineContainer *container = [lines_ objectAtIndex: i];
        CGFloat x = NSMaxX(bounds)-(i+1)*boundingWidth-padding.width+boxSize/2.0;
        CGPoint origin = CGPointMake(x ,dy);
        const LogicalLine *line = [container line];
        line->draw(context, origin);
        line->draw_return_mark_if_need(context, origin, border);
    }
}

- (void)drawSelectionInRect:(NSRect)dirtyRect
             characterRange:(NSRange)characterRange
{
    NSRect bounds = [textView_ bounds];
    CGFloat boundingWidth = format_.boundingWidth;
    CGFloat boxSize = format_.boxSize;
    NSSize padding = format_.padding;
    
    NSRange lineRange = lineRangeForRect(bounds, dirtyRect, padding, boundingWidth);
    NSUInteger limit = [lines_ count];
    if( lineRange.location >= limit )
        return;
    
    // 有意な行の範囲
    lineRange.length = MIN(lineRange.length, limit-lineRange.location);
    
    // 座標準備1
    CGFloat h = boxSize*format_.boxCount; // 行高
    CGFloat y = floor((NSHeight(bounds)-h)/2)+1; // 枠線の分だけシフト

    CGContextRef context = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];

    // 色設定
    if([[textView_ window] isMainWindow])
        [[NSColor selectedTextBackgroundColor] set];
    else
        [[NSColor secondarySelectedControlColor] set];

    MSLayoutBlock *block = nil;
    NSRange overrap; // block localのオーバーラップ範囲
    for(NSUInteger i=lineRange.location;i<NSMaxRange(lineRange);i++)
    {
        VTLogicalLineContainer *container = [lines_ objectAtIndex: i];
        id aBlock = [container container];
        if( aBlock != block )
        {
            // ブロックの乗り換え
            block = aBlock;
            NSRange blockRange = [content_ characterRangeOfBlock: block];
            overrap = NSIntersectionRange(blockRange, characterRange);
            if(overrap.length == 0)
            {
                // このブロックは無関係
                // 行数分だけカウンターを進めるべき(ループの増分を引いておく)
                NSRange blockLineRange = [content_ lineRangeOfBlock: block];
                i = NSMaxRange(blockLineRange)-1;
                continue;
            }
            overrap.location -= blockRange.location; // ブロック内のローカルに変換
        }
        const LogicalLine *line = container.line;
        CFRange localCharacterRange = line->get_character_range();
        NSRange localOverrap = ({
            NSRange nLineRange = NSMakeRange(localCharacterRange.location, localCharacterRange.length); // ちょっとズルをするCFRange->NSRange
            NSIntersectionRange(nLineRange, overrap);
        });
        if(localOverrap.length == 0)
            continue;
        
        CFRange localRange = CFRangeMake(localOverrap.location-localCharacterRange.location, localOverrap.length);
        
        CGFloat x = NSMaxX(bounds)-(i+1)*boundingWidth-padding.width+boxSize/2.0;
        line->fill_background(context, CGPointMake(x,y), localRange);
    }
}

- (void)drawGrid:(NSRect)dirtyRect
{
    NSRect bounds = [textView_ bounds];
    NSUInteger aBoxCount  = format_.boxCount;
    CGFloat boxSize = format_.boxSize;
    CGFloat boundingWidth = format_.boundingWidth;
    NSSize padding = format_.padding;
    
    // 枠オブジェクトを原点に構築
    NSRect lineRect = NSMakeRect(0, 0, boxSize, boxSize*aBoxCount);
    lineRect = NSInsetRect(lineRect, 0.5,0.5);
    id bp = [NSBezierPath bezierPathWithRect: lineRect];
    CGFloat y=0.5;
    for(NSUInteger i=0;i<aBoxCount-1;i++)
    {
        y += boxSize;
        [bp moveToPoint: NSMakePoint(0.5, y)];
        [bp lineToPoint: NSMakePoint(boxSize-1.0, y)];
    }
    
    CGFloat dashs[2] = {2.0, 2.0};
    [bp setLineDash: dashs count: 2 phase: 0.0];
    
    // 描画領域の計算
    NSRange lineRange = lineRangeForRect(bounds, dirtyRect, padding, boundingWidth);
    [NSGraphicsContext saveGraphicsState];
    NSColor *color = [NSColor colorWithCalibratedRed:79.0/255
                                         green:80.0/255
                                          blue:109.0/255 
                                         alpha:1.0];
    [color set];

    // 基準原点へ移動
    CGFloat dx = floor(NSMaxX(bounds)-padding.width-boundingWidth*(MAX(0,lineRange.location)));
    CGFloat dy = floor((NSHeight(bounds)-NSHeight(lineRect))/2);
    id trans = [NSAffineTransform transform];
    [trans translateXBy: dx
                    yBy: dy];
    [trans concat];
    
    // X-move
    trans = [NSAffineTransform transform];
    [trans translateXBy: -boundingWidth yBy: 0.0];    
    for(NSUInteger i=lineRange.location;i<NSMaxRange(lineRange);i++)
    {
        [trans concat];
        [bp stroke];
    }
    
    [NSGraphicsContext restoreGraphicsState];
}

- (void)_replaceLinesInRange:(NSRange)oldLineRange
                  withBlocks:(NSRange)blockRange
{
    id blocks = [content_ subblocksWithRange: blockRange];
    
    typesetter_.boxSize = format_.boxSize;
    typesetter_.boxCount = format_.boxCount;
    
    id replaced = [NSMutableArray array];
    
    // この範囲をレイアウト
    for(MSLayoutBlock *block in blocks)
    {
        CompiledAttributedString *str = block.string;
        id lines = [typesetter_ linesFromString: str];
        block.lines = lines;
        [replaced addObjectsFromArray: lines];
    }
    
    // 行の入れ替え
    if(oldLineRange.location == NSNotFound)
    {
        [lines_ addObjectsFromArray: replaced];
    }
    else
    {
        [lines_ replaceObjectsInRange: oldLineRange
                 withObjectsFromArray: replaced];
    }
    
    // viewの大きさを再調整
    [self updateViewSize];

}

- (void)layoutBlockWillChange:(id)notif
{
    NSRange change = [content_ editedRange];
    touchedRange_ = [content_ lineRangeForBlockRange: change];
    // 消滅する行を記憶
}

- (void)layoutBlockDidChange:(id)notif
{
    NSRange change = [content_ editedRange];
    [self _replaceLinesInRange: touchedRange_
                    withBlocks: change];
}
@end
