//
//  MSTextView.m
//  Manuscript
//
//  Created by 二鏡 on 12/02/11.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSTextView.h"
#import "MSTextView_Private.h"

static CGFloat cCaretInterval = 0.6;


@implementation MSTextView

@synthesize minSize = minSize_, pager = pager_, delegate = delegate_, lock;
#pragma mark Property
- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame: frame];
    if(self)
    {
        [self registerForDraggedTypes: [self acceptableDraggingTypes]];
    }
    return self;
}
- (BOOL)isFlipped
{
    return YES;
}

- (BOOL)isOpaque
{
    return YES;
}

- (NSRange)selectedRange
{
    return inputState.selectedRange;
}

- (NSRange)markedRange
{
    return inputState.markedRange;
}

- (void)setSelectedRange:(NSRange)aRange
{
    inputState.selectedRange = aRange;
    caretState.locator = inputState.selectedRange.location; // 原則としてselectionと一致する
    [self updateCaretRect];
}

- (void)setFrameSize:(NSSize)size
{
    [super setFrameSize: size];
    if(caretState.is_locked() == false)
        [self updateCaretRect];
}

- (void)setMarkedRange:(NSRange)aRange
{
    inputState.markedRange = aRange;
}

- (NSArray*)acceptableDraggingTypes
{
    return [NSArray arrayWithObjects: NSPasteboardTypeString, NSPasteboardTypeRTF,nil];
}

- (NSArray*)availablePboardClasses
{
    return [NSArray arrayWithObjects: [NSAttributedString class], [NSString class], nil];
}

#pragma mark Event
- (BOOL)acceptsFirstResponder {
    return YES;
}

- (BOOL)becomeFirstResponder {
    [self startCaretTimer];
    [self setNeedsDisplay: YES];
    return YES;
}

- (BOOL)resignFirstResponder {
    [self stopCaretTimer];
    [self setNeedsDisplay: YES];
    return YES;
}

- (void)keyDown:(NSEvent *)theEvent {
    if(self.lock)
        return;
    
    [self _resetCaretTimer: nil];
    NSUInteger flags = [theEvent modifierFlags];
    unichar key = [[theEvent characters] characterAtIndex: 0];
    switch(key)
    {
        case NSUpArrowFunctionKey:
            if(flags & NSAlternateKeyMask && flags & NSShiftKeyMask)
            {
                [self moveWordBackwardAndModifySelection: nil];
                return;
            }
            if(flags & NSAlternateKeyMask)
            {
                [self moveWordBackward: nil];
                return;
            }
            if(flags & NSCommandKeyMask && flags & NSShiftKeyMask)
            {
                [self moveToBeginningOfLineAndModifySelection: nil];
                return;
            }
            if(flags & NSCommandKeyMask)
            {
                [self moveToBeginningOfLine: nil];
                return;
            }
            [[self inputContext] handleEvent: theEvent];
            return;
        case NSDownArrowFunctionKey:
            if(flags & NSAlternateKeyMask && flags & NSShiftKeyMask)
            {
                [self moveWordForwardAndModifySelection: nil];
                return;
            }
            if(flags & NSAlternateKeyMask)
            {
                [self moveWordForward: nil];
                return; 
            }
            if(flags & NSCommandKeyMask && flags & NSShiftKeyMask)
            {
                [self moveToEndOfLineAndModifySelection: nil];
                return;
            }
            if(flags & NSCommandKeyMask)
            {
                [self moveToEndOfLine: nil];
                return ;
            }
            [[self inputContext] handleEvent: theEvent];
            return;
        case NSLeftArrowFunctionKey:
            if(flags & NSAlternateKeyMask && flags & NSShiftKeyMask)
            {
                [self moveToEndOfParagraphAndModifySelection: nil];
                return;
            }
            if(flags & NSAlternateKeyMask)
            {
                [self moveToEndOfParagraph: nil];
                return; 
            }
            if(flags & NSCommandKeyMask && flags & NSShiftKeyMask)
            {
                [self moveToEndOfDocumentAndModifySelection: nil];
                return;
            }
            if(flags & NSCommandKeyMask)
            {
                [self moveToEndOfDocument: nil];
                return ;
            }
            [[self inputContext] handleEvent: theEvent];
            return;
        case NSRightArrowFunctionKey:
            if(flags & NSAlternateKeyMask && flags & NSShiftKeyMask)
            {
                [self moveToBeginningOfParagraphAndModifySelection: nil];
                return;
            }
            if(flags & NSAlternateKeyMask)
            {
                [self moveToBeginningOfParagraph: nil];
                return; 
            }
            if(flags & NSCommandKeyMask && flags & NSShiftKeyMask)
            {
                [self moveToBeginningOfDocumentAndModifySelection: nil];
                return;
            }
            if(flags & NSCommandKeyMask)
            {
                [self moveToBeginningOfDocument: nil];
                return ;
            }
            [[self inputContext] handleEvent: theEvent];
            return;
        default:
            [[self inputContext] handleEvent:theEvent];
    }
}

- (void)mouseDown:(NSEvent *)theEvent {
    if(self.lock)
        return;
    
    if([self hasMarkedText])
    {
        [[self inputContext] handleEvent:theEvent];
        return;
    }
    
    NSPoint point = [self convertPoint: [theEvent locationInWindow]
                              fromView: nil];
    _dragState.onSelection = NO;
    
    NSInteger clickCount = [theEvent clickCount];
    if(clickCount == 2)
    {
        NSUInteger index = [pager_ indexOfCharacterAtPoint: point];
        if(index != NSNotFound)
        {
            // ワード選択
            NSRange range = [pager_.content.textStorage doubleClickAtIndex: index];
            self.selectedRange = range;
            [self setNeedsDisplay: YES];
            return ;
        }
    }
    
    if(clickCount >= 3)
    {
        // パラグラフ選択
        NSUInteger index = [pager_ indexOfCharacterAtPoint: point];
        if(index != NSNotFound)
        {
            NSRange range = [pager_.content.textStorage.mutableString paragraphRangeForRange: NSMakeRange(index,0)];
            self.selectedRange = range;
            [self setNeedsDisplay: YES];
            return ;
        }        
    }
    
    // ドラッグ判定
    // selectionをドラッグするのはbouding判定で行う
    NSUInteger index = [pager_ indexOfCharacterAtPoint: point];
    if(index != NSNotFound)
    {
        if(inputState.is_selected_range_nonzero())
        {
            if(NSLocationInRange(index, inputState.selectedRange))
            {
                // クリックの初期化点がselection上である場合、
                // 内容ドラッグの起点とみなす
                _dragState.onSelection = YES;
                return ;
            }
        }
    }
    
    // それ以外のクリック
    // 領域選択は最も近いボーダーを基準とする
    BOOL isTail;
    index = [pager_ indexOfCharacterOfNearestBorderAtPoint: point
                                                    isTail: &isTail];
    if(index == NSNotFound)
    {
        if(inputState.is_selected_range_nonzero())
            [self deselect: nil];
    }
    else
    {
        if(isTail)
            index--;
        self.selectedRange = NSMakeRange(index,0);
        inputState.selectionAnchor = index;
        [self setNeedsDisplay: YES];
    }
    return ;
}

- (void)mouseDragged:(NSEvent *)theEvent {
    if([self hasMarkedText])
    {
        [[self inputContext] handleEvent:theEvent];
        return;
    }
    
    NSPoint point = [self convertPoint: [theEvent locationInWindow]
                              fromView: nil];
    
    if(_dragState.onSelection)
    {
        // selection上でドラッグ 開始
        // コピー
        NSRect rect;
        NSImage *image = [pager_ generateTextImageForDraggingOfRange: inputState.selectedRange
                                             inRect: &rect];
        NSPasteboard *pboard = [NSPasteboard pasteboardWithName: NSDragPboard];
        [self _copySelectionTo: pboard];
        NSPoint image_position = NSMakePoint(NSMinX(rect),NSMaxY(rect));
        [self dragImage: image
                     at: image_position
                 offset: NSZeroSize
                  event: theEvent
             pasteboard: pboard
                 source: self
              slideBack: YES];
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            return;
        
        BOOL isTail;
        NSUInteger index = [pager_ indexOfCharacterOfNearestBorderAtPoint: point
                                                                   isTail: &isTail];
        if(index != NSNotFound)
        {
            inputState.selectionLocator = index;
            self.selectedRange = inputState.get_anchored_range();
            //        [self scrollToReferenceIndex: index]; // ドラッグ中のスクロールはこれだと不味いはず
            [self setNeedsDisplay: YES];
        }
    }
}

- (void)mouseUp:(NSEvent *)theEvent {
    if([self hasMarkedText])
    {
        [[self inputContext] handleEvent:theEvent];
        return;
    }
    inputState.reset_selection_anchor();
}

#pragma mark Notification Handler
- (void)viewDidMoveToSuperview
{
    id center = [NSNotificationCenter defaultCenter];
    id scroll = [self enclosingScrollView];
    id clipView = [scroll contentView];
    if(scroll)
    {
        [center addObserver: self
                   selector: @selector(scrollFrameDidChange:)
                       name: NSViewFrameDidChangeNotification
                     object: scroll];
        [center addObserver: self
                   selector: @selector(clipBoundsDidChange:)
                       name: NSViewBoundsDidChangeNotification
                     object: clipView];
        [self sizeToFit];
    }
    else
    {
        [center removeObserver: self 
                          name: NSViewFrameDidChangeNotification
                        object: nil];
        [center removeObserver: self
                          name: NSViewBoundsDidChangeNotification
                        object: nil];
    }
}

- (void)clipBoundsDidChange:(id)notif
{
    // 現在のスクロール状態を覚える
    [self memoryScrollRegion];
}

- (void)scrollFrameDidChange:(id)notif
{
    // 右上を維持するようにスクロールさせる
    [self memoryScrollRegion]; // ここでmemoryされたvisibleはscroll frame変更の影響を受けている
    [self sizeToFit];

    NSRect oldVisibleRect = scrollRegion_.visible;
    oldVisibleRect.size = scrollRegion_.clipSize;
    CGFloat dx = NSMaxX(scrollRegion_.bounds) - NSMaxX(oldVisibleRect);
    CGFloat dy = NSMinY(oldVisibleRect);
    
    NSRect visible = [self visibleRect];
    visible.origin = [[[self enclosingScrollView] contentView] bounds].origin;
    NSRect bounds  = [self bounds];
    CGFloat x = NSMaxX(bounds)-dx-NSWidth(visible);
    [self scrollPoint: NSMakePoint(x,dy)];
    
    scrollRegion_.clipSize = visible.size;
}

#pragma mark General
- (void)sizeToFit
{
    NSSize contentSize = [[self enclosingScrollView] contentSize];
    CGFloat w = MAX( minSize_.width, contentSize.width);
    CGFloat h = MAX( minSize_.height, contentSize.height);
    [self setFrameSize: NSMakeSize(w,h)];
}

- (void)scrollToReferenceIndex:(NSUInteger)charIndex
{
    NSRange range;
    NSRect charRect = [pager_ firstRectForCharacterRange: NSMakeRange(charIndex,1)
                                             actualRange: &range];
    
    if( NSEqualRects(NSZeroRect, charRect) )
        [self scrollPoint: NSZeroPoint];
    
    NSRect idealRect = scrollRegion_.visible;
    // 右辺合わせ
    CGFloat oldRight = NSMaxX(scrollRegion_.bounds)-NSMaxX(scrollRegion_.visible);
    NSRect bounds = [self bounds];
    idealRect.origin.x = NSMaxX(bounds)-oldRight-NSWidth(idealRect);
    if( NSContainsRect(idealRect, charRect) )
    {
        [self scrollPoint: idealRect.origin];
        return; 
    }
    
    CGFloat x;
    if( NSMinX(charRect)-NSMinX(idealRect) > 0 &&
        NSMaxX(idealRect)-NSMaxX(charRect) > 0 )
    {
        // x は入ってる. 現状維持
        x = NSMinX(idealRect);
    }
    else
    {
        // なるべく中央に
        x = NSMidX(charRect)-NSWidth(idealRect)/2.0;
    }
    CGFloat y;
    if( NSMinY(charRect)-NSMinY(idealRect) > 0 &&
       NSMaxY(idealRect)-NSMaxY(charRect) > 0 )
    {
        y = NSMinY(idealRect);
    }
    else
    {
        y = NSMidY(charRect)-NSHeight(idealRect)/2.0;
    }
    [self scrollPoint: NSMakePoint(x, y)];
}

- (void)drawRect:(NSRect)dirtyRect
{
    NSEraseRect(dirtyRect);
    
    // selection
    if(inputState.is_selected_range_nonzero())
        [pager_ drawSelectionInRect: dirtyRect
                     characterRange: inputState.selectedRange];
    
    // main
    if(pager_.format.showsGrid)
        [pager_ drawGrid: dirtyRect];
    [pager_ drawLinesInRect: dirtyRect];

    // caret
    if([self _shouldDrawCaret])
    {
        CGContextRef context = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
        NSRect rect = caretState.get_rect();
        [[NSColor blackColor] set];
        CGFloat y = floor(NSMinY(rect))+1.5;
        CGFloat x0 = NSMinX(rect);
        CGFloat x1 = NSMaxX(rect);
        CGContextMoveToPoint(context, x0, y);
        CGContextAddLineToPoint(context, x1, y);
        CGContextSetLineWidth(context, 1.0);
        CGContextStrokePath(context);
    }
}

#pragma mark Timer
- (void)startCaretTimer
{
    if(caretTimer)
        return;
    caretTimer = [NSTimer scheduledTimerWithTimeInterval: cCaretInterval
                                                  target: self
                                                selector: @selector(_tickCaretTimer:)
                                                userInfo: nil
                                                 repeats: YES];
    caretState.show();
    [self setNeedsDisplay: YES];
}

- (void)stopCaretTimer
{
    [caretTimer invalidate];
    caretTimer = nil;
    caretState.hide();
    [self setNeedsDisplay: YES];
}

- (void)_tickCaretTimer:(NSTimer*)timer
{
    caretState.toggle_hidden();
    [self setNeedsDisplay: YES];
}

- (void)_resetCaretTimer:(id)notif
{
    caretState.show();
    [caretTimer setFireDate: [NSDate dateWithTimeIntervalSinceNow: cCaretInterval]];
}

#pragma mark Private

- (BOOL)_shouldDrawCaret
{
    if(caretState.hidden())
        return NO;
    if(self.lock)
        return NO;
    return caretState.is_valid_rect();
}

- (void)updateCaretRect
{
    caretState.cancel_extend(); // 原則位置に戻す
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        caretState.set_rect([pager_ caretRectForCharacterIndex: caretState.locator]);
    }
    else
    {
        caretState.invalide_rect();
    }
}
// 移動用のリファレンス
- (NSRect)_referenceCaretRect
{
    if( caretState.is_valid_rect() )
        return caretState.get_rect(); 
    
    // 現在のキャレットを参考に基準高を計算する
    // selectionから基準キャレットを取って値とする
    NSUInteger cIndex;
    if(inputState.is_selected_range_nonzero())
        cIndex = inputState.selectionLocator;
    else
        cIndex = inputState.selectedRange.location;
    return [pager_ caretRectForCharacterIndex: cIndex];
}

- (void)_memoryCaretHeightIfNeed
{
    if( caretState.has_valid_height() == false)
    {
        // 現在のキャレットを参考に基準高を計算する
        NSRect rect = [self _referenceCaretRect];
        
        // 適当な行の原点を取る(Yのみが必要)
        NSPoint lineOrigin = [pager_ lineOriginAtIndex: 0];
        caretState.set_height(NSMidY(rect)-lineOrigin.y);
        
        // 以下この高さが位置の算出基準となる
    }    
}

- (void)memoryScrollRegion
{
    scrollRegion_.bounds = [self bounds];
    scrollRegion_.visible = [self visibleRect];
}

- (void)_copySelectionTo:(NSPasteboard*)pboard
{
    if(inputState.is_selected_range_zero())
        return;
    
    [pboard clearContents];
    NSTextStorage *storage = pager_.content.textStorage;
    id aStr = [[storage attributedSubstringFromRange: inputState.selectedRange] string];
    id objects = [NSArray arrayWithObjects: aStr,nil];
    [pboard writeObjects: objects];    
}

- (void)_moveTextInRange:(NSRange)range
                      to:(NSUInteger)index
{
    MSLayoutBlockManager *content = pager_.content;
    NSTextStorage *storage = content.textStorage;
    id undoManager = [delegate_ undoManager];
    
    if(index >= NSMaxRange(range))
    {
        index -= range.length;
        [[undoManager prepareWithInvocationTarget: self]
         _moveTextInRange: NSMakeRange(index, range.length)
         to: range.location];
    }
    else
    {
        [[undoManager prepareWithInvocationTarget: self]
         _moveTextInRange: NSMakeRange(index, range.length)
         to: range.location+range.length];        
    }    

    id subString = [storage attributedSubstringFromRange: range];
    [undoManager disableUndoRegistration];
    [self deleteCharactersInRange: range];
    [self insertText: subString
    replacementRange: NSMakeRange(index,0)];
    [undoManager enableUndoRegistration];
    
    self.selectedRange = NSMakeRange(index, range.length);
    inputState.reset_selection_anchor();
    [self setNeedsDisplay: YES];
}

#pragma mark Action
- (void)deleteCharactersInRange:(NSRange)range 
{
    // Update the marked range
    NSRange newMarkedRange = inputState.markedRange;
    if (NSLocationInRange(NSMaxRange(range), inputState.markedRange)) {
        // 何故こうなるのかよくわからないがサンプルがこうなっている
        // 日本語以外の入力システムでは機能する？
        newMarkedRange.length -= NSMaxRange(range) - inputState.markedRange.location;
        newMarkedRange.location = range.location;
        self.markedRange = newMarkedRange;
    } else if (inputState.markedRange.location > range.location) {
        newMarkedRange.location -= range.length;
        self.markedRange = newMarkedRange;
    }
    
    if (inputState.markedRange.length == 0) {
        [self unmarkText];
    }
    
    caretState.lock();
    [self memoryScrollRegion];
    MSLayoutBlockManager *content = pager_.content;
    NSTextStorage *storage = content.textStorage;
    
    // Prepare undoing
    id undoManager = [delegate_ undoManager];
    [[undoManager prepareWithInvocationTarget: self]
     insertText: [storage attributedSubstringFromRange: range]
     replacementRange: NSMakeRange(range.location,0)];
    
    [content replaceCharactersInRange: range
                 withAttributedString: nil];
    
    caretState.unlock();
    self.selectedRange = NSMakeRange(range.location,0);
    inputState.reset_selection_anchor();
    caretState.clear_height();
    
    // scroll
    NSUInteger ref = range.location == 0 ? 0 : range.location-1;
    [self scrollToReferenceIndex: ref];
    
    [[self inputContext] invalidateCharacterCoordinates];
    [self setNeedsDisplay:YES];
}

- (void)pasteAttributedString:(NSAttributedString*)aStr
{
    // アタッチメントが含まれる場合は除染を必要とする。
    // mutableCopyを作ってアタッチメントキャラクターを除けばオッケー
    if(delegate_)
        aStr = [delegate_ normalizeAttributedString: aStr];
    [self insertText: aStr
    replacementRange: NSMakeRange(NSNotFound, NSNotFound)];
}

@end

@implementation MSTextView (ScriptInterface)
- (NSString*)selectedString
{
    if(inputState.is_selected_range_zero())
        return @"";
    
    id storage = pager_.content.textStorage;
    id ret = [[storage mutableString] substringWithRange: inputState.selectedRange];
    return ret;
}

- (NSString*)string
{
    id storage = pager_.content.textStorage;
    return [[storage mutableString] copy];
}

- (void)replaceSelectionWithString:(NSString*)aString
{
    NSUInteger length = [aString length];
    NSUInteger location = inputState.get_head_of_selection();
    [self insertText: aString replacementRange: inputState.selectedRange];
    [self setSelectedRange: NSMakeRange(location, length)];
}

- (void)replaceAllWithString:(NSString*)aString
{
    id storage = pager_.content.textStorage;
    [self insertText: aString replacementRange: NSMakeRange(0, [storage length])];
    [self setSelectedRange: NSMakeRange(0, [aString length])];
}

- (void)insertStringAfterSelection:(NSString*)aString
{
    NSUInteger location = NSMaxRange(inputState.selectedRange);
    [self insertText: aString replacementRange: NSMakeRange(location,0)];
    [self setSelectedRange: NSMakeRange(location, [aString length])];
}

- (void)appendString:(NSString*)aString
{
    id storage = pager_.content.textStorage;
    NSUInteger location = [storage length];
    [self insertText: aString replacementRange: NSMakeRange(location,0)];
    [self setSelectedRange: NSMakeRange(location, [aString length])];
}
@end
