//
//  MSTypesetter.m
//  Manuscript
//
//  Created by 二鏡 on 12/03/11.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSTypesetter.h"
#import "GlyphCluster.h"
#import "StandardCharacterSet.h"
#import "VTVerticalGlyphSubstitution.h"
#import "VTLogicalLineContainer.h"
#import "CFRangeExt.h"

static CFCharacterSetRef lFullWidthSet;
static CFCharacterSetRef lHeadProhibitionSet;
static CFCharacterSetRef lTailProhibitionSet;

@implementation MSTypesetter
@synthesize boxCount, boxSize;

+ (void)initialize
{
    if(self == [MSTypesetter class])
    {
        GlyphCluster::intialize_rederning_system();
        lFullWidthSet = CreateFullWidthCharacterSet();
        lHeadProhibitionSet = CFCharacterSetCreateWithCharactersInString(NULL, JISHeadProhibitionCharacters);
        lTailProhibitionSet = CFCharacterSetCreateWithCharactersInString(NULL, JISTailProhibitionCharacters);
    }
}

- (id)init
{
    bufferSpace = new _GlyphBufferSpace;
    glyphLocator = new VerticalGlyphLocator(bufferSpace);
    bufferSpace->setDelegate(self);
        
    return self;
}

- (void)substituteGlyphs:(CGGlyph*)glyphs
              characters:(const UniChar*)chars
                   count:(CFIndex)count
                  inFont:(NSFont*)font
{
    if(!font)
        return;
    
    @try {
        id subst = [VTVerticalGlyphSubstitution verticalGlyphSubstitutionWithFont: font];
        [subst substituteGlyphs: glyphs count: count];
    }
    @catch(id ex)
    {
    }
}

- (void)parseCharacters:(const UniChar*)characters
           toAttributes:(uint8_t*)attributes
                 length:(CFIndex)length
{
    CFCharacterSetRef nl = CFCharacterSetGetPredefined(kCFCharacterSetNewline);
    CFCharacterSetRef ws = CFCharacterSetGetPredefined(kCFCharacterSetWhitespace);
    CFCharacterSetRef punc = CFCharacterSetGetPredefined(kCFCharacterSetPunctuation);
    
    // attribute parsing.
    CFIndex i;
    for(i=0;i<length;i++)
    {
        // initialize
        UniChar c = characters[i];
        attributes[i] = 0;
        
        // 強い改行候補は改行文字だけである。
        if(CFCharacterSetIsCharacterMember(nl, c))
        {
            attributes[i] = cStrongBreak;
            continue;
        }
        
        // 全角文字は全て弱い改行候補でもある
        if(CFCharacterSetIsCharacterMember(lFullWidthSet, c))
        {
            attributes[i] |= cFixedWidth + cWeakBreak;
        }
        
        // 行頭禁則文字
        if(CFCharacterSetIsCharacterMember(lHeadProhibitionSet, c))
        {
            attributes[i] |= cHeadProhibition;
        }
        
        // 行末禁則文字
        if(CFCharacterSetIsCharacterMember(lTailProhibitionSet, c))
        {
            attributes[i] |= cTailProhibition;
        }
        
        // 空白と分断記号は弱い改行候補
        if(CFCharacterSetIsCharacterMember(ws, c) ||
           CFCharacterSetIsCharacterMember(punc, c))
        {
            attributes[i] |= cWeakBreak;
        }
    }
}

#pragma mark -

- (GlyphCluster*)makeGlyphBlockInRange:(CFRange)range
                             attribute:(const AttributesSet&)attr
{
    const CGGlyph* glyphs = bufferSpace->getGlyphs()+range.location;
    const CGPoint* locations = bufferSpace->getPositions()+range.location;
    
    GlyphCluster *ret = new GlyphCluster(attr.font,glyphs,locations,range.length);
    ret->setColor(attr.foregroundColor);
    ret->underline_style = attr.underlineStyle;
    ret->strikethrough_style = attr.strikethroughStyle;
    ret->underline_color = attr.underlineColor;
    ret->strikethrough_color = attr.strikethroughColor;
    ret->segment = attr.segment != -1;
    
    // 属性解析
    return ret;
}


- (VTLogicalLineContainer*)generateLogicalLineInRange:(CFRange)lineRange
{
    const CompiledAttributedString *string = bufferSpace->getString();
    size_t limit = string->attributes.size();
    size_t i;
    // 目的の場所までから回し
    for(i=0;i<limit;i++)
    {
        CFRange range = string->attributes[i].first;
        if(_CFLocationInRange(lineRange.location, range))
            break;
    }
    
    // 閉じているかは行の情報
    BOOL isClosed;
    {
        unichar last = string->characters[_CFRangeLimit(lineRange)-1];
        CFCharacterSetRef newlineSet = CFCharacterSetGetPredefined(kCFCharacterSetNewline);
        isClosed = CFCharacterSetIsCharacterMember(newlineSet, last);
    }
    
    // 最初に行オブジェクトを生成
    const vector<CGFloat>& borders = bufferSpace->getCaretOffsetsRef();
    LogicalLine *line = new LogicalLine(lineRange,boxSize,isClosed,borders);
    
    // 所属クラスタを計算
    for(;i<limit;i++)
    {
        // ここでレンジを制約しているので、属性をまたいだ場合行末で分断される
        CFRange runRange = string->attributes[i].first;
        CFRange range = _CFRangeConstrain(runRange, lineRange);
        
        // 対象クラスタを生成
        const AttributesSet& attr = string->attributes[i].second;
        GlyphCluster *render_unit = [self makeGlyphBlockInRange: range 
                                                      attribute: attr];
        // 行ローカルの処理
        CFRange localRange = range;
        localRange.location -= lineRange.location;
        render_unit->bounds_top    = borders[localRange.location];
        render_unit->bounds_bottom = borders[_CFRangeLimit(localRange)];
        
        // 単位の確定
        line->add(render_unit);
        
        // 先頭処理の理由により、必ず等号で処理が終る
        if(_CFRangeLimit(range) == _CFRangeLimit(lineRange))
            break;
    }
    
    // boxingして完了
    return [[VTLogicalLineContainer alloc] initWithLine: line];
}

- (NSArray*)linesFromString:(CompiledAttributedString*)aString
{
    bufferSpace->loadString(aString);
    bufferSpace->loadStage1substitute();
    bufferSpace->loadStage2metrics(kCTFontVerticalOrientation);
    bufferSpace->loadStage3attributes();
    
    id lines = [NSMutableArray array];
    CFRange targetRange = CFRangeMake(0, aString->characters.size());
    
    for(;;)
    {
        // locatorを初期化
        glyphLocator->reloadLine(boxSize,boxCount,targetRange.location);
        
        glyphLocator->setShouldCenteringStabilize(true);
        bool line_completed = glyphLocator->analyzeGlyphsInRange(targetRange);
        glyphLocator->setShouldCenteringStabilize(line_completed);
        glyphLocator->stabilizeCurrentChainIfNeed();
        if(line_completed)
        {
            if(glyphLocator->tryExtraLayoutIfNeed() == false)
                glyphLocator->tryWrapToNextIfNeed();
        }
        glyphLocator->finalizeStabilizedRange();
        CFRange stabilizedRange = glyphLocator->getStabilizedRange();
        glyphLocator->hasHyphenation(); // ハイフン処理が入ったか？
        
        id line = [self generateLogicalLineInRange: stabilizedRange];
        [lines addObject: line];
        
        if(_CFRangeLimit(stabilizedRange) == _CFRangeLimit(targetRange))
            break;
        targetRange = _CFRangeSubtract(targetRange, stabilizedRange);
    }
    
    return [lines copy];
}

@end
