//
//  MSLayoutBlockManager.h
//  Okusa
//
//  Created by 木谷 洋 on 12/03/02.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "ClusterMappingCache.h"

extern NSString *MSLayoutBlockManagerWillChangeBlocksNotification;
extern NSString *MSLayoutBlockManagerDidChangeBlocksNotification;

@class MSLayoutBlock;

class BlockMappingValidator
{
    // それぞれ正しくマッピングされているindexを指す
    // NSNotFoundの時は何もない
    NSUInteger lineMapping_;
    NSUInteger characterMapping_;
public:
    BlockMappingValidator()
    {
        lineMapping_ = NSNotFound;
        characterMapping_ = NSNotFound;
    }
    
    bool has_valid_line_mapping() const
    {
        return lineMapping_ != NSNotFound;
    }
    
    bool is_line_mapped(NSUInteger i) const
    {
        if(lineMapping_ == NSNotFound)
            return false;
        return lineMapping_ >= i;
    }
    
    bool is_character_mapped(NSUInteger i) const
    {
        if(characterMapping_ == NSNotFound)
            return false;
        return characterMapping_ >= i;
    }
    
    bool has_valid_character_mapping() const
    {
        return characterMapping_ != NSNotFound;
    }
    
    NSUInteger get_valid_line_mapping() const
    {
        return lineMapping_;
    }
    
    NSUInteger get_valid_character_mapping() const
    {
        return characterMapping_;
    }
    
    inline void validate_line_mapping(NSUInteger i)
    {
        lineMapping_ = i;
    }
    
    inline void validate_line_mapping_to_limit(NSUInteger limit)
    {
        if(limit == 0)
            lineMapping_ = NSNotFound;
        else
            lineMapping_ = limit-1;
    }
    
    inline void cut_line_mapping_to_limit(NSUInteger limit)
    {
        // 最大で上限limitまでの範囲にする。もともと小さければそちらに合わす
        if(limit == 0)
            lineMapping_ = NSNotFound;
        else
            lineMapping_ = MIN(lineMapping_, limit-1);
    }
    
    inline void invalidate_line_mapping()
    {
        lineMapping_ = NSNotFound;
    }
    
    inline void validate_character_mapping(NSUInteger i)
    {
        characterMapping_ = i;
    }
    
    inline void validate_character_mapping_to_limit(NSUInteger limit)
    {
        if(limit == 0)
            characterMapping_ = 0;
        else
            characterMapping_ = limit-1;
    }
    
    inline void cut_character_mapping_to_limit(NSUInteger limit)
    {
        // 最大で上限limitまでの範囲にする。もともと小さければそちらに合わす
        if(limit == 0)
            characterMapping_ = NSNotFound;
        else
            characterMapping_ = MIN(characterMapping_, limit-1);
    }
    
    inline void invalidate_character_mapping()
    {
        characterMapping_ = NSNotFound;
    }
};

@interface MSLayoutBlockManager : NSObject <NSTextStorageDelegate>
{
    NSTextStorage *textStorage_;
    NSMutableArray *blocks_;
    ClusterMappingCache syncCache_;
    BlockMappingValidator mapping_;
    struct {
        CFIndex removalAffect;
        CFIndex creationAffect;
    } synchronizeAffection_;
    NSRange editedRange_;
}

@property (readonly) NSTextStorage *textStorage;
@property (readonly) NSArray *blocks;

// 更新されるブロックの範囲を返す
// will changeではなくなる範囲
// did changeでは新しくなった範囲を指す
@property (readonly) NSRange editedRange;
//@property (readonly) NSUInteger changeInLength;
- (NSRange)lineRangeForBlockRange:(NSRange)aBlockRange;
- (MSLayoutBlock*)lastBlock;
- (NSArray*)subblocksWithRange:(NSRange)blockRange;
- (NSRange)characterRangeOfBlock:(MSLayoutBlock*)aBlock;
- (NSRange)lineRangeOfBlock:(MSLayoutBlock*)aBlock;
- (MSLayoutBlock*)blockAtCharacterIndex:(NSUInteger)index; // このblockはcharacter validである
@end


@interface MSLayoutBlockManager (TextStorageProxy)
// nilを渡すと削除扱い
- (void)replaceCharactersInRange:(NSRange)range
            withAttributedString:(NSAttributedString*)aStr;
- (void)loadString:(NSAttributedString*)aString;
- (NSData*)dataWithEncoding:(NSStringEncoding)aEncoding;
@end

