package org.dyndns.nuda.management;

import java.io.Serializable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ツールのバージョンを表すJavaBeansです.<br />
 * 
 * バージョンは以下の４つの要素からなります.<br />
 * 
 * ・メジャーバージョン:major-version<br />
 * ・マイナーバージョン:minor-version<br />
 * ・ブランチバージョン:branch-version<br />
 * ・製品状態:product-status<br />
 * 
 * 基本的に、バージョン番号は10桁のint値からなります。<br />
 * 以下に、バージョン番号の意味を記載します
 * <table border=1>
 * 	<tr>
 * 		<th>桁</th>
 *      <th>意味</th>
 *  </tr>
 * 		<th>1～3桁</th><td>メジャーバージョンをあらわす"1"<br /> + メジャーバージョン番号(2桁)</td>
 * </tr><tr>
 *      <th>4～6桁</th><td>マイナーバージョンをあらわす"2"<br /> + マイナーバージョン番号(2桁)</td>
 *      </tr><tr>
 *      <th>7～9桁</th><td>ブランチバージョンをあらわす"3"<br /> + ブランチバージョン番号(2桁)</td>
 *      </tr><tr>
 *      <th>10桁</th><td>製品状態を表す1桁の番号</td>
 *  </tr>
 * </table>
 * 
 * 
 * 
 * なお、製品状態はバージョンの大小に影響しません<br />
 * 
 * 
 * @author koseki
 * 
 */
public class Version implements Comparable<Version>, Serializable {
	/**
	 * シリアルバージョンUID
	 */
	private static final long serialVersionUID = 1L;

	public static final int version = 1301;

	/**
	 * バージョンをパースするための正規表現文字列
	 */
	private static final String versionParsePatternStr = "1(\\d{2})2(\\d{2})3(\\d{2})(\\d{1})";

	/**
	 * バージョンをパースするための正規表現
	 */
	private static final Pattern VERSION_PATTERN = Pattern
			.compile(versionParsePatternStr);

	private int currentVersion = 0;
	private int majorVersion = 0;
	private int minorVersion = 0;
	private int branchVersion = 0;
	private int productStatus = 0;

	/**
	 * 引数に指定したバージョン番号を元にインスタンスを生成します
	 * @param version
	 * 
	 */
	public Version(int version) {
		String strVersion = Integer.valueOf(version).toString();
		Matcher matcher = VERSION_PATTERN.matcher(strVersion);
		if (!matcher.matches()) {
			// 引数が正規のバージョン番号でない場合は
			// バージョン番号を4桁のint値と認識して、インスタンス生成を行う。
			
			currentVersion = 0;
			int toolVersion = 0;
			if (version < 0) {
				currentVersion = 0;
				toolVersion = 0;
			} else {
				currentVersion = version;
				toolVersion = version;
			}

			majorVersion = (toolVersion - (toolVersion % 1000)) / 1000;

			if (majorVersion < 0) {
				majorVersion = 0;
			}

			toolVersion = toolVersion - majorVersion * 1000;

			minorVersion = (toolVersion - (toolVersion % 100)) / 100;

			toolVersion = toolVersion - minorVersion * 100;
			branchVersion = (toolVersion - (toolVersion % 10)) / 10;

			toolVersion = toolVersion - branchVersion * 10;
			productStatus = (toolVersion - (toolVersion % 1)) / 1;
		} else {
			majorVersion = Integer.valueOf(matcher.group(1));
			minorVersion = Integer.valueOf(matcher.group(2));
			branchVersion = Integer.valueOf(matcher.group(3));
			productStatus = Integer.valueOf(matcher.group(4));
		}
	}

	/**
	 * 現在のバージョンを表すint値を取得します
	 * 
	 * @return
	 */
	public int getCurrentVersion() {
		return currentVersion;
	}

	/**
	 * 現在のメジャーバージョンを表すint値を取得します
	 * 
	 * @return
	 */
	public int getMajorVersion() {
		return majorVersion;
	}

	/**
	 * 現在のマイナーバージョンを表すint値を取得します
	 * 
	 * @return
	 */
	public int getMinorVersion() {
		return minorVersion;
	}

	/**
	 * 現在のブランチバージョン(枝番)を表すint値を取得します
	 * 
	 * @return
	 */
	public int getBranchVersion() {
		return branchVersion;
	}

	/**
	 * 現在の製品状態を取得します
	 * 
	 * @return
	 */
	public PRODUCT_STATUS getProductStatus() {
		return PRODUCT_STATUS.create(productStatus);
	}

	/**
	 * バージョンアップ(メジャーバージョンアップ)を行います
	 * 
	 * @return
	 */
	public Version versionUp() {
		int currentMajorVersion = majorVersion;
		int currentMinorVersion = minorVersion;
		int currentBranchVersion = branchVersion;

		return null;
	}

	/**
	 * リビジョンアップ(マイナーバージョンアップ)を行います
	 * 
	 * @return
	 */
	public Version revisionUp() {
		int currentMajorVersion = majorVersion;
		int currentMinorVersion = minorVersion;
		int currentBranchVersion = branchVersion;
		return null;
	}

	/**
	 * サフィックスアップ(ブランチバージョンアップ)を行います
	 * 
	 * @return
	 */
	public Version suffixUp() {
		int currentMajorVersion = majorVersion;
		int currentMinorVersion = minorVersion;
		int currentBranchVersion = branchVersion;
		return null;
	}

	/**
	 * 現在のバージョンに対して製品状態の変更を行います
	 * 
	 * @param status
	 */
	public void changeProductStatus(PRODUCT_STATUS status) {

	}

	@Override
	public String toString() {
		StringBuilder b = new StringBuilder();
		if (majorVersion < 1) {
			b.append(Integer.valueOf(majorVersion)).append(".");
		} else {
			b.append(Integer.valueOf(majorVersion - 1)).append(".");
		}

		b.append(Integer.valueOf(minorVersion)).append(".");
		b.append(Integer.valueOf(branchVersion));

		b.append(PRODUCT_STATUS.create(productStatus).getStatusName());

		return b.toString();
	}

	@Override
	public int compareTo(Version arg0) {
		if (arg0 == null) {
			return 1;
		}
		int major = this.majorVersion - arg0.majorVersion;
		int minor = this.minorVersion - arg0.minorVersion;
		int branch = this.branchVersion - arg0.branchVersion;
		return major * 100 + minor * 10 + branch;
	}
}
