package org.dyndns.nuda.tools.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.dyndns.nuda.tools.util.filter.AutoReplacementFilter;

/**
 * 
 * 文字列を操作するための様々なユーティリティメソッドが含まれます
 * 
 * 
 * 
 * 
 * <style type="text/css"> .r { color:red; } </style>
 * 
 * @author nkoseki
 * 
 */
public class StringUtil {
	private StringUtil() {
		// インスタンス化させない
	}
	
	public static String loadString(String path) throws IOException {
		StringBuilder b = new StringBuilder();
		if(path == null) {
			return b.toString();
		}
		if(path.isEmpty()) {
			return b.toString();
		}
		
		ClassLoader loader = Thread.currentThread().getContextClassLoader();
		
		InputStream is = null;
		InputStreamReader isr = null;
		BufferedReader br = null;
		
		is = loader.getResourceAsStream(path);
		if(is == null) {
			return b.toString();
		}
		try {
			try {
				isr = new InputStreamReader(is);
				br = new BufferedReader(isr);
				
				while(br.ready()) {
					String line = br.readLine();
					b.append(line).append("\n");
				}
			} finally {
				if(br != null) {
					br.close();
				}
				if(isr != null) {
					isr.close();
				}
				if(is != null) {
					is.close();
				}
			}
		} catch (IOException e) {
			throw e;
		}
		
		return b.toString();
	}
	
	/**
	 * 引数に指定された文字列のインデント解除を行います
	 * 
	 * @param content
	 *            対象文字列
	 * @return インデント解除された文字列
	 */
	public static String deIndent(final String content) {
		String content0 = content;
		String patternStr0 = "\\t+";
		Pattern p0 =
			Pattern.compile(patternStr0, Pattern.DOTALL | Pattern.MULTILINE);
		
		Matcher m0 = p0.matcher(content);
		String replacement = "";
		
		if (m0.find()) {
			replacement = m0.group();
			
			content0 = m0.replaceFirst("");
		}
		
		Pattern p =
			Pattern.compile("^" + replacement, Pattern.DOTALL
				| Pattern.MULTILINE);
		
		Matcher m = p.matcher(content0);
		
		String replaceItem = "";
		// int i = 0;
		while (m.find()) {
			replaceItem = m.replaceFirst(REPLACEMENT_HOLDER);
			
			m.reset(replaceItem);
		}
		
		replaceItem = replaceItem.replace(REPLACEMENT_HOLDER, "");
		
		return replaceItem;
	}
	
	/**
	 * 第一引数の文字列を第二引数に指定された数値分, タブ文字でインデントを行います
	 * 
	 * @param content
	 *            対象文字列
	 * @param tabCount
	 *            インデントを行う際のタブ個数
	 * @return インデントされた文字列
	 */
	public static String indent(final String content, final int tabCount) {
		String content0 = content;
		// ^((?:[^(<REPLACEMENT>)]).*)$
		Pattern p =
			Pattern.compile("^((?!<REPLACEMENT>).+?)$", Pattern.DOTALL
				| Pattern.MULTILINE);
		
		Matcher m = p.matcher(content0);
		
		String replaceItem = "";
		// int i = 0;
		while (m.find()) {
			replaceItem = m.replaceFirst("<REPLACEMENT>" + m.group(1));
			
			m.reset(replaceItem);
			// i++;
		}
		
		//LoggerFactory.getLogger("StringUtil").info("\n{}\n", replaceItem);
		
		StringBuilder tab = new StringBuilder();
		
		for (int i = 0; i < tabCount; i++) {
			tab.append("\t");
		}
		
		replaceItem = replaceItem.replace("<REPLACEMENT>", tab.toString());
		
		return replaceItem;
	}
	
	/**
	 * 第一引数の文字列を第二引数に指定された数値数値分, タブ文字でインデントを行います.</br>
	 * このメソッドはindentメソッドとは異なり行の先頭に行番号を付加します
	 * 
	 * @param content
	 *            対象文字列
	 * @param tabCount
	 *            インデントを行う際のタブ個数
	 * @return インデントされた文字列
	 */
	public static String indentWithNum(final String content, final int tabCount) {
		String content0 = content;
		// ^((?:[^(<REPLACEMENT>)]).*)$
		Pattern p =
			Pattern.compile("^((?!<REPLACEMENT>|\\d|\\:).+?)$", Pattern.DOTALL
				| Pattern.MULTILINE);
		
		Matcher m = p.matcher(content0);
		
		String replaceItem = "";
		// int i = 0;
		int lineNum = 1;
		
		while (m.find()) {
			String lineNumStr = String.format(FORMAT_P03D, lineNum);
			replaceItem =
				m.replaceFirst(lineNumStr + " :<REPLACEMENT>" + m.group(1));
			
			m.reset(replaceItem);
			lineNum++;
		}
		
		//LoggerFactory.getLogger("StringUtil").info("\n{}\n", replaceItem);
		
		StringBuilder tab = new StringBuilder();
		
		for (int i = 0; i < tabCount; i++) {
			tab.append(TAB);
		}
		
		replaceItem = replaceItem.replace("<REPLACEMENT>", tab.toString());
		
		return replaceItem;
	}
	
	private static final String		FORMAT_P03D				= "%03d";
	//private static final String		CRLF					= "";
	private static final String		TAB						= "\t";
	
	private static final String		ptnStr					= "\\{\\}";
	private static final Pattern	PTN						=
																Pattern
																	.compile(
																		ptnStr,
																		Pattern.DOTALL
																			| Pattern.MULTILINE);
	
	private static final String		NOT_REPLACE_HOLDER		=
																"<StringUtil.NOT_REPLACE>";
	private static final String		ptnStr2					=
																"<StringUtil\\.NOT_REPLACE>";
	private static final Pattern	PTN2					=
																Pattern
																	.compile(
																		ptnStr2,
																		Pattern.DOTALL
																			| Pattern.MULTILINE);
	
	private static final String		ptnStr3					=
																"\\{([^\\{|^\\}]+?)\\}";
	private static final Pattern	PTN3					=
																Pattern
																	.compile(
																		ptnStr3,
																		Pattern.DOTALL
																			| Pattern.MULTILINE);
	
	private static final String		BLACKET_LEFT_REPLACE	=
																"<<<org.dyndns.nuda.repserv.util.StringUtil.BLACKET_LEFT>>>";
	private static final String		BLACKET_RIGHT_REPLACE	=
																"<<<org.dyndns.nuda.repserv.util.StringUtil.BLACKET_RIGHT>>>";
	
	private static final String		BLACKET_LEFT			= "{";
	private static final String		BLACKET_RIGHT			= "}";
	private static final String		BLACKET					= "{}";
	private static final String		REPLACEMENT_HOLDER		=
																"<<<org.dyndns.nuda.repserv.util.StringUtil.REPLACEMENT_HOLDER>>>";
	//	private static final String		REPLACEMENT_HOLDER_PTN	=
	//																"<<<org\\.dyndns\\.nuda\\.repserv\\.util\\.StringUtil\\.REPLACEMENT_HOLDER>>>";
	
//	private static final String		YEN						= "\\";
//	private static final String		YEN_REPLACEMENT_HOLDER	= "#YEN#";
	
	/**
	 * C形式のformatメソッドです. </br>
	 * {}をプレースホルダーとして、第二引数以降に指定したオブジェクトを</br>
	 * 順次割り当てます.
	 * 
	 * @param format
	 *            フォーマット文字列です
	 * @param params
	 *            パラメータ
	 * @return フォーマット済み文字列
	 */
	public static String format(final String format, final Object... params) {
		String result = "";
		if (format == null) {
			return null;
		}
		if (params == null || params.length == 0) {
			result = format;
		} else {
			Matcher m = PTN.matcher(format);
			
			int i = 0;
			while (m.find()) {
				if (i < params.length) {
					if (params[i] != null) {
						
						String replacement = params[i].toString();
//						if (replacement.contains(YEN)) {
//							
//							replacement =
//								replacement
//									.replace(YEN, YEN_REPLACEMENT_HOLDER);
//						}
//						if (replacement.contains("(")) {
//							
//							replacement =
//								replacement
//									.replace("(", "#KAKKOL#");
//						}
//						if (replacement.contains(")")) {
//							
//							replacement =
//								replacement
//									.replace(")", "#KAKKOR#");
//						}
//						if (replacement.contains(".")) {
//							
//							replacement =
//								replacement
//									.replace(".", "#DOT#");
//						}
//						if (replacement.contains("$")) {
//							
//							replacement =
//								replacement
//									.replace("$", "#DOLLER#");
//						}
						replacement = AutoReplacementFilter.getInstance().replace(replacement);
						
						//System.out.println("replacement:" + replacement + " " + m.group());
						String replaceResult = m.replaceFirst(replacement);
						
						m.reset(replaceResult);
						result = replaceResult;
						
					} else {
						String replaceResult =
							m.replaceFirst(NOT_REPLACE_HOLDER);
						
						m.reset(replaceResult);
						result = replaceResult;
					}
					
				}
				
				i++;
			}
			
			Matcher m2 = PTN2.matcher(result);
			
			while (m2.find()) {
				String r = m2.replaceFirst(BLACKET);
				m2.reset(r);
				result = r;
			}
		}
		
//		result = result.replace(YEN_REPLACEMENT_HOLDER, YEN);
//		result = result.replace("#KAKKOL#", "(");
//		result = result.replace("#KAKKOR#", ")");
//		result = result.replace("#DOT#", ".");
//		result = result.replace("#DOLLER#", "$");
		result = AutoReplacementFilter.getInstance().repair(result);
		
		
//		if (replacement.contains("(")) {
//			
//			replacement =
//				replacement
//					.replace("(", "#KAKKOL#");
//		}
//		if (replacement.contains(")")) {
//			
//			replacement =
//				replacement
//					.replace(")", "#KAKKOR#");
//		}
//		if (replacement.contains(".")) {
//			
//			replacement =
//				replacement
//					.replace(".", "#DOT#");
//		}
//		if (replacement.contains("$")) {
//			
//			replacement =
//				replacement
//					.replace("$", "#DOLLER#");
//		}
		
		return result;
	}
	
	/**
	 * C形式のformatメソッドです.<br>
	 * {KEY}をMAP<KEY, VALUE>によって「VALUE」に変換します
	 * 
	 * @param format
	 *            フォーマット文字列
	 * @param map
	 *            変換マップ
	 * @return フォーマット済み文字列
	 */
	public static String format(final String format,
			final Map<String, Object> map) {
		String result = "";
		if (format == null) {
			return result;
		}
		if (map == null || map.size() == 0) {
			result = format;
		} else {
			Matcher m = PTN3.matcher(format);
			
			//int i = 0;
			while (m.find()) {
				String g = m.group(1);
				Object val = map.get(g);
				
				if (map.containsKey(g)) {
					String replacement = val.toString();
//					if (replacement.contains(YEN)) {
//						
//						replacement =
//							replacement.replace(YEN, YEN_REPLACEMENT_HOLDER);
//					}
					replacement = AutoReplacementFilter.getInstance().replace(replacement);
					
					String replaceResult = m.replaceFirst(replacement);
					
					m.reset(replaceResult);
					result = replaceResult;
					
				} else if (g.isEmpty()) {
					String replaceResult = m.replaceFirst(NOT_REPLACE_HOLDER);
					
					m.reset(replaceResult);
					result = replaceResult;
				} else {
					
					String group = m.group();
					
					group =
						group
							.replace(BLACKET_LEFT, BLACKET_LEFT_REPLACE)
							.replace(BLACKET_RIGHT, BLACKET_RIGHT_REPLACE);
					
					String replaceResult = m.replaceFirst(group);
					
					m.reset(replaceResult);
					result = replaceResult;
				}
				
				//i++;
			}
			
			Matcher m2 = PTN2.matcher(result);
			
			while (m2.find()) {
				String r = m2.replaceFirst(BLACKET);
				m2.reset(r);
				result = r;
			}
			
			result =
				result.replace(BLACKET_LEFT_REPLACE, BLACKET_LEFT).replace(
					BLACKET_RIGHT_REPLACE,
					BLACKET_RIGHT);
		}
		
		//result = result.replace(YEN_REPLACEMENT_HOLDER, YEN);
		result = AutoReplacementFilter.getInstance().repair(result);
		
		return result;
	}
	
	// ============================= splitCSVRow =================================
	private static final String		DC					= "\"";
	private static final String		splitCSVRowPTNSTR	=
															"((?!\")((\\p{InCJKUnifiedIdeographs}|\\p{InHiragana}|\\p{InKatakana}|\\p{InHalfwidthAndFullwidthForms}|[a-zA-Z0-9])+)(?!\"))|((?:\")([,\\w\\s\\W]+?)(?:\"))"
																.replace(
																	"@",
																	DC);
	private static final String		splitCSVRowPTNSTR2	= "@(.+?)@".replace(
															"@",
															DC);
	private static final Pattern	splitCSVRowPTN		=
															Pattern
																.compile(
																	splitCSVRowPTNSTR,
																	Pattern.MULTILINE
																		| Pattern.DOTALL);
	
	private static final Pattern	splitCSVRowPTN2		=
															Pattern
																.compile(
																	splitCSVRowPTNSTR2,
																	Pattern.MULTILINE
																		| Pattern.DOTALL);
	
	/**
	 * <style type="text/css"> .r { color:red; } </style>
	 * 
	 * CSV行を表す文字列を各カラムごとに分解します.</br> CSV行はカンマで区切られた文字列データを表し それぞれのカラムは
	 * <ul>
	 * <li><span class="r">カンマを含まない</span>英数大文字/小文字/, 半角/全角カタカナ, 半角/全角ひらがな, 漢字</li>
	 * <li>「ダブルクォートで囲まれた」<span class="r">カンマを含む</span>英数大文字/小文字/, 半角/全角カタカナ, 半角/全角ひらがな, 漢字, 改行以外の空白文字</li>
	 * </ul>
	 * より成ります
	 * 
	 * @param csvRow
	 *            CSV行
	 * @return カラムリスト
	 */
	public static List<String> splitCSVRow(final String csvRow) {
		List<String> result = new LinkedList<String>();
		
		Matcher m = splitCSVRowPTN.matcher(csvRow);
		
		while (m.find()) {
			String r = m.group();
			Matcher m2 = splitCSVRowPTN2.matcher(r);
			
			if (m2.find()) {
				// ""(ダブルクォートで囲まれている場合)
				result.add(m2.group(1));
			} else {
				// ""(ダブルクォートで囲まれていない場合)
				result.add(r.trim());
			}
			
		}
		
		return result;
	}
	
}
