package org.dyndns.nuda.dynamic.compiler;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.tools.DiagnosticListener;
import javax.tools.JavaCompiler;
import javax.tools.JavaCompiler.CompilationTask;
import javax.tools.JavaFileObject;
import javax.tools.ToolProvider;

/**
 * Javaソースを動的コンパイルするためのクラスです
 * 
 * @author nkoseki
 * 
 */
public class CompileTask {
	protected DiagnosticListener<? super JavaFileObject>	listener				= new ErrorListener();
	
	protected JavaCompiler									compiler				= ToolProvider
																							.getSystemJavaCompiler();
	
	/**
	 * コンパイラ引数：クラスパス
	 */
	public static final String								ARGS_CLASSPATH			= "-classpath";
	
	/**
	 * システムプロパティキー：クラスパス
	 */
	public static final String								SYS_PROP_CLASS_PATH		= "java.class.path";
	
	/**
	 * システムプロパティキー：動的生成クラスパス
	 */
	public static final String								APP_PROP_CLASS_PATH		= "app.class.path";
	
	/**
	 * システムプロパティキー：動的生成ソースパス
	 */
	public static final String								APP_PROP_SOURCE_PATH	= "app.source.path";
	
	/**
	 * Javaソースのコンパイルを行います
	 * 
	 * @param <T>
	 * @param source
	 * @return コンパイル結果
	 */
	public <T> Class<T> compile(final SourceBean source) {
		JavaFileObject fo = new JavaSourceFromString(source);
		return this.compileInner(fo, source.getClassName(),
				source.getSourceCode());
	}
	
	/**
	 * Javaソースのコンパイルを行います
	 * 
	 * @param <T>
	 * @param resolver
	 * @return コンパイル結果
	 */
	public <T> Class<T> compile(final SourceResolver resolver) {
		SourceBean sourceBean = resolver.resolve();
		JavaFileObject fo = new JavaSourceFromString(sourceBean);
		
		return this.compileInner(fo, sourceBean.getClassName(),
				sourceBean.getSourceCode());
	}
	
	/**
	 * Javaソースのコンパイルを行います
	 * 
	 * @param <T>
	 * @param className
	 * @param sourceCode
	 * @return コンパイル結果
	 */
	public <T> Class<T> compile(final String className, final String sourceCode) {
		JavaFileObject fo = new JavaSourceFromString(className, sourceCode);
		return this.compileInner(fo, className, sourceCode);
	}
	
	private <T> Class<T> compileInner(final JavaFileObject fo,
			final String className, final String sourceCode) {
		List<JavaFileObject> compilationUnits = Arrays.asList(fo);
		
		List<String> options = Arrays.asList(
				ARGS_CLASSPATH,
				System.getProperty(SYS_PROP_CLASS_PATH) + ";"
						+ System.getProperty(APP_PROP_CLASS_PATH));
		
		ClassFileManager manager = new ClassFileManager(this.compiler,
				this.listener);
		ClassLoader cl = manager.getClassLoader(null);
		if (manager.exists(className)) {
			try {
				@SuppressWarnings("unchecked")
				Class<T> c = (Class<T>) cl.loadClass(className);
				return c;
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		} else {
			CompilationTask task = this.compiler.getTask(null, manager,
					this.listener, options, null, compilationUnits);
			
			// コンパイル実行
			boolean successCompile = task.call();
			
			if (!successCompile) {
				throw new RuntimeException("コンパイル失敗：" + className);
			} else {
				// System.out.println("コンパイル成功");
				// ソースフォルダにjavaファイルを出力
				//				System.out.println(className);
				
				String sourceRoot = System.getProperty(APP_PROP_SOURCE_PATH);
				
				if (sourceRoot != null && !sourceRoot.isEmpty()) {
					Pattern p = Pattern.compile(".+(\\..+)");
					Matcher m = p.matcher(className);
					if (m.find()) {
						String replacement = m.group(1);
						
						String packageName = className.replace(replacement, "");
						
						//						System.out.println(packageName);
						
						File sourceRootDir = new File(sourceRoot + "\\"
								+ packageName.replace(".", "\\"));
						
						if (!sourceRootDir.exists()) {
							sourceRootDir.mkdirs();
						}
						String fileName = replacement.replace(".", "")
								+ ".java";
						
						File outputFile = new File(sourceRoot + "\\"
								+ packageName.replace(".", "\\") + "\\"
								+ fileName);
						FileOutputStream fos = null;
						try {
							fos = new FileOutputStream(outputFile);
							fos.write(sourceCode.getBytes("UTF-8"));
						} catch (FileNotFoundException e) {
							// TODO 自動生成された catch ブロック
							e.printStackTrace();
						} catch (UnsupportedEncodingException e) {
							// TODO 自動生成された catch ブロック
							e.printStackTrace();
						} catch (IOException e) {
							// TODO 自動生成された catch ブロック
							e.printStackTrace();
						} finally {
							if (fos != null) {
								try {
									fos.close();
								} catch (IOException e) {
									// TODO 自動生成された catch ブロック
									e.printStackTrace();
								}
							}
						}
					}
				}
				
			}
			
			try {
				@SuppressWarnings("unchecked")
				Class<T> c = (Class<T>) cl.loadClass(className);
				return c;
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		}
		
	}
}
