#include "RtStk.h"
#include <algorithm>
#include <cmath>
#include "common.h"
#include "SynthFactory.h"

using namespace std;

// Tick Callback
int tick( void *outputBuffer, void *inputBuffer, unsigned int nBufferFrames,
          double streamTime, RtAudioStreamStatus status, void *dataPointer )
{
	RtStk *rt = (RtStk *) dataPointer;
	rt->rtTick(outputBuffer, inputBuffer, nBufferFrames, streamTime);
	return 0;
}

RtStk::RtStk(float sampleRate, int channels, float tempo) :
	metro(tempo, sampleRate, DEFAULT_DIAC),
	masterCount(0),
	sampleRate(sampleRate),
	channels(channels),
	volume(0.5),
	counter(0),
	peakCounter(0),
	haveMessage(false),
	done(false)
{
	printfunc
	Stk::setSampleRate(sampleRate);
	Stk::showWarnings(true);
	Stk::setRawwavePath(STK_RAWWAVE_PATH);
}

RtStk::~RtStk()
{
	printf("RtStk:: destructor\n");
}

int RtStk::openDac()
{
	printfunc
	RtAudioFormat format = ( sizeof(StkFloat) == 8 ) ? RTAUDIO_FLOAT64 : RTAUDIO_FLOAT32;
	RtAudio::StreamParameters parameters;
	parameters.deviceId = dac.getDefaultOutputDevice();
	parameters.nChannels = channels;
	bufferFrames = RT_BUFFER_SIZE;
//	unsigned int bufferFrames = RT_BUFFER_SIZE;
//	unsigned int bufferFrames = 4096;

	cout << "RtStk::openDac" << endl;
	cout << "- device id    : " << parameters.deviceId << endl;
	cout << "- sample rate  : " << Stk::sampleRate() << endl;
	cout << "- sample rate  : " << sampleRate << endl;
	cout << "- channels     : " << parameters.nChannels << endl;
	cout << "- bufferFrames : " << bufferFrames << endl;
	string format_str = (format == RTAUDIO_FLOAT64) ? "RTAUDIO_FLOAT64" : "RTAUDIO_FLOAT32";
	cout << "- format       : " << format_str << endl;
	cout << "- rawwave path : " << Stk::rawwavePath() << endl;
	
	// initialize FrameBuffer
	framePtr = new float*[channels];
	frameBuffer = new float*[channels];
	for(int i = 0 ; i < channels; i++) {
		frameBuffer[i] = new float[bufferFrames];
	}

	try {
		dac.openStream( &parameters, NULL, format, (unsigned int)Stk::sampleRate(), &bufferFrames, &tick, (void *) this );
	}
	catch ( RtError& error ) {
		error.printMessage();
		return -1;
	}

	mixer.init(channels, bufferFrames);

	return 0;
}

int RtStk::startDac()
{
	metro.start(messager, masterCount);

	try {
		dac.startStream();
	}
	catch ( RtError &error ) {
		error.printMessage();
		return -1;
	}
	return 0;
}

int RtStk::closeDac()
{
	try {
		dac.stopStream();
		dac.closeStream();
	}
	catch( RtError &error ) {
		error.printMessage();
		printf("RtSTK : DAC closed Error\n");
		return -1;
	}
	printf("RtSTK : DAC closed\n");
	return 0;
}

void RtStk::finish()
{
//	closeDac();

	//! ׂĂSynth폜
	for(int channel = 0; channel < NUM_SYNTH_CHANNEL; channel++) {
		Synth *synth = SynthFactory::getSynthAt(channel);
		if(synth) {
			cout << "delete synth : " << synth  << endl;
			delete synth;
		}
	}
}

int RtStk::rtTick( void *outputBuffer, void *inputBuffer,
				   unsigned int nBufferFrames, double streamTime)
{
	register StkFloat *samples = (StkFloat *) outputBuffer;
	int cnt, nTicks = (int) nBufferFrames;
	int frameCount = 0;


	while ( nTicks > 0 && !done ) {
		if ( !haveMessage ) {
			messager.popMessage( message );
			if ( message.type > 0 ) { // no empty
#ifdef __RTSTK_MSGTIME_INT__
				long msgCount = message.intValues[SKINI_MSG_INT_TICKTIME];
#else
				long msgCount = (long)(message.time * getSamplingRate());
#endif
				long relativeMsgCount = msgCount - masterCount;
#ifdef DEBUG
				cout << "relativeMsgCount" << relativeMsgCount << endl;
#endif
				if(relativeMsgCount >= 0) {
					counter = relativeMsgCount;
					haveMessage = true;
#ifdef DEBUG
					rtcout << "=====message " << relativeMsgCount << "  type:" << message.type << endl;
#endif
				} else {
//					counter = DELTA_CONTROL_TICKS;
					counter = 0;
					haveMessage = true;
#ifdef DEBUG
					rtcout << "=====old message " << relativeMsgCount << "  type:" << message.type << endl;
#endif
				}
			} else {
				counter = DELTA_CONTROL_TICKS;
			}
		}

		cnt = std::min( nTicks, counter );

		if(cnt) {
			counter -= cnt;
			masterCount += cnt;

			for(int i = 0; i < channels; i++) {
				framePtr[i] = frameBuffer[i] + frameCount;
			}
			mixer.tick(framePtr, channels, cnt);
			nTicks -= cnt;
			frameCount += cnt;
		}

		if ( nTicks == 0 ) break;

		// Process control messages.
		if ( haveMessage ) {
			processMessage();
		}
	}

	// interleave
	for(unsigned int i = 0; i < nBufferFrames; i++) {
		*samples++ = frameBuffer[0][i] * volume;
		*samples++ = frameBuffer[1][i] * volume;
	}

	return 0;
}

// The processMessage() function encapsulates the handling of control
// messages.  It can be easily relocated within a program structure
// depending on the desired scheduling scheme.
void RtStk::processMessage()
{
//	printfunc

	// If only one instrument, allow messages from all channels to control it.
	int channel = message.channel;

	switch( message.type ) {
	case __SK_Exit_:
		done = true;
		return;
	case __SK_Metro_:
		metro.beat(messager, masterCount);

		if(peakCounter != 0) {
			printf("peak count! : %d", peakCounter);
			peakCounter = 0;
		}
		break;
	case __SK_ProgramChange_:
		break;
	default:
		if(channel < NUM_SYNTH_CHANNEL) {
			Synth *synth = SynthFactory::getSynthAt(channel);
			cout << "synth message : channel " << channel << "  synth " << synth  << endl;
			if(synth) {
				synth->processMessage(message);
			}
		}
		break;
	} // end of switch

	haveMessage = false;
	return;
}

