/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file FacebookCommunication.java
 *
 * @author
 */

package com.sonyericsson.eventstream.facebookplugin;

import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.provider.ContactsContract;
import android.provider.ContactsContract.RawContacts;
import android.util.Log;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.ConfigState;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.EventTable;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.FriendTable;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.PluginTable;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.SourceTable;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.StatusSupport;
import com.sonyericsson.eventstream.facebookplugin.Facebook.EventStorage;
import com.sonyericsson.eventstream.facebookplugin.Facebook.FriendStorage;

import static com.sonyericsson.eventstream.facebookplugin.Constants.LOG_TAG;
import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.EVENT_PROVIDER_URI;
import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.FRIEND_PROVIDER_URI;
import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.PLUGIN_PROVIDER_URI;
import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.SOURCE_PROVIDER_URI;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Map.Entry;

public class Database implements FriendStorage, EventStorage {
    private Context mContext;

    private ContentResolver mCr;

    /**
     * List of new events to store
     */
    private List<ContentValues> mEventContentValueList = new ArrayList<ContentValues>();

    /**
     * List of new friends to store
     */
    private List<ContentValues> mFriendContentValueList = new ArrayList<ContentValues>();

    /**
     * This is a cache, containing friend-keys.
     */
    private Set<String> mRegisteredFriends = null;

    private long mSourceId = -1;

    private static final int BULK_INSERT_MAX_COUNT = 50;

    private static final int BULK_INSERT_DELAY = 20; //ms

    /**
     * The event_key field in the event table is used differently for the
     * events that we support. The reason for this is that we need more data
     * than the Facebook object ID to handle deep linking.
     *
     * LINK
     * <facebook friend ID>_<facebook object ID>#<URL encoded link address>;LINK
     *
     * STATUS
     * <facebook friend ID>_<facebook object ID>;STATUS
     *
     * PHOTO
     * <facebook friend ID>_<facebook object ID>;PHOTO
     *
     * MESSAGE
     * <facebook friend ID>_<facebook object ID>;MESSAGE
     */
    private static final String EVENT_ID_DELIMITER = ";";

    private static final String EVENT_TYPE_LINK_AS_STRING = "LINK";

    private static final String EVENT_TYPE_STATUS_AS_STRING = "STATUS";

    private static final String EVENT_TYPE_PHOTO_AS_STRING = "PHOTO";

    private static final String EVENT_TYPE_MESSAGE_AS_STRING = "MESSAGE";
    private static final String FRIEND_ID_FACBOOK_OBJECT_ID_DELIMITER = "_";

    public Database(Context context) {
        mContext = context;
        mCr = context.getContentResolver();
    }

    public boolean isRegistered() {
        Cursor cursor = null;
        boolean result = false;

        try {
            cursor = mCr.query(PLUGIN_PROVIDER_URI, null, null, null, null);
            result = (cursor != null && cursor.moveToFirst());
        } catch (Exception exception) {
            result = false;
        } finally {
            if (cursor != null) {
                cursor.close();
                cursor = null;
            }
        }

        return result;
    }

    public void clearData() {
        // Clear friend
        mCr.delete(FRIEND_PROVIDER_URI, null, null);

        // Clear event
        mCr.delete(EVENT_PROVIDER_URI, null, null);

        // Reset source
        ContentValues sourceValues = new ContentValues();

        sourceValues.put(SourceTable.ENABLED, 1);
        sourceValues.putNull(SourceTable.CURRENT_STATUS);
        sourceValues.putNull(SourceTable.STATUS_TIMESTAMP);
        mCr.update(SOURCE_PROVIDER_URI, sourceValues, null, null);

        // Reset plugin
        ContentValues pluginValues = new ContentValues();

        pluginValues.put(PluginTable.CONFIGURATION_STATE, ConfigState.NOT_CONFIGURED);
        pluginValues.put(PluginTable.CONFIGURATION_TEXT, mContext.getResources().getString(
                R.string.facebook_register_txt));
        mCr.update(PLUGIN_PROVIDER_URI, pluginValues, null, null);
    }

    public long registerPlugin() {
        Settings settings = new Settings(mContext);
        ContentValues pluginValues = new ContentValues();
        long sourceId = 0;
        pluginValues.put(PluginTable.PLUGIN_KEY, Constants.PLUGIN_KEY);
        pluginValues.put(PluginTable.NAME, mContext.getResources().getString(R.string.ts_facebook_service_name));
        pluginValues.put(PluginTable.STATUS_TEXT_MAX_LENGTH, Constants.STATUS_TEXT_MAX_LENGTH);
        pluginValues.put(PluginTable.ICON_URI,
                ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + mContext.getPackageName() + "/" + R.drawable.facebook_source_icn);
        pluginValues.put(PluginTable.API_VERSION, 1);
        ComponentName componentName = new ComponentName(mContext, FacebookPluginConfig.class);
        pluginValues.put(PluginTable.CONFIGURATION_ACTIVITY, componentName.flattenToString());
        pluginValues.put(PluginTable.STATUS_SUPPORT, StatusSupport.HAS_SUPPORT_TRUE);
        ContentValues sourceValues = new ContentValues();
        sourceValues.put(SourceTable.NAME, mContext.getResources().getString(
                R.string.ts_facebook_service_name));
        sourceValues.put(SourceTable.ICON_URI, ContentResolver.SCHEME_ANDROID_RESOURCE + "://"
                + mContext.getPackageName() + "/"
                + R.drawable.facebook_source_icn);

        Cursor pluginCursor = null;
        try {
            // Register or update plugin table
            ContentResolver contentResolver = mContext.getContentResolver();

            pluginCursor = contentResolver.query(PLUGIN_PROVIDER_URI, null, null, null, null);
            if (pluginCursor != null && pluginCursor.moveToFirst()) {
                int configState = pluginCursor.getInt(pluginCursor
                        .getColumnIndex(PluginTable.CONFIGURATION_STATE));
                if (configState == ConfigState.NOT_CONFIGURED) {
                    pluginValues.put(PluginTable.CONFIGURATION_TEXT, mContext.getResources()
                            .getString(R.string.facebook_register_txt));
                } else {
                    String displayName = settings.getDisplayName();
                    pluginValues.put(PluginTable.CONFIGURATION_TEXT, mContext
                            .getString(R.string.ts_facebook_logout_label)
                            + " " + displayName);
                }
                contentResolver.update(PLUGIN_PROVIDER_URI, pluginValues, null, null);
            } else {
                pluginValues.put(PluginTable.CONFIGURATION_TEXT, mContext.getResources().getString(
                        R.string.facebook_register_txt));
                pluginValues.put(PluginTable.CONFIGURATION_STATE, ConfigState.NOT_CONFIGURED);

                contentResolver.insert(PLUGIN_PROVIDER_URI, pluginValues);
            }
            // Register or update source table
            Cursor sourceCursor = null;
            try {
                sourceCursor = contentResolver.query(SOURCE_PROVIDER_URI, null, null, null, null);
                if (sourceCursor != null && sourceCursor.moveToFirst()) {
                    // We should only have a single source
                    sourceId = sourceCursor.getInt(sourceCursor
                            .getColumnIndexOrThrow(SourceTable.ID_COLUMN));
                    contentResolver.update(SOURCE_PROVIDER_URI, sourceValues, SourceTable.ID_COLUMN
                            + " = ?", new String[] {
                        String.valueOf(sourceId)
                    });
                } else {
                    sourceValues.put(SourceTable.ENABLED, 1);
                    Uri uri = mContext.getContentResolver().insert(SOURCE_PROVIDER_URI,
                            sourceValues);
                    sourceId = ContentUris.parseId(uri);
                }
            } finally {
                if (sourceCursor != null) {
                    sourceCursor.close();
                }
            }
        } finally {
            if (pluginCursor != null) {
                pluginCursor.close();
            }
        }

        mSourceId = sourceId;
        return sourceId;
    }

    public void setOwnStatus(String message, long timestamp) {
        ContentValues values = new ContentValues();
        values.put(SourceTable.CURRENT_STATUS, message);
        values.put(SourceTable.STATUS_TIMESTAMP, timestamp);
        mContext.getContentResolver().update(SOURCE_PROVIDER_URI, values, null, null);
    }

    public static String getEventId(String databaseEventId) {
        String result = databaseEventId;
        int index = databaseEventId.lastIndexOf(EVENT_ID_DELIMITER);

        if (index != -1) {
            result = databaseEventId.substring(0, index);
        }
        return result;
    }

    public static EventType getEventType(String databaseEventId) {
        String value = null;
        int index = databaseEventId.lastIndexOf(EVENT_ID_DELIMITER);
        EventType eventType = EventType.LINK_EVENT;

        if (index != -1) {
            value = databaseEventId.substring(index + 1);
        }
        if (value != null) {
            if (EVENT_TYPE_LINK_AS_STRING.equals(value)) {
                eventType = EventType.LINK_EVENT;
            } else if (EVENT_TYPE_STATUS_AS_STRING.equals(value)) {
                eventType = EventType.STATUS_EVENT;
            } else if (EVENT_TYPE_PHOTO_AS_STRING.equals(value)) {
                eventType = EventType.PHOTO_EVENT;
            } else if (EVENT_TYPE_MESSAGE_AS_STRING.equals(value)) {
                eventType = EventType.MESSAGE_EVENT;
            }
        }

        return eventType;
    }

    public void addEvent(String eventId, String message, String title, String picture,
            String friendId, long timestamp, EventType eventType) {
        ContentValues eventValues = new ContentValues();
        String databaseEventId = generateEventId(eventId, eventType);

        eventValues.put(EventTable.EVENT_KEY, databaseEventId);
        eventValues.put(EventTable.SOURCE_ID, getSourceId());
        if (message != null) {
            eventValues.put(EventTable.MESSAGE, message);
        }
        if (picture != null) {
            eventValues.put(EventTable.IMAGE_URI, picture);
        }
        eventValues.put(EventTable.PUBLISHED_TIME, timestamp);
        if (title != null) {
            eventValues.put(EventTable.TITLE, title);
        }
        if (friendId != null) {
            eventValues.put(EventTable.FRIEND_KEY, friendId);
        }
        eventValues.put(EventTable.OUTGOING, 0);
        eventValues.put(EventTable.PERSONAL, 0);
        mEventContentValueList.add(eventValues);
    }

    public int storeEvents() {
        int inserted = 0;

        List<ContentValues> bulkValues = new ArrayList<ContentValues>();
        for (int i = 0; i < mEventContentValueList.size(); i++) {
            bulkValues.add(mEventContentValueList.get(i));

            if (bulkValues.size() >= BULK_INSERT_MAX_COUNT) {
                inserted += mCr.bulkInsert(EVENT_PROVIDER_URI,
                        (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
                bulkValues.clear();
                // Give eventstream some time to run queries
                try {
                    Thread.sleep(BULK_INSERT_DELAY);
                } catch (InterruptedException e) {
                    // Do nothing
                }
            }
        }

        if (bulkValues.size() > 0) {
            inserted += mCr.bulkInsert(EVENT_PROVIDER_URI,
                    (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
        }

        if (Config.DEBUG) {
            Log.d(LOG_TAG, "Inserted " + inserted + " Facebook statuses.");
        }

        mEventContentValueList.clear();

        return inserted;
    }

    public void addFriend(String friendId, String profileImageUri, String friendName) {
        ContentValues cv = new ContentValues();
        Set<String> myFriends = getRegisteredFriends();

        if (myFriends != null) {
            if (!myFriends.contains(friendId)) {
                cv.put(FriendTable.SOURCE_ID, getSourceId());
                cv.put(FriendTable.PROFILE_IMAGE_URI, profileImageUri);
                cv.put(FriendTable.FRIEND_KEY, friendId);
                cv.put(FriendTable.DISPLAY_NAME, friendName);
                mFriendContentValueList.add(cv);
                myFriends.add(friendId);
            } else {
                mCr.update(FRIEND_PROVIDER_URI, cv, FriendTable.FRIEND_KEY
                        + " = ? AND " + FriendTable.SOURCE_ID + " = ?", new String[] {
                        friendId, String.valueOf(getSourceId())
                });
            }
        }
    }

    public Set<String> getFriendIds() {
        Set<String> source = getRegisteredFriends();
        Set<String> copy = new HashSet<String>();

        if (source != null && (! source.isEmpty())) {
            copy.addAll(source);
        }

        return copy;
    }

    public boolean isFriend(String friendId) {
        Set<String> myFriends = getRegisteredFriends();
        if (myFriends != null) {
            return myFriends.contains(friendId);
        } else {
            return false;
        }
    }


    public int storeFriends() {
        int inserted = 0;

        List<ContentValues> bulkValues = new ArrayList<ContentValues>();
        for (int i = 0; i < mFriendContentValueList.size(); i++) {
            bulkValues.add(mFriendContentValueList.get(i));

            if (bulkValues.size() >= BULK_INSERT_MAX_COUNT) {
                inserted += mCr.bulkInsert(FRIEND_PROVIDER_URI,
                        (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
                bulkValues.clear();
                // Give eventstream some time to run queries
                try {
                    Thread.sleep(BULK_INSERT_DELAY);
                } catch (InterruptedException e) {
                    // Do nothing
                }
            }
        }

        if (bulkValues.size() > 0) {
            inserted += mCr.bulkInsert(FRIEND_PROVIDER_URI,
                    (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
        }

        if (Config.DEBUG) {
            Log.d(LOG_TAG, "Inserted " + inserted + " Facebook friends.");
        }

        mFriendContentValueList.clear();

        return inserted;
    }

    /**
     * Remove a friend using the facebook ID
     *
     * @param friendId the facebook friend ID
     */
    public void removeFriend(String friendId) {
        // Delete the friend
        mCr.delete(FRIEND_PROVIDER_URI, FriendTable.FRIEND_KEY + "=?", new String[] {
            friendId
        });

        // Delete all events for the friend
        mCr.delete(EVENT_PROVIDER_URI, EventTable.FRIEND_KEY + "=?", new String[] {
            friendId
        });

        // Delete in cache
        removeRegisterFriend(friendId);
    }

    /**
     * Set configuration text for the plugin (displayed in service list)
     *
     * @param configText the new configuration text
     * @return true if the update succeeded
     */
    public boolean setConfigurationText(String configText) {
        ContentValues values = new ContentValues();
        int updated = 0;

        try {
            values.put(PluginTable.CONFIGURATION_TEXT, configText);
            updated = mCr.update(PLUGIN_PROVIDER_URI, values, null, null);
        } catch (Exception exception) {
            // Do nothing, this is most likely caused by the fact that we are
            // not registerd
        }

        return (updated == 1);
    }

    public void setConfigurationState(int state) {
        ContentValues values = new ContentValues();

        try {
            values.put(EventStreamConstants.PluginTable.CONFIGURATION_STATE, state);
            mCr.update(EventStreamConstants.PLUGIN_PROVIDER_URI, values, null, null);
        } catch (Exception exception) {
            // Do nothing, this is most likely caused by the fact that we are
            // not registerd
        }
    }

    private Set<String> getRegisteredFriendIds(long sourceId) {
        Set<String> registeredFriends = new HashSet<String>();
        Cursor friendsCursor = null;

        try {
            friendsCursor = mCr.query(FRIEND_PROVIDER_URI, null, FriendTable.SOURCE_ID + " = ?",
                    new String[] {
                        String.valueOf(sourceId)
                    }, null);
            while (friendsCursor != null && friendsCursor.moveToNext()) {
                registeredFriends.add(friendsCursor.getString(friendsCursor
                        .getColumnIndexOrThrow(FriendTable.FRIEND_KEY)));
            }
        } finally {
            if (friendsCursor != null) {
                friendsCursor.close();
            }
        }
        return registeredFriends;
    }

    private long getSourceId() {
        long result = mSourceId;
        if (result == -1) {
            Cursor c = null;
            try {
                c = mCr.query(SOURCE_PROVIDER_URI, null, null, null, null);
                if (c != null && c.moveToFirst()) {
                    result = c.getLong(c.getColumnIndex(SourceTable.ID_COLUMN));
                }
            } finally {
                if (c != null) {
                    c.close();
                }
            }
        }
        return result;
    }

    /**
     * Auto link friends to contacts by looking at the Facebook raw contacts
     */
    public void autoLinkFriends() {
        Cursor c = null;
        HashMap<String, String> lookupMap = new HashMap<String, String>();
        Set<String> myFriends = getRegisteredFriends();

        try {
            c = mCr.query(RawContacts.CONTENT_URI, new String[] {
                    RawContacts._ID, RawContacts.SOURCE_ID
            }, RawContacts.ACCOUNT_TYPE + "=\"com.facebook.auth.login\"" + " AND "
                    + RawContacts.DELETED + "= 0", null, null);
            if (c != null) {
                while (c.moveToNext()) {
                    String uid = c.getString(c.getColumnIndex(RawContacts.SOURCE_ID));

                    if (Config.DEBUG) {
                        Log.d(LOG_TAG, "Trying to autolink " + uid != null ? uid : "null");
                    }
                    if (uid != null && myFriends != null && myFriends.contains(uid)) {
                        Long rawContactId = c.getLong(c.getColumnIndex(RawContacts._ID));

                        if (rawContactId != null) {
                            if (Config.DEBUG) {
                                Log.d(LOG_TAG, "Match found in " + rawContactId);
                            }
                            Uri rawContactUri = Uri.withAppendedPath(
                                    ContactsContract.RawContacts.CONTENT_URI, String
                                            .valueOf(rawContactId));

                            lookupMap.put(uid, rawContactUri.toString());
                        }
                    }
                }
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }

        for (Entry<String, String> entry : lookupMap.entrySet()) {
            ContentValues values = new ContentValues();

            values.put(FriendTable.CONTACTS_REFERENCE, entry.getValue());
            if (mCr.update(FRIEND_PROVIDER_URI, values, FriendTable.FRIEND_KEY + " = " + "'"
                    + entry.getKey() + "'", null) > 0) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Added link (" + entry.getValue() + ") to friend "
                            + entry.getKey());
                }
            } else if (Config.DEBUG) {
                Log.d(LOG_TAG, "Failed to update " + entry.getKey());
            }
        }
    }

    private String generateEventId(String facebookEventId, EventType eventType) {
        String eventTypeString = null;

        switch (eventType) {
            case LINK_EVENT:
                eventTypeString = EVENT_TYPE_LINK_AS_STRING;
                break;
            case STATUS_EVENT:
                eventTypeString = EVENT_TYPE_STATUS_AS_STRING;
                break;
            case PHOTO_EVENT:
                eventTypeString = EVENT_TYPE_PHOTO_AS_STRING;
                break;
            case MESSAGE_EVENT:
                eventTypeString = EVENT_TYPE_MESSAGE_AS_STRING;
                break;
            default:
                break;
        }
        if (eventTypeString == null) {
            return null;
        } else {
            return facebookEventId + EVENT_ID_DELIMITER + eventTypeString;
        }
    }

    public static String extractFriendKeyFromEventId(String facebookEventId) {
        String result = null;

        if (facebookEventId != null) {
            int index = facebookEventId.indexOf(FRIEND_ID_FACBOOK_OBJECT_ID_DELIMITER);

            if (index != -1) {
                result = facebookEventId.substring(0, index);
            }
        }

        return result;
    }

    private Set<String> getRegisteredFriends() {
        if (mRegisteredFriends == null) {
            long sourceId = getSourceId();

            mRegisteredFriends = getRegisteredFriendIds(sourceId);
        }
        return mRegisteredFriends;
    }

    private void removeRegisterFriend(String id) {
        if (mRegisteredFriends != null && id != null) {
            try {
                mRegisteredFriends.remove(id);
            } catch (Exception exception) {

            }
        }
    }
}
