/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin;

import android.content.Context;
import android.test.InstrumentationTestCase;

import com.sonyericsson.eventstream.facebookplugin.Facebook.FacebookException;
import com.sonyericsson.eventstream.facebookplugin.Facebook.ServiceState;
import com.sonyericsson.eventstream.facebookplugin.util.DatabaseHelper;
import com.sonyericsson.eventstream.facebookplugin.util.FileReader;
import com.sonyericsson.eventstream.facebookplugin.util.MockCommunication;
import com.sonyericsson.eventstream.facebookplugin.util.MockFacebookSecurity;
import com.sonyericsson.eventstream.facebookplugin.util.ReflectionUtilities;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;

/**
 *  Tests the facebook implementation
 */
public class UFacebookPluginTestFacebook extends InstrumentationTestCase {
    private Settings mSetting = null;
    private MockFacebookSecurity mSecurityMock = new MockFacebookSecurity();

    static final private String AUTH_TOKEN = "3333333333333";
    static final private String MY_DISPLAY_NAME = "kalle anka";
    static final private String MY_FACEBOOK_ID = "00000000";
    static private final String KAJSA_ID = "111111111"; //< Don't change this...
    static private final String JOAKIM_ID = "2222222";  //< Don't change this...
    static private final String LUKAS_ID = "333333333"; //< Don't change this...
    static private final String SESSION_KEY = "1111111-00000000";
    static final private String ME_PROFILE =
        "{\"id\":\"00000000\"," +
        "\"name\":\"" + MY_DISPLAY_NAME + "\",\"first_name\":\"kalle\",\"last_name\":\"anka\"," +
        "\"link\":\"http:\\/\\/www.facebook.com\\/profile.php?id=" + MY_FACEBOOK_ID + "\"," +
        "\"birthday\":\"01\\/01\\/2010\"," +
        "\"location\":{\"id\":\"\",\"name\":null}," +
        "\"gender\":\"man\"," +
        "\"timezone\":2," +
        "\"locale\":\"sv_SE\"," +
        "\"updated_time\":\"2000-01-01T00:00:01+0000\"}";

    static private final String AUTH_CONTENT =
        "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" +
        "<auth_login_response xmlns=\"http://api.facebook.com/1.0/\" " +
            "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" " +
            "xsi:schemaLocation=\"http://api.facebook.com/1.0/ http://api.facebook.com/1.0/facebook.xsd\">" +
            "<session_key>" + SESSION_KEY + "</session_key>" +
            "<uid>" + MY_FACEBOOK_ID + "</uid>" +
            "<secret>22222222222222</secret>" +
            "<access_token>" + AUTH_TOKEN + "</access_token>" +
        "</auth_login_response>";

    static private final String FRIENDS =
        "{\"data\":" +
            "[{\"name\":\"Kajsa anka\",\"id\":\"" + KAJSA_ID + "\"}," +
            "{\"name\":\"Farbror Joakim\",\"id\":\"" + JOAKIM_ID + "\"}," +
            "{\"name\":\"Alexander Lukas\",\"id\":\"" + LUKAS_ID + "\"}]}";

    static private final String EVENTS =
        "{\"data\":[" +
        "{\"id\":\"111111111_77777777777\"," +
        "\"from\":{\"name\":\"kajsa anka\",\"id\":\"111111111\"}," +
        "\"message\":\"test2\"," +
        "\"actions\":[{\"name\":\"Comment\",\"link\":\"http:\\/\\/www.facebook.com\\/111111111\\/posts\\/77777777777\"}," +
              "{\"name\":\"Like\",\"link\":\"http:\\/\\/www.facebook.com\\/111111111\\/posts\\/77777777777\"}]," +
        "\"type\":\"status\"," +
        "\"created_time\":\"2010-09-01T08:00:00+0000\"," +
        "\"updated_time\":\"2010-09-01T08:00:00+0000\"}," +
        "{\"id\":\"2222222_88888888888\"," +
        "\"from\":{\"name\":\"Farbror Joakim\",\"id\":\"2222222\"}," +
        "\"message\":\"Testing\"," +
        "\"actions\":[{\"name\":\"Comment\",\"link\":\"http:\\/\\/www.facebook.com\\/2222222\\/posts\\/88888888888\"}," +
                     "{\"name\":\"Like\",\"link\":\"http:\\/\\/www.facebook.com\\/2222222\\/posts\\/88888888888\"}]," +
        "\"type\":\"status\"," +
        "\"created_time\":\"2010-09-21T11:16:18+0000\"," +
        "\"updated_time\":\"2010-09-22T05:36:02+0000\"," +
        "\"likes\":2," +
        "\"comments\":{\"data\":[{\"id\":\"2222222_88888888888_121212121\", " +
                       "\"from\":{\"name\":\"Alexander Lucas\",\"id\":\"333333333\"}," +
                       "\"message\":\"Test verified.\"," +
                       "\"created_time\":\"2010-09-21T20:01:15+0000\"}]," +
               "\"count\":5}" + "}]}";

    /**
     * Event list containing photo type event with no picture as the second out
     * of three events
     */
    static private final String EVENTS_WITH_TYPE_PHOTO_EVENT_NO_PICTURE = "{\"data\":["
            + "{\"id\":\"111111111_77777777777\","
            + "\"from\":{\"name\":\"kajsa anka\",\"id\":\"111111111\"},"
            + "\"message\":\"test2\","
            + "\"actions\":[{\"name\":\"Comment\",\"link\":\"http:\\/\\/www.facebook.com\\/111111111\\/posts\\/77777777777\"},"
            + "{\"name\":\"Like\",\"link\":\"http:\\/\\/www.facebook.com\\/111111111\\/posts\\/77777777777\"}],"
            + "\"type\":\"status\","
            + "\"created_time\":\"2010-09-01T08:00:00+0000\","
            + "\"updated_time\":\"2010-09-01T08:00:00+0000\"},"
            + "{\"id\":\"2323232_78787878787\",\"icon\":\"http:\\/\\/static.ak.fbcdn.net\\/111.gif\""
            + ",\"name\":\"den 28 december 2010\",\"link\":\"http:\\/\\/www.facebook.com\\/222.php\""
            + ",\"properties\":[{\"href\":\"http:\\/\\/www.facebook.com\\/profile.php?id=1234\",\"text\":\"Johan Johnsson\",\"name\":\"By\"}"
            + ",{\"text\":\"3\",\"name\":\"Photos\"}],\"from\":{\"id\":\"123456789\",\"name\":\"Johan Johnsson\"},\"created_time\":\"2010-12-28T21:14:31+0000\""
            + ",\"type\":\"photo\",\"updated_time\":\"2010-12-28T21:14:31+0000\",\"actions\":[{\"link\":\"http:\\/\\/www.facebook.com\\/123456789\\/posts\\/7878787878\""
            + ",\"name\":\"Comment\"},{\"link\":\"http:\\/\\/www.facebook.com\\/123456789\\/posts\\/78787878787878\",\"name\":\"Like\"}]},"
            + "{\"id\":\"2222222_88888888888\","
            + "\"from\":{\"name\":\"Farbror Joakim\",\"id\":\"2222222\"},"
            + "\"message\":\"Testing\","
            + "\"actions\":[{\"name\":\"Comment\",\"link\":\"http:\\/\\/www.facebook.com\\/2222222\\/posts\\/88888888888\"},"
            + "{\"name\":\"Like\",\"link\":\"http:\\/\\/www.facebook.com\\/2222222\\/posts\\/88888888888\"}],"
            + "\"type\":\"status\"," + "\"created_time\":\"2010-09-21T11:16:18+0000\","
            + "\"updated_time\":\"2010-09-22T05:36:02+0000\"," + "\"likes\":2,"
            + "\"comments\":{\"data\":[{\"id\":\"2222222_88888888888_121212121\", "
            + "\"from\":{\"name\":\"Alexander Lucas\",\"id\":\"333333333\"},"
            + "\"message\":\"Test verified.\","
            + "\"created_time\":\"2010-09-21T20:01:15+0000\"}]," + "\"count\":5}" + "}]}";

    static private final String STATUS_UPDATE = "{\n   \"data\": [\n      {\n         \"id\": \"100000669254664_154309147934726\",\n         \"from\": {\n            \"name\": \"Erik T\\u00e9stman\",\n            \"id\": \"00000000\"\n         },\n         \"to\": {\n            \"data\": [\n               {\n                  \"name\": \"Robert Testman\",\n                  \"id\": \"100000669254664\"\n               }\n            ]\n         },\n         \"message\": \"Writing something on your wall Dude!\",\n         \"actions\": [\n            {\n               \"name\": \"Comment\",\n               \"link\": \"http://www.facebook.com/100000669254664/posts/154309147934726\"\n            },\n            {\n               \"name\": \"Like\",\n               \"link\": \"http://www.facebook.com/100000669254664/posts/154309147934726\"\n            }\n         ],\n         \"type\": \"status\",\n         \"created_time\": \"2010-09-28T12:49:14+0000\",\n         \"updated_time\": \"2010-09-28T12:49:14+0000\"\n      },\n      {\n         \"id\": \"100000623654571_156608927696030\",\n         \"from\": {\n            \"name\": \"Erik T\\u00e9stman\",\n            \"id\": \"00000000\"\n         },\n         \"message\": \"New status!\",\n         \"actions\": [\n            {\n               \"name\": \"Comment\",\n               \"link\": \"http://www.facebook.com/100000623654571/posts/156608927696030\"\n            },\n            {\n               \"name\": \"Like\",\n               \"link\": \"http://www.facebook.com/100000623654571/posts/156608927696030\"\n            }\n         ],\n         \"privacy\": {\n            \"description\": \"Everyone\",\n            \"value\": \"EVERYONE\"\n         },\n         \"type\": \"status\",\n         \"created_time\": \"2010-09-28T12:47:44+0000\",\n         \"updated_time\": \"2010-09-28T12:47:44+0000\"\n      }\n   ]\n}";

    static private final String STATUSES_FEED = "{\n   \"data\": [\n      {\n         \"id\": \"100000623654571_156608927696030\",\n         \"from\": {\n            \"name\": \"Erik T\\u00e9stman\",\n            \"id\": \"00000000\"\n         },\n         \"message\": \"New status!\",\n         \"actions\": [\n            {\n               \"name\": \"Comment\",\n               \"link\": \"http://www.facebook.com/100000623654571/posts/156608927696030\"\n            },\n            {\n               \"name\": \"Like\",\n               \"link\": \"http://www.facebook.com/100000623654571/posts/156608927696030\"\n            }\n         ],\n         \"privacy\": {\n            \"description\": \"Everyone\",\n            \"value\": \"EVERYONE\"\n         },\n         \"type\": \"status\",\n         \"created_time\": \"2010-09-28T12:47:44+0000\",\n         \"updated_time\": \"2010-09-28T12:47:44+0000\"\n      }\n   ]\n}";

    @Override
    protected void setUp() throws Exception {
        super.setUp();

        if (mSetting == null) {
            mSetting = new Settings(getInstrumentation().getTargetContext());
        }

        // Clear settings and database...
        mSetting.removeSettings();
        DatabaseHelper.removePluginFromEventStream(getInstrumentation().getTargetContext());

        // Register...
        DatabaseHelper.registerPlugin(getInstrumentation().getTargetContext());
    }

    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    private void initializeTest(Facebook facebook, MockCommunication mock) {
        FacebookFactory.terminate(false);

        try {
            Field field = ReflectionUtilities.getField(facebook, "mCommunication");

            field.set(facebook, mock);
            field = ReflectionUtilities.getField(FacebookFactory.class, "sFacebookInstance");
            field.set(new Object(), facebook);
        } catch (Exception e) {
            e.printStackTrace();
            fail("Couldn't set mock class!");
        }

        try {
            Field field = ReflectionUtilities.getField(FacebookFactory.class, "sFacebookSecurityInstance");

            field.set(new Object(), mSecurityMock);
        } catch (Exception e) {
            e.printStackTrace();
            fail("Couldn't set mock class!");
        }
    }

    /**
     * Test that simulates a successful authentication
     */
    public void testAuthentication() {
        String displayName = null;
        MockCommunication mock = new MockCommunication(new int[]{200, 200},
                                                       new String[]{"Ok", "Ok"},
                                                       new String[]{AUTH_CONTENT, ME_PROFILE});
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            displayName = facebook.authenticate("test@test", "PWD");
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }

        // Ensure that the token is stored...
        String token = mSetting.getAuthenticationToken();
        assertEquals("Authentication failed", AUTH_TOKEN, token);

        // Ensure the session key is stored...
        String sessionToken = mSetting.getSessionKey();
        assertEquals("Session key failed", SESSION_KEY, sessionToken);

        // Ensure that the display-name was properly parsed...
        assertEquals("Display name wasn't properly returned", MY_DISPLAY_NAME, displayName);
    }

    /**
     * Test that simulates authentication failure due to no network
     */
    public void testAuthenticationNoNetwork() {
        boolean gotException = false;
        String displayName = null;
        MockCommunication mock = new MockCommunication(new IOException());
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            displayName = facebook.authenticate("test@test", "PWD");
        } catch (FacebookException exception) {
            // Should get an exception!
            gotException = true;
        }

        // Ensure that auth-token doesn't exist...
        String token = mSetting.getAuthenticationToken();

        assertNull("Authentication string not null", token);
        assertNull("Display name isn't null", displayName);
        assertTrue("We didn't get an exception", gotException);
    }

    /**
     * Test that simulates authentication failure due to null
     */
    public void testAuthenticationNull() {
        boolean gotException = false;
        String displayName = null;
        MockCommunication mock = new MockCommunication(new IOException());
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            displayName = facebook.authenticate(null, null);
        } catch (FacebookException exception) {
            // Should get an exception!
            gotException = true;
        }

        // Ensure that auth-token doesn't exist...
        String token = mSetting.getAuthenticationToken();

        assertNull("Authentication string not null", token);
        assertNull("Display name isn't null", displayName);
        assertTrue("We didn't get an exception", gotException);
    }

    /**
     * Test that simulates authentication failure due to empty strings
     */
    public void testAuthenticationNoDataEntered() {
        boolean gotException = false;
        String displayName = null;
        MockCommunication mock = new MockCommunication(new IOException());
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            displayName = facebook.authenticate("", "");
        } catch (FacebookException exception) {
            // Should get an exception!
            gotException = true;
        }

        // Ensure that auth-token doesn't exist...
        String token = mSetting.getAuthenticationToken();

        assertNull("Authentication string not null", token);
        assertNull("Display name isn't null", displayName);
        assertTrue("We didn't get an exception", gotException);
    }

    /**
     * Test that simulates a 500 Internal server error
     */
    public void testAuthenticationFailure() {
        String displayName = null;
        boolean gotException = false;
        MockCommunication mock = new MockCommunication(500, "Internal server error", "");
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            displayName = facebook.authenticate("test@test", "PWD");
        } catch (FacebookException exception) {
            // Should get an exception!
            gotException = true;
        }

        // Ensure that auth-token doesn't exist...
        String token = mSetting.getAuthenticationToken();

        assertNull("Authentication string not null", token);
        assertNull("Display name isn't null", displayName);
        assertTrue("We didn't get an exception", gotException);
    }

    public void testAuthenticationStrangePassword() {
        String displayName = null;
        MockCommunication mock = new MockCommunication(new int[]{200, 200},
                                                       new String[]{"Ok", "Ok"},
                                                       new String[]{AUTH_CONTENT, ME_PROFILE});
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            displayName = facebook.authenticate("test@test", "PWD åäö () !!");
        } catch (Exception exception) {
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }

        // Ensure that the token is stored...
        String token = mSetting.getAuthenticationToken();
        assertEquals("Authentication failed", AUTH_TOKEN, token);

        // Ensure the session key is stored...
        String sessionToken = mSetting.getSessionKey();
        assertEquals("Session key failed", SESSION_KEY, sessionToken);

        // Ensure that the display-name was properly parsed...
        assertEquals("Display name wasn't properly returned", MY_DISPLAY_NAME, displayName);
    }

    /**
     * Test fetching of friends
     */
    public void testFetchFriends() {
        MockCommunication mock = new MockCommunication(200, "Ok", FRIENDS);
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            // Make sure we are "logged in"...
            Database database = new Database(getInstrumentation().getTargetContext());

            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");

            facebook.refreshFacebookFriends(database);
            database.storeFriends();
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }
        int actualNbrOfFriends = DatabaseHelper.getFriendCountFromEventStream(getInstrumentation().getTargetContext());

        assertEquals("Couldn't store the correct nbr of friends", 3, actualNbrOfFriends);
    }

    /**
     * Test fetching friends but communication is bad
     */
    public void testFetchFriendsCausesException() {
        MockCommunication mock = new MockCommunication(new IOException());
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            // Make sure we are "logged in"...
            Database database = new Database(getInstrumentation().getTargetContext());

            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");

            facebook.refreshFacebookFriends(database);
            fail("Exception was not thrown!");
        } catch (FacebookException exception) {
            // Yes, we would like to get an exception
        }
    }

    /**
     * Test that a the un-friend algorithm works.
     *
     * Precondition: Have some friends and events in the database
     *
     * Action: Sync friends, where two are removed
     * (two is the key; this to ensure that any iterator doesn't throw
     * ConcurrentModificationException)
     *
     * Postcondition: All data associated with the removed friend in
     * the previous step should be removed.
     */
    public void testUnFriend() {
        String TINGELING_ID = "10010101010";
        String PETERPAN_ID = "2221212123";
        String CAPTAIN_HOOK_ID = "343434343";
        String THE_DOG_ID = "4753485739";
        final String UN_FRIEND_DATA =
            "{\"data\":" +
                "[{\"name\":\"Tingeling\",\"id\":\"" + TINGELING_ID + "\"}," +
                "{\"name\":\"Peter Pan\",\"id\":\"" + PETERPAN_ID + "\"}]}";
        Context context = getInstrumentation().getTargetContext();
        Database database = new Database(context);
        long sourceId = database.registerPlugin();
        int friendId = DatabaseHelper.addFriend(context, sourceId, "Tingeling", null, TINGELING_ID, null);
        int unFriendId = DatabaseHelper.addFriend(context, sourceId, "Captain Hook", null, CAPTAIN_HOOK_ID, null);
        int friendId2 = DatabaseHelper.addFriend(context, sourceId, "The dog", null, THE_DOG_ID, null);
        int friendId3 = DatabaseHelper.addFriend(context, sourceId, "Peter Pan", null, PETERPAN_ID, null);
        int eventId1 = DatabaseHelper.addEvent(context, sourceId, "4444444", "The message", null, CAPTAIN_HOOK_ID, System.currentTimeMillis());
        int eventId2 = DatabaseHelper.addEvent(context, sourceId, "444442232", "Fly away", null, PETERPAN_ID, System.currentTimeMillis());

        // Verify that pre-condition is valid...
        assertTrue("Couldn't store the plugin", sourceId > 0);
        assertTrue("Couldn't store friend 1", friendId > 0);
        assertTrue("Couldn't store friend 2", friendId2 > 0);
        assertTrue("Couldn't store friend 3", friendId3 > 0);
        assertTrue("Couldn't store un-friend ", unFriendId > 0);
        assertTrue("Couldn't store event", eventId1 > 0);
        assertTrue("Couldn't store event", eventId2 > 0);

        // Set up communication...
        MockCommunication mock = new MockCommunication(200, "Ok", UN_FRIEND_DATA);
        initializeTest(new Facebook(mSetting), mock);
        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {

            // Make sure we are logged in...
            mSetting.setAuthenticationToken(AUTH_TOKEN);

            facebook.refreshFacebookFriends(database);
            database.storeFriends();
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }
        int actualNbrOfFriends = DatabaseHelper.getFriendCountFromEventStream(context);

        //Should be 2 since two friends has been de-friended...
        assertEquals("Could successfully un-friend", 2, actualNbrOfFriends);

        int actualNbrOfEvents = DatabaseHelper.getEventCount(context);

        //Should be 1 since two friends has been de-friended...
        assertEquals("Couldn't successfully remove events when un-friending", 1, actualNbrOfEvents);
    }

    /**
     * Test fetching of events
     */
    public void testFetchEvents() {
        int actualNbrOfEvents = -1;
        MockCommunication mock = new MockCommunication(200, "Ok", EVENTS);
        mock.allowDefaultResponse(true);
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            // Make sure we are "logged in"...
            Database database = new Database(getInstrumentation().getTargetContext());
            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");

            facebook.retrieveLatestPosts(database, database);
            actualNbrOfEvents = database.storeEvents();
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }

        assertEquals("Couldn't store the correct nbr of events", 2, actualNbrOfEvents);
    }

    /**
     * Test fetching of events, when there is a type Photo event that has no Picture
     */
    public void testFetchEvents_typePhotoNoPicture() {
        int actualNbrOfEvents = -1;
        MockCommunication mock = new MockCommunication(200, "Ok", EVENTS_WITH_TYPE_PHOTO_EVENT_NO_PICTURE);
        mock.allowDefaultResponse(true);
        initializeTest(new Facebook(mSetting), mock);

        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        try {
            // Make sure we are "logged in"...
            Database database = new Database(getInstrumentation().getTargetContext());
            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");

            facebook.retrieveLatestPosts(database, database);
            actualNbrOfEvents = database.storeEvents();
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }

        assertEquals("Couldn't store the correct nbr of events", 3, actualNbrOfEvents);
    }

    /**
     * Added test case here just to be able to quick test JSON data
     * stored in the file system
     *
     * The file should be stored in
     * /data/data/com.sonyericsson.eventstream.facebookplugin/files
     */
    public void testFetchEventFromFile() {
        String result = FileReader.readFile(getInstrumentation().getTargetContext(),
                "mattias.txt"); //"<your file>");

        if (result != null) {
            MockCommunication mock = new MockCommunication(200, "Ok", result);
            mock.allowDefaultResponse(true);

            initializeTest(new Facebook(mSetting), mock);

            Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
            try {
                // Make sure we are "logged in"...
                Database database = new Database(getInstrumentation().getTargetContext());
                database.registerPlugin();
                mSetting.setAuthenticationToken("SECRET_TOKEN");

                facebook.retrieveLatestPosts(database, database);
            } catch (FacebookException exception) {
                exception.printStackTrace();
                fail("An exception was thrown\n:" + exception.toString() + "\n");
            }
        }
    }

    /**
     * Test that we can retrieve our own status text
     */
    public void testFetchStatus() {
        MockCommunication mock = new MockCommunication(200, "OK", STATUS_UPDATE);
        initializeTest(new Facebook(mSetting), mock);
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        try {
            // Make sure we are "logged in"...
            Database database = new Database(targetContext);
            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");
            mSetting.setOwnId(MY_FACEBOOK_ID);
            facebook.retrieveLatestPosts(database, database);
            database.storeEvents();
            assertEquals("New status!", DatabaseHelper.getOwnStatus(targetContext));
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }
    }

    /**
     * Test fetching own status from the statuses feed if it is not included
     * in the new feed (happens when there are too many other events).
     */
    public void testFetchStatusFromStatusesFeed() {
        MockCommunication mock = new MockCommunication(new int[] {200, 200}, new String[] {"OK", "OK"}, new String[] {EVENTS, STATUSES_FEED});
        initializeTest(new Facebook(mSetting), mock);
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        try {
            // Make sure we are "logged in"...
            Database database = new Database(targetContext);
            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");
            mSetting.setOwnId(MY_FACEBOOK_ID);
            facebook.retrieveLatestPosts(database, database);
            database.storeEvents();
            assertEquals("New status!", DatabaseHelper.getOwnStatus(targetContext));
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }
    }

    /**
     * Test that we, when trying to fetch status and we get a 404 from the server,
     * we should get an exception and we shouldn't reset our own status
     */
    public void testFetchStatusFails() {
        MockCommunication mock = new MockCommunication(404, "Not allowed", null);
        initializeTest(new Facebook(mSetting), mock);
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        final String OWN_STATUS = "my_new_status";

        try {
            // Make sure we are "logged in"...
            Database database = new Database(targetContext);
            database.registerPlugin();
            mSetting.setAuthenticationToken("SECRET_TOKEN");
            mSetting.setOwnId(MY_FACEBOOK_ID);

            // Ensure that we have a valid status text
            boolean ok = DatabaseHelper.setOwnStatus(getInstrumentation().getTargetContext(), OWN_STATUS);
            assertTrue("Could set own status", ok);

            // Fetch status...
            facebook.retrieveLatestPosts(database, database);
            database.storeEvents();
            fail("Didn't get an exception");
        } catch (FacebookException exception) {
            // we want an exception
        }
        String oldStatus = DatabaseHelper.getOwnStatus(targetContext);

        assertNotNull("Couldn't fetch status", oldStatus);
        assertTrue("Own status changed!", OWN_STATUS.equals(oldStatus));
    }

    /**
     * Test that we can upload a new status!
     */
    public void testUploadNewStatus() {
        final String OWN_STATUS = "my_new_status";
        MockCommunication mock = new MockCommunication(200, "Ok", null);
        initializeTest(new Facebook(mSetting), mock);
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        boolean result = false;

        // Make sure we are "logged in"...
        Database database = new Database(targetContext);
        database.registerPlugin();
        mSetting.setAuthenticationToken("SECRET_TOKEN");
        mSetting.setOwnId(MY_FACEBOOK_ID);

        try {
            result = facebook.updateStatus(OWN_STATUS);
        } catch (FacebookException exception) {
            fail("An exception!!");
        }
        assertTrue("Couldn't update my status!", result);
    }

    /**
     * Test that we can handle no network situation when uploading a status
     */
    public void testUploadNewStatusFails() {
        final String OWN_STATUS = "my_new_status";
        MockCommunication mock = new MockCommunication(new IOException());
        initializeTest(new Facebook(mSetting), mock);
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        boolean result = false;

        // Make sure we are "logged in"...
        Database database = new Database(targetContext);
        database.registerPlugin();
        mSetting.setAuthenticationToken("SECRET_TOKEN");
        mSetting.setOwnId(MY_FACEBOOK_ID);

        try {
            result = facebook.updateStatus(OWN_STATUS);
            fail("I want an exception!");
        } catch (FacebookException exception) {
            assertEquals("Not a communication error?", exception.cause, FacebookException.COMMUNICATION_ERROR);
        }

        assertFalse("Update was successful", result);
    }

    /**
     * Test that we can upload a new status containing special characters!
     */
    public void testUploadNewStatusWithSpecialCharacters() {
        final String OWN_STATUS = "my new status with special characters öäå © ñ ß Ł אָלֶף-בֵּית עִבְרִי بجدية عربية‎";
        MockCommunication mock = new MockCommunication(200, "Ok", null);
        initializeTest(new Facebook(mSetting), mock);
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        boolean result = false;

        // Make sure we are "logged in"...
        Database database = new Database(targetContext);
        database.registerPlugin();
        mSetting.setAuthenticationToken("SECRET_TOKEN");
        mSetting.setOwnId(MY_FACEBOOK_ID);

        try {
            result = facebook.updateStatus(OWN_STATUS);
        } catch (FacebookException exception) {
            fail("An exception!!");
        }
        assertTrue("Couldn't update my status!", result);
    }

    /**
     * Test that the set/remove methods for the listener works
     */
    public void testListenerMethods() {
        Context targetContext = getInstrumentation().getTargetContext();
        Facebook facebook = FacebookFactory.getFacebook(targetContext);
        Facebook.ServiceStateChangeListener listener = new Facebook.ServiceStateChangeListener() {
            public void onServiceStateChanged(ServiceState oldState, ServiceState newState) {
                // Do nothing....
            }
        };
        Facebook.ServiceStateChangeListener secondListener = new Facebook.ServiceStateChangeListener() {
            public void onServiceStateChanged(ServiceState oldState, ServiceState newState) {
                // Do nothing....
            }
        };

        facebook.setServiceStateListener(null);
        facebook.setServiceStateListener(listener);
        facebook.setServiceStateListener(secondListener);
        facebook.removeServiceStateListener(null);
        facebook.removeServiceStateListener(secondListener);
        facebook.removeServiceStateListener(listener);
    }

    public class MyStateChangeListener implements Facebook.ServiceStateChangeListener {
        public ArrayList<ServiceState> mOldState = new ArrayList<ServiceState>();
        public ArrayList<ServiceState> mNewState = new ArrayList<ServiceState>();

        public void onServiceStateChanged(ServiceState oldState, ServiceState newState) {
            mOldState.add(oldState);
            mNewState.add(newState);
        }
    }

    /**
     * Test that we get the appropriate callback when doing some
     * communication with the server
     */
    public void testListenerCallback() {
        String displayName = null;
        Context targetContext = getInstrumentation().getTargetContext();
        MyStateChangeListener listener = new MyStateChangeListener();
        MockCommunication mock = new MockCommunication(new int[]{200, 200},
                new String[]{"Ok", "Ok"},
                new String[]{AUTH_CONTENT, ME_PROFILE});

        initializeTest(new Facebook(mSetting), mock);
        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());

        // Make sure we are "logged in"...
        Database database = new Database(targetContext);
        database.registerPlugin();
        mSetting.setAuthenticationToken("SECRET_TOKEN");
        mSetting.setOwnId(MY_FACEBOOK_ID);

        try {
            facebook.setServiceStateListener(listener);
            displayName = facebook.authenticate("test@test", "PWD");
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }

        assertNotNull("No display name!", displayName);

        // Authentication: authenticate + get me-profile
        // Authentication:
        //      We should be going from passive to authenticating back to passive
        // Me-profile:
        //      We should be going from passive to operation back to passive
        assertEquals("Not authenticating (old)?", listener.mOldState.get(0), ServiceState.PASSIVE);
        assertEquals("Not authenticating (new)?", listener.mNewState.get(0), ServiceState.AUTHENTICATION_IN_PROGRESS);
        assertEquals("Not authenticating (old)?", listener.mOldState.get(1), ServiceState.AUTHENTICATION_IN_PROGRESS);
        assertEquals("Not authenticating (new)?", listener.mNewState.get(1), ServiceState.PASSIVE);

        assertEquals("Not me-profile (old)?", listener.mOldState.get(2), ServiceState.PASSIVE);
        assertEquals("Not me-profile (new)?", listener.mNewState.get(2), ServiceState.SERVER_OPERATION_IN_PROGRESS);
        assertEquals("Not me-profile (old)?", listener.mOldState.get(3), ServiceState.SERVER_OPERATION_IN_PROGRESS);
        assertEquals("Not me-profile (new)?", listener.mNewState.get(3), ServiceState.PASSIVE);

        assertEquals("Too many callbacks", 4, listener.mOldState.toArray().length);
    }

    /**
     * Test that we are in logged in state after a successful authentication
     */
    public void testIsLoggedIn() {
        MockCommunication mock = new MockCommunication(new int[]{200, 200},
                new String[]{"Ok", "Ok"},
                new String[]{AUTH_CONTENT, ME_PROFILE});

        initializeTest(new Facebook(mSetting), mock);
        Facebook facebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        boolean isLoggedIn = facebook.isLoggedIn();

        assertFalse("What, are we logged in!", isLoggedIn);

        try {
            facebook.authenticate("test@test", "PWD");
        } catch (FacebookException exception) {
            exception.printStackTrace();
            fail("An exception was thrown\n:" + exception.toString() + "\n");
        }

        isLoggedIn = facebook.isLoggedIn();
        assertTrue("What, are we *not* logged in!", isLoggedIn);
    }
}