/*
 * Copyright (C) 2010 Sony Ericsson Mobile Communications AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 *
 */

package com.sonyericsson.eventstream.rssplugin;

import com.sonyericsson.eventstream.rssplugin.db.RssFeedProvider;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.EventStream;
import android.app.Dialog;
import android.app.ListActivity;
import android.content.ContentValues;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.EditText;
import android.widget.SimpleCursorAdapter;
import android.widget.Toast;

/**
 * The Setup Page for the Rss Plugin Application
 */
public class RssPluginConfig extends ListActivity {

    /** The dialog used for adding a new Rss feed */
    private static final int DIALOG_ADD_FEED = 1;

    /** Add feed menu item*/
    private static final int MENU_ITEM_ADD_FEED = 1;

    /** Remove feed menu item*/
    private static final int MENU_ITEM_REMOVE_FEED = 2;

    /** Add NYTimes feed menu item*/
    private static final int MENU_ITEM_ADD_NYTIMES_FEED = 3;

    /** Add BBC feed menu item*/
    private static final int MENU_ITEM_ADD_BBC_FEED = 4;

    /** Add BBC business feed menu item*/
    private static final int MENU_ITEM_ADD_BBC_BUSINESS_FEED = 5;

    /** The Cursor connected to the SimpleCursorAdapter */
    private Cursor mCursor = null;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setContentView(R.layout.channel_list);

        mCursor = getAllChannels();
        SimpleCursorAdapter adapter = new SimpleCursorAdapter(
                this, android.R.layout.simple_list_item_1, mCursor,
                new String[] {RssFeedProvider.CHANNEL_URL_COLUMN}, new int[]{android.R.id.text1});
        setListAdapter(adapter);
    }

    @Override
    protected void onDestroy() {
        if (mCursor != null) {
            mCursor.close();
            mCursor = null;
        }
        super.onDestroy();
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        super.onCreateOptionsMenu(menu);

        menu.add(Menu.NONE, MENU_ITEM_ADD_FEED, Menu.NONE,
                getResources().getText(R.string.menu_add_feed)).setIcon(
                        android.R.drawable.ic_menu_add);
        menu.add(Menu.NONE, MENU_ITEM_REMOVE_FEED, Menu.NONE,
                getResources().getText(R.string.menu_remove_feed)).setIcon(
                        android.R.drawable.ic_menu_delete);
        menu.add(Menu.NONE, MENU_ITEM_ADD_BBC_FEED, Menu.NONE,
                getResources().getText(R.string.menu_add_feed_bbc)).setIcon(
                        android.R.drawable.ic_menu_add);
        menu.add(Menu.NONE, MENU_ITEM_ADD_NYTIMES_FEED, Menu.NONE,
                getResources().getText(R.string.menu_add_feed_nytimes)).setIcon(
                        android.R.drawable.ic_menu_add);
        menu.add(Menu.NONE, MENU_ITEM_ADD_BBC_BUSINESS_FEED, Menu.NONE,
                getResources().getText(R.string.menu_add_feed_bbc_business)).setIcon(
                        android.R.drawable.ic_menu_add);

        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {

            case MENU_ITEM_ADD_FEED:
                onMenuAddFeed();
                break;

            case MENU_ITEM_REMOVE_FEED:
                removeAllChannels();
                break;

            case MENU_ITEM_ADD_BBC_FEED:
                addChannel("http://newsrss.bbc.co.uk/rss/newsonline_world_edition/europe/rss.xml");
                break;

            case MENU_ITEM_ADD_NYTIMES_FEED:
                addChannel("http://www.nytimes.com/services/xml/rss/userland/Education.xml");
                break;

            case MENU_ITEM_ADD_BBC_BUSINESS_FEED:
                addChannel("http://feeds.bbci.co.uk/news/business/rss.xml");
                break;

            default:
                break;
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    public Dialog onCreateDialog(int id) {
        switch (id) {
            case DIALOG_ADD_FEED:
                final Dialog addFeedDialog = new Dialog(this);
                addFeedDialog.setContentView(R.layout.channel_add);
                addFeedDialog.setTitle(R.string.menu_add_feed);
                addFeedDialog.findViewById(R.id.button_add_feed_ok)
                        .setOnClickListener(new View.OnClickListener() {
                    public void onClick(View v) {
                        addFeedDialog.dismiss();
                        EditText feedUrlEditText = (EditText) addFeedDialog.findViewById(R.id.edit_feed_url);
                        addChannel(feedUrlEditText.getText().toString());
                    }
                });

                EditText feedUrlEditText = (EditText) addFeedDialog.findViewById(R.id.edit_feed_url);
                feedUrlEditText.setText(R.string.add_feed_edit_default);
                return addFeedDialog;

            default:
                throw new IllegalArgumentException("unknown dialog");
        }
    }

    /**
     * Adds a channel url to the database.
     * If the url already exists, an info toast is shown instead.
     *
     * @param url The url to store
     */
    private void addChannel(String url) {
        try {
            Uri row = addChannelToDB(url);
            if (row == null) {
                showToast(getString(R.string.add_feed_already_exists));
            }
        } catch (Exception e) {
            if(Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Exception: " + e.toString());
            }
            showToast(e.toString());
        }
    }

    /**
     * Launches a dialog for adding a new feed. If the user has reached the feed limit,
     * it shows an info toast instead.
     */
    private void onMenuAddFeed() {
        if (mCursor != null && mCursor.getCount() >= RssFeedProvider.MAX_CHANNEL_ROWS) {
            showToast(getString(R.string.add_feed_limit_exceeded));
        } else {
            showDialog(DIALOG_ADD_FEED);
        }
    }

    /**
     * Shows a toast.
     *
     * @param message The message to display.
     */
    private void showToast(CharSequence message) {
        Toast.makeText(this, message, Toast.LENGTH_SHORT).show();
    }

    /**
     * Stores a new Channel url
     *
     * @param url The url to store
     * @return the Uri to the inserted row, null if not inserted
     */
    private Uri addChannelToDB(String url) {
        Uri uri = null;
        Cursor cursor = null;
        try {
            cursor = getContentResolver().query(
                    RssFeedProvider.CHANNEL_CONTENT_URI,
                    null,
                    RssFeedProvider.CHANNEL_URL_COLUMN + " = ?",
                    new String[]{DatabaseUtils.sqlEscapeString(url.toString())},
                    null);
            if (cursor != null && cursor.getCount() == 0) {
                ContentValues channelValues = new ContentValues();
                channelValues.put(RssFeedProvider.CHANNEL_URL_COLUMN, url);
                uri = getContentResolver().insert(RssFeedProvider.CHANNEL_CONTENT_URI, channelValues);
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return uri;
    }

    /**
     * Gets all the channels.
     *
     * @return a Cursor or null.
     */
    private Cursor getAllChannels() {
        return getContentResolver().query(RssFeedProvider.CHANNEL_CONTENT_URI, null, null, null, null);
    }

    /**
     * Deletes all the channels
     */
    private void removeAllChannels() {
        getContentResolver().delete(RssFeedProvider.CHANNEL_CONTENT_URI, null, null);
        getContentResolver().delete(RssFeedProvider.ITEM_CONTENT_URI, null, null);
        getContentResolver().delete(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null);
    }
}