/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.simpleui.activity;

import com.sonyericsson.simpleui.R;
import com.sonyericsson.simpleui.activity.SimpleUiInternal.Config;
import com.sonyericsson.simpleui.provider.EventStreamContentProvider;
import com.sonyericsson.simpleui.utils.ImageUtilities;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.ExpandableListAdapter;
import android.widget.ExpandableListView;
import android.widget.ImageView;
import android.widget.SimpleCursorTreeAdapter;

public class SimpleUiLaunchActivity extends Activity implements View.OnClickListener {

    private ExpandableListView mExpandableListView;

    private Bitmap mDefaultSourceIconBitmap;

    private AsyncTask<String, Void, Void> mSendTask = null;

    /**
     * Called when the activity is first created.
     */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        findViewById(R.id.send_refresh_broadcast).setOnClickListener(this);
        findViewById(R.id.send_registration_broadcast).setOnClickListener(this);
        findViewById(R.id.see_events).setOnClickListener(this);
        findViewById(R.id.go_to_settings).setOnClickListener(this);
        findViewById(R.id.go_to_status_update).setOnClickListener(this);

        mDefaultSourceIconBitmap = BitmapFactory.decodeResource(getResources(),
                R.drawable.generic_service);
        if (mDefaultSourceIconBitmap == null) {
            if (Config.DEBUG) {
                Log.i(Config.LOG_TAG, "Failed to decode default source icon image!");
            }
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        String[] pluginFrom = {
                EventStreamContentProvider.PluginColumns.NAME,
                EventStreamContentProvider.PluginColumns.CONFIGURATION_TEXT
        };

        int[] pluginTo = new int[] {
                R.id.plugin_name, R.id.plugin_config_text
        };

        String[] sourceFrom = {
            EventStreamContentProvider.SourceColumns.NAME
        };

        int[] sourceTo = new int[] {
            R.id.source_name
        };

        mExpandableListView = (ExpandableListView)findViewById(R.id.plugin_list);

        Cursor pluginCursor = managedQuery(EventStreamContentProvider.Uris.PLUGIN_URI, null, null,
                null, EventStreamContentProvider.PluginColumns.NAME);

        if (pluginCursor != null) {
            // Declared final here for use by the expandable list adapter
            // to do managed queries for children (not possible to supply in
            // constructor)
            final Activity activity = this;

            ExpandableListAdapter expandableListAdapter = new SimpleCursorTreeAdapter(this,
                    pluginCursor, R.layout.plugin, pluginFrom, pluginTo, R.layout.source,
                    sourceFrom, sourceTo) {
                @Override
                protected Cursor getChildrenCursor(Cursor groupCursor) {
                    long id = groupCursor.getLong(groupCursor
                            .getColumnIndex(android.provider.BaseColumns._ID));
                    return activity.managedQuery(EventStreamContentProvider.Uris.SOURCE_URI, null,
                            "plugin_id = " + id, null, null);
                }

                @Override
                protected void bindGroupView(View view, Context context, Cursor cursor,
                        boolean isExpanded) {
                    super.bindGroupView(view, context, cursor, isExpanded);

                    String iconUriString = cursor.getString(cursor
                            .getColumnIndex(EventStreamContentProvider.PluginColumns.ICON_URI));
                    Bitmap pluginIconBitmap = null;
                    if (iconUriString != null && !iconUriString.equals("")) {
                        pluginIconBitmap = ImageUtilities.decodeImageFromUri(context, Uri
                                .parse(iconUriString));
                    }
                    if (pluginIconBitmap != null) {
                        ((ImageView)view.findViewById(R.id.plugin_icon))
                                .setImageBitmap(pluginIconBitmap);
                    } else {
                        ((ImageView)view.findViewById(R.id.plugin_icon))
                                .setImageBitmap(mDefaultSourceIconBitmap);
                    }
                    String packageName = cursor
                            .getString(cursor
                                    .getColumnIndexOrThrow(EventStreamContentProvider.PluginColumns.PACKAGE_NAME));
                    if (packageName != null && packageName.length() > 0) {
                        view.findViewById(R.id.plugin_button).setTag(packageName);
                        int configState = cursor
                                .getInt(cursor
                                        .getColumnIndexOrThrow(EventStreamContentProvider.PluginColumns.CONFIGURATION_STATE));
                        if (configState == EventStreamContentProvider.ConfigState.NOT_CONFIGURED) {
                            ((Button)view.findViewById(R.id.plugin_button))
                                    .setText("not configured");
                        } else if (configState == EventStreamContentProvider.ConfigState.CONFIGURED) {
                            ((Button)view.findViewById(R.id.plugin_button)).setText("configured");
                        } else {
                            ((Button)view.findViewById(R.id.plugin_button)).setText("config");
                        }
                    }
                    String configActivityName = cursor
                            .getString(cursor
                                    .getColumnIndexOrThrow(EventStreamContentProvider.PluginColumns.CONFIGURATION_ACTIVITY));
                    if (configActivityName != null && configActivityName.length() > 0) {
                        view.findViewById(R.id.plugin_button).setEnabled(true);
                    } else {
                        view.findViewById(R.id.plugin_button).setEnabled(false);
                    }
                }

                @Override
                protected void bindChildView(View view, Context context, Cursor cursor,
                        boolean isLastChild) {
                    try {
                        super.bindChildView(view, context, cursor, isLastChild);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return;
                    }
                    String iconUriString = cursor.getString(cursor
                            .getColumnIndex(EventStreamContentProvider.SourceColumns.ICON_URI));
                    Bitmap sourceIconBitmap = null;
                    if (iconUriString != null && !iconUriString.equals("")) {
                        sourceIconBitmap = ImageUtilities.decodeImageFromUri(context, Uri
                                .parse(iconUriString));
                    }
                    if (sourceIconBitmap != null) {
                        ((ImageView)view.findViewById(R.id.source_icon))
                                .setImageBitmap(sourceIconBitmap);
                    } else {
                        ((ImageView)view.findViewById(R.id.source_icon))
                                .setImageBitmap(mDefaultSourceIconBitmap);
                    }
                }
            };

            mExpandableListView.setAdapter(expandableListAdapter);
        } else {
            Log.e(Config.LOG_TAG, "Not possible to list plugins and sources.");
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
    }

    public void onClick(View view) {
        if (view.getId() == R.id.send_registration_broadcast) {
            Intent broadcast = new Intent(
                    EventStreamContentProvider.OutgoingIntents.REGISTER_PLUGINS_REQUEST_INTENT);
            sendBroadcast(broadcast);
            Log.d(Config.LOG_TAG, "Sending register broadcast");
        } else if ((view.getId() == R.id.send_refresh_broadcast) && (mSendTask == null)) {
            mSendTask = new SendRefreshBroadcast()
                    .execute(EventStreamContentProvider.OutgoingIntents.REFRESH_REQUEST_INTENT, SimpleUiInternal.DISTRIBUTION_TIME);
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Sending refresh broadcast ");
            }
        } else if (view.getId() == R.id.see_events) {
            startActivity(new Intent(this, SimpleUiEventsActivity.class));
        } else if (view.getId() == R.id.go_to_settings) {
            Intent settingsIntent = new Intent();
            settingsIntent.setClass(getApplicationContext(), SimpleUiSettingsActivity.class);
            startActivity(settingsIntent);
        } else if (view.getId() == R.id.go_to_status_update) {
            Intent statusUpdateIntent = new Intent();
            statusUpdateIntent
                    .setClass(getApplicationContext(), SimpleUiStatusUpdateActivity.class);
            startActivity(statusUpdateIntent);
        } else if (view.getId() == R.id.plugin_button) {
            String packageName = (String)view.getTag();
            String configActivityName;
            Cursor pluginCursor = null;
            try {
                pluginCursor = managedQuery(EventStreamContentProvider.Uris.PLUGIN_URI,
                        new String[] {
                            EventStreamContentProvider.PluginColumns.CONFIGURATION_ACTIVITY,
                        }, EventStreamContentProvider.PluginColumns.PACKAGE_NAME + " LIKE ?",
                        new String[] {
                            packageName
                        }, EventStreamContentProvider.PluginColumns.NAME);
                configActivityName = null;
                if (pluginCursor != null && pluginCursor.moveToFirst()) { // We
                                                                          // should
                                                                          // only
                                                                          // get
                                                                          // one
                                                                          // single
                                                                          // hit
                                                                          // here...
                    configActivityName = pluginCursor
                            .getString(pluginCursor
                                    .getColumnIndexOrThrow(EventStreamContentProvider.PluginColumns.CONFIGURATION_ACTIVITY));
                }
            } finally {
                if (pluginCursor != null) {
                    pluginCursor.close();
                }
            }
            ComponentName componentName = ComponentName.unflattenFromString(configActivityName);
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Trying to start plugin config activity: " + packageName);
            }
            Intent intent = new Intent();
            intent.setComponent(componentName);
            intent.putExtra(EventStreamContentProvider.IntentData.EXTRA_PACKAGENAME, packageName);
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Start activity with intent: " + intent.toString());
            }
            try {
                startActivity(intent);
            } catch (ActivityNotFoundException e) {
                // TODO: make sure this try/catch is added to the real UI since
                // we can't make sure
                // that the component exists
                Log.e(Config.LOG_TAG, "Activity now found, could not be started: " + e);
            }
        }
    }

    /**
     * Sends broadcasts to all enabled plugins. The broadcasts are distributed over time to avoid choking
     * the system.
     */
    private class SendRefreshBroadcast extends AsyncTask<String, Void, Void> {

        @Override
        protected Void doInBackground(String... strings) {
            int distributionTime = new Integer(strings[1]).intValue();
            Intent broadcast = new Intent(strings[0]);
            Cursor pluginCursor = null;
            try {
                pluginCursor = getContentResolver().query(
                        EventStreamContentProvider.Uris.PLUGIN_URI,
                        new String[] {
                                EventStreamContentProvider.PluginColumns.PACKAGE_NAME,
                                EventStreamContentProvider.PluginColumns.PLUGIN_KEY
                        },
                        EventStreamContentProvider.ID_COLUMN + " IN (SELECT "
                                + EventStreamContentProvider.SourceColumns.PLUGIN_ID + " FROM "
                                + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + " WHERE "
                                + EventStreamContentProvider.SourceColumns.ENABLED + " = 1 " + ")",
                        null, null);

                if (pluginCursor != null && pluginCursor.moveToFirst()) {
                    String packageName = null;
                    String pluginKey = null;
                    do {
                        packageName = pluginCursor
                                .getString(pluginCursor
                                        .getColumnIndexOrThrow(EventStreamContentProvider.PluginColumns.PACKAGE_NAME));
                        pluginKey = pluginCursor
                                .getString(pluginCursor
                                        .getColumnIndexOrThrow(EventStreamContentProvider.PluginColumns.PLUGIN_KEY));
                        broadcast.setPackage(packageName);

                        if (pluginKey != null && !pluginKey.equals("")) {
                            broadcast.putExtra(
                                    EventStreamContentProvider.IntentData.EXTRA_PLUGIN_KEY,
                                    pluginKey);
                        }
                        if (Config.DEBUG) {
                            Log.d(Config.LOG_TAG, "Refresh intent: " + packageName + ", key: " + pluginKey);
                            Log.d(Config.LOG_TAG, "Distribution time: " + distributionTime);
                        }
                        sendBroadcast(broadcast);
                        try {
                            Thread.sleep(distributionTime);
                        } catch (InterruptedException e) {
                        }
                    } while (pluginCursor.moveToNext());
                }
            } finally {
                if (pluginCursor != null) {
                    pluginCursor.close();
                }
            }
            return null;
        }

        @Override
        protected void onPostExecute(Void result) {
            mSendTask = null;
            super.onPostExecute(result);
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "onPostExecute done.");
            }
        }
    }

}
