/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.testplugin;

import com.sonyericsson.eventstream.provider.EventStreamProvider;
import com.sonyericsson.eventstream.provider.EventStreamProvider.Friend;

import android.app.Activity;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.ContentObserver;
import android.database.Cursor;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.provider.BaseColumns;
import android.util.Config;
import android.util.Log;
import android.view.View;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.SimpleCursorAdapter;
import android.widget.Spinner;
import android.widget.Toast;

import java.util.Date;

/**
 * Activity for manually adding some events and data to the Event Stream&#x99;
 * provider.
 *
 * @author 23007672
 */
public class TestPlugin extends Activity {

    // / String used as prefix when logging.
    public static final String LOG_TAG = "TestPlugin[Main]";

    // / Counter for id of events written to Event Stream.
    private long mEventId = 0;

    // / Counter that tries to keep up with source_id.
    private long mSourceCount = 1;

    // / Incremented to create unique friend names.
    private long mFriendCount = 0;

    // When creating events increment this to "exercise"
    // all friends in source.
    private long mFriendLooper = 0;

    private static final String PREF_NAME = "TestPluginAlpha";

    private static final String PREF_FRIEND_KEY = "friend_count";

    private static final String PREF_SOURCE_KEY = "source_count";

    private static final String PREF_EVENT_KEY = "event_id";

    private static final int OP_SUCCESS = 0; // Also index in resource string

    // array.

    private static final int OP_IGNORED = 1;

    private static final int OP_ERROR = 2;

    private static final int EVENT_DEFAULT_MSG_LEN = 140;

    // States

    // / Spinner for selecting between created sources.
    private Spinner mSourceSpinner;

    private SimpleCursorAdapter mSimpleCursorAdapterSources;

    /**
     * Observer to keep UI spinner with sources up to date.
     */
    private class SourceTableObserver extends ContentObserver {

        public SourceTableObserver(Handler handler) {
            super(handler);
        }

        @Override
        public void onChange(boolean selfChange) {
            super.onChange(selfChange);
            runOnUiThread(new Thread() {
                public void run() {
                    /*
                     * for some reason it is not enough to requery the existing
                     * cursor, that is why a new is created instead, should not
                     * matter for this test plugin
                     */
                    Cursor c = getContentResolver().query(EventStreamProvider.URIs.SOURCES,
                            new String[] {
                                BaseColumns._ID
                            }, null, null, null);
                    if (c != null) {
                        mSimpleCursorAdapterSources.changeCursor(c);
                    }
                }
            });
        }
    }

    SourceTableObserver mSourceContentObserver = new SourceTableObserver(null);

    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setContentView(R.layout.main);

        mSourceSpinner = (Spinner)findViewById(R.id.source_spinner);

        getContentResolver().registerContentObserver(EventStreamProvider.URIs.SOURCES, false,
                mSourceContentObserver);

        Cursor c = getContentResolver().query(EventStreamProvider.URIs.SOURCES, new String[] {
            BaseColumns._ID
        }, null, null, null);

        mSimpleCursorAdapterSources = new SimpleCursorAdapter(this.getApplicationContext(),
                android.R.layout.simple_spinner_item, c, new String[] {
                    BaseColumns._ID
                }, new int[] {
                    android.R.id.text1
                });

        mSourceSpinner.setAdapter(mSimpleCursorAdapterSources);

        // Get stored data (counters)
        SharedPreferences sharedPref = this.getSharedPreferences(PREF_NAME, MODE_PRIVATE);

        if (sharedPref != null) {

            mEventId = sharedPref.getLong(PREF_EVENT_KEY, 0);

            mSourceCount = sharedPref.getLong(PREF_SOURCE_KEY, 1);

            mFriendCount = sharedPref.getLong(PREF_FRIEND_KEY, 0);
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        getContentResolver().unregisterContentObserver(mSourceContentObserver);
    }

    @Override
    public void onResume() {
        super.onResume();
        getContentResolver().registerContentObserver(EventStreamProvider.URIs.SOURCES, false,
                mSourceContentObserver);
    }

    public void onClick(View view) {

        // Fire operation to asynchronous task...
        new DatabaseOperationTask().execute(view.getId());
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
    }

    @Override
    public void onRestoreInstanceState(Bundle savedInstanceState) {
        super.onRestoreInstanceState(savedInstanceState);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        SharedPreferences.Editor pe = this.getSharedPreferences(PREF_NAME, 0).edit();
        pe.putLong(PREF_EVENT_KEY, mEventId);
        pe.putLong(PREF_SOURCE_KEY, mSourceCount);
        pe.putLong(PREF_FRIEND_KEY, mFriendCount);
        pe.commit();
    }

    private class DatabaseOperationTask extends AsyncTask<Integer, Integer, Integer> {

        private long mCurrentSource = -1;

        private boolean mEventBulkInsertEnabled = false;

        int mEventBulkCount = 0;

        private boolean mEventMessageLengthEnabled = false;

        int mEventMessageLength = 0;

        private boolean mSourceAutoGenerateDataEnabled = false;

        private boolean mFriendAddNewContactWithExistingNameEnabled = false;

        private boolean mFriendAddNewContactWithUniqueName = false;

        @Override
        protected void onPreExecute() {
            // Get values and setting from UI components.
            // Method runs in UI thread so accessing views is OK.

            mCurrentSource = -1;

            if ((mSourceSpinner != null) && (mSourceSpinner.getCount() > 0)) {
                Object o = mSourceSpinner.getSelectedItem();
                if (o instanceof Cursor) {
                    Cursor c = (Cursor)o;
                    mCurrentSource = c.getLong(c.getColumnIndex(BaseColumns._ID));
                    if (Config.DEBUG) {
                        Log.d(LOG_TAG, "Current source is: " + mCurrentSource + " at cursor pos: "
                                + c.getPosition());
                    }
                }
            }

            CheckBox checkBox = (CheckBox)findViewById(R.id.id_source_check_box);
            if (checkBox != null && checkBox.isChecked())
                mSourceAutoGenerateDataEnabled = true;
            checkBox = (CheckBox)findViewById(R.id.id_event_bulk_insert_check_box);
            if (checkBox != null && checkBox.isChecked())
                mEventBulkInsertEnabled = true;
            checkBox = (CheckBox)findViewById(R.id.id_event_msg_length_check_box);
            if (checkBox != null && checkBox.isChecked())
                mEventMessageLengthEnabled = true;
//            checkBox = (CheckBox)findViewById(R.id.id_friend_check_box);
            if (checkBox != null && checkBox.isChecked()) {
//                mFriendAddNewContactWithExistingNameEnabled = true;
            }
            checkBox = (CheckBox)findViewById(R.id.id_contact_check_box);
            if (checkBox != null && checkBox.isChecked()) {
                mFriendAddNewContactWithUniqueName = true;
            }

            // Get preferred message length
            if (mEventMessageLengthEnabled) {
                mEventMessageLength = EVENT_DEFAULT_MSG_LEN;
                EditText msgLengthView = (EditText)findViewById(R.id.event_msg_length);
                if (msgLengthView != null) {
                    int msgLengthTmp = Integer.parseInt(msgLengthView.getText().toString());
                    if (msgLengthTmp != -1) {
                        mEventMessageLength = msgLengthTmp;
                    }
                }
            }

            if (mEventBulkInsertEnabled) {
                EditText bulkEditView = (EditText)findViewById(R.id.event_bulk_size);
                if (bulkEditView != null) {
                    mEventBulkCount = Integer.parseInt(bulkEditView.getText().toString());
                }
            }
        }

        @Override
        protected Integer doInBackground(Integer... op) {
            int status = OP_ERROR;

            switch (op[0]) {
                case R.id.plugin_add:
                    status = pluginAdd();
                    break;
                case R.id.plugin_remove:
                    status = pluginRemove();
                    break;
                case R.id.source_add:
                    status = sourceAdd();
                    break;
                case R.id.source_remove:
                    status = sourceRemove();
                    break;
                case R.id.event_add:
                    status = eventAdd();
                    break;
                case R.id.event_add_all_sources:
                    status = addEventForAllSources();
                    break;
                case R.id.event_remove:
                    status = eventRemove();
                    break;
                case R.id.friend_add:
                    status = friendAdd();
                    break;
            }
            return status;
        }

        @Override
        protected void onProgressUpdate(Integer... progress) {
            // setProgressPercent(progress[0]);
        }

        @Override
        protected void onPostExecute(Integer result) {
            String toastMessage = null;
            String[] msgArray = getResources().getStringArray(R.array.status_msg_array);
            toastMessage = msgArray[result];

            if (toastMessage != null) {
                Toast.makeText(TestPlugin.this.getApplicationContext(), toastMessage,
                        Toast.LENGTH_SHORT).show();
            }
        }

        /**
         * Insert row in plug-in table ("register" this plug-in).
         *
         * @return
         */
        private int pluginAdd() {

            int result = OP_ERROR;
            Cursor cursor = null;
            ContentValues pluginValues = new ContentValues();

            try {
                // Check if we already have instance in plug-in table.
                cursor = getContentResolver().query(EventStreamProvider.URIs.PLUGINS, null, null,
                        null, null);

                pluginValues.put(EventStreamProvider.Plugin.COLUMN_FRIENDLY_NAME, getResources()
                        .getString(R.string.str_plugin_name));

                pluginValues.put(EventStreamProvider.Plugin.COLUMN_ICON,
                        "android.resource://com.sonyericsson.eventstream.testplugin/"
                                + Integer.toString(R.drawable.icon));
                pluginValues.put(EventStreamProvider.Plugin.COLUMN_API_VERSION, 1);
                pluginValues.put(EventStreamProvider.Plugin.COLUMN_CONFIGURATION_STATE,
                        EventStreamProvider.ConfigState.CONFIGURATION_NOT_NEEDED);

                if (cursor != null && cursor.getCount() > 0) {
                    // Already row in table. Lets update the plug-in data.
                    getContentResolver().update(EventStreamProvider.URIs.PLUGINS, pluginValues,
                            null, null);

                    result = OP_IGNORED;
                    if (Config.DEBUG) {
                        Log.d(TestPlugin.LOG_TAG, "Updated data of test plug-in in Event Stream.");
                    }
                } else {
                    // Register ourself for the first time.
                    Uri uri = getContentResolver().insert(EventStreamProvider.URIs.PLUGINS,
                            pluginValues);

                    if (Config.DEBUG) {
                        Log.d(TestPlugin.LOG_TAG, "Inserted test plug-in into Event Stream.");
                    }
                    if (uri != null) {
                        result = OP_SUCCESS;
                    }
                }
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(TestPlugin.LOG_TAG, "Event Stream provider error: " + e.getMessage());
                }
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
            return result;
        }

        private int pluginRemove() {
            int result = OP_ERROR;
            // A general delete on plug-in table should only delete row
            // inserted by this application (no permission for deleting the
            // others).
            try {
                int i = getContentResolver().delete(EventStreamProvider.URIs.PLUGINS, null, null);

                if (i == 1) {
                    result = OP_SUCCESS;
                } else if (i > 1) {
                    throw new IllegalStateException();
                } else { // i < 1
                    result = OP_IGNORED;
                }
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Remove plugin: " + e.getMessage());
                }
                result = OP_ERROR;
            }
            return result;
        }

        /**
         * Adds source to source table.
         *
         * @return
         */
        private int sourceAdd() {
            int result = OP_ERROR;
            // Insert new source into source table.
            // Use counter to create 'unique' source name. Keep track of
            // returned source_id by storing it in source adapter.
            try {
                String sourceName = "Source " + mSourceCount;
                long sourceId = -1;

                ContentValues sourceRegValues = new ContentValues();
                sourceRegValues.put(EventStreamProvider.Source.COLUMN_NAME, sourceName);
                sourceRegValues.put(EventStreamProvider.Source.COLUMN_ICON,
                        "android.resource://com.sonyericsson.eventstream.testplugin/"
                                + Integer.toString(R.drawable.icon));

                sourceRegValues.put(EventStreamProvider.Source.COLUMN_ENABLED, true);

                Uri sourceUri = getContentResolver().insert(EventStreamProvider.URIs.SOURCES,
                        sourceRegValues);

                // Update source name with correct (returned) ID (to make debug
                // easier)
                if (sourceUri != null) {
                    sourceId = ContentUris.parseId(sourceUri);

                    if (mSourceCount != sourceId) {
                        mSourceCount = sourceId;
                        sourceName = "Source " + sourceId;

                        sourceRegValues.clear();
                        sourceRegValues.put(EventStreamProvider.Source.COLUMN_NAME, sourceName);

                        getContentResolver().update(sourceUri, sourceRegValues, null, null);
                    }

                    mSourceCount++;

                    result = OP_SUCCESS;

                    // Start auto generating content for this source
                    if (mSourceAutoGenerateDataEnabled) {
                        Intent intent = new Intent(
                                GenerateContentService.TESTPLUGINSOURCE_ACTION_GENERATE_CONTENT);
                        intent.putExtra("source_id", sourceId);
                        intent.putExtra("start_code", 2);
                        TestPlugin.this.startService(intent);
                    }
                } else {
                    result = OP_IGNORED;
                }
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Add source error (Exception: " + e.getMessage() + ").");
                }
                result = OP_ERROR;
            }
            return result;
        }

        private int sourceRemove() {
            int result = OP_ERROR;
            // Delete current (check UI spinner) source.
            // This operation will trig database to remove
            // all events created by this source.

            if (mCurrentSource == -1) {
                return OP_IGNORED;
            }

            try {
                int i = getContentResolver().delete(EventStreamProvider.URIs.SOURCES,
                        BaseColumns._ID + " = ?", new String[] {
                            Long.toString(mCurrentSource)
                        });

                if (i == 1) {
                    if (Config.DEBUG) {
                        Log.d(LOG_TAG, "Source \"" + mCurrentSource + "\" was removed.");
                    }
                    result = OP_SUCCESS;
                } else {
                    // Item was not found.
                    result = OP_IGNORED;
                }
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Source delete error: " + e.getMessage());
                }
            }

            return result;
        }

        private int eventAdd() {
            int result = OP_ERROR;
            // Add event originating from current source (check UI spinner).
            // We use counter to generate a somewhat unique event identifier and
            // name.
            // If sources have friends then add friend column. (Loop through
            // friends.)

            // Check spinner
            if (mCurrentSource == -1) {
                return OP_IGNORED;
            } else {
                return addEventForSpecificSource(mCurrentSource);
            }
        }

        private int addEventForSpecificSource(long sourceId) {
            int result = OP_ERROR;
            int nEvents = 1;

            // Check if bulk insert and loop on count value
            if (mEventBulkInsertEnabled) {
                nEvents = mEventBulkCount;
            }

            ContentValues[] eventValues = new ContentValues[nEvents];

            // Check if (current) source have any friends...
            int sourceFriendId = -1;
            Cursor friendCursor = null;
            friendCursor = getContentResolver().query(EventStreamProvider.URIs.FRIENDS, null,
                    EventStreamProvider.Friend.COLUMN_SOURCE_ID + " = ? ", new String[] {
                        Long.toString(sourceId)
                    }, null);

            for (int bc = 0; bc < nEvents; bc++) {
                eventValues[bc] = new ContentValues();

                // Pick a friend...
                if ((friendCursor != null) && (friendCursor.getCount() > 0)) {
                    // Just alter this counter so that we uses
                    // somewhat different rows in the friend table.
                    mFriendLooper = (mFriendLooper + 1) % friendCursor.getCount();

                    if (friendCursor.moveToPosition((int)mFriendLooper)) {
                        int idColumn = friendCursor.getColumnIndexOrThrow(BaseColumns._ID);
                        sourceFriendId = friendCursor.getInt(idColumn);
                    }
                }
                // Create event data.
                if (mEventMessageLengthEnabled) {
                    // We put "lorum ipsum" text as message. Repeated if
                    // necessary.
                    // Wanted length is taken from input text component in view.

                    String strLoremIpsum = TestPlugin.this.getResources().getString(
                            R.string.str_lorem_ipsum);
                    StringBuffer msgBuilder = new StringBuffer(strLoremIpsum.substring(0, Math.min(
                            mEventMessageLength, strLoremIpsum.length())));
                    msgBuilder.ensureCapacity(mEventMessageLength);
                    while (msgBuilder.length() < mEventMessageLength) {
                        msgBuilder.append(strLoremIpsum.substring(0, Math
                                .min((mEventMessageLength - msgBuilder.length()), strLoremIpsum
                                        .length())));
                    }
                    eventValues[bc].put(EventStreamProvider.Event.COLUMN_MESSAGE, msgBuilder
                            .toString());
                } else {
                    eventValues[bc].put(EventStreamProvider.Event.COLUMN_MESSAGE, "Source "
                            + sourceId + " event " + mEventId);
                }

                eventValues[bc].put(EventStreamProvider.Event.COLUMN_SOURCE_ID, sourceId);
                eventValues[bc].put(EventStreamProvider.Event.COLUMN_IMAGE_URI,
                        "android.resource://com.sonyericsson.eventstream.testplugin/"
                                + Integer.toString(R.drawable.icon));
                eventValues[bc].put(EventStreamProvider.Event.COLUMN_PUBLISHED_TIME, new Date()
                        .getTime());

                eventValues[bc].put(EventStreamProvider.Event.COLUMN_STATUS_ICON_URI,
                        "android.resource://com.sonyericsson.eventstream.testplugin/"
                                + Integer.toString(R.drawable.status_icon));

                eventValues[bc].put(EventStreamProvider.Event.COLUMN_PERSONAL, 0);
                eventValues[bc].put(EventStreamProvider.Event.COLUMN_OUTGOING, 0);
                if (sourceFriendId != -1) {
                    eventValues[bc].put(EventStreamProvider.Event.COLUMN_FRIEND_KEY, sourceFriendId);
                }

                eventValues[bc].put(EventStreamProvider.Event.COLUMN_OUTGOING, 0);
                eventValues[bc].put(EventStreamProvider.Event.COLUMN_PERSONAL, 0);
                mEventId++;
            }

            if (friendCursor != null) {
                friendCursor.close();
            }

            // Insert new event(s) into provider.
            try {
                int n = 0;

                if (nEvents == 1) {
                    if (getContentResolver()
                            .insert(EventStreamProvider.URIs.EVENTS, eventValues[0]) != null) {
                        n = 1;
                    }
                } else {
                    n = getContentResolver().bulkInsert(EventStreamProvider.URIs.EVENTS,
                            eventValues);
                }

                if (n > 0) {
                    if (Config.DEBUG) {
                        Log.d(LOG_TAG, "Added " + n + " event(s) from: Source " + sourceId);
                    }
                    result = OP_SUCCESS;
                } else {
                    result = OP_ERROR;
                }
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Event insert error: " + e.getMessage());
                }
                result = OP_ERROR;
            }
            return result;

        }

        private int addEventForAllSources() {
            int result = OP_SUCCESS;
            int resultForSpecificSource = OP_SUCCESS;

            Cursor c = getContentResolver().query(EventStreamProvider.URIs.SOURCES, new String[] {
                    BaseColumns._ID
                }, null, null, null);

            boolean cursorOk = c.moveToFirst();
            long currentSourceId = -1;

            while (cursorOk) {
                currentSourceId = c.getLong(c.getColumnIndex(BaseColumns._ID));
                resultForSpecificSource = addEventForSpecificSource(currentSourceId);
                Log.d(LOG_TAG, "addEventForSpecificSource: " + currentSourceId + " returned the result: " + resultForSpecificSource);
                if (resultForSpecificSource != OP_SUCCESS) {
                    result = resultForSpecificSource;
                }
                cursorOk = c.moveToNext();
            }

            return result;
        }

        private int eventRemove() {
            int result = OP_ERROR;

            // Delete all events for current source.
            if (mCurrentSource == -1) {
                return OP_IGNORED;
            }

            try {
                int i = getContentResolver().delete(EventStreamProvider.URIs.EVENTS,
                        EventStreamProvider.Event.COLUMN_SOURCE_ID + " = ?", new String[] {
                            Long.toString(mCurrentSource)
                        });

                if (Config.DEBUG) {
                    Log.d(LOG_TAG, i + " event(s) was deleted for Source " + mCurrentSource);
                }
                result = OP_SUCCESS;
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Event delete error: " + e.getMessage());
                }
                result = OP_ERROR;
            }
            return result;
        }

        /**
         * Add friend (for current source).
         *
         * @return integer status code.
         */
        private int friendAdd() {
            int result = OP_ERROR;

            if (mCurrentSource == -1) {
                return OP_IGNORED;
            }
            ContentValues friendsRegValues = new ContentValues();
            String candidateRawContact = null;
            Uri rawContactUri = null;

            // We would like to pick an already existing raw contact from time to
            // time. We check UI check-box if this is desired.Check if we should
            // try to reuse raw contact.
            if (mFriendAddNewContactWithExistingNameEnabled) {
                // Get existing raw contact from friend table
                Cursor cursor = null;

                try {
                    // List all friends.
                    cursor = getContentResolver().query(EventStreamProvider.URIs.FRIENDS, null,
                            null, // all
                            null, null);

                    if (cursor != null) {
                        if (Config.DEBUG) {
                            Log.d(LOG_TAG, "Total number of friends: " + cursor.getCount() + ".");
                        }
                    }

                    int candidateIndex = -1;
                    boolean abort = false;

                    // Loop through friends and look for friend added by other
                    // source.
                    if ((cursor != null) && (cursor.getCount() > 0)) {
                        int columnIndexSourceId = cursor
                                .getColumnIndexOrThrow(EventStreamProvider.Friend.COLUMN_SOURCE_ID);
                        int columnIndexContactReference = cursor
                                .getColumnIndexOrThrow(EventStreamProvider.Friend.COLUMN_CONTACT_RAW_ID);

                        while (!abort) {
                            while (cursor.moveToNext()) {
                                // Was this friend added by other than current
                                // source?
                                if (cursor.getInt(columnIndexSourceId) != mCurrentSource) {
                                    // Use this friend as candidate.
                                    candidateIndex = cursor.getPosition();
                                    candidateRawContact = cursor.getString(columnIndexContactReference);
                                    break;
                                }
                            }

                            // Check so that the raw contact is unique for current
                            // source.
                            if ((candidateIndex != -1) && (candidateRawContact != null)) {
                                cursor.moveToPosition(-1);
                                while (cursor.moveToNext()) {
                                    // Found candidate must not already be used
                                    // by current source.
                                    if (candidateRawContact.equals(columnIndexContactReference)
                                            && (cursor
                                                    .getLong(cursor
                                                            .getColumnIndexOrThrow(Friend.COLUMN_SOURCE_ID)) == mCurrentSource)) {
                                        // Found another friend with same
                                        // Contacts Lookup key. Skip this candidate and
                                        // its Contacts Lookup key. Move back cursor to
                                        // previous candidate and look for new
                                        // candidate
                                        cursor.moveToPosition(candidateIndex);
                                        candidateIndex = -1;
                                        candidateRawContact = null;
                                        break;
                                    }
                                }

                                // We have our final 'candidate' that will be
                                // used.
                                if (candidateIndex != -1) {
                                    if (candidateRawContact != null) {
                                        abort = true;
                                        rawContactUri = Uri.parse(candidateRawContact);
                                    }
                                }
                            } else {
                                abort = true;
                            }
                        }
                    }
                } finally {
                    if (cursor != null) {
                        cursor.close();
                    }
                }
            } else if (mFriendAddNewContactWithUniqueName) {
                // Add contact in phone contacts provider
                // and store the generated ContactRawId.

                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Add new contact to phone...");
                }

                TestContact testContact = new TestContact(
                        "com.sonyericsson.eventstream.testplugin", "Source " + mCurrentSource
                                + mFriendCount, "Firstname", "Lastname" + mFriendCount,
                        getContentResolver());

                testContact.setRingtone(TestPlugin.this.getApplicationContext());
                testContact.setNotes();
                testContact.setNumbers();
                testContact.setEmail();
                testContact.setIM(8);
                testContact.setPostal();
                testContact.setOrganization();

                rawContactUri = testContact.getRawUri();
            }

            if (rawContactUri != null) {
                String friendKey = System.currentTimeMillis() + rawContactUri.toString();

                friendsRegValues.put(EventStreamProvider.Friend.COLUMN_FRIEND_KEY, friendKey);
                friendsRegValues.put(EventStreamProvider.Friend.COLUMN_CONTACT_RAW_ID, rawContactUri.toString());
                friendsRegValues.put(EventStreamProvider.Friend.COLUMN_SOURCE_ID, mCurrentSource);

                Uri uri = getContentResolver().insert(EventStreamProvider.URIs.FRIENDS,
                            friendsRegValues);

                if (uri != null) {
                    mFriendCount++;
                    if (Config.DEBUG) {
                        Log.d(LOG_TAG, "Friend was added. (Raw contact uri: " + rawContactUri.toString() + ")");
                    }

                    return OP_SUCCESS;
                }
            }
            return result;
        }
    }
}
