/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.service.TwitterPluginService;
import com.sonyericsson.eventstream.twitterplugin.utility.EventStreamHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;

import android.app.Application;
import android.content.ComponentName;
import android.content.Intent;
import android.content.res.Configuration;
import android.util.Log;

import java.util.HashSet;
import java.util.Set;

public class TwitterPluginApplication extends Application {

    /**
     * The plug-in states
     */
    public enum State {
        SERVICE_NOT_CONFIGURED,
        SERVICE_AUTHENTICATED,
        SERVICE_NOT_AUTHENTICATED,
        SERVICE_AUTHENTICATION_IN_PROGRESS,
        SERVICE_AUTHENTICATION_FAILED,
        SERVICE_AUTHENTICATION_SUCCESS,
        ACTIVITY_SHOW_DISCLAIMER,
        ACTIVITY_INVALID_ACCOUNT,
        ACTIVITY_MISSING_PASSWORD,
        ACTIVITY_MISSING_USERNAME,
        ACTIVITY_NETWORK_FAILED,
        ACTIVITY_CREATE_ACCOUNT,
    }

    /** Keeps the current configuration activity visibility */
    private boolean mIsConfigurationShown = false;

    /** The current state */
    private State mState = State.SERVICE_NOT_CONFIGURED;

    /** Set of all the state listeners */
    private Set<Listener> mListeners;

    /**
     * This is the main entry to this application.
     * If we can find a token from Twitter then
     * we set the state to be authenticated.
     */
    @Override
    public void onCreate() {
        super.onCreate();
        mListeners = new HashSet<Listener>();
        if (PreferencesHelper.hasToken(this)
                && EventStreamHelper.isPluginRegistered(getContentResolver())) {
            setState(State.SERVICE_AUTHENTICATED);
        }
    }

    /**
     * Interface for the state listener
     */
    public interface Listener {
        /**
         * Indicate that the current state has changed
         * @param oldState The old state
         * @param newState The new changed state
         */
        void onStateChange(State oldState, State newState);
    }

    /**
     * Add a state listener
     * @param listener The listener to set
     * @return true if the listener was added
     */
    public boolean addListener(Listener listener) {
        if (listener != null) {
            return mListeners.add(listener);
        }
        return false;
    }

    /**
     * Remove the state listener
     * @param listener The listener to remove
     * @return true if the listener exist and was removed
     */
    public boolean removeListener(Listener listener) {
        if (listener != null) {
            return mListeners.remove(listener);
        }
        return false;
    }

    /**
     * Get the current plug-in state
     * @return The current state
     */
    public State getState() {
        return mState;
    }

    /**
     * Set the current plug-in state
     * @param state The current state to set
     */
    public void setState(State state) {
        State oldState = mState;
        mState = state;
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "state: " + state.name());
        }
        for (Listener listener : mListeners) {
            listener.onStateChange(oldState, mState);
        }
    }

    /**
     * Will set the configuration activity visibility
     * @param visible true if the view is shown
     */
    public void setConfigurationActivityVisible(boolean visible) {
        mIsConfigurationShown = visible;
    }

    /**
     * Will return the current activity visibility state
     * @return true if the view is shown.
     */
    public boolean isConfigurationActivityVisible() {
        return mIsConfigurationShown;
    }

}
