/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/
package com.sonyericsson.eventstream.twitterplugin.utility;

import android.content.Context;
import android.content.SharedPreferences;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;

public class PreferencesHelper {

    /** Config store id */
    public static final String CONFIG_STORE = "twitter-plugin-store.config";

    /** Preference key, the Twitter token key */
    private static final String TOKEN_KEY = "twitter.oauth.token";

    /** Preference key, the Twitter token secret key */
    private static final String TOKEN_SECRET_KEY = "twitter.oauth.token_secret";

    /** Preference key, the Twitter paging message */
    private static final String PAGING_SINCE_MESSAGE_ID = "twitter.paging.since_message_id";

    /** Preference key, the Twitter paging status */
    private static final String PAGING_SINCE_STATUS_ID = "twitter.paging.since_statuses_id";

    /** Preference key, true if disclaimer is accepted */
    private static final String DISCLAIMER = "accepted_disclaimer";

    /** Indicates that we have registered the plugin */
    private static final String REGISTERED = "registered";

    /** current language setting for plugin */
    private static final String LANGUAGE_SETTING = "language";

    /**
     * If the disclaimer has been accepted.
     *
     * @param context   the context
     *
     * @return  true if the disclaimer has been accepted
     */
    public static boolean hasAcceptedDisclaimer(Context context) {
        if (context == null) {
            return false;
        }
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            return preferences.getBoolean(DISCLAIMER, false);
        } else {
            return false;
        }
    }

    /**
     * Set the twitter disclaimer as accepted
     *
     * @param context   the context
     * @param accepted  the value
     */
    public static void setAcceptedDisclaimer(Context context) {
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            editor.putBoolean(DISCLAIMER, true);
            editor.commit();
        }
    }

    /**
     * Check if it exist a token
     * @param context The Context
     * @return true if token exist else false
     */
    public static boolean hasToken(Context context) {
        if (context == null) {
            return false;
        }
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            return preferences.contains(TOKEN_KEY) && preferences.contains(TOKEN_SECRET_KEY);
        } else {
            return false;
        }
    }

    /**
     * Get the token key
     * @param context The Context
     * @return The token key if exist else empty string
     */
    public static String getTokenKey(Context context) {
        if (context == null) {
            return "";
        }
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            return preferences.getString(TOKEN_KEY, "");
        } else {
            return "";
        }
    }

    /**
     * Get the token secret key
     * @param context The Context
     * @return The token key if exist else empty string
     */
    public static String getTokenSecretKey(Context context) {
        if (context == null) {
            return "";
        }
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            return preferences.getString(TOKEN_SECRET_KEY, "");
        } else {
            return "";
        }
    }

    /**
     * Set the token key and secret key
     * @param context The Context
     * @param key The token key
     * @param secretKey The token secret key
     */
    public static void setToken(Context context, String key, String secretKey) {
        if (context == null || key == null || secretKey == null) {
            return;
        }
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.putString(TOKEN_KEY, key);
                editor.putString(TOKEN_SECRET_KEY, secretKey);
                editor.commit();
            }
        }
    }

    /**
     * Get the paging since id
     * @param context The Context
     * @param pagingType TwitterConf.PAGING_TYPE_MESSAGE or TwitterConf.PAGING_TYPE_STATUS
     * @return The since id for the type else 0
     */
    public static long getPagingSinceId(Context context, String pagingType) {
        if (context == null) {
            return 0;
        }

        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences == null) {
            return 0;
        }

        if (pagingType.equals(TwitterConf.PAGING_TYPE_MESSAGE)) {
            return preferences.getLong(PAGING_SINCE_MESSAGE_ID, 0);
        } else if (pagingType.equals(TwitterConf.PAGING_TYPE_STATUS)) {
            return preferences.getLong(PAGING_SINCE_STATUS_ID, 0);
        } else  {
            return 0;
        }
    }

    /**
     * Set the paging since id for the type
     * @param context The Context
     * @param pagingType TwitterConf.PAGING_TYPE_MESSAGE or TwitterConf.PAGING_TYPE_STATUS
     * @param lastId The id to set
     */
    public static void setPagingSinceId(Context context, String pagingType, long lastId) {
        if (context == null) {
            return;
        }

        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences == null) {
            return;
        }

        SharedPreferences.Editor editor = preferences.edit();
        if (editor != null) {
            if (pagingType.equals(TwitterConf.PAGING_TYPE_MESSAGE)) {
                editor.putLong(PAGING_SINCE_MESSAGE_ID, lastId);
                editor.commit();
            } else if (pagingType.equals(TwitterConf.PAGING_TYPE_STATUS)) {
                editor.putLong(PAGING_SINCE_STATUS_ID, lastId);
                editor.commit();
            }
        }
    }

    /**
     * Set us as registered
     *
     * @param context The Context
     */
    public static void setRegistered(Context context) {
        if (context == null) {
            return;
        }

        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences == null) {
            return;
        }

        SharedPreferences.Editor editor = preferences.edit();
        if (editor != null) {
            editor.putBoolean(REGISTERED, true);
            editor.commit();
        }
    }

    /**
     * Set us as registered
     *
     * @param context The Context
     */
    public static boolean isRegistered(Context context) {
        if (context == null) {
            return false;
        }

        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences == null) {
            return false;
        }

        return preferences.contains(REGISTERED);
    }

    /**
     * Will clear the authentication information stored
     * @param context The Context
     */
    public static void clearAuthentication(Context context) {
        if (context == null) {
            return;
        }
        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.remove(TOKEN_KEY);
                editor.remove(TOKEN_SECRET_KEY);
                editor.remove(PAGING_SINCE_MESSAGE_ID);
                editor.remove(PAGING_SINCE_STATUS_ID);
                editor.commit();
            }
        }
    }

    /**
     * Will clear the authentication information stored
     * @param context The Context
     */
    public static void clearAll(Context context) {
        if (context == null) {
            return;
        }

        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.clear();
                editor.commit();
            }
        }
    }

    /**
     * Will clear the authentication information stored
     * @param context The Context
     */
    public static void logout(Context context) {
        if (context == null) {
            return;
        }

        SharedPreferences preferences =
            context.getSharedPreferences(CONFIG_STORE, Context.MODE_PRIVATE);

        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.clear();
                editor.putBoolean(REGISTERED, true);
                editor.commit();
            }
        }
    }
}
