#!/usr/bin/env python
# -*- coding: utf-8 -*-

import rp
from util import *

# トークン基底
class IToken:
	_app  = None # アプリケーション
	_g    = None # グラフィック
	_f    = None # フォント
	_i    = None # 入力
	_m    = None # サウンド
	_game = None # セーブデータ

	@staticmethod
	def set(app, game=None):
		IToken._app  = app
		IToken._g    = app.graphics
		IToken._f    = app.font
		IToken._i    = app.input
		IToken._m    = app.music
		IToken._game = game

# トークン実装
class Token(IToken):
	# 当たり判定モード
	_MODE = "rect" # 矩形

	# 移動量
	_VEL_DEFAULT = 1.0
	vel_ratio = _VEL_DEFAULT

	# 画面サイズ(描画サイズ)
	SX1 = 0 # 左上座標(X)
	SY1 = 0 # 左上座標(Y)
	SX2 = 0 # 右下座標(X)
	SY2 = 0 # 右下座標(Y)
	# 画面サイズ(メインゲームサイズ)
	MX1 = 0 # 左上座標(X)
	MY1 = 0 # 左上座標(Y)
	MX2 = 0 # 右下座標(X)
	MY2 = 0 # 右下座標(Y)
	@classmethod
	def setMainArea(cls, x1, y1, x2, y2):
		cls.MX1, cls.MY1 = x1, y1
		cls.MX2, cls.MY2 = x2, y2
	@classmethod
	def setArea(cls, x1, y1, x2, y2):
		cls.SX1, cls.SY1 = x1, y1
		cls.SX2, cls.SY2 = x2, y2
	@classmethod
	def getMainArea(cls):
		return (cls.MX1, cls.MY1, cls.MX2, cls.MY2)
	@classmethod
	def getArea(cls):
		return (cls.SX1, cls.SY1, cls.SX2, cls.SY2)
	@classmethod
	def setSlow(cls, ratio=_VEL_DEFAULT):
		cls.vel_ratio = ratio
	@classmethod
	def isSlow(cls):
		return cls.vel_ratio != cls._VEL_DEFAULT

	def __init__(self):
		self.x     = 0
		self.y     = 0
		self.vx    = 0
		self.vy    = 0
		self.ax    = 0
		self.ay    = 0
		self.deg   = 0
		self.speed = 0
		self.exist = False
		self.w     = 0
		self.h     = 0
		self.size  = 0
		self.pause = False
		self.hitmode = Token._MODE

	def dump(self):
		print("(x,y)=(%d,%d) (vx,vy)=(%d,%d) (ax,ay)=(%d,%d)"%(self.x, self.y, self.vx, self.vy, self.ax, self.ay))

	def create(self, x, y, deg=0, speed=0, ax=0, ay=0):
		self.x, self.y  = x, y
		self.vx    = Util.cosEx(deg) * speed
		self.vy    = -Util.sinEx(deg) * speed
		self.deg   = deg
		self.speed = speed
		self.ax, self.ay = ax, ay
		self.exist = True

	def vanish(self):
		self.exist = False

	# 移動する
	def move(self, bSlow=False):
		ratio = 1.0
		if bSlow:
			# スローモーション有効
			ratio = Token.vel_ratio

		self.vx += self.ax * ratio
		self.vy += self.ay * ratio
		self.x  += self.vx * ratio
		self.y  += self.vy * ratio

	# 移動量減衰
	def decay(self, a):
		self.vx *= a
		self.vy *= a

	# サイズ設定(矩形)
	def setSize(self, w, h):
		self.w = w
		self.h = h
		self.hitmode = "rect"

	# サイズ設定(円)
	def setSize2(self, size):
		self.size = size
		self.hitmode = "circle"

	# 画面内にいるかどうかチェック
	def isInArea(self, type="screen"):

		x1, y1, x2, y2 = (0, 0, 0, 0)

		sx1, sy1, sx2, sy2 = Token.getArea()
		if type == "main":
			sx1, sy1, sx2, sy2 = Token.getMainArea()

		if(self.hitmode == "rect"):

			# 矩形で判定
			x1, y1 = sx1 - self.w, sy1 - self.h
			x2, y2 = sx2 + self.w, sy2 + self.h
		else:

			# 円で判定
			x1, y1 = sx1 - self.w, sy1 - self.h
			x2, y2 = sx2 + self.w, sy2 + self.h

		if(self.x < x1): return False # 画面外
		if(self.y < y1): return False # 画面外
		if(self.x > x2): return False # 画面外
		if(self.y > y2): return False # 画面外

		# 画面内
		return True

	def isInAreaMain(self):
		return self.isInArea(self, "main")

	# 画面内に押し戻す
	def clipInArea(self, type="screen"):

		# 画面サイズ
		sx1, sy1, sx2, sy2 = Token.getArea()
		if type == "main":
			sx1, sy1, sx2, sy2 = Token.getMainArea()

		# 矩形で判定
		sx1 = sx1 + self.w
		sy1 = sy1 + self.h
		sx2 = sx2 - self.w
		sy2 = sy2 - self.h
		if self.hitmode == "circle":
			# 円で判定
			sx1 = sx1 + self.size
			sy1 = sy1 + self.size
			sx2 = sx2 - self.size
			sy2 = sy2 - self.size

		x1 = self.x
		y1 = self.y
		bClipX, bClipY = False, False
		if x1 < sx1: self.x = sx1; bClipX = True
		if y1 < sy1: self.y = sy1; bClipY = True
		if x1 > sx2: self.x = sx2; bClipX = True
		if y1 > sy2: self.y = sy2; bClipY = True

		return bClipX, bClipY

	# 画面端で反射する
	def reflectInArea(self, type="screen"):
		bClipX, bClipY = self.clipInArea(type)
		if bClipX: self.vx *= -1
		if bClipY: self.vy *= -1

	# 当たり判定チェック
	def isHit(self, t):

		if(self.hitmode == "rect"):

			# 矩形同士
			left1,  top1    = self.x,          self.y
			right1, bottom1 = self.x + self.w, self.y + self.h
			left2,  top2    = t.x,             t.y
			right2, bottom2 = t.x + t.w,       t.y + t.h
			return (left1 < right2) and (right1 > left2) and (top1 < bottom2) and (bottom1 > top2)
		else:

			# 円同士
			dx = t.x - self.x
			dy = t.y - self.y
			return (self.size**2 + t.size**2) < (dx**2 + dy**2)

	# 当たり判定チェック(x,yが中心座標)
	def isHitEx(self, t):

		if(self.hitmode == "rect"):

			# 矩形同士
			left1,  top1    = self.x,          self.y
			right1, bottom1 = self.x + self.w, self.y + self.h
			left2,  top2    = t.x - t.w,       t.y - t.h
			right2, bottom2 = t.x + t.w,       t.y + t.h
			return (left1 < right2) and (right1 > left2) and (top1 < bottom2) and (bottom1 > top2)
		else:

			# 円同士
			dx = t.x - self.x
			dy = t.y - self.y
			return (self.size**2 + t.size**2) > (dx**2 + dy**2)

	def getAim(self, t):
		return self.getAimEx(t.x, t.y)

	def getAimEx(self, x, y):
		dx = x - self.x
		dy = y - self.y
		aim = Util.atan2Ex(dy, -dx)
		return aim

	def getAimLength(self, t):
		dx = t.x - self.x
		dy = t.y - self.y
		return Util.sqrt(dx**2 + dy**2)

	# 描画
	def drawEx(self, r, g, b, a=255):
		self._g.setColor(r, g, b, a)
		if self.hitmode == "rect":
			self._g.fillRectEx(self.x, self.y, self.w, self.h)
		else:
			self._g.fillCircle(self.x, self.y, self.size)
	def drawEx2(self, color):
		self._g.setColor2(color)
		if self.hitmode == "rect":
			self._g.fillRectEx(self.x, self.y, self.w, self.h)
		else:
			self._g.fillCircle(self.x, self.y, self.size)


# Token管理
class TokenManager:
	def __init__(self, path_to_class, size):
		self.pool = []
		target = path_to_class.split(".")
		package  = target[0]
		module   = ".".join(target[:-1])
		cls_name = target[-1]
		mod = __import__(module, fromlist=[package])
		for i in range(size):
			t = getattr(mod, cls_name)()
			self.pool.append(t)
	def add(self):
		for t in self.pool:
			if t.exist == False:
				return t
		return None
	def pause(self):
		for t in self.pool:
			t.pause = True
	def unpause(self):
		for t in self.pool:
			t.pause = False
	def count(self):
		ret = 0
		for t in self.pool:
			if t.exist: ret += 1
		return ret
	def update(self):
		for t in self.pool:
			if t.pause == True: continue
			if t.exist == False: continue
			t.update()
	def draw(self):
		for t in self.pool:
			if t.exist == False: continue
			t.draw()
	def vanish(self):
		for t in self.pool:
			t.exist = False
