/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.android.mediaframeworktest.functional;

import android.content.Context;
import android.media.MediaScannerConnection;
import android.media.MediaScannerConnection.MediaScannerConnectionClient;
import android.net.Uri;

import android.util.Log;


/**
 * Class for scanning of a DRM file (can be used for non DRM file too).
 * Its a helper class and doesn't implement any test case.
 */
public class DrmMediaScan {

    private static final String TAG = "DrmMediaScan";

    private MediaScannerConnection mConnection;
    // This is not used but kept it for keeping track of scanning
    private volatile boolean isScanOver = false;
    private String filePath;
    private String mimeType;
    public String resultPath;
    public Uri resultUri;

    class DrmMediaConnectionClient implements MediaScannerConnectionClient {
        Object syncObj;

        /**
         * Constructs a new DrmMediaConnectionClient.
         * It accepts an object as a parameter. This object is notified
         * when the scan is over.
         *
         * @param syncObj object for synchronization.
         */
        DrmMediaConnectionClient(Object syncObj) {
            this.syncObj = syncObj;
        }

        /**
         * call back when connected to Media Scanner service. After connection,
         * scanning is started
         */
        public void onMediaScannerConnected() {
            mConnection.scanFile(filePath, mimeType);
        }

        /**
         * call back when scanning gets completed. It notifies the caller
         * by syncObj.
         * It accepts an object as a parameter. This object is notified
         * when the scan is over.
         *
         * @param path scanned object path
         * @param uri content uri of scanned object.
         */
        public void onScanCompleted(String path, Uri uri) {
            resultPath = path;
            resultUri = uri;
            Log.v(TAG, "Scan Complete for File " + path);
            synchronized(syncObj) {
                isScanOver = true;
                syncObj.notify();
            }

        }
    };

    private DrmMediaConnectionClient drmMediaConnectionClient;

    /**
     * Constructs a new DrmMediaScan object and hence MediaScannerConnection object.
     *
     * @param context the Context object, required for establishing a connection to
     * the media scanner service.
     */
    public DrmMediaScan(Context context) {
        drmMediaConnectionClient = new DrmMediaConnectionClient(this);
        mConnection = new MediaScannerConnection(context, drmMediaConnectionClient);
    }

    /**
     * Scans a DRM file. It waits maximum 10 seconds for scan to complete.
     *
     * @param filePath file path for scanning
     * @param mimeType mime type for file.
     */
    public void scanDrmFile(String filePath, String mimeType) {
        if(null != filePath && null != mimeType) {
            this.filePath = filePath;
            this.mimeType = mimeType;

            mConnection.connect();

            try {
                synchronized(this) {
                    // giving 10 seconds so it wont block
                    this.wait(10000);
                }
            } catch(InterruptedException e) {
                e.printStackTrace();
            }

            mConnection.disconnect();
        }
    }

}
