/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file FacebookCommunication.java
 *
 * @author
 */
package com.sonyericsson.eventstream.facebookplugin;

import android.net.http.AndroidHttpClient;

import android.util.Log;
import android.util.Xml;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;
import com.sonyericsson.eventstream.facebookplugin.Facebook.Communication.CommunicationResult;
import com.sonyericsson.eventstream.facebookplugin.Facebook.EventStorage.EventType;

import static com.sonyericsson.eventstream.facebookplugin.Constants.*;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.message.BasicNameValuePair;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

public class Facebook {

    /**
     * Exception to indicate a failure in the communication
     * with the server
     */
    static public class FacebookException extends Exception {
        private static final long serialVersionUID = 6408906217064226916L;

        /**
         * Failure to authenticate
         */
        public static final int AUTHENTICATION_FAILED = 0;

        /**
         * Invalid credentials provided by the user
         */
        public static final int AUTHENTICATION_INVALID_CREDENTIALS = 1;

        /**
         * A http error occured.
         * Check httpResponseCode for details
         */
        public static final int HTTP_ERROR = 2;

        /**
         * Couldn't connect to the server, check data connectivity
         */
        public static final int COMMUNICATION_ERROR = 3;

        /**
         * Error parsing document retrieved from server
         */
        public static final int PARSER_ERROR = 4;

        public int cause;
        public int httpResponseCode;

        public FacebookException(int cause) {
            this.cause = cause;
        }

        public FacebookException(int cause, int httpResponseCode) {
            this.cause = cause;
            this.httpResponseCode = httpResponseCode;
        }

        public String toString() {
            return "cause=" + cause + " httpResponseCode=" + httpResponseCode;
        }
    }

    public enum ServiceState {
        AUTHENTICATION_IN_PROGRESS,
        SERVER_OPERATION_IN_PROGRESS,
        PASSIVE;
    }

    /**
     * Interface for storing different kinds of events
     */
    public interface EventStorage {
        enum EventType {
            LINK_EVENT,
            STATUS_EVENT,
            PHOTO_EVENT,
            MESSAGE_EVENT
        };
        public void setOwnStatus(String message, long timestamp);
        public void addEvent(String eventId, String message, String title, String picture,
                String friendId, long timestamp, EventType eventType);
    }

    /**
     * Interface for storing friends
     */
    public interface FriendStorage {
        public Set<String> getFriendIds();
        public void removeFriend(String friendId);
        public boolean isFriend(String friendId);
        public void addFriend(String friendId, String profileImageUri, String friendName);
    }

    /**
     * Callback for state changes
     */
    public interface ServiceStateChangeListener {
        void onServiceStateChanged(ServiceState oldState, ServiceState newState);
    }

    public interface Settings {
        public String getOwnId();
        public void setOwnId(String id);
        public String getAuthenticationToken();
        public void setAuthenticationToken(final String token);
        public Long getYoungestTimestamp();
        public void setYoungestTimestamp(Long newValue);
        public void setSessionKey(String sessionKey);
        public void setSessionSecret(String sessionSecret);
    }

    public static class Communication {
        private AndroidHttpClient mHttpClient;

        public static class CommunicationResult {
            private int mHttpResult;
            private String mHttpPhrase;
            private String mHttpContent;

            public CommunicationResult(int result, String phrase, String content) {
                mHttpResult = result;
                mHttpPhrase = phrase;
                mHttpContent = content;
            }

            public int getHttpResult() {
                return mHttpResult;
            }

            public String getHttpPhrase() {
                return mHttpPhrase;
            }

            public String getHttpContent() {
                return mHttpContent;
            }
        }

        public Communication() {
            // Deliberate empty
        }

        public Communication(String userAgentString) {
            mHttpClient = AndroidHttpClient.newInstance(userAgentString);
        }

        public CommunicationResult executePost(String uri) throws ClientProtocolException, IOException {
            HttpPost httpPost = new HttpPost(uri);

            return execute(httpPost);
        }

        public CommunicationResult executeGet(String uri) throws ClientProtocolException, IOException {
            HttpGet httpGet = new HttpGet(uri);

            return execute(httpGet);
        }

        public void closeConnection() {
            mHttpClient.getConnectionManager().closeExpiredConnections();
        }

        public void shutdown() {
            mHttpClient.close();
            mHttpClient = null;
        }

        private CommunicationResult execute(HttpUriRequest operation) throws ClientProtocolException, IOException {
            HttpResponse httpResponse = mHttpClient.execute(operation);
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            String phrase = httpResponse.getStatusLine().getReasonPhrase();
            String content = getContent(httpResponse);

            return new CommunicationResult(statusCode, phrase, content);
        }

        private String getContent(HttpResponse response) throws IllegalStateException, IOException {
            StringBuilder builder = new StringBuilder();
            HttpEntity entity = response.getEntity();

            if (entity != null) {
                InputStream inputStream = entity.getContent();

                try {
                    byte[] buffer = new byte[512];
                    int bytesRead;

                    while ((bytesRead = inputStream.read(buffer)) != -1) {
                        builder.append(new String(buffer, 0, bytesRead, ENCODING));
                    }
                } finally {
                    if (inputStream != null) {
                        inputStream.close();
                    }

                    entity.consumeContent();
                }
            }
            return builder.toString();
        }
    }

    private static final String ENCODING = "UTF-8";
    private Communication mCommunication = null;
    private ServiceState mState;
    private Set<ServiceStateChangeListener> mListeners;
    private Settings mSettings;

    public Facebook(Settings settings) {
        mListeners = new HashSet<ServiceStateChangeListener>();
        mSettings = settings;
        setServiceState(ServiceState.PASSIVE);
    }

    public void initialize() {
        if (mCommunication == null) {
            mCommunication = new Communication("UserAgent"); // TODO Change this later on??!?
        }
        setServiceState(ServiceState.PASSIVE);
    }

    public boolean verifyAccessToken(String accessToken) {
        boolean verified = false;
        if (accessToken == null) {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Plugin not fully registered or user not logged in. Accesstoken == null");
            }
        }  else {
            verified = true;
        }
        return verified;
    }

    public synchronized boolean isLoggedIn() {
        String accessToken = mSettings.getAuthenticationToken();
        return verifyAccessToken(accessToken);
    }

    public synchronized Facebook.ServiceState getState() {
        return mState;
    }

    /**
     * Authenticate the user
     *
     * @param email
     * @param password
     * @return the display name
     */
    public String authenticate(final String email, final String password) throws FacebookException {
        String result = null;

        //verify if any data was inserted
        if (email == null || password == null || email.length() == 0 || password.length() == 0) {
            throw new FacebookException(FacebookException.AUTHENTICATION_INVALID_CREDENTIALS);
        }

        try {
            String accessToken = null;

            // Clear all data associated with a session!
            clearSessionData();
            accessToken = callFacebookAuthLogin(email, password);

            if (isLoggedIn()) {
                JSONObject currentUser = httpGetToGraphApi(MYSELF_PATH, "");

                if (currentUser != null) { // TODO check that the fields are available...
                    result = currentUser.getString("name");
                    String ownId = currentUser.getString("id");
                    mSettings.setOwnId(ownId);
                }
            } else {
                throw new FacebookException(FacebookException.AUTHENTICATION_FAILED);
            }
        } catch (JSONException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error reading current username from Facebook API.");
            }
            // Ensure that we are in logged out state as we couldn't authenticate/get the me-profile
            clearSessionData();

            throw new FacebookException(FacebookException.AUTHENTICATION_FAILED);
        } catch (IOException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "I/O error reading data from Facebook.");
            }
            // Ensure that we are in logged out state as we couldn't authenticate/get the me-profile
            clearSessionData();

            throw new FacebookException(FacebookException.AUTHENTICATION_FAILED);
        }
        return result;
    }

    public boolean updateStatus(String message) throws FacebookException {
        boolean result = false;

        if (! isLoggedIn()) {
            return false;
        }

        try {
            final String accessToken = mSettings.getAuthenticationToken();

            final String uri = GRAPH_API_BASE_URL + MY_FEED_PATH + "?access_token="
                    + accessToken + "&message="
                    + URLEncoder.encode(message, ENCODING);
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Calling " + GRAPH_API_BASE_URL + MY_FEED_PATH);
                Log.d(LOG_TAG, "Message is:" + URLEncoder.encode(message, ENCODING));
            }

            setServiceState(ServiceState.SERVER_OPERATION_IN_PROGRESS);

            CommunicationResult operationResult = mCommunication.executePost(uri);
            int statusCode = operationResult.getHttpResult();

            if (statusCode == HttpStatus.SC_OK) {
                result = true;
            } else {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Failed to update current status in Event Stream: "
                            + statusCode + " "
                            + operationResult.getHttpPhrase());
                }
                FacebookException fbError;

                fbError = new FacebookException(FacebookException.COMMUNICATION_ERROR,
                        statusCode);
                throw fbError;
            }
        } catch (IOException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error calling Facebook API.");
            }
            throw new FacebookException(FacebookException.COMMUNICATION_ERROR);
        } finally {
            setServiceState(ServiceState.PASSIVE);
        }

        return result;
    }

    public boolean retrieveLatestPosts(EventStorage eventStorage, FriendStorage friendStorage) throws FacebookException {
        boolean result = false;

        if (Config.DEBUG) {
            Log.d(LOG_TAG, "Retrieve latest Facebook statuses.");
        }

        if (! isLoggedIn()) {
            return false;
        }
        try {
            // Parser for date strings of format "2010-05-12T12:11:02+0000"
            // Facebook wants a timestamp in seconds (no millisec as Eventstream)
            Long since = mSettings.getYoungestTimestamp() / 1000;
            String parameters = null;
            if (since.equals(0L)) {
                parameters = "&since=" + since + "&limit=" + LIMIT_FIRST_TIME;
            } else {
                parameters = "&since=" + since + "&limit=" + LIMIT_DEFAULT;
            }

            JSONObject jsonObject = httpGetToGraphApi(NEWS_FEED_PATH, parameters);
            long youngestTimestamp = insertFacebookPostsInEventStream(eventStorage, friendStorage, jsonObject);

            mSettings.setYoungestTimestamp(youngestTimestamp);
            result = true;
        } catch (IOException ioe) {
            throw new FacebookException(FacebookException.COMMUNICATION_ERROR);
        } catch (JSONException jse) {
            throw new FacebookException(FacebookException.PARSER_ERROR);
        } catch (ParseException pe) {
            throw new FacebookException(FacebookException.PARSER_ERROR);
        }
        return result;
    }

    /**
     * Refresh our local friend list, handle added and removed friends
     */
    public boolean refreshFacebookFriends(FriendStorage storage) throws FacebookException {
        boolean result = false;

        if (! isLoggedIn()) {
            return false;
        }

        try {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Register friends.");
            }
            String accessToken = mSettings.getAuthenticationToken();

            Set<String> serverIds = new HashSet<String>();
            JSONObject jsonObject = httpGetToGraphApi(FRIENDS_FEED_PATH, "");
            if (jsonObject != null) {

                JSONArray dataArray = jsonObject.getJSONArray(Constants.DATA_PARAMTER);
                int length = dataArray.length();

                for (int i = 0; i < length; i++) {
                    try {
                        JSONObject friendsObject = dataArray.getJSONObject(i);
                        String friendId = friendsObject.getString("id");

                        if (friendId != null ) {
                            String profileImageUri = GRAPH_API_BASE_URL + '/' + friendId +
                                                    "/picture?access_token=" + accessToken + "&type=large";
                            String friendName = friendsObject.getString("name");

                            storage.addFriend(friendId, profileImageUri, friendName);
                            serverIds.add(friendId);
                        }
                    } catch (JSONException e) {
                        if (Config.DEBUG) {
                            Log.e(LOG_TAG, "Error parsing JSON data.");
                        }
                    }
                }
            }
            // Removed unfriended persons from the friend table
            Set<String> localIds = storage.getFriendIds();

            localIds.removeAll(serverIds);
            for (String friendId : localIds) {
                storage.removeFriend(friendId);
            }
            result = (jsonObject != null);
        } catch (JSONException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error parsing JSON data.");
            }
            throw new FacebookException(FacebookException.PARSER_ERROR);
        } catch (IOException ioe) {
            ioe.printStackTrace();
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Error cause:" + ioe.toString());
            }
            throw new FacebookException(FacebookException.COMMUNICATION_ERROR);
        } finally {
            setServiceState(ServiceState.PASSIVE);
        }

        return result;
    }

    public synchronized void setServiceStateListener(ServiceStateChangeListener callback) {
        if (callback != null) {
            mListeners.add(callback);
        }
    }

    public synchronized void removeServiceStateListener(ServiceStateChangeListener callback) {
        if (callback != null) {
            mListeners.remove(callback);
        }
    }

    public void shutdown() {
        mCommunication.closeConnection();
        mCommunication.shutdown();
        mCommunication = null;
    }

    public void close() {
        mCommunication.closeConnection();
    }

    private synchronized void setServiceState(ServiceState state) {
        if (mState != state) {
            ServiceState oldState = mState;

            mState = state;
            for (ServiceStateChangeListener l : mListeners) {
                l.onServiceStateChanged(oldState, mState);
            }
        }
    }

    private long insertFacebookPostsInEventStream(EventStorage eventStorage, FriendStorage friendStorage,
            JSONObject jsonObject) throws JSONException, ParseException, FacebookException, IOException {
        // English locale is used here because English (either UK or US) is
        // guaranteed to exist in all phones sold over the world.
        // If using Default locale, parsing will fail when using Arabic language
        // and possibly other languages
        SimpleDateFormat facebookDateParser = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZZZZ", Locale.ENGLISH);
        long youngestTimestamp = mSettings.getYoungestTimestamp();

        if (jsonObject != null) {
            JSONArray dataArray = jsonObject.getJSONArray(Constants.DATA_PARAMTER);

            if (dataArray == null || dataArray.length() == 0) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Empty result set from Facebook - no need to parse the data.");
                }
                return youngestTimestamp;
            }

            int length = dataArray.length();
            long latestStatusUpdate = 0;
            boolean ownStatusSet = false;

            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Retrieved " + length + " Facebook objects.");
            }
            for (int i = 0; i < length; i++) {
                try {
                    JSONObject eventObject = dataArray.getJSONObject(i);
                    String facebookEventId = eventObject.getString("id");
                    String fromId = eventObject.getJSONObject("from").getString("id");
                    String fromName = eventObject.getJSONObject("from").getString("name");
                    String friendId = null;
                    String message = null;
                    String picture = null;

                    if (friendStorage.isFriend(fromId)) {
                        friendId = fromId;
                    }
                    long publishedTimestamp = 0;
                    if (eventObject.has("created_time")) {
                        String createdTime = eventObject.getString("created_time");
                        Date time = facebookDateParser.parse(createdTime);

                        if (time == null) { // Let's do the best we can...
                            time = new Date(System.currentTimeMillis());
                        }

                        publishedTimestamp = time.getTime();
                        if (publishedTimestamp > youngestTimestamp) {
                            youngestTimestamp = publishedTimestamp;
                        }
                    }
                    if (eventObject.has("type")) {
                        String type = eventObject.getString("type");
                        if (fromId.equals(mSettings.getOwnId())) {
                            if (type.equals("status") && !eventObject.has("to")
                                    && (publishedTimestamp > latestStatusUpdate)) {
                                if (eventObject.has("message")) {
                                    message = eventObject.getString("message");
                                } else {
                                    message = "";
                                }
                                eventStorage.setOwnStatus(message, publishedTimestamp);
                                ownStatusSet = true;
                                latestStatusUpdate = publishedTimestamp;
                            }
                        } else {
                            if (type.equals("status")) {
                                if (eventObject.has("message")) {
                                    message = eventObject.getString("message");
                                }
                                eventStorage.addEvent(facebookEventId, message, fromName, null,
                                        friendId, publishedTimestamp, EventType.STATUS_EVENT);
                            } else if (type.equals("link")) {
                                // Message
                                if (eventObject.has("message")) {
                                    message = eventObject.getString("message");
                                } else if (eventObject.has("caption")){
                                    message = eventObject.getString("caption");
                                } else if (eventObject.has("name")){
                                    message = eventObject.getString("name");
                                }
                                if (message == null) {
                                    message = "";
                                }
                                String link = eventObject.getString("link");
                                facebookEventId = facebookEventId + "#" + URLEncoder.encode(link, "UTF-8");
                                message += "\n" + link;
                                // Picture
                                if (eventObject.has("picture")) {
                                    picture = eventObject.getString("picture");
                                }
                                eventStorage.addEvent(facebookEventId, message, fromName,
                                        picture, friendId, publishedTimestamp, EventType.LINK_EVENT);
                            } else if (type.equals("photo")) {
                                if (eventObject.has("picture")) {
                                    picture = eventObject.getString("picture");
                                }
                                if (eventObject.has("name")) {
                                    message = eventObject.getString("name");
                                } else if (eventObject.has("caption")) {
                                    message = eventObject.getString("caption");
                                } else if (eventObject.has("message")) {
                                    message = eventObject.getString("message");
                                }
                                eventStorage.addEvent(facebookEventId, message, fromName, picture,
                                        friendId, publishedTimestamp, EventType.PHOTO_EVENT);
                            } else {
                                // Message
                                if (eventObject.has("message")) {
                                    message = eventObject.getString("message");
                                } else if (eventObject.has("name")) {
                                    message = eventObject.getString("name");
                                } else if (eventObject.has("caption")) {
                                    message = eventObject.getString("caption");
                                }
                                // Friend key
                                if (friendStorage.isFriend(fromId)) {
                                    friendId = fromId;
                                }
                                // Photo
                                if (eventObject.has("picture")) {
                                    picture = eventObject.getString("picture");
                                }
                                eventStorage.addEvent(facebookEventId, message, fromName, picture,
                                        friendId, publishedTimestamp, EventType.PHOTO_EVENT);
                            }
                        }
                    }
                } catch (JSONException e) {
                    if (Config.DEBUG) {
                        Log.e(LOG_TAG, "Error parsing JSON data." + e);
                        Log.d(LOG_TAG, dataArray.getJSONObject(i).toString(3));
                    }
                } catch (ParseException e) {
                    if (Config.DEBUG) {
                        Log.e(LOG_TAG, "Error parsing JSON data." + e);
                        Log.d(LOG_TAG, dataArray.getJSONObject(i).toString(3));
                    }
                }
            }
            if (!ownStatusSet) {
                setLatestStatus(eventStorage);
            }
        }
        return youngestTimestamp;
    }

    private void setLatestStatus(EventStorage eventStorage) throws FacebookException, IOException, JSONException, ParseException {
        JSONObject jsonObject = httpGetToGraphApi(STATUSES_PATH, "&limit=1");
        if (jsonObject != null) {
            JSONArray dataArray = jsonObject.getJSONArray(Constants.DATA_PARAMTER);
            if (dataArray.length() > 0) {
                // English locale is used here because English (either UK or US) is
                // guaranteed to exist in all phones sold over the world.
                // If using Default locale, parsing will fail when using Arabic language
                // and possibly other languages
                SimpleDateFormat facebookDateParser = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZZZZ", Locale.ENGLISH);
                JSONObject status = dataArray.getJSONObject(0);
                Date timestampDate = facebookDateParser.parse(status.getString("updated_time"));
                long timestamp = 0;
                if (timestampDate != null) {
                    timestamp = timestampDate.getTime();
                } else {
                    // Use current time as fall back
                    timestamp = System.currentTimeMillis();
                }
                String message = status.getString("message");
                eventStorage.setOwnStatus(message, timestamp);
            }
        }
    }

    private JSONObject httpGetToGraphApi(String graphMethod, String additionalParameters)
        throws FacebookException, IOException, JSONException {
        String accessToken = mSettings.getAuthenticationToken();
        if (isLoggedIn() && graphMethod != null) {
            try {
                String uri = GRAPH_API_BASE_URL + graphMethod + "?access_token=" + accessToken;

                //We assume that the additionalParameters are encoded correctly. If not we will
                //fail when trying to use the url
                if (additionalParameters != null) {
                    uri += additionalParameters;
                }

                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Calling " + GRAPH_API_BASE_URL + graphMethod);
                }

                setServiceState(ServiceState.SERVER_OPERATION_IN_PROGRESS);
                CommunicationResult result = mCommunication.executeGet(uri);
                int statusCode =  result.getHttpResult();

                if (statusCode == HttpStatus.SC_OK) {
                    String facebookData = result.getHttpContent();
                    JSONObject jsonObject = new JSONObject(facebookData);

                    return jsonObject;
                } else if (statusCode >= HttpStatus.SC_BAD_REQUEST && statusCode < HttpStatus.SC_INTERNAL_SERVER_ERROR) {
                    throw new FacebookException(FacebookException.AUTHENTICATION_FAILED, statusCode);
                } else {
                    // Server error
                    if (Config.DEBUG) {
                        Log.e(LOG_TAG, "Server error: " + statusCode);
                    }
                    throw new FacebookException(FacebookException.COMMUNICATION_ERROR, statusCode);
                }
            } catch (IOException e) {
                if (Config.DEBUG) {
                    Log.e(LOG_TAG, "Error reading from Facebook server.");
                }
                throw e;
            } catch (JSONException e) {
                if (Config.DEBUG) {
                    Log.e(LOG_TAG, "Error parsing JSON data.");
                }
                throw e;
            } finally {
                setServiceState(ServiceState.PASSIVE);
            }
        }
        return null;
    }

    private String callFacebookAuthLogin(String email, String password) throws IOException, FacebookException {
        //Parameters will be url form encoded before sent on to Facebook.
        //thus we need all parameters in a non encoded format here
        String accessToken = null;
        FacebookSecurity security = FacebookFactory.getFacebookSecurity();

        try {
            List<NameValuePair> parameters = new ArrayList<NameValuePair>();
            String apiKey = security.generateAPIKey();
            parameters.add(new BasicNameValuePair(API_KEY_PARAM, apiKey));
            parameters.add(new BasicNameValuePair(EMAIL_PARAM, email));
            parameters.add(new BasicNameValuePair(METHOD_PARAM, AUTH_LOGIN_METHOD));
            parameters.add(new BasicNameValuePair(PASSWORD_PARAM, password));
            parameters.add(new BasicNameValuePair(SIG_PARAM, security.generateAuthenticationSignature(parameters)));

            String uri = "";
            StringBuffer buffer = new StringBuffer(RESTFUL_API_BASE_URL + RESTFUL_PATH + '?');
            for (NameValuePair parameter : parameters) {
                buffer.append(URLEncoder.encode(parameter.getName(), ENCODING));
                buffer.append('=');
                buffer.append(URLEncoder.encode(parameter.getValue(), ENCODING));
                buffer.append('&');
            }

            int lastIndex = buffer.lastIndexOf("&");
            if (lastIndex == -1) {
                lastIndex = buffer.length();
            }
            uri = buffer.substring(0, lastIndex);

            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Calling " + RESTFUL_API_BASE_URL + RESTFUL_PATH);
            }

            setServiceState(ServiceState.AUTHENTICATION_IN_PROGRESS);

            CommunicationResult result = mCommunication.executeGet(uri);
            int statusCode = result.getHttpResult();

            if (statusCode == HttpStatus.SC_OK) {
                String sessionToken = null;
                String sessionSecretToken = null;
                String facebookData = result.getHttpContent();
                XmlPullParser parser = Xml.newPullParser();
                parser.setInput(new StringReader(facebookData));
                while (parser.next() != XmlPullParser.END_DOCUMENT) {
                    if (parser.getEventType() == XmlPullParser.START_TAG) {
                        if (ACCESS_TOKEN_TAG.equals(parser.getName())) {
                            accessToken = parser.nextText();
                            setAuthenticationToken(accessToken);
                        } else if (SESSION_KEY_TAG.equals(parser.getName())) {
                            sessionToken = parser.nextText();
                            setSessionToken(sessionToken);
                        } else if (SESSION_SECRET_TAG.equals(parser.getName())) {
                            sessionSecretToken = parser.nextText();
                            setSessionSecretToken(sessionSecretToken);
                        }
                    }
                }

                // Not validating on session token and session secret since they are only used
                // when taping on tile (deep linking) and isn't critical to
                // when fetching data from the server
                if (accessToken == null) {
                    throw new FacebookException(FacebookException.AUTHENTICATION_INVALID_CREDENTIALS);
                }
            } else { // Failed!
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Unexpected response from server: "
                            + statusCode + " "
                            + result.getHttpPhrase());
                    Log.e(LOG_TAG, "Error when authenticating a Facebook user.");
                }
                throw new FacebookException(FacebookException.AUTHENTICATION_FAILED,
                        statusCode);
            }
        } catch (IllegalArgumentException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error when authenticatnig Facebook user. ");
            }
            throw new IOException();
        } catch (IOException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error when authenticating Facebook user.");
            }
            throw e;
        } catch (XmlPullParserException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error when authenticatnig Facebook user.");
            }
            throw new IOException();
        } finally {
            setServiceState(ServiceState.PASSIVE);
        }

        return accessToken;
    }

    private void setAuthenticationToken(String accessToken) throws FacebookException {
        try {
            accessToken = URLEncoder.encode(accessToken, ENCODING);
        } catch (UnsupportedEncodingException exception) {
            mSettings.setAuthenticationToken(null);
            throw new FacebookException(FacebookException.PARSER_ERROR);
        }
        mSettings.setAuthenticationToken(accessToken);
    }

    private void setSessionToken(String sessionToken) {
        try {
            sessionToken = URLEncoder.encode(sessionToken, ENCODING);
        } catch (UnsupportedEncodingException exception) {
            // Not much to do...
        }
        mSettings.setSessionKey(sessionToken);
    }

    private void setSessionSecretToken(String sessionSecretToken) {
        try {
            sessionSecretToken = URLEncoder.encode(sessionSecretToken, ENCODING);
        } catch (UnsupportedEncodingException exception) {
            // Not much to do...
        }
        mSettings.setSessionSecret(sessionSecretToken);
    }

    private void clearSessionData() {
        mSettings.setAuthenticationToken(null);
        mSettings.setOwnId(null);
        mSettings.setSessionKey(null);
        mSettings.setSessionSecret(null);
    }
}
