/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY_PARAMETER;

import com.sonyericsson.eventstream.facebookplugin.util.DatabaseHelper;
import com.sonyericsson.eventstream.facebookplugin.util.MockFacebook;
import com.sonyericsson.eventstream.facebookplugin.util.ReflectionUtilities;

import android.app.Instrumentation;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.test.ServiceTestCase;

import java.lang.reflect.Field;

/**
 * Tests involving the com.sonyericsson.eventstream.facebookplugin.FacebookService class.
 *
 * Test that the facebook service can handle all expected
 * intents in a correct way and that intents sent by
 * the "event stream state" machine can arrive in any order.
 */
public class UFacebookPluginTestService extends ServiceTestCase<FacebookService> {

    private MockFacebook mFacebookMock = null;
    private FacebookPluginApplication mApplication = null;

    public UFacebookPluginTestService() {
        super(FacebookService.class);
    }

    @Override
    protected void setUp() throws Exception {
        super.setUp();

        // Remove everything from event stream framework...
        DatabaseHelper.removePluginFromEventStream(getContext());
        new Settings(getContext()).removeSettings();

        // Set up facebook mock object...
        mFacebookMock = new MockFacebook(new Settings(getSystemContext()));
        try {
            Field field = ReflectionUtilities.getField(FacebookFactory.class, "sFacebookInstance");

            // we can not simple overwrite the facebook object since
            // that will cause a memory leak.
            // So, if it's a real live object ensure it is destroyed appropriately
            Object object = field.get(new Object());
            if (object != null && ! (object instanceof MockFacebook)) {
                FacebookFactory.terminate(false);
            }
            field.set(new Object(), mFacebookMock);
        } catch (Exception e) {
            e.printStackTrace();
            fail("Couldn't set mock class!");
        }

        // We can accept any call to "listener"-methods and to the init method...
        mFacebookMock.mMockObject.expect("initialize");
        mFacebookMock.mMockObject.expect("isLoggedIn");
        mFacebookMock.mMockObject.expect("setServiceStateListener");
        mFacebookMock.mMockObject.expect("removeServiceStateListener");
        mFacebookMock.mMockObject.expect("shutdown");

        // As we need the facebook application up and running, lets create one...
        if (mApplication == null) {
            try {
                mApplication = (FacebookPluginApplication) Instrumentation.newApplication(FacebookPluginApplication.class,
                        getSystemContext());
                mApplication.onCreate();
                setApplication(mApplication);
            } catch (Exception exception) {
                fail("Couldn't create application!");
            }
        }
        // Set system context...
        mContext = mApplication.getApplicationContext();

        // Clear expectations...
        mFacebookMock.mMockObject.clearExpectations();
    }

    @Override
    protected void tearDown() throws Exception {
        DatabaseHelper.removePluginFromEventStream(getContext());

        super.tearDown();
    }

    /**
     * Test that the facebook service acts correctly upon
     * a register intent.
     * Pre-condition: not registered and not logged in
     */
    public void testPluginRegisterIntent() {
        FacebookService service = null;

        // Initialize service
        Intent initIntent = new Intent();
        startService(initIntent);
        service = getService();

        mFacebookMock.mMockObject.expect("isLoggedIn", 1).andReturn(false);

        // Send registration intent
        Intent intent = new Intent();
        intent.setAction("com.sonyericsson.facebook.REGISTER_PLUGIN");
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Check database, plug-in must be registered
        boolean isRegistered = DatabaseHelper.isRegistered(getContext());
        assertTrue("We are not registered", isRegistered);

        // Assert that correct calls to facebook class are made
        mFacebookMock.mMockObject.verifyAllExpectaions();
    }

    /**
     * Tests that the facebook service can correctly handle
     * multiple register intents
     */
    public void testPluginMultipleRegisterIntents() {
        // Initialize service
        Intent initIntent = new Intent();
        FacebookService service;

        startService(initIntent);
        service = getService();

        // Clear database...
        DatabaseHelper.removePluginFromEventStream(getContext());

        // Allow calls to isLoggedIn()...
        mFacebookMock.mMockObject.expect("isLoggedIn").andReturn(false);

        // Send registration intent
        Intent intent = new Intent();
        intent.setAction("com.sonyericsson.facebook.REGISTER_PLUGIN");
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Check database, plug-in must be registered
        boolean isRegistered = DatabaseHelper.isRegistered(getContext());
        assertTrue("We are not registered", isRegistered);

        // Send registration intent again
        intent = new Intent();
        intent.setAction("com.sonyericsson.facebook.REGISTER_PLUGIN");
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Check database, plug-in must be registered
        isRegistered = DatabaseHelper.isRegistered(getContext());
        assertTrue("We are not registered", isRegistered);

        // Assert that correct calls to facebook class are made
        mFacebookMock.mMockObject.verifyAllExpectaions();
    }

    /**
     * Tests that the service can correctly handle
     * if a user tries to clear data on event-stream
     */
    public void testClearData() {
        // Initialize service
        Intent initIntent = new Intent();
        FacebookService service;

        startService(initIntent);
        service = getService();

        // Simulate that we where logged in prior to clear data
        mFacebookMock.mMockObject.expect("isLoggedIn").andReturn(true);
        mFacebookMock.mMockObject.expect("shutdown", 1);
        mFacebookMock.mMockObject.expect("initialize", 1);

        // Send refresh intent
        Intent intent = new Intent();

        intent.setAction("com.sonyericsson.facebook.REGISTER_PLUGIN");
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Assert that correct calls to facebook class are made
        mFacebookMock.mMockObject.verifyAllExpectaions();

        // Should be registered
        boolean isRegistered = DatabaseHelper.isRegistered(getContext());
        assertTrue("We are not registered", isRegistered);
    }

    /**
     * Test that the facebook service can handle a refresh request intent
     */
    public void testPluginRefreshIntent() {
        // Initialize service
        Intent initIntent = new Intent();
        FacebookService service;

        startService(initIntent);
        service = getService();

        // Ensure plug-in is registered
        int sourceId = DatabaseHelper.registerPlugin(getContext());
        assertTrue("Couldn't register the plugin", sourceId != -1);

        // Set up mock facebook object
        // Upon a refresh intent we should try to:
        //      Retrieve new facebook contacts
        //      Retrieve new events
        // This is only possible if:
        //      We are logged in
        mFacebookMock.mMockObject.expect("isLoggedIn").andReturn(true);
        mFacebookMock.mMockObject.expect("refreshFacebookFriends", 1).andReturn(true);
        mFacebookMock.mMockObject.expect("retrieveLatestPosts", 1).andReturn(true);

        // Send refresh intent
        Intent intent = new Intent();

        intent.setAction("com.sonyericsson.facebook.REFRESH_REQUEST");
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Assert that correct calls to facebook class are made
        mFacebookMock.mMockObject.verifyAllExpectaions();
    }

    /**
     * Test the logout intent
     * This intent must remove all facebook content and settings
     */
    public void testPluginLogoutIntent() {
        // Initialize service
        Intent initIntent = new Intent();
        FacebookService service;

        startService(initIntent);
        service = getService();

        // Ensure plug-in is registered
        DatabaseHelper.registerPlugin(getContext());

        // Set up mock facebook object
        mFacebookMock.mMockObject.expect("shutdown", 1);
        mFacebookMock.mMockObject.expect("initialize", 1);

        // Carry out intent
        Intent intent = new Intent();

        intent.setAction("com.sonyericsson.facebook.LOGOUT");
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Assert that correct calls to facebook class is made
        mFacebookMock.mMockObject.verifyAllExpectaions();

        // Ensure that we still are registered and that
        // no facebook data is stored!
        Settings setting = new Settings(getContext());

        // Ensure that all settings have their default values
        assertTrue("The authentication token isn't null", setting.getAuthenticationToken() == null);
        assertTrue("The last communication time isn't null", setting.getLastCommunicationTime() == 0L);
        assertTrue("The own id is set", setting.getOwnId().equals(""));
        assertTrue("The youngest timestamp is set", setting.getYoungestTimestamp() == 0L);

        // Upon logout, we want to remove all data from facebook but
        // still be registered
        boolean isRegistered = DatabaseHelper.isRegistered(getContext());
        assertTrue("The plug-in isn't registered", isRegistered);

        // Put the object back again, otherwise we will clean-up the
        // real implementation of facebook...
        try {
            ReflectionUtilities.getField(FacebookFactory.class, "sFacebookInstance").set(new Object(),
                    mFacebookMock);
        } catch (Exception e) {
            e.printStackTrace();
            fail("Couldn't set mock class!");
        }
    }

    /**
     * Test the authenticate intent
     *
     * The facebook service should call the facebook class and the
     * we should be considered logged in
     */
    public void testPluginAuthenticateIntent() {
        final String username = "Kalle";
        final String pwd = "my_pwd";
        final String displayName = "Kalle Kula";

        // Initialize service
        Intent initIntent = new Intent();
        FacebookService service;

        startService(initIntent);
        service = getService();

        // Ensure plug-in is registered
        DatabaseHelper.registerPlugin(getContext());

        // Set up mock facebook object
        mFacebookMock.mMockObject.expect("authenticate", 1, username + ";" + pwd).andReturn(displayName);
        mFacebookMock.mMockObject.expect("isLoggedIn").andReturn(new Boolean(true));
        mFacebookMock.mMockObject.expect("refreshFacebookFriends", 1);
        mFacebookMock.mMockObject.expect("retrieveLatestPosts", 1);

        // "Send" intent...
        Intent intent = new Intent();

        intent.setAction("com.sonyericsson.facebook.AUTHENTICATE");
        intent.putExtra("username", username);
        intent.putExtra("password", pwd);
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Assert that correct calls to facebook class is made
        mFacebookMock.mMockObject.verifyAllExpectaions();

        // Assert that the correct authentication token is stored
        String name = DatabaseHelper.getDisplayName(getContext());
        assertEquals("The display name isn't set properly", mContext.getString(R.string.ts_facebook_logout_label) + " " + displayName, name);
    }

    /**
     * Test that the facebook service can properly handle a
     * send status update intent.
     * The facebook class should be called and the
     * status update string should be stored in the event stream
     * framework
     */
    public void testPluginSendStatusUpdateIntent() {
        final String statusUpdateString = "Hello world!";

        // Initialize service
        Intent initIntent = new Intent();
        FacebookService service;

        startService(initIntent);
        service = getService();

        // Ensure plug-in is registered
        DatabaseHelper.registerPlugin(getContext());

        // Set up mock facebook object
        mFacebookMock.mMockObject.expect("updateStatus", 1, statusUpdateString).andReturn(true);

        // "Send" intent..
        Intent intent = new Intent();

        intent.setAction("com.sonyericsson.facebook.SEND_STATUS_UPDATE");
        intent.putExtra("new_status_message", statusUpdateString);
        intent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        service.onHandleIntent(intent);

        // Assert that correct calls to facebook class is made
        mFacebookMock.mMockObject.verifyAllExpectaions();

        String statusText = DatabaseHelper.getStatusUpdate(getContext());
        assertEquals("Couldn't store status update", statusUpdateString, statusText);
    }

    public void testLaunchWebIntent() {
        Context context = new FakeContext(getContext());
        FacebookService service = new FacebookService();

        // Simulate logged in state...
        Settings settings = new Settings(getContext());
        settings.setAuthenticationToken("MY_SECRET_TOKEN");
        settings.setOwnId("101010101");
        settings.setSessionKey("8888888");
        settings.setSessionSecret("SESSION_SECRET)");

        service.launchBrowser(context, "1101010101010_2323232323232323;STATUS", "");
        assertNotNull("No intent!", ((FakeContext) context).serviceIntent);
    }

    public class FakeContext extends ContextWrapper {
        public Intent serviceIntent = null;

        public FakeContext(Context context) {
            super(context);
        }

        @Override
        public void startActivity(Intent service) {
            serviceIntent = service;
        }
    }
}
