/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.simpleui.utils;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.AsyncTask;
import android.util.Log;
import android.widget.ImageView;

import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;
import java.util.HashMap;

public class ImageDownloader {

    private static final String LOG_TAG = "SimpleUiImageDownloader";
    private static HashMap<Uri, SoftReference<Bitmap>> sArtCache =
        new HashMap<Uri, SoftReference<Bitmap>>();
    private Bitmap mDefaultImage;

    public ImageDownloader(Bitmap defaultImage) {
        mDefaultImage = defaultImage;
    }

    /**
     * Checks if the wanted image is already available in the in-memory cache.
     *
     * @param imageUri
     * @param view
     * @return
     */
    public boolean getImageFromCache(Uri imageUri, ImageView view) {
        boolean result = false;
        Bitmap drawable = null;
        SoftReference<Bitmap> reference = sArtCache.get(imageUri);
        if (reference != null) {
            Log.d(LOG_TAG, "Image is in the in-memory cache");
            drawable = reference.get();
            if (drawable == null) {
                // Drawable has been garbage collected, so remove it from the cache
                sArtCache.remove(imageUri);
            } else {
                // Cancel potential downloads attached to this URI since the image is already cached
                cancelPotentialDownloads(imageUri, view);
                view.setImageBitmap(drawable);
                result = true;
            }
        } else {
            Log.d(LOG_TAG, "Image is not in the in-memory cache");
        }
        return result;
    }

    /**
     * Checks if any ongoing downloads should be cancelled for the reasons stated
     * below.
     *
     * @param imageUri the URI of the image to be downloaded
     * @param view the ImageView associated with the image URI
     * @return true if an image download task is cancelled
     */
    private static boolean cancelPotentialDownloads(Uri imageUri, ImageView view) {
        // Get a handle to the current DownloadImageTask
        // Query the URI that is associated with the DownloadImageTask
        // Check if that URI is the same as the one passed in
        boolean result = true;
        Log.d(LOG_TAG, "Checking if there is any image download task to cancel");
        DownloadImageTask task = getDownloadImageTask(view);
        if (task != null) {
            Uri currentDownloadingImageUri = task.uri;
            if ((currentDownloadingImageUri == null) || (!currentDownloadingImageUri.equals(imageUri))) {
                // Cancel the task since there is nothing to download or the 'wrong' image is being downloaded
                task.cancel(true);
                result = true;
                Log.d(LOG_TAG, "Current image download is cancelled");
            } else {
                // continue
                result = false;
                Log.d(LOG_TAG, "Nothing to cancel");
            }
        } else {
            Log.d(LOG_TAG, "No associated DownloadImageTask");
        }
        return result;
    }

    /**
     * Retrieve a handle to the DownloadImageTask that is attached to the wanted
     * ImageView.
     *
     * @param view the ImageView to find the associated DownloadImageTask
     */
    private static DownloadImageTask getDownloadImageTask(ImageView view) {
        if (view != null) {
            Drawable drawable = view.getDrawable();
            if (drawable instanceof DownloadingDrawable) {
                DownloadingDrawable downloadingDrawable = (DownloadingDrawable)drawable;
                return downloadingDrawable.getAssociatedDownloadImageTask();
            }
        }
        return null;
    }

    private void addImageToCache(Uri imageUri, Bitmap imageBitmap) {
        SoftReference<Bitmap> reference = new SoftReference<Bitmap>(imageBitmap);
        sArtCache.put(imageUri, reference);
        Log.d(LOG_TAG, "Image added to cache for URI " + imageUri);
    }

    /**
     * Creates a task for downloading/retrieving the image via the OnlineImageCacheHandler.
     * NOTE: the condition checks are weak now so it does not detect if there is already a
     * task to download the same image, so there may be several tasks fired up to download the
     * same image and this will in turn cause the same image to be attached to the in-memory
     * cache :((((. Needs refactoring.
     *
     * @param context the operating context
     * @param imageUri the URI to the image to be downloaded/retrieved
     * @param view the ImageView to attach the final image to
     */
    public void triggerDownload(Context context, Uri imageUri, ImageView view) {
        // Create task to call OnlineImageCacheHandler
        // When OnlineImageCacheHandler returns, attach the image to the view and
        // place in the in-memory cache
        if (cancelPotentialDownloads(imageUri, view)) {
            Log.d(LOG_TAG, "Image download triggered for URI " + imageUri);
            DownloadImageTask downloadTask = new DownloadImageTask(view, context);
            // Set some colour drawable as a placeholder in the mean time while downloading
            DownloadingDrawable downloadingDrawable = new DownloadingDrawable(downloadTask);
            view.setImageDrawable(downloadingDrawable);
            downloadTask.execute(imageUri);
        }
    }

    /**
     * Wanted: remove all information stored in the in-memory cache
     */
    public static void cleanupCache() {
        Log.i(LOG_TAG, "cleanupCache called");
        sArtCache.clear();
    }

    class DownloadingDrawable extends ColorDrawable {
        private WeakReference<DownloadImageTask> downloadImageTaskReference;

        public DownloadingDrawable(DownloadImageTask task) {
            super(Color.GREEN);
            downloadImageTaskReference = new WeakReference<DownloadImageTask>(task);
        }
        public DownloadImageTask getAssociatedDownloadImageTask() {
            return downloadImageTaskReference.get();
        }
    }

    class DownloadImageTask extends AsyncTask<Uri, Void, Bitmap> {
        private WeakReference<ImageView> imageViewReference;
        private Context mContext;
        private Uri uri;

        public DownloadImageTask(ImageView view, Context context) {
            imageViewReference = new WeakReference<ImageView>(view);
            mContext = context;
        }

        @Override
        protected Bitmap doInBackground(Uri... params) {
            Log.d(LOG_TAG, "Number of URIs in params array = " + params.length);
            uri = params[0];
            Log.d(LOG_TAG, "doInBackground for URI: " + uri);
            Bitmap bitmap = ImageUtilities.decodeImageFromUri(mContext, uri);
            if (bitmap != null) {
                addImageToCache(uri, bitmap);
            }
            return bitmap;
        }

        @Override
        protected void onPostExecute(Bitmap result) {
            if (result == null) {
                Log.d(LOG_TAG, "Failed to decode image!");
            } else {
                // Want to check if this bitmap result is for an imageView reference that still exists
                if (imageViewReference != null) {
                    ImageView associatedImageView = imageViewReference.get();
                    if (associatedImageView != null) {
                        Drawable associatedDrawable = associatedImageView.getDrawable();
                        if (associatedDrawable instanceof DownloadingDrawable) {
                            DownloadingDrawable drawable = (DownloadingDrawable)associatedDrawable;
                            // Check which DownloadImageTask this associatedDrawable belongs to
                            DownloadImageTask associatedTask = drawable.getAssociatedDownloadImageTask();
                            if (this == associatedTask) {
                                associatedImageView.setImageBitmap(result);
                            } else {
                                associatedImageView.setImageBitmap(mDefaultImage);
                            }
                        } else {
                            associatedImageView.setImageBitmap(mDefaultImage);
                        }
                    }
                }
            }
        }
    }
}
