/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.service;

import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_EVENT_PROVIDER_URI;
import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_FRIEND_PROVIDER_URI;
import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_PLUGIN_PROVIDER_URI;
import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_SOURCE_PROVIDER_URI;

import android.app.Instrumentation;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;
import android.database.Cursor;
import android.net.Uri;
import android.net.Uri.Builder;
import android.test.ServiceTestCase;
import android.util.Log;

import twitter4j.DirectMessage;
import twitter4j.PagableResponseList;
import twitter4j.Status;
import twitter4j.User;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants;
import com.sonyericsson.eventstream.twitterplugin.R;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.EventColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.EventStreamIntentExtraParams;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.FriendColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.InternalIntentExtraParams;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.PluginColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.SourceColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication.State;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException.StatusCode;
import com.sonyericsson.eventstream.twitterplugin.utility.EventStreamHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;
import com.sonyericsson.eventstream.twitterplugin.utils.MockTwitter;
import com.sonyericsson.eventstream.twitterplugin.utils.MockTwitterCommunication;
import com.sonyericsson.eventstream.twitterplugin.utils.ReflectionUtilities;
import com.sonyericsson.eventstream.twitterplugin.view.ConfigurationActivity;

import junit.framework.Assert;

import java.util.List;

public class FTwitterPluginServiceTests extends ServiceTestCase<TwitterPluginService> {

    private static final String PACKAGE_NAME = "com.sonyericsson.eventstream.twitterplugin";

    private TwitterPluginService mService;

    private TwitterPluginApplication mApplication;

    MockTwitterCommunication mTwitterCommunication;

    private static final String USERNAME = "username";

    private static final String PASSWORD = "password";

    private static final int CONFIGURED = 0;

    private static final int NOT_CONFIGURED = 1;

    private static final int NBROFUSERS = 2;

    private static final int NBROFMSGPERUSER = 10;

    private static final int NBROFSTATUSPERUSER = 10;

    private static final int NBROFEVENTS = NBROFUSERS * (NBROFMSGPERUSER + NBROFSTATUSPERUSER);

    private static final int NOCONTENT = 0;

    public FTwitterPluginServiceTests() {
        super(TwitterPluginService.class);
    }

    @Override
    protected void setUp() throws Exception {
        cleanup();

        try {
            mApplication = (TwitterPluginApplication) Instrumentation.newApplication(TwitterPluginApplication.class,
                    getContext());
            mApplication.onCreate();
            setApplication(mApplication);
            setContext(getContext());
        } catch (Exception exception) {
            fail("Couldn't create application!");
        }

        mTwitterCommunication = new MockTwitterCommunication(getContext());
        mTwitterCommunication.trackMethodCalls("no_method");
        mTwitterCommunication.performThrowExceptionOnCall(false, StatusCode.SERVER_UNAVAILABLE);
        mTwitterCommunication.returnErrorOnCall(false);

        bindService(new Intent());
        mService = getService();

        PreferencesHelper.clearAll(getContext());
        EventStreamHelper.clearAll(getContext().getContentResolver());
        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        cleanup();
        super.tearDown();
    }

    private void cleanup() {
        ContentResolver contentResolver = getContext().getContentResolver();
        contentResolver.delete(EVENTSTREAM_EVENT_PROVIDER_URI, null, null);
        contentResolver.delete(EVENTSTREAM_FRIEND_PROVIDER_URI, null, null);
        contentResolver.delete(EVENTSTREAM_SOURCE_PROVIDER_URI, null, null);
        contentResolver.delete(EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null);
        PreferencesHelper.clearAll(getContext());
    }

    /**
     * Should be called to start the service with a intent.
     * Will replace the twitter communication.
     * @param intent The intent to start the service with.
     */
    private void executeServiceIntent(Intent intent, boolean setMockObject) {
        MockTwitterCommunication twitterCommunication = null;
        if (setMockObject) {
            twitterCommunication = mTwitterCommunication;
        }
        try {
            ReflectionUtilities.getField(mService, "mTwitterCommunication").set(mService,
                    twitterCommunication);
        } catch (Exception e) {
            Assert.fail("Could not change the factory field");
        }
        mService.onHandleIntent(intent);
    }

    @Override
    public void testServiceTestCaseSetUpProperly() throws Exception {
    }

    /**
     * Verify that the plug-in is registered correct and its source.
     * Will also check that sync if performed after the registration
     */
    public void testSinglePluginRegistration() {
        performRegistration(1);
    }

    /**
     * Verify that the plug-in is registered correct and its source.
     * Will also check that sync if performed after the registration
     */
    public void testMultiPluginRegistration() {
        performRegistration(10);
    }

    public void testPluginColumnNames() throws Exception {
        Context context = getContext();
        ContentResolver contentResolver = context.getContentResolver();
        Cursor c = contentResolver.query(EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null, null, null);
        assertNotNull(c);
        try {
            c.getColumnIndexOrThrow(PluginColumns.API_VERSION);
            c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_ACTIVITY);
            c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_STATE);
            c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_TEXT);
            c.getColumnIndexOrThrow(PluginColumns.ICON_URI);
            c.getColumnIndexOrThrow(PluginColumns.NAME);
            c.getColumnIndexOrThrow(PluginColumns.PLUGIN_KEY);
            c.getColumnIndexOrThrow(PluginColumns.STATUS_SUPPORT);
            c.getColumnIndexOrThrow(PluginColumns.STATUS_TEXT_MAX_LENGTH);
        } catch (IllegalArgumentException e) {
            fail("Invalid column: " + e.getMessage());
            e.printStackTrace();
        } finally {
            if (c != null) {
                c.close();
            }
        }
    }

    public void testSourceColumnNames() throws Exception {
        Context context = getContext();
        ContentResolver contentResolver = context.getContentResolver();
        Cursor c = contentResolver.query(EVENTSTREAM_SOURCE_PROVIDER_URI, null, null, null, null);
        assertNotNull(c);
        try {
            c.getColumnIndexOrThrow(SourceColumns._ID);
            c.getColumnIndexOrThrow(SourceColumns.CURRENT_STATUS);
            c.getColumnIndexOrThrow(SourceColumns.ENABLED);
            c.getColumnIndexOrThrow(SourceColumns.ICON_URI);
            c.getColumnIndexOrThrow(SourceColumns.NAME);
            c.getColumnIndexOrThrow(SourceColumns.STATUS_TIMESTAMP);
        } catch (IllegalArgumentException e) {
            fail("Invalid column: " + e.getMessage());
            e.printStackTrace();
        } finally {
            if (c != null) {
                c.close();
            }
        }
    }

    public void testFriendColumnNames() throws Exception {
        Context context = getContext();
        ContentResolver contentResolver = context.getContentResolver();
        Cursor c = contentResolver.query(EVENTSTREAM_FRIEND_PROVIDER_URI, null, null, null, null);
        assertNotNull(c);
        try {
            c.getColumnIndexOrThrow(FriendColumns._ID);
            c.getColumnIndexOrThrow(FriendColumns.CONTACTS_REFERENCE);
            c.getColumnIndexOrThrow(FriendColumns.DISPLAY_NAME);
            c.getColumnIndexOrThrow(FriendColumns.FRIEND_KEY);
            c.getColumnIndexOrThrow(FriendColumns.PROFILE_IMAGE_URI);
            c.getColumnIndexOrThrow(FriendColumns.SOURCE_ID);
        } catch (IllegalArgumentException e) {
            fail("Invalid column: " + e.getMessage());
            e.printStackTrace();
        } finally {
            if (c != null) {
                c.close();
            }
        }
    }


    public void testEventColumnNames() throws Exception {
        Context context = getContext();
        ContentResolver contentResolver = context.getContentResolver();
        Cursor c = contentResolver.query(EVENTSTREAM_EVENT_PROVIDER_URI, null, null, null, null);
        assertNotNull(c);
        try {
            c.getColumnIndexOrThrow(EventColumns._ID);
            c.getColumnIndexOrThrow(EventColumns.IMAGE_URI);
            c.getColumnIndexOrThrow(EventColumns.EVENT_KEY);
            c.getColumnIndexOrThrow(EventColumns.FRIEND_KEY);
            c.getColumnIndexOrThrow(EventColumns.GEODATA);
            c.getColumnIndexOrThrow(EventColumns.MESSAGE);
            c.getColumnIndexOrThrow(EventColumns.OUTGOING);
            c.getColumnIndexOrThrow(EventColumns.PERSONAL);
            c.getColumnIndexOrThrow(EventColumns.PUBLISHED_TIME);
            c.getColumnIndexOrThrow(EventColumns.SOURCE_ID);
            c.getColumnIndexOrThrow(EventColumns.STATUS_ICON_URI);
            c.getColumnIndexOrThrow(EventColumns.TITLE);

        } catch (IllegalArgumentException e) {
            fail("Invalid column: " + e.getMessage());
            e.printStackTrace();
        } finally {
            if (c != null) {
                c.close();
            }
        }
    }

    /**
     *  Verify that a change of language updates the applicable fields in
     *  the EventStream. Case when user is logged in.
     */
    public void testChangeOfLanguageLoggedIn() {
        changeOfLanguage(true);
    };

    /**
     *  Verify that a change of language updates the applicable fields in
     *  the EventStream. Case when user is logged out.
     */
    public void testChangeOfLanguageLoggedOut() {
        changeOfLanguage(false);
    };



    /**
     * Verify that refresh intent trigger a new event
     */
    public void testRefreshEventData() {
        performRegistration(1);

        mTwitterCommunication.trackMethodCalls("getHomeTimeline");

        // Set token to fake a already registered and authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        // Add event data
        generateUsersAndEvents(false);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REFRESH_REQUEST);
        executeServiceIntent(serviceIntent, true);

        assertTrue(checkRegistration());

        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NBROFEVENTS);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NBROFUSERS);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Verify that refresh intent trigger a new event and no content is stored if
     * twitter throw a exception. The mock object will throw exception on all
     * return values from Twitter communication
     */
    public void testRefreshEventDataTwitterReturnException() {
        performRegistration(1);

        mTwitterCommunication.trackMethodCalls("getHomeTimeline");
        // Set token to fake a already registered and authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        // Add event data
        generateUsersAndEvents(false);

        mTwitterCommunication.performThrowExceptionOnCall(true, StatusCode.INTERNAL_ERROR);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REFRESH_REQUEST);
        executeServiceIntent(serviceIntent, true);

        assertTrue(checkRegistration());

        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Unexpected method call", 0, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Verify that refresh intent trigger a new event and no content is stored if
     * twitter return null from a value. The mock object will return null on all
     * return values from Twitter communication
     */
    public void testRefreshEventDataTwitterReturnNull() {
        performRegistration(1);

        mTwitterCommunication.trackMethodCalls("getHomeTimeline");
        // Set token to fake a already registered and authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        // Add event data
        generateUsersAndEvents(false);

        mTwitterCommunication.returnErrorOnCall(true);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REFRESH_REQUEST);
        executeServiceIntent(serviceIntent, true);

        assertTrue(checkRegistration());

        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Unexpected method call", 0, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Verify that refresh intent trigger a new event and no content is stored if
     * twitter return no content
     */
    public void testRefreshEventDataTwitterNoContent() {
        // Will refresh and perform registration
        testRefreshEventData();

        mTwitterCommunication.trackMethodCalls("getHomeTimeline");
        // Add event data
        generateUsersAndEvents(true);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REFRESH_REQUEST);
        executeServiceIntent(serviceIntent, true);

        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Verify that after a logout all content is cleared
     */
    public void testLogoutFromTwitter() {
        // Will refresh and perform registration
        testRefreshEventData();

        mApplication.setState(State.SERVICE_AUTHENTICATED);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_LOGOUT);
        executeServiceIntent(serviceIntent, false);

        // The plug-in should still be registered after logout
        assertTrue(checkRegistration());

        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Wrong state, should not be authenticated", State.SERVICE_NOT_AUTHENTICATED, mApplication.getState());
        assertEquals("No token should exist", false, PreferencesHelper.hasToken(mContext));
    }

    /**
     * Valid status update should result in event-stream update
     */
    public void testPerformStatusUpdateValid() {
        String statusText = "test_status";
        mTwitterCommunication.trackMethodCalls("updateStatus");

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        performRegistration(1);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
        serviceIntent.putExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE, statusText);
        executeServiceIntent(serviceIntent, true);

        checkStatusText(statusText);

        assertEquals("Expected method call", 1, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Status update should not be performed if max size of status
     * is reached.
     */
    public void testPerformStatusUpdateMaxTextSize() {
        // Create a string of max size plus one
        String statusText = "";
        for (int i = 0; i <= TwitterConf.MAX_CHARACTERS; i++) {
            statusText += "a";
        }

        mTwitterCommunication.trackMethodCalls("updateStatus");

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        performRegistration(1);

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE, statusText);
        serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
        executeServiceIntent(serviceIntent, true);

        checkStatusText(null);

        assertEquals("Unexpected method call", 0, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Valid status update should not result in a event-stream update
     * in case of the extras value is null.
     */
    public void testPerformStatusUpdateNull() {
        mTwitterCommunication.trackMethodCalls("updateStatus");

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        performRegistration(1);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
        executeServiceIntent(serviceIntent, true);

        checkStatusText(null);

        assertEquals("Unexpected method call", 0, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Valid status update should not result in a event-stream update
     * in case of the extras value is null.
     */
    public void testPerformStatusUpdateNotAuthenticated() {
        String statusText = "test_status";
        mTwitterCommunication.trackMethodCalls("updateStatus");

        // Remove the token
        PreferencesHelper.clearAuthentication(getContext());

        performRegistration(1);

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE, statusText);
        serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
        executeServiceIntent(serviceIntent, true);

        checkStatusText(null);

        assertEquals("Unexpected method call", 0, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Valid status update should not result in a event-stream update
     * in case of the extras value is null.
     */
    public void testPerformStatusUpdateNotRegistered() {
        String statusText = "test_status";
        mTwitterCommunication.trackMethodCalls("updateStatus");

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE, statusText);
        serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
        executeServiceIntent(serviceIntent, true);

        assertEquals("Unexpected method call", 0, mTwitterCommunication.getMethodCalls());
    }

    /**
     * Valid status update should not result in a event-stream update
     * in case of the Twitter throw exception.
     */
    public void testPerformStatusUpdateExceptionThrown() {
        String statusText = "test_status";

        mTwitterCommunication.trackMethodCalls("updateStatus");
        mTwitterCommunication.performThrowExceptionOnCall(true, StatusCode.SERVER_UNAVAILABLE);

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        performRegistration(1);

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE, statusText);
        serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
        executeServiceIntent(serviceIntent, true);

        checkStatusText(null);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
    }

    public void testAuthenticationValid() {
        mTwitterCommunication.trackMethodCalls("authenticateTwitterAccount");

        performRegistration(1);

        // Add event data
        generateUsersAndEvents(false);

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(InternalIntentExtraParams.USERNAME, USERNAME);
        serviceIntent.putExtra(InternalIntentExtraParams.PASSWORD, PASSWORD);
        serviceIntent.setAction(PluginConstants.TWITTER_AUTHENTICATE);
        executeServiceIntent(serviceIntent, true);

        checkConfigurationSet(CONFIGURED);
        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NBROFEVENTS);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NBROFUSERS);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
        Assert.assertEquals(State.SERVICE_AUTHENTICATION_SUCCESS, mApplication.getState());
    }

    public void testAuthenticationTwitterThrowException() {
        performRegistration(1);

        mTwitterCommunication.performThrowExceptionOnCall(true, StatusCode.INTERNAL_ERROR);
        mTwitterCommunication.trackMethodCalls("authenticateTwitterAccount");

        // Add event data
        generateUsersAndEvents(false);

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(InternalIntentExtraParams.USERNAME, USERNAME);
        serviceIntent.putExtra(InternalIntentExtraParams.PASSWORD, PASSWORD);
        serviceIntent.setAction(PluginConstants.TWITTER_AUTHENTICATE);
        executeServiceIntent(serviceIntent, true);

        checkConfigurationSet(NOT_CONFIGURED);
        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
        Assert.assertEquals(State.SERVICE_AUTHENTICATION_FAILED, mApplication.getState());
    }

    public void testAuthenticationTwitterReturnNull() {
        performRegistration(1);

        mTwitterCommunication.returnErrorOnCall(true);
        mTwitterCommunication.trackMethodCalls("authenticateTwitterAccount");

        // Add event data
        generateUsersAndEvents(false);

        Intent serviceIntent = new Intent();
        serviceIntent.putExtra(InternalIntentExtraParams.USERNAME, USERNAME);
        serviceIntent.putExtra(InternalIntentExtraParams.PASSWORD, PASSWORD);
        serviceIntent.setAction(PluginConstants.TWITTER_AUTHENTICATE);
        executeServiceIntent(serviceIntent, true);

        checkConfigurationSet(NOT_CONFIGURED);
        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
        Assert.assertEquals(State.SERVICE_AUTHENTICATION_FAILED, mApplication.getState());
    }

    public void testAuthenticationTwitterNoUsernameAndPassword() {
        performRegistration(1);

        mTwitterCommunication.returnErrorOnCall(true);
        mTwitterCommunication.trackMethodCalls("authenticateTwitterAccount");

        // Add event data
        generateUsersAndEvents(false);

        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_AUTHENTICATE);
        executeServiceIntent(serviceIntent, true);

        checkConfigurationSet(NOT_CONFIGURED);
        checkAmountOfData(EVENTSTREAM_EVENT_PROVIDER_URI, NOCONTENT);
        checkAmountOfData(EVENTSTREAM_FRIEND_PROVIDER_URI, NOCONTENT);

        assertEquals("Unexpected method call", 1, mTwitterCommunication.getMethodCalls());
        Assert.assertEquals(State.SERVICE_AUTHENTICATION_FAILED, mApplication.getState());
    }

    /**
     * If the user do clear data on the plugin we should
     * be in a registered state after the TWITTER_REFRESH_REQUEST
     * intent but not logged in.
     */
    public void testIsInBadStatePluginHasClearedData() {

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        // Do EventStream registration
        performRegistration(1);
        assertTrue(PreferencesHelper.isRegistered(mContext));

        // Remove all our preference data
        EventStreamHelper.clearAll(mContext.getContentResolver());

        // Call register
        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REFRESH_REQUEST);
        executeServiceIntent(serviceIntent, true);

        // We should be registered
        assertTrue(checkRegistration());
        assertTrue(PreferencesHelper.isRegistered(mContext));
    }

    /**
     * If the user do clear data on the EventStream although the
     * plugin thinks it's logged in. We should
     * be in a registered state after the TWITTER_REGISTER_PLUGIN
     * intent.
     */
    public void testIsInBadStateEventStreamHasClearData() {

        // Set token to fake a already authenticated user
        PreferencesHelper.setToken(getContext(), "token", "key");

        // We should not be registered
        assertFalse(checkRegistration());

        // Call register
        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REGISTER_PLUGIN);
        executeServiceIntent(serviceIntent, true);

        // We should be registered
        assertTrue(checkRegistration());
        assertTrue(PreferencesHelper.isRegistered(mContext));
    }

    private void checkStatusText(String expectedStatus) {
        Cursor c = null;
        try {
            c = getContext().getContentResolver().query(EVENTSTREAM_SOURCE_PROVIDER_URI, null,
                    null, null, null);
            assertNotNull(c);
            assertTrue(c.moveToFirst());
            String status = c.getString(c.getColumnIndexOrThrow(SourceColumns.CURRENT_STATUS));
            Assert.assertEquals("Expected status worng", expectedStatus, status);
        } finally {
            if (c != null) {
                c.close();
                c = null;
            }
        }
    }

    private void checkConfigurationSet(int expectedConfiguration) {
        Cursor c = null;
        try {
            c = getContext().getContentResolver().query(EVENTSTREAM_PLUGIN_PROVIDER_URI,
                    null, null, null, null);
            assertNotNull(c);
            assertTrue(c.moveToFirst());
            Assert.assertEquals(expectedConfiguration, c.getInt(c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_STATE)));
        } finally {
            if (c != null) {
                c.close();
                c = null;
            }
        }
    }

    private void checkAmountOfData(Uri uri, int expectedAmount) {
        Cursor c = null;
        try {
            c = getContext().getContentResolver().query(uri, null, null, null, null);
            assertNotNull(c);
            assertEquals("Wrong expected amount", expectedAmount, c.getCount());
        } finally {
            if (c != null) {
                c.close();
                c = null;
            }
        }
    }

    /**
     * Will perform a plug-in registration
     * @param service The Twitter service
     */
    private void performRegistration(int repeatedRequests) {
        Intent serviceIntent = new Intent();
        serviceIntent.setAction(PluginConstants.TWITTER_REGISTER_PLUGIN);

        for (int i = 0; i < repeatedRequests; i++) {
            executeServiceIntent(serviceIntent, true);
            assertTrue(checkRegistration());
        }
    }

    /**
     * Check if the plug-in is registered
     * @return true if registered
     */
     private boolean checkRegistration() {
         Context context = getSystemContext();
         Cursor c = null;
         try {
             c = getContext().getContentResolver().query(EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null, null, null);
             if (c != null && c.moveToFirst() && c.getCount() == 1) {

                 // Check the name
                 String expectedPluginName = context.getResources().getString(R.string.ts_twitter_name_txt);
                 String realPluginName = c.getString(c.getColumnIndexOrThrow(PluginColumns.NAME));
                 Assert.assertEquals(expectedPluginName, realPluginName);

                 // Check the icon
                 Builder iconUriBuilder = new Uri.Builder().scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
                 .authority(PACKAGE_NAME).appendPath(Integer.toString(R.drawable.twitter_icn));
                 String expectedPluginIconUri = iconUriBuilder.toString();
                 String realPluginIconUri = c.getString(c.getColumnIndexOrThrow(PluginColumns.ICON_URI));
                 Assert.assertEquals(expectedPluginIconUri, realPluginIconUri);

                 // Check the API version
                 int expectedVersion = 1;
                 Assert.assertEquals(expectedVersion, c.getInt(c.getColumnIndexOrThrow(PluginColumns.API_VERSION)));

                 // Check the configuration activity
                 ComponentName expectedConfigComponentName = new ComponentName(PACKAGE_NAME, ConfigurationActivity.class.getName());
                 ComponentName realConfigComponentName = ComponentName.unflattenFromString(c.getString(c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_ACTIVITY)));
                 Assert.assertEquals(expectedConfigComponentName, realConfigComponentName);

                 // Check the config state
                 Assert.assertEquals(NOT_CONFIGURED, c.getInt(c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_STATE)));

                 // Check the config text
                 String expectedConfigText = context.getResources().getString(R.string.ts_twitter_register_txt);
                 String realConfigText = c.getString(c.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_TEXT));
                 Assert.assertEquals(expectedConfigText, realConfigText);

                 // Check the status support
                 int expectedStatus = 1;
                 Assert.assertEquals(expectedStatus, c.getInt(c.getColumnIndexOrThrow(PluginColumns.STATUS_SUPPORT)));

                 // Check the max length
                 Assert.assertEquals(TwitterConf.MAX_CHARACTERS, c.getInt(c.getColumnIndexOrThrow(PluginColumns.STATUS_TEXT_MAX_LENGTH)));

                 if (checkSourceRegistration()) {
                     return true;
                 }
             }
         } finally {
             if (c != null) {
                 c.close();
             }
         }
         return false;
     }

     /**
      * Check if the source is registered
      * @return true if registered
      */
     private boolean checkSourceRegistration() {
         Context context = getSystemContext();
         Cursor c = null;
         try {
             c = getContext().getContentResolver().query(
                     EVENTSTREAM_SOURCE_PROVIDER_URI, null, null, null,
                     null);
             if (c != null && c.moveToFirst() && c.getCount() == 1) {

                 // Check the name
                 String expectedPluginName = context.getResources().getString(R.string.ts_twitter_name_txt);
                 String realPluginName = c.getString(c.getColumnIndexOrThrow(SourceColumns.NAME));
                 Assert.assertEquals(expectedPluginName, realPluginName);

                 // Check the icon
                 Builder iconUriBuilder = new Uri.Builder().scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
                 .authority(PACKAGE_NAME).appendPath(Integer.toString(R.drawable.twitter_icn));
                 String expectedPluginIconUri = iconUriBuilder.toString();
                 String realPluginIconUri = c.getString(c.getColumnIndexOrThrow(SourceColumns.ICON_URI));
                 Assert.assertEquals(expectedPluginIconUri, realPluginIconUri);

                 // Check the enable status
                 int expectedEnabled = 1;
                 Assert.assertEquals(expectedEnabled, c.getInt(c.getColumnIndexOrThrow(SourceColumns.ENABLED)));

                 return true;
             }
         } finally {
             if (c != null) {
                 c.close();
             }
         }
         return false;
     }

     /**
      * Generates and set the Twitter communication user and event values.
      * @param allocatedButNoContent If lists should be created but with no content
      */
     private void generateUsersAndEvents(boolean allocatedButNoContent) {
         int nbrOfUsers = allocatedButNoContent ? 0 : NBROFUSERS;
         int nbrOfStatusPerUser = allocatedButNoContent ? 0 : NBROFSTATUSPERUSER;
         int nbrOfMessagesPerUser = allocatedButNoContent ? 0 : NBROFMSGPERUSER;

         MockTwitter twitter = new MockTwitter();
         PagableResponseList<User> users = twitter.generateSomeUsers(nbrOfUsers);
         List<Status> statuses = twitter.generateStatuses(users, nbrOfStatusPerUser);
         List<DirectMessage> messages = twitter.generateMessages(users, nbrOfMessagesPerUser);
         User ownUser = twitter.generateOwnUser();
         mTwitterCommunication.setOwnUser(ownUser);
         mTwitterCommunication.setFriends(users);
         mTwitterCommunication.setStatuses(statuses);
         mTwitterCommunication.setMessages(messages);
     }

     private String getUserString() {
         User user = null;
         String userStr = "";
         try {
             user = mTwitterCommunication.getOwnUser();
             if (user != null) {
                 userStr = user.getScreenName();
             }
         }
         catch (TwitterPluginException e) {
         }
         return userStr;
     }

     private void setLocaleColumns(String configName, String configText) {
         Cursor cursor = null;
         try {
             ContentResolver contentResolver = getContext().getContentResolver();
             cursor = contentResolver.query(EVENTSTREAM_SOURCE_PROVIDER_URI, null, null, null, null);
             if (cursor != null && cursor.moveToFirst() && cursor.getCount() == 1) {
                 ContentValues values = new ContentValues();
                 values.put(PluginColumns.NAME, configName);
                 values.put(PluginColumns.CONFIGURATION_TEXT, configText);
                 contentResolver.update(EVENTSTREAM_PLUGIN_PROVIDER_URI, values, null, null);
             }
         } finally {
             if (cursor != null) {
                 cursor.close();
             }
         }
     }

     private void checkLocaleColumns(String configName, String configText, boolean shouldFail) {
         Context context = getSystemContext();
         Cursor cursor = null;
         try {
             ContentResolver contentResolver = getContext().getContentResolver();
             cursor = contentResolver.query(EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null, null, null);
             if (cursor != null && cursor.moveToFirst() && cursor.getCount() == 1) {
                 // Check the name
                 String tempName = cursor.getString(cursor.getColumnIndexOrThrow(PluginColumns.NAME));
                 if (shouldFail) {
                     Assert.assertTrue(!configName.equals(tempName));
                 } else {
                     Assert.assertTrue(configName.equals(tempName));
                 }

                 // Check the text
                 String tempText = cursor.getString(cursor.getColumnIndexOrThrow(PluginColumns.CONFIGURATION_TEXT));
                 if (shouldFail) {
                     Assert.assertTrue(!configText.equals(tempText));
                 } else {
                     Assert.assertTrue(configText.equals(tempText));
                 }
             }
         } finally {
             if (cursor != null) {
                 cursor.close();
             }
         }
     }

     private void changeOfLanguage(boolean loggedIn) {
         performRegistration(1);
         Context context = getSystemContext();
         // Create a user in Twitter
         generateUsersAndEvents(true);

         String expectedConfigName = context.getResources().getString(R.string.ts_twitter_name_txt);
         String expectedConfigText = context.getResources().getString(R.string.ts_twitter_register_txt);
         // Verify that the database contains the initial strings
         checkLocaleColumns(expectedConfigName, expectedConfigText, false);

         // Set fake values in EventStream
         setLocaleColumns("Twitter-xxxx", "Testing-yyyyy");
         // Verify that fake values was written
         checkLocaleColumns(expectedConfigName, expectedConfigText, true);

         if (!loggedIn) {
             // Force exception. Emulates that the user is not authorized
             mTwitterCommunication.performThrowExceptionOnCall(true,
                     TwitterPluginException.StatusCode.UNAUTHORIZED);
         }

         // Notify the service that language has changed. This should update database
         // with correct texts for a logged in user. For Twitter only the login/logout
         // text is affected by language changes.
         Intent serviceIntent = new Intent();
         serviceIntent.setAction(PluginConstants.TWITTER_LOCALE_CHANGED);
         executeServiceIntent(serviceIntent, true);

         // Get the expected texts.
         String expectedText;
         if (loggedIn) {
             // Get the user name
             // Create the correct text shown on a logged in account
             expectedText = context.getResources().getString(R.string.ts_twitter_logout_txt);
             expectedText += " " + getUserString();
         } else {
             expectedText = context.getResources().getString(R.string.ts_twitter_register_txt);
         }

         // Now the text should be correct again
         checkLocaleColumns(expectedConfigName, expectedText, false);
     }
}
