
/* --------------------------------------------- */
/*  H8-3069F USB Mass Storage Class              */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "usb_ms.h"

#define MS_DATA_DIR_IN   0x80,
#define MS_DATA_DIR_OUT  0x00,
#define MS_DATA_DIR_NONE 0x00

#define  CBW_SIZE                      31
#define  CSW_SIZE                      13

#define  SCSI_CMD_REQUEST_SENSE      0x03
#define  SCSI_CMD_TEST_UNIT_READY    0x00
#define  SCSI_CMD_READ_10            0x28
#define  SCSI_CMD_READ_CAPACITY      0x25
#define  SCSI_CMD_WRITE_10           0x2A

#define  SCSI_LEN_REQUEST_SENSE      0x06
#define  SCSI_LEN_TEST_UNIT_READY    0x06
#define  SCSI_LEN_READ_CAPACITY      0x0A


// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
// Global
void usbms_1ms_handler(void);
int  usbms_initialize(void);
int  usbms_process(void);
int ui_function_usbms_debug(UI_COMMAND uicmd);

// Locals

// -------------------------------------------
//  Variables
// -------------------------------------------
// Global
// Locals
//  +------------+-----+-----+-----+-----+-----+-----+-----+-----+
//  |   Byte     |  7  |  6  |  5  |  4  |  3  |  2  |  1  |  0  |
//  +============+=====+=====+=====+=====+=====+=====+=====+=====+
//  |   0-3      |             dCBWSignature                     |
//  +------------+-----------------------------------------------+
//  |   4-7      |                 dCBWTag                       |
//  +------------+-----------------------------------------------+
//  |   8-11     |         dCBWDataTransferLength                |
//  +------------+-----------------------------------------------+
//  |   12       |               bmCBWFlags                      |
//  +------------+-----------------------+-----------------------+
//  |   13       |      Reserved(0)      |        bCBWLUN        |
//  +------------+-----------------+-----+-----------------------+
//  |   14       |   Reserved(0)   |         bCBWCBLength        |
//  +------------+-----------------+-----------------------------+
//  |   15-30    |                     CBWCB                     |
//  +------------+-----------------------------------------------+
static unsigned char CmdBlkWrapper[CBW_SIZE];

//  +------------+-----+-----+-----+-----+-----+-----+-----+-----+
//  |   Byte     |  7  |  6  |  5  |  4  |  3  |  2  |  1  |  0  |
//  +============+=====+=====+=====+=====+=====+=====+=====+=====+
//  |   0-3      |             dCSWSignature                     |
//  +------------+-----------------------------------------------+
//  |   4-7      |                 dCSWTag                       |
//  +------------+-----------------------------------------------+
//  |   8-11     |              dCSWDataResidue                  |
//  +------------+-----------------------------------------------+
//  |   12       |                 bCSWStatus                    |
//  +------------+-----------------------------------------------+
static unsigned char CmdStsWrapper[CSW_SIZE];


static int usbms_wait_timer;
static int usbms_disable_timer;
static unsigned char usbms_proc;
static unsigned char usbms_error;

static unsigned char max_lun;

enum usbms_error_code {
	USBMS_ERR_NONE,
	USBMS_ERR_TUR_FAIL,
	USBMS_PROCESS_ERROR = 100,
};

enum usbms_process_mode {
	USBMS_READY_WAIT,
	USBMS_GET_MAX_LUN_01,
	USBMS_GET_MAX_LUN_02,
	USBMS_TESTUNIT_READY_01,
	USBMS_TESTUNIT_READY_02,
	USBMS_TESTUNIT_READY_03,
	USBMS_TESTUNIT_READY_04,
	USBMS_READ_CAPACITY_01,
	USBMS_READ_CAPACITY_02,
	USBMS_READ_CAPACITY_03,
	USBMS_READ_CAPACITY_04,
	USBMS_IDLE,
	USBMS_ERROR_STOP = 0xFF,
};

typedef struct {
	int	num;
	unsigned char	type;
	char			display[MAX_COLUMN];
	void*			p_variable;
	unsigned int	interval_time;
} DEBUG_VAR_TABLE;

static const DEBUG_VAR_TABLE debug_var_table[] = {
	{	 0,		 1,	"Process number: ",	&usbms_proc,							 300		},
	{	 1,		 1,	"Error Code:     ",	&usbms_error,							 300		},
	{	 1,		 1,	"Max LUN:        ",	&max_lun,								 300		},
	{	-1,		 1,	"0123456789ABCDEF", 0,										   0		},
};


// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void usbms_1ms_handler(void) {
	if( usbms_wait_timer )    usbms_wait_timer--;
	if( usbms_disable_timer ) usbms_disable_timer--;
}

// -------------------------------------------
//  Initialize
// -------------------------------------------
int  usbms_initialize(void) {
	usbms_proc = USBMS_READY_WAIT;
	usbms_error = USBMS_ERR_NONE;
}

// -------------------------------------------
//  Main process
// -------------------------------------------
int  usbms_process(void) {
	unsigned char status, i;
	SETUP_FORMAT	setup_fmt;
	
	if(usbms_disable_timer) return 0;
	
	switch( usbms_proc )
	{
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Waiting for device class ready                                            *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_READY_WAIT:
			status = sl811_status(SL811_REQ_NONE);
			if( status == SL811_STS_READY ) {
				usbms_proc = USBMS_GET_MAX_LUN_01;
			} else if( status == SL811_STS_RESET_IN_PROGRESS ) {
					usbms_disable_timer = 100;   // 100ms Wait
			} else {
				if( sl811_status( SL811_REQ_RESET ) == SL811_STS_RESET_ACCEPTED )
					usbms_disable_timer = 1000;  // 1S Wait
			}
			break;
		
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Get max number of LUN                                                     *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_GET_MAX_LUN_01:
			setup_fmt.bmRequestType 	= DIR_DEV2HOST | USB_REQUEST_TYPE_CLASS| RECEIPIENT_INTERFACE;
			setup_fmt.bRequest 			= REQ_TYPE_GET_MAX_LUN;
			setup_fmt.wValue 			= 0;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= 1;
			
			if( sl811_ctrl_receive(&setup_fmt) == USB_REQ_ACCEPTED )
				usbms_proc = USBMS_GET_MAX_LUN_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
				
			break;
			
		case USBMS_GET_MAX_LUN_02:
			status = sl811_status(SL811_REQ_NONE);
			if( status == SL811_STS_READY ) {
				usbms_proc = USBMS_TESTUNIT_READY_01;
				sl811_buffer_copy( &max_lun, 1);
			} else if( status == SL811_STS_ERROR_STOP ) {
				// sl811_status( SL811_REQ_RESET );
				// usbms_proc = USBMS_GET_MAX_LUN_01;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
		
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Test Unit Ready                                                           *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_TESTUNIT_READY_01:
			for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
			CmdBlkWrapper[ 0] = 0x44;	CmdBlkWrapper[ 1] = 0x42;	CmdBlkWrapper[ 2] = 0x53;	CmdBlkWrapper[ 3] = 0x55;
			CmdBlkWrapper[ 8] = 0x00;
			CmdBlkWrapper[12] = MS_DATA_DIR_NONE;
			CmdBlkWrapper[14] = SCSI_LEN_TEST_UNIT_READY;
			CmdBlkWrapper[15] = SCSI_CMD_TEST_UNIT_READY;
			
			if( sl811_bulk_out_request(CmdBlkWrapper, CBW_SIZE) == USB_REQ_ACCEPTED )
				usbms_proc = USBMS_TESTUNIT_READY_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_TESTUNIT_READY_02:
			status = sl811_status(SL811_REQ_NONE);
			if( status == SL811_STS_READY ) {
				usbms_proc = USBMS_TESTUNIT_READY_03;
			} else if( status == SL811_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_TESTUNIT_READY_03:
			if( sl811_bulk_in_request(CmdStsWrapper, CSW_SIZE) == USB_REQ_ACCEPTED )
				usbms_proc = USBMS_TESTUNIT_READY_04;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_TESTUNIT_READY_04:
			status = sl811_status(SL811_REQ_NONE);
			if( status == SL811_STS_READY ) {
				if( CmdStsWrapper[12] 	!= 0 ) {
					usbms_error = USBMS_ERR_TUR_FAIL;
					usbms_proc  = USBMS_ERROR_STOP;
				} else {
					usbms_proc = USBMS_IDLE;
				}
			} else if( status == SL811_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Read Capacity                                                             *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USBMS_READ_CAPACITY_01:
			for(i=0; i<CBW_SIZE; i++) CmdBlkWrapper[i] = 0;
			CmdBlkWrapper[ 0] = 0x44;	CmdBlkWrapper[ 1] = 0x42;	CmdBlkWrapper[ 2] = 0x53;	CmdBlkWrapper[ 3] = 0x55;
			CmdBlkWrapper[ 8] = 0x00;
			CmdBlkWrapper[12] = MS_DATA_DIR_NONE;
			CmdBlkWrapper[14] = SCSI_LEN_READ_CAPACITY;
			CmdBlkWrapper[15] = SCSI_CMD_READ_CAPACITY;
			
			if( sl811_bulk_out_request(CmdBlkWrapper, CBW_SIZE) == USB_REQ_ACCEPTED )
				usbms_proc = USBMS_READ_CAPACITY_02;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_READ_CAPACITY_02:
			status = sl811_status(SL811_REQ_NONE);
			if( status == SL811_STS_READY ) {
				usbms_proc = USBMS_READ_CAPACITY_03;
			} else if( status == SL811_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_READ_CAPACITY_03:
			if( sl811_bulk_in_request(CmdStsWrapper, CSW_SIZE) == USB_REQ_ACCEPTED )
				usbms_proc = USBMS_READ_CAPACITY_04;
			else
				usbms_disable_timer = 100;   // 100ms Wait for retry
			break;
			
		case USBMS_READ_CAPACITY_04:
			status = sl811_status(SL811_REQ_NONE);
			if( status == SL811_STS_READY ) {
				if( CmdStsWrapper[12] 	!= 0 ) {
					usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
					usbms_proc  = USBMS_ERROR_STOP;
				} else {
					usbms_proc = USBMS_IDLE;
				}
			} else if( status == SL811_STS_ERROR_STOP ) {
				usbms_error = USBMS_PROCESS_ERROR + usbms_proc;
				usbms_proc = USBMS_ERROR_STOP;
			} else {
				usbms_disable_timer = 100;   // 100ms Wait for retry
			}
			break;
			
		case USBMS_IDLE:
			break;
		
		case USBMS_ERROR_STOP:
			break;
	}
}

// -------------------------------------------
//  UI Function - USB-MS debug
// -------------------------------------------
int ui_function_usbms_debug(UI_COMMAND uicmd) {
	static unsigned char current_index;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
	case UI_CMD_KEY_PRESS_OK:
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, debug_var_table[current_index].display );
		switch(debug_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_dump (1, debug_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}
