﻿//
//  MainWindow.cs
//
//  Author:
//       tsntsumi <tsntsumi at tsntsumi.com>
//
//  Copyright (c) 2015 tsntsumi
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// GTK# ベースの簡単な UDP チャットクライアントのウィンドウを実装。
/// </summary>
/// @since 2015.8.23
using System;
using System.Text;
using System.Net;
using System.Net.Sockets;
using Gtk;
using SocketNet;

/// <summary>
/// SocketNet ライブラリのサンプルプログラム (GTK# ベースの簡単な UDP チャットクライアント)。
/// </summary>
public partial class MainWindow: Gtk.Window
{
	UdpServer udpServer;

	/// <summary>
	/// コンストラクタ。
	/// </summary>
	public MainWindow () : base(Gtk.WindowType.Toplevel)
	{
		Build();

		sendAddressEntry.TooltipText = "送信先アドレスには、特定のアドレスのほかにブロードキャストアドレスを指定できます。";
	}

	/// <summary>
	/// ウィンドウのデリートイベントを処理します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="a">イベントデータを格納したオブジェクト。</param>
	protected void OnDeleteEvent(object sender, DeleteEventArgs a)
	{
		Application.Quit();
		a.RetVal = true;
	}

	/// <summary>
	/// 受信開始ボタンがクリックされたときに、受信開始処理を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnStartReceiveButtonClicked(object sender, EventArgs e)
	{
		try
		{
			int port = int.Parse(receivePortNoEntry.Text);

			if (udpServer == null)
			{
				udpServer = UdpServer.CreateServer(port);
				udpServer.DataReceived += OnMessageReceived;
			}
			udpServer.Start();
			receivePortNoEntry.Sensitive = false;
		}
		catch (FormatException ex)
		{
			receiveTextView.Buffer.Text = ex.Message;
		}
	}

	/// <summary>
	/// 受信停止ボタンがクリックされたときに、停止処理を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnStopReceiveButtonClicked(object sender, EventArgs e)
	{
		if (udpServer != null)
		{
			udpServer.Stop();
			udpServer = null;
		}
		receivePortNoEntry.Sensitive = true;
	}

	/// <summary>
	/// 送信ボタンをクリックされたときに、メッセージ入力エリアの文字列をサーバに送信します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	/// <remarks>
	/// UdpClient のインスタンスを作成するときに使用する送信先アドレスは、
	/// 固有アドレス、ブロードキャストアドレス、ローカルブロードキャストアドレスを指定することができる。
	/// </remarks>
	protected async void OnSendButtonClicked(object sender, EventArgs e)
	{
		try
		{
			int port = int.Parse(sendPortNoEntry.Text);
			Encoding encoding = new UTF8Encoding(false);
			byte[] data = encoding.GetBytes(messageEntry.Text);
			UdpClient udpClient = new UdpClient(sendAddressEntry.Text, port);

			udpClient.EnableBroadcast = true;
			udpClient.DontFragment = true;

			await udpClient.SendAsync(data, data.Length);
		}
		catch (FormatException ex)
		{
			receiveTextView.Buffer.Text = ex.Message;
		}
		catch (Exception ex)
		{
			receiveTextView.Buffer.Text = ex.Message;
		}
	}

	/// <summary>
	/// 受信ポート番号入力エリアが変更されたときに、開始/停止ボタンの有効化・無効化を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnReceivePortNoEntryChanged(object sender, EventArgs e)
	{
		bool hasText = receivePortNoEntry.Text.Length > 0;

		startReceiveButton.Sensitive = hasText;
		stopReceiveButton.Sensitive = hasText;
	}
	
	/// <summary>
	/// 送信先アドレス入力エリアが変更されたときに、送信ボタンの有効化・無効化を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnSendAddressEntryChanged(object sender, EventArgs e)
	{
		sendButton.Sensitive = (sendAddressEntry.Text.Length > 0 &&
								sendPortNoEntry.Text.Length > 0 &&
								messageEntry.Text.Length > 0);
	}

	/// <summary>
	/// 送信ポート番号入力エリアが変更されたときに、送信ボタンの有効化・無効化を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnSendPortNoEntryChanged(object sender, EventArgs e)
	{
		sendButton.Sensitive = (sendAddressEntry.Text.Length > 0 &&
								sendPortNoEntry.Text.Length > 0 &&
								messageEntry.Text.Length > 0);
	}

	/// <summary>
	/// メッセージ入力エリアが変更されたときに、送信ボタンの有効化・無効化を行います。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnMessageEntryChanged(object sender, EventArgs e)
	{
		sendButton.Sensitive = (sendAddressEntry.Text.Length > 0 &&
								sendPortNoEntry.Text.Length > 0 &&
								messageEntry.Text.Length > 0);
	}

	/// <summary>
	/// チャットメッセージを受信したときに、テキストビューにメッセージを追加します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnMessageReceived(object sender, UdpDataReceivedEventArgs e)
	{
		Encoding encoding = new UTF8Encoding(false);
		string message = encoding.GetString(e.Data);

		receiveTextView.Buffer.Text += e.SourceEndPoint.ToString() + "> " + message + "\n";
	}

	/// <summary>
	/// メッセージ入力エリアでリターンキーが押されたときに、送信ボタンをクリックする動作を実行します。
	/// </summary>
	/// <param name="sender">センダ。</param>
	/// <param name="e">イベントデータを格納したオブジェクト。</param>
	protected void OnMessageEntryActivated(object sender, EventArgs e)
	{
		sendButton.Click();
	}
}
