﻿// ChatClient.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// SocketNet ライブラリのサンプルのチャットクライアント。
/// </summary>
/// @since 2015.8.14
using System;
using System.Net;
using System.Net.Sockets;
using SocketNet;

/// <summary>
/// SocketNet ライブラリのサンプルプログラム (簡単なチャットクライアント)。
/// </summary>
namespace SampleChatClient
{
	/// <summary>
	/// SocketNet ライブラリのサンプルプログラム (簡単なチャットクライアント)。
	/// </summary>
	public class ChatClient
	{
		/// <summary>
		/// チャットメッセージを受信した時に発行されます。
		/// </summary>
		public event EventHandler<ChatMessageReceivedEventArgs> ChatMessageReceived;

		/// <summary>
		/// 接続先のIPアドレスを取得します。
		/// </summary>
		/// <value>The remote IP address.</value>
		public IPAddress RemoteIPAddress { get; private set; }

		/// <summary>
		/// 接続先のポート番号を取得します。
		/// </summary>
		/// <value>The remote port.</value>
		public int RemotePort { get; private set; }

		/// <summary>
		/// 接続を取得します。
		/// </summary>
		public TcpConnection Connection { get; private set; }

		/// <summary>
		/// 接続しているかどうかを取得します。
		/// </summary>
		public bool IsConnected {
			get { return Connection != null && Connection.Client.Connected; }
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		public ChatClient ()
		{
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="connection">接続。</param>
		public ChatClient (TcpConnection connection)
		{
			Connection = connection;
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="serverEndPoint">サーバのエンドポイント。</param>
		public ChatClient (IPEndPoint serverEndPoint)
			: this(serverEndPoint.Address, serverEndPoint.Port)
		{
		}

		/// <summary>
		/// コンストラクタ。
		/// </summary>
		/// <param name="serverIPAddress">サーバのIPアドレス。</param>
		/// <param name="serverPort">サーバのポート番号。</param>
		public ChatClient (IPAddress serverIPAddress, int serverPort)
			: this()
		{
			RemoteIPAddress = serverIPAddress;
			RemotePort = serverPort;
		}

		/// <summary>
		/// サーバに接続します。
		/// </summary>
		public void Connect()
		{
			Connect(RemoteIPAddress, RemotePort);
		}

		/// <summary>
		/// 指定されたエンドポイントのサーバに接続します。
		/// </summary>
		/// <param name="serverEndPoint">サーバのエンドポイント。</param>
		public void Connect(IPEndPoint serverEndPoint)
		{
			Connect(serverEndPoint.Address, serverEndPoint.Port);
		}

		/// <summary>
		/// 指定されたIPアドレスとポート番号のサーバに接続します。
		/// </summary>
		/// <param name="serverIPAddress">サーバのIPアドレス。</param>
		/// <param name="serverPort">サーバのポート番号。</param>
		public void Connect(IPAddress serverIPAddress, int serverPort)
		{
			if (Connection == null)
			{
				TcpClient client = new TcpClient();
				RemoteIPAddress = serverIPAddress;
				RemotePort = serverPort;
				client.Connect(RemoteIPAddress, RemotePort);
				Connection = new TcpConnection(client.Client, new ChatMessage());
				Connection.DataReceived += OnDataReceived;
			}
		}

		/// <summary>
		/// 接続を閉じます。
		/// </summary>
		public void Close()
		{
			if (Connection != null)
			{
				Connection.Dispose();
				Connection = null;
			}
		}

		/// <summary>
		/// メッセージの受信を開始します。
		/// </summary>
		public void BeginReceiveMessage()
		{
			Connection.ReceiveDataAsync();
		}

		/// <summary>
		/// 指定されたチャットメッセージを送信します。
		/// </summary>
		/// <param name="message">メッセージ。</param>
		public void Send(ChatMessage message)
		{
			byte[] data = message.ToByteArray();
			Connection.Writer.Write(data);
		}

		/// <summary>
		/// Tcpパケットを受信したときにチャットメッセージに変換して、メッセージ受信イベントを発行します。
		/// </summary>
		/// <param name="sender">センダ。</param>
		/// <param name="e">イベントデータを格納するオブジェクト。</param>
		private void OnDataReceived(object sender, TcpDataReceivedEventArgs e)
		{
			if (ChatMessageReceived != null)
			{
				ChatMessage chatMessage = ChatMessage.FromByteArray(e.Data);
				chatMessage.Connection = e.Connection;
				ChatMessageReceived(this, new ChatMessageReceivedEventArgs(chatMessage));
			}
		}
	}
}
