#include <comp/comp.h>
#include "comp_bitio.h"
#ifdef DEBUG
#include <stdio.h>
#endif

static void COMP_orderHuffmanHeap(COMP_huffman *huffman, int heap_index);
static void COMP_writeHuffmanTree(
    COMP_huffman *huffman, int node);
static int COMP_readHuffmanTree(COMP_huffman *huffman);


COMP_result COMP_createHuffman(COMP_huffman **huffman) {

    COMP_result result = COMP_ok;

    (*huffman) = (COMP_huffman *)COMP_malloc(sizeof(COMP_huffman));
    if ((*huffman) == NULL) {
#ifdef DEBUG
        fprintf(stderr, "file=%s, line=%d\n", __FILE__, __LINE__);
#endif
        result = COMP_err;
        goto err;
    }    

    /* 0 indicates the node has no parent. the node is root node. */
    COMP_memset((*huffman)->tree_parent, 0, 
        sizeof(int) * (2 * COMP_CODE_PATTERN_MAX - 1));
    COMP_memset((*huffman)->freq, 0, 
        sizeof(int) * COMP_CODE_PATTERN_MAX);
    (*huffman)->heap_size = 0;
    (*huffman)->inbuf_byte_pos = 0;
    (*huffman)->inbuf_bit_space = 8;
    (*huffman)->outbuf[0] = 0;
    (*huffman)->outbuf_data_size = 0;
    (*huffman)->outbuf_byte_pos = 0;
    (*huffman)->outbuf_bit_space = 8;
    (*huffman)->decoded_size = 0;
err:
    return result;
}

void COMP_disposeHuffman(COMP_huffman *huffman) {

    COMP_free(huffman);

    return;
}

static void COMP_orderHuffmanHeap(COMP_huffman *huffman, int heap_index) {
    int pos = heap_index, pos_child;
    int heap_value = huffman->heap[pos];

    /* pos_child is pos's left  child */
    for(pos_child = 2 * pos; 
        pos_child <= huffman->heap_size;
        pos_child = 2 * pos) {
        if (pos_child < huffman->heap_size) {
            /* select a smaller child */
            if (huffman->freq[huffman->heap[pos_child]] > 
                huffman->freq[huffman->heap[pos_child + 1]]) {
                /* pos child is pos's right child */
                pos_child++;
            }
        }
        if (huffman->freq[heap_value] <= 
            huffman->freq[huffman->heap[pos_child]]) {
            /* heap_value enters this position */
            break;
        }
        /* heap_value tries to enter the child position */
        huffman->heap[pos] = huffman->heap[pos_child];
        pos = pos_child;
    }
    huffman->heap[pos] = heap_value;

    return;
}

static void COMP_writeHuffmanTree(
    COMP_huffman *huffman, int node) {
    size_t n;

#ifdef DEBUG
    int i;
    for(i = COMP_CODE_PATTERN_MAX; i < COMP_CODE_PATTERN_MAX * 2; i++) {
        fprintf(stderr, "tree_left[%d]=%d\n", i, huffman->tree_left[i]);
    }
    for(i = COMP_CODE_PATTERN_MAX; i < COMP_CODE_PATTERN_MAX * 2; i++) {
        fprintf(stderr, "tree_right[%d]=%d\n", i, huffman->tree_right[i]);
    }
#endif

    if (node < COMP_CODE_PATTERN_MAX) {
        /* leaf, write bit 0 */
        huffman->outbuf_bit_space--;
        if (huffman->outbuf_bit_space == 0) {
            huffman->outbuf_byte_pos++;
            huffman->outbuf[huffman->outbuf_byte_pos] = 0;
            huffman->outbuf_bit_space = 8;
        }
        /* write the leaf's character */
        n = 8;
        while(n >= huffman->outbuf_bit_space) {
            n -= huffman->outbuf_bit_space;
            huffman->outbuf[huffman->outbuf_byte_pos] |=
                COMP_GET_RIGHT_BITS((node >> n), huffman->outbuf_bit_space);
            huffman->outbuf_byte_pos++;
            huffman->outbuf[huffman->outbuf_byte_pos] = 0;
            huffman->outbuf_bit_space = 8;
        }
        huffman->outbuf_bit_space -= n;
        huffman->outbuf[huffman->outbuf_byte_pos] |=
            (COMP_GET_RIGHT_BITS(node, n) << huffman->outbuf_bit_space);
    }
    else {
        /* not leaf, write bit 1 */
        huffman->outbuf_bit_space--;
        COMP_SET_BIT(
            huffman->outbuf[huffman->outbuf_byte_pos], 
            huffman->outbuf_bit_space);
        if (huffman->outbuf_bit_space == 0) {
            huffman->outbuf_byte_pos++;
            huffman->outbuf[huffman->outbuf_byte_pos] = 0;
            huffman->outbuf_bit_space = 8;
        }
        COMP_writeHuffmanTree(huffman, huffman->tree_left[node]);
        COMP_writeHuffmanTree(huffman, huffman->tree_right[node]);
    }
    return;
}

COMP_result COMP_encodeHuffman(COMP_huffman *huffman) {
    int i, j, k, avail, tabesize, node, node_left, node_right, node_avail;
    int heap_index;
    unsigned long int incount, cr;
    static char codebit[COMP_CODE_PATTERN_MAX];
    size_t n;
    COMP_result result = COMP_ok;

    /* count frequency */
    for(i = 0; i < huffman->inbuf_data_size; i++) {
        huffman->freq[huffman->inbuf[i]]++;
    }

    /* create heap for huffman */
    /* heap array starts from index 1 */
    for(node = 0; node < COMP_CODE_PATTERN_MAX; node++) {
        if (huffman->freq[node] != 0) {
            huffman->heap_size++;
            huffman->heap[huffman->heap_size] = node;
        }
    }
    for(heap_index = huffman->heap_size / 2; heap_index > 0; heap_index--) {
        COMP_orderHuffmanHeap(huffman, heap_index);
    }

    /* create huffman tree */
    if (huffman->heap_size < 2) { 
        node = huffman->heap[1];
    }
    else {
        for(node_avail = COMP_CODE_PATTERN_MAX; 
            huffman->heap_size >= 2;) {
            /* get the least frequnecy node */
            node_left = huffman->heap[1];
            huffman->heap[1] = huffman->heap[huffman->heap_size]; 
            huffman->heap_size--;
            COMP_orderHuffmanHeap(huffman, 1);
            /* get the least frequnecy node */
            node_right = huffman->heap[1];
            /* crete new node */
            node = node_avail;
            node_avail++;
            huffman->freq[node] = 
                huffman->freq[node_left] + huffman->freq[node_right];
            huffman->heap[1] = node;
            COMP_orderHuffmanHeap(huffman, 1);
            /* create huffman tree */
            huffman->tree_parent[node_left] = node;
            huffman->tree_parent[node_right] = -node;
            huffman->tree_left[node] = node_left;
            huffman->tree_right[node] = node_right;
        }
    }

    /* decode original data size */
    huffman->decoded_size += huffman->inbuf_data_size;
    /* write original data size */
    COMP_memcpy(huffman->outbuf,
        &huffman->decoded_size, sizeof(unsigned long int));
    huffman->outbuf_byte_pos += sizeof(unsigned long int);
    /* decode huffman tree */
    /* node is root node */
    COMP_writeHuffmanTree(huffman, node);
    /* decode data */
    for(i = 0; i < huffman->inbuf_data_size; i++) {
        /* from leaf to root */
#ifdef DEBUG
        fprintf(stderr, "encode %c: ", huffman->inbuf[i]);
#endif
        node = huffman->tree_parent[huffman->inbuf[i]];
        for(j = 0; node != 0; node = huffman->tree_parent[node], j++) {
            if (node > 0) {
                /* from left_child to parent */
                codebit[j] = 0;
            }
            else {
                /* from right_child to parent */
                codebit[j] = 1;
                node = -node;
            }
        }
        for(j = j - 1; j >= 0; j--) {
            huffman->outbuf_bit_space--;
            if(codebit[j] == 0) {
                /* write bit 0 */
#ifdef DEBUG
                fprintf(stderr, "0");
#endif
            }
            else {
                /* write bit 1 */
#ifdef DEBUG
                fprintf(stderr, "1");
#endif
                COMP_SET_BIT(
                    huffman->outbuf[huffman->outbuf_byte_pos], 
                    huffman->outbuf_bit_space);
            }
            if (huffman->outbuf_bit_space == 0) {
                huffman->outbuf_byte_pos++;
                huffman->outbuf[huffman->outbuf_byte_pos] = 0;
                huffman->outbuf_bit_space = 8;
            }
        }
#ifdef DEBUG
        fprintf(stderr, "\n");
#endif
    }
    /* padding */
    /* this does not work well on buffer output */
    /* sample works on file output and internal buffer */
    n = 7;
    while(n >= huffman->outbuf_bit_space) {
        n -= huffman->outbuf_bit_space;
        huffman->outbuf[huffman->outbuf_byte_pos] |=
            COMP_GET_RIGHT_BITS((0 >> n), huffman->outbuf_bit_space);
        huffman->outbuf_byte_pos++;
        huffman->outbuf[huffman->outbuf_byte_pos] = 0;
        huffman->outbuf_bit_space = 8;
    }
    huffman->outbuf_bit_space -= n;
    huffman->outbuf[huffman->outbuf_byte_pos] |=
        (COMP_GET_RIGHT_BITS(0, n) << huffman->outbuf_bit_space);

    huffman->outbuf_data_size = huffman->outbuf_byte_pos;
    return result;
}

static int COMP_readHuffmanTree(COMP_huffman *huffman) {
    int node;
    unsigned int x = 0, bit;
    /* TODO */
    /* 
     * (x < COMP_CODE_PATTERN_MAX) mean leaf value
     * (COMP_CODE_PATTERM_MAX <= x) mean array index
     */
    static int next_node = COMP_CODE_PATTERN_MAX;
    size_t n;

    /* tree is written from root to leaf */
    /* get bit */
    if (huffman->inbuf_bit_space == 0) {
        huffman->inbuf_byte_pos++;
        huffman->inbuf_bit_space = 7;
    }
    else {
        huffman->inbuf_bit_space--;
    }
    bit = (huffman->inbuf[huffman->inbuf_byte_pos] >>
            huffman->inbuf_bit_space) & 1U;
    /* */ 
    if(bit == 1) {
        /* not leaf node */
        node = next_node;
        next_node++;
        if (node >= 2 * COMP_CODE_PATTERN_MAX - 1) {
#ifdef DEBUG
            fprintf(stderr, "wrong table");
#endif
        }
        huffman->tree_left[node] = COMP_readHuffmanTree(huffman);
        huffman->tree_right[node] = COMP_readHuffmanTree(huffman);
        return node;
    }
    else {
        /* leaf node */
        n = 8;
        for(; n > huffman->inbuf_bit_space;) {
            n -= huffman->inbuf_bit_space;
            x |= (COMP_GET_RIGHT_BITS(
                huffman->inbuf[huffman->inbuf_byte_pos],
                huffman->inbuf_bit_space) << n);
            huffman->inbuf_byte_pos++;
            huffman->inbuf_bit_space = 8;
        }
        huffman->inbuf_bit_space -= n;
        return x | COMP_GET_RIGHT_BITS(
            huffman->inbuf[huffman->inbuf_byte_pos] >> 
            huffman->inbuf_bit_space, n);
    }
}

COMP_result COMP_decodeHuffman(COMP_huffman *huffman) {
    int node, root;
    unsigned long int i;
    unsigned int bit;
    COMP_result result = COMP_ok;

    /* read original data size */
    COMP_memcpy(&huffman->decoded_size, 
        huffman->inbuf, sizeof(unsigned long int));
    huffman->inbuf_byte_pos += sizeof(unsigned long int);
    /* read huffman tree */
    root = COMP_readHuffmanTree(huffman);

#ifdef DEBUG
    for(i = COMP_CODE_PATTERN_MAX; i < COMP_CODE_PATTERN_MAX * 2; i++) {
        fprintf(stderr, "tree_left[%d]=%d\n", i, huffman->tree_left[i]);
    }
    for(i = COMP_CODE_PATTERN_MAX; i < COMP_CODE_PATTERN_MAX * 2; i++) {
        fprintf(stderr, "tree_right[%d]=%d\n", i, huffman->tree_right[i]);
    }
#endif

    for(i = 0; i < huffman->decoded_size; i++) {
        node = root;
        while (node >= COMP_CODE_PATTERN_MAX) {
            /* node is not leaf */
            /* get bit */
            if (huffman->inbuf_bit_space == 0) {
                huffman->inbuf_byte_pos++;
                huffman->inbuf_bit_space = 7;
            }
            else {
                huffman->inbuf_bit_space--;
            }
            bit = (huffman->inbuf[huffman->inbuf_byte_pos] >>
                    huffman->inbuf_bit_space) & 1U;
            /* */
            if (bit == 1) {
                node = huffman->tree_right[node];
            }
            else {
                node = huffman->tree_left[node];
            }
        }
        /* node is leaf, decoded character */
        huffman->outbuf[i] = node;
    }

    huffman->outbuf_byte_pos = i;
    huffman->outbuf_data_size = huffman->outbuf_byte_pos;
    return result;
}
