/**************************************************
Opengate server
 module for Communication through CGI 

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
Modified Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp

Programmed by Yoshiaki WATANABE
Modified by Shin-ichi TADAKI
Modified by Katsuhiko Eguchi
Modified by Makoto Otani
**************************************************/

#include	"opengatesrv.h"

/* convert two-char-hex "aa" to one-hex-number 0Xaa */ 
#define hex2num(x)  ((x)>='A' ? ((x) & 0XDF) - 'A' +10 : ((x) - '0'))

int isHttpWatchEnableClient(void);
void split(char content[], char *name[], char *value[], char *next[]);
void decode(char *string);

/*******************************/
/* get the client address      */
/*******************************/
void getClientAddr(char *clientAddr)
{
  strlcpy(clientAddr, getenv("REMOTE_ADDR"), ADDRMAXLN);
}

/********************************************/
/* get Post data from the client  */
/********************************************/
int getPostData(char *userid, char *password, char *clientAddr4, int *durationPtr, int *durationEntered, char *language, char *redirectedUrl)
{
  int contentLen=0;
  int queryStrLen=0;
  char content[BUFFMAXLN]="";
  char queryStr[BUFFMAXLN]="";
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;
  char durationStr[WORDMAXLN]="";
  char langList[BUFFMAXLN]="";
  char encodeAddr4[ADDRMAXLN]="";
  char accessAddr[ADDRMAXLN]="";

  /* get POST content sent from web input */
  if(getenv("CONTENT_LENGTH")!=NULL && (contentLen=atoi(getenv("CONTENT_LENGTH")))!=0){
    
    /* if too long string, cut off tail */
    contentLen++; /* for counting terminate char */
    if(contentLen > BUFFMAXLN) contentLen=BUFFMAXLN;
    if(fgets(content, contentLen, stdin) == NULL){
      content[0]='\0';
    }
    
    /* get items from string */
    /* default is null string */
    userid[0]='\0';
    password[0]='\0';
    encodeAddr4[0]='\0';
    language[0]='\0';
    durationStr[0]='\0';
    redirectedUrl[0]='\0';

    /* pick up the name and its value from the POST content 'name=value&name=value..' */
    ptr=content;
    while(ptr!=NULL){
      split(ptr, name, value, next);
      
      if(strstr(name[0], "userid")!=NULL){
	strlcpy(userid, value[0], USERMAXLN);
      }else if(strstr(name[0], "password")!=NULL){
	strlcpy(password, value[0], PASSMAXLN);
      }else if(strstr(name[0],"remote_addr")!=NULL){
	strlcpy(encodeAddr4,value[0],ADDRMAXLN);
      }else if(strstr(name[0], "language")!=NULL){
	strlcpy(language, value[0], WORDMAXLN);
      }else if(strstr(name[0], "duration")!=NULL){
	strlcpy(durationStr, value[0], WORDMAXLN);
      }else if(strstr(name[0], "redirected_url")!=NULL){
	strlcpy(redirectedUrl, value[0], BUFFMAXLN);
      }
      ptr=next[0];
    }
  }

  /* get paremeters from query string */
  /* described in url such as http://site/path?name=value&name=value */
  if(getenv("QUERY_STRING")!=NULL && (queryStrLen=strlen(getenv("QUERY_STRING")))!=0){

    /* get html access parameter string */
    strlcpy(queryStr, getenv("QUERY_STRING"), BUFFMAXLN);

    /* split language and address in GET content
       [addr=0-0-0&lang=ja&redirectedurl=xxxx] */
    ptr=queryStr;
    while(ptr!=NULL){
      split(ptr, name, value, next);
      if(strstr(name[0], "addr")!=NULL){
	strlcpy(encodeAddr4,value[0],ADDRMAXLN);
      }else if(strstr(name[0], "lang")!=NULL){
	strlcpy(language, value[0], WORDMAXLN);
      }else if(strstr(name[0], "redirectedurl")!=NULL){
	strlcpy(redirectedUrl, value[0], BUFFMAXLN);
      }
      ptr=next[0];
    }
  }

  /* if no content and no query string, return error */
  if(contentLen==0 && queryStrLen==0){
    err_msg("ERR at %s#%d: no parameter is aquired",__FILE__,__LINE__);
    return FALSE;
  }

  /* decode the HTTP encoding */
  decode(userid);
  decode(password);
  decode(encodeAddr4);
  decode(language);
  decode(durationStr);
  decode(redirectedUrl);

  /* if indicated language is null or not available */
  /* (language="xy" but html document for "xy" language is not prepared), */
  /* use first language in conf file (eg "en") */
  strlcpy(langList, GetConfValue("HtmlLangs"), BUFFMAXLN);  /* list of available languages */
  if(isNull(language) || strstr(langList,language)==NULL){
    sscanf(langList,"%s",language);
  }
  
  /* convert duration string to integer and minutes to seconds */
  *durationPtr = atoi(durationStr)*60;
  if(*durationPtr > 0) *durationEntered = TRUE;
  else *durationEntered = FALSE;
  
  /* encoded address starting as "0-0-0" means no addr info */
  /* it indicates needless of getting dual stack addresses (IPv4&IPv6) */
  /* and only use getenv("REMOTE_ADDR") address */
  if(strnstr(encodeAddr4, "0-0-0", ADDRMAXLN)==encodeAddr4){
    clientAddr4[0]='\0';
  }
      
  /* decode client address to dot separated form */
  else if(AddrDecode(clientAddr4, encodeAddr4)==1){
    /* if can't decode, retry */
    err_msg("ERR at %s#%d: Cannot decode client address",__FILE__,__LINE__);
    return FALSE;
  }

  /* if the decoded IPv4 addr is not same as access IPv4 addr, use later */
  strlcpy(accessAddr, getenv("REMOTE_ADDR"), ADDRMAXLN);
  if((strnstr(accessAddr, ".", ADDRMAXLN)!=NULL)   /* access is IPv4 */
     && strncmp(accessAddr, clientAddr4, ADDRMAXLN)!=0){  /* and not same */
    strlcpy(clientAddr4, accessAddr, ADDRMAXLN);
  }

  /* if no userid, set useid="?" */
  if(isNull(userid)){
    userid[0]='?';
    userid[1]='\0';
  }

  return TRUE;
}


/********************************************/
/* get userid from environment variable     */
/*  in the case of shibboleth or httpbasic, */
/*  the userid is aquired from env variable.*/
/********************************************/
int getUserIdFromEnv(char *userid){

  int ret=FALSE;
  char* pEnv=NULL;

  /* if shibboleth or httpbasic, get uid from environment var */
  /* shibboleth */
  if(strcmp(GetConfValue("AuthServer/Protocol"), "shibboleth")==0){
    pEnv=getenvEx(GetConfValue("AuthServer/UidAttribute"),TRUE,TRUE);
    if(!isNull(pEnv)){
      strlcpy(userid, pEnv, USERMAXLN);

      /* if idp string can be get from env variable, concatinate it as uid@org */
      pEnv=getenvEx(GetConfValue("AuthServer/OrgAttribute"),TRUE,TRUE);
      if(!isNull(pEnv)){
	strlcat(userid, GetConfValue("UserIdSeparator"), USERMAXLN);
	strlcat(userid, pEnv, USERMAXLN);
      } 
      ret=TRUE;
    }

    /* get from env-variable for EPPN(edu person principal name) attribute */
    else{
      pEnv=getenvEx(GetConfValue("AuthServer/EppnAttribute"),TRUE,FALSE);
      if(!isNull(pEnv)){
	strlcat(userid, pEnv, USERMAXLN);
	ret=TRUE;
      }
    }

    /* no env-variable is found */
    if(ret==FALSE){
      err_msg("ERR at %s#%d: Cannot get user info from shibboleth",__FILE__,__LINE__);
      PutClientMsg("Cannot get user info from shibboleth<br>Check shibboleth setting in .htaccess and other");
      exit(0);
    }    
  }

  /* httpbasic */
  else if(strcmp(GetConfValue("AuthServer/Protocol"), "httpbasic")==0){
    if(!isNull(getenv("REMOTE_USER"))){
      strlcpy(userid,getenv("REMOTE_USER"),USERMAXLN);
      ret=TRUE;
    }else{
      err_msg("ERR at %s#%d: Cannot get user info from httpbasic",__FILE__,__LINE__);
      ret=FALSE;
      PutClientMsg("Cannot get user info from http basic<br>Check http basic setting in .htaccess and other");
      exit(0);
    }
  }
  return ret;
}

/******************************************/
/* get cookie auth                        */
/*  get session-key & userid from client  */
/*  get related information from Db       */
/*  if info is not correct, return FALSE. */
/******************************************/
int getCookieData(char *userid, char *clientAddr4, int *duration, int *durationEntered, char *language, char* closeTime)
{
  char cookie[SIDMAXLN]=""; /* md5 session key from cookie */
  char useridInCookie[USERMAXLN]=""; /* userid from cookie */
  char macAddr[ADDRMAXLN]=""; /* mac address from arp */
  char macAddrInDb[ADDRMAXLN]=""; /* mac address in DB */

  /* get cookie values */
  if(!GetAuthCookie(cookie,useridInCookie)) return FALSE; 

  /* get related info from DB */ 
  if(!GetSessionInfoFromDb(cookie, userid, clientAddr4, macAddrInDb, 
			   duration, durationEntered, language, closeTime)) return FALSE;
    
  /* if userid is changed, cookie auth is failed */
  if(strcmp(useridInCookie, userid)!=0) return FALSE;

  /* if mac addr is changed, cookie auth is failed */
  GetMacAddrFromArp(clientAddr4, macAddr);
  if(strcmp(macAddr, macAddrInDb)!=0) return FALSE;

  return TRUE;
}

/*********************************************/
/* get HTTP-Cookie for OpengateAuth          */
/*********************************************/
  /* OpengateAuth(session-key) and UserId are included in the cookie string.
  for examples 
  "OpengateAuth=de..ac1&Userid=user1"
  "OpengateAuth=de..ac1&Userid=user1; xxx=..; yyy=.."
  "xxx=..; yyy=..; OpengateAuth=de..ac1&Userid=user1"
  */
int getAuthCookie(char *cookie, char *userid){
  char content[BUFFMAXLN];
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr=NULL;
  char *ptrNext=NULL;

  /* reset buffer */
  cookie[0]='\0';
  userid[0]='\0';

 /* if cookie exists, copy it to work area */
  if(isNull(getenv("HTTP_COOKIE"))) return FALSE;
  strlcpy(content, getenv("HTTP_COOKIE"), BUFFMAXLN);
  ptr=content;

  /* search 'OpengateAuth' cookie string (terminated by ; or \0) */
  while(ptr!=NULL){
    if((ptrNext=strstr(ptr, "; "))==NULL) break;          /* search "; " */
    *ptrNext='\0';                               /* overwrite string end */
    ptrNext++;                                 /* pointer to next string */
    while(!isNull(ptrNext)&&*ptrNext==' ') ptrNext++;     /* skip spaces */
    if(strstr(ptr, COOKIENAME)==ptr) break;          /* exit at matching */
    ptr=ptrNext;                                    /* check next string */
  }

  /* get valuses of cookie from "OpengateAuth=de..ac1&Userid=user1" */
  /* COOKIENAME is set as "OpengateAuth" in header file */
  while(ptr!=NULL){
    split(ptr, name, value, next);

    if(strstr(name[0], COOKIENAME)!=NULL){
      strlcpy(cookie, value[0], SIDMAXLN);
    }else if(strstr(name[0], "Userid")!=NULL){
      strlcpy(userid, value[0], USERMAXLN);
    }
    ptr=next[0];
  }
  
  if(isNull(cookie)) return FALSE;
  else return TRUE;
}

/*****************************************************/
/* deny message to the client                        */
/*  DenyDoc in conf file indicates the html template */
/*****************************************************/
void putClientDeny(char *clientAddr4, char *language)
{
  char denydoc[BUFFMAXLN]="";
  char authCgiUrl[BUFFMAXLN]="";
  char encodeAddr[ADDRMAXLN]="";
  char opengateDir[BUFFMAXLN]="";
  char protocol[WORDMAXLN]="";

  /* keyword pairs */
  /*  the left key in template file is replaced by the right value */
  struct html_key keys[]=
    {
      {"%%OPENGATEDIR%%", opengateDir},
      {"%%AUTHCGIURL%%", authCgiUrl},
      {"%%ADDR4%%", encodeAddr},
      {"%%LANGUAGE%%", language},
      {"%%PROTOCOL%%", protocol},
      {"",""}  /* DON'T REMOVE THIS LINE */
    };
 
  /* create authcgi URL string */
  snprintf(authCgiUrl, BUFFMAXLN, "%s%s%s/%s",
	   GetConfValue("OpengateServerName"),
	   GetConfValue("CgiDir"),
	   GetConfValue("OpengateDir"),
	   GetConfValue("AuthCgi"));

  /* create opengate dir */
  snprintf(opengateDir, BUFFMAXLN, "%s", GetConfValue("OpengateDir"));

  /* create encoded addr4 */
  if(AddrEncode(encodeAddr, clientAddr4)==1){
    encodeAddr[0]='\0';
  }

  /* protocol is ssl or non-ssl */
  if(strcmp(getenv("SERVER_PORT"),GetServicePortStr("https"))==0){
    strlcpy(protocol, "https",WORDMAXLN);
  }else{
    strlcpy(protocol, "http",WORDMAXLN);
  }

  /* document path to deny doc */
  snprintf(denydoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	    GetConfValue("OpengateDir"),language,GetConfValue("DenyDoc"));

  /* replace keyword and send out the file */
  printf("Content-type: text/html\r\n");

  /* if cookie auth is enabled, delete the cookie */
  if( (*GetConfValue("EnableCookieAuth")!='0') ){
    printf("Set-Cookie: %s=;expires=Thu, 01-Jan-1970 00:00:00 GMT;path=/;\r\n", COOKIENAME);
  }

  /* end of html header */
  printf("\r\n");

  /* send denydoc content */
  HtmlTemplate(denydoc, keys);

  return;
}
/******************************************************/
/* put retry message to the client                    */
/*  RetryDoc in conf file indicates the html template */
/******************************************************/
void putClientRetry(char *language)
{
  char retrydoc[BUFFMAXLN];
  char externalUrl[BUFFMAXLN];
  char authCgiUrl[BUFFMAXLN];
  char opengateDir[BUFFMAXLN];

  /* keyword pairs */
  /*  the left key in template file is replaced by the right value */
  struct html_key keys[]=
    {
      {"%%OPENGATEDIR%%", opengateDir},
      {"%%EXTERNALURL%%", externalUrl},
      {"%%AUTHCGIURL%%", authCgiUrl},
      {"%%LANGUAGE%%", language},
      {"",""}  /* DON'T REMOVE THIS LINE */
    };

  /* create opengate Dir */
  snprintf(opengateDir, BUFFMAXLN, "%s", GetConfValue("OpengateDir"));
 
  /* create external URL string */
  strlcpy(externalUrl, GetConfValue("ExternalUrl"), BUFFMAXLN);

  /* create authcgi URL string */
  snprintf(authCgiUrl, BUFFMAXLN, "%s%s%s/%s",
	   GetConfValue("OpengateServerName"),
	   GetConfValue("CgiDir"),
	   opengateDir,
	   GetConfValue("AuthCgi"));

  /* make read in path to the retry document */
  snprintf(retrydoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  opengateDir,language,GetConfValue("RetryDoc"));

  /* replace keyword and send out the file */
  printf("Content-type: text/html\r\n\r\n\r\n");
  HtmlTemplate(retrydoc, keys);

  return;
}

/*********************************************/
/* put some message to the client            */
/*********************************************/
void putClientMsg(char *message)
{
  printf("Content-type: text/html\r\n\r\n");
  printf("<HTML><HEAD><TITLE>OpengateMsg</TITLE></HEAD> \r\n");
  printf("<BODY>         \r\n");
  printf("%s\r\n",     message);
  printf("</BODY></HTML> \r\n\r\n");

}


/****************************************************************/
/* put accept messageto the client                              */
/*  AcceptDocHttp/Time in conf file indicate the html templates.*/
/*  AcceptDocTime page is shown when javascript is disabled.    */ 
/*  AcceptDocHttp page requests httpkeep page on specific port. */
/*  httpkeep page is sent in watch-client.c                     */
/****************************************************************/
void putClientAccept(char *userid, char *sessionId, int port, int pid, char *clientAddr4, char *clientAddr6, int ipStatus, int duration, int durationEntered, char *language, char *cookie, int cookieAuth, char *redirectedUrl)
{

  FILE *fp;
  char buff[BUFFMAXLN];
  char acceptDocPath[BUFFMAXLN];
  char acceptDoc2Url[BUFFMAXLN];
  char terminateUrl[BUFFMAXLN];
  char httpkeepUrl[BUFFMAXLN];
  char portStr[WORDMAXLN];
  char durationStr[WORDMAXLN];
  char *startPageUrl="";
  char *pAcceptDoc;
  char *opengateDir=GetConfValue("OpengateDir");
  char *opengateServerName=GetConfValue("OpengateServerName");
  int startPageType=atoi(GetConfValue("StartPage/Type"));
  char useridshort[USERMAXLN];
  char extraId[USERMAXLN];

  /* select proper accept doc */
  switch(toupper(*GetConfValue("WatchMode"))){

    /* HTTP watch mode */
  case 'H':
  default:
    if(isHttpWatchEnableClient()){
      pAcceptDoc=GetConfValue("AcceptDocHttp");
    }else{
      pAcceptDoc=GetConfValue("AcceptDocTime");
    }
    break;

    /* TIMEOUT watch mode */
  case 'T':
    pAcceptDoc=GetConfValue("AcceptDocTime");
    break;
  }

  /* if positive value is set in duration, TIME watch mode is selected */
  if(durationEntered) pAcceptDoc=GetConfValue("AcceptDocTime");

  /* split id to display short format of userid */
  SplitId(userid, useridshort, extraId);

  /* create path to acceptdoc */
  snprintf(acceptDocPath, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	 GetConfValue("OpengateDir"),language,pAcceptDoc);


  snprintf(acceptDoc2Url, BUFFMAXLN, 
	  "http://%s%s/%s/%s",GetConfValue("OpengateServerName"),
	  GetConfValue("OpengateDir"),language,GetConfValue("AcceptDoc2"));

  /* create terminate url [http://<servaddr>:<port>/terminate-<pid>] */
  snprintf(terminateUrl, BUFFMAXLN, "http://%s:%d/terminate-%d", 
	  GetConfValue("OpengateServerName"), port, getpid());

  /* create httpkeep page url
     ['http://<servaddr>:<port>/httpkeep-<userid>'] */
  snprintf(httpkeepUrl, BUFFMAXLN,
	   "'http://%s:%d/httpkeep-%s-%s'", 
	  GetConfValue("OpengateServerName"), port, userid,sessionId);

  /* create port string */
  snprintf(portStr, WORDMAXLN, "%d", port);

  /* create duration string (duration=sec, display value=min) */
  if(duration<60) snprintf(durationStr, WORDMAXLN, "%.2f", duration/60.);
  else snprintf(durationStr, WORDMAXLN, "%d", duration/60);

  /* open acceptdoc */
  if((fp=fopen(acceptDocPath, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot open %s",__FILE__,__LINE__,acceptDocPath);
    PutClientMsg("Cannot find html document");
    return;
  }

  /* if redirect page is not set, use other setting */
  if(isNull(redirectedUrl)){
    if(isNull(GetConfValue("StartPage/Url"))) redirectedUrl=acceptDoc2Url;
    else  redirectedUrl=GetConfValue("StartPage/Url");
  }

  /* create start page url to put information */
  if(cookieAuth==1) startPageUrl=redirectedUrl;
  else if(startPageType==0) startPageUrl=acceptDoc2Url;
  else if(startPageType==1) startPageUrl=GetConfValue("StartPage/Url");
  else if(startPageType==2) startPageUrl=redirectedUrl;
  else startPageUrl=acceptDoc2Url;

  /* write out html headers */
  printf("Content-type: text/html\r\n");
  printf("Set-Cookie: %s=%s&Userid=%s;path=/;\r\n\r\n", COOKIENAME, cookie, userid);
  
  /* read html document from file and send to web */
  while(fgets(buff, BUFFMAXLN, fp)!=NULL){
    
    /* length check */
    if(strlen(buff)>=BUFFMAXLN-1){
      err_msg("ERR at %s#%d: too long line in %s",__FILE__,__LINE__,acceptDocPath);
    }
    
    /* replace mark */
    htmlReplace(buff, "%%OPENGATESERVERNAME%%", opengateServerName);
    htmlReplace(buff, "%%OPENGATEDIR%%", opengateDir);
    htmlReplace(buff, "%%OPENGATEPORT%%", portStr);
    htmlReplace(buff, "%%DURATION%%", durationStr);
    htmlReplace(buff, "%%USERID%%", useridshort);
    htmlReplace(buff, "%%SESSIONID%%", sessionId);
    htmlReplace(buff, "%%LANGUAGE%%", language);
    htmlReplace(buff, "%%TERMINATEURL%%", terminateUrl);
    htmlReplace(buff, "%%HTTPKEEPURL%%", httpkeepUrl);
    htmlReplace(buff, "%%STARTURL%%", startPageUrl);
    htmlReplace(buff, "%%REDIRECTEDURL%%", redirectedUrl);
    
    /* write out */
    printf("%s",buff);
  }
  
  /* write end */
  fputs("\r\n\r\n",stdout);
  fclose(fp);

  return;
}

/*****************************************************/
/* is the client enable to keep long http connection */
/*****************************************************/
int isHttpWatchEnableClient(void)
{
  /* HTTP Keep-Alive is not standard in http/1.0 */
  if(strcmp(getenv("SERVER_PROTOCOL"),"HTTP/1.0")==0) return FALSE;

  /* some user agent does not support long HTTP Keep-Alive */
  /* last param 0 means case insensitive */
  if(RegExMatch(getenv("HTTP_USER_AGENT"),
		 GetConfValue("HttpWatch/SkipAgentPattern"),0)) return FALSE;

  return TRUE;
}

/************************************/
/* split value for indicated name   */
/* in content  "name=value&..."     */
/************************************/
void split(char content[], char *name[], char *value[], char *next[])
{
  char *pstr;
  
  /* set default */
  name[0]=content;
  value[0]=content+strlen(content);
  next[0]=value[0];

  /* set name end */
  if((pstr=strchr(name[0],(int)'='))==NULL){
    next[0]=NULL;
    return;
  }
  *pstr='\0';
  
  /* set value start */
  pstr++;
  value[0]=pstr;
  
  /* set value end */
  if((pstr=strchr(value[0],'&'))==NULL){
    next[0]=NULL;
    return;
  }
  *pstr='\0';

  /* set next start */
  pstr++;
  next[0]=pstr;

  return;
}

/**********************************/
/* decode text coding in web post */
/**********************************/
void decode(char *string)
{
  char *pcheck, *pinsert;

  pcheck=pinsert=string;
  while(*pcheck != '\0'){
    if(*pcheck == '+'){
      *pinsert = ' ';
    }else if(*pcheck == '%'){
      *pinsert=(char)(hex2num(*(pcheck+1))*16 + hex2num(*(pcheck+2)));
      pcheck+=2;
    }else{
      *pinsert=*pcheck;
    }
    pcheck++;
    pinsert++;
  }
  *pinsert=*pcheck;
}

/***************************************/
/* get HTTP_REFERER and check true url */
/***************************************/
int checkReferer(void)
{
  char url[BUFFMAXLN]="";
  if(getenv("HTTP_REFERER")!=NULL){
    strlcpy(url,getenv("HTTP_REFERER"),BUFFMAXLN);
    if(strstr(url,GetConfValue("OpengateServerName"))==NULL){
      return FALSE;
    }
  }
  return TRUE;
}

/****************************************************/
/* trim duration value between zero and max in conf */
/****************************************************/
int trimDuration(int duration, int durationEntered){

  /* if no entry, set default. if huge value, set upper-limit */
  if(!durationEntered){
    duration = atoi(GetConfValue("Duration/Default"));
  }else{
    int durmax=atoi(GetConfValue("Duration/Max"));
    if(duration > durmax) duration=durmax;
  }
  return duration;
}

/*******************************/
/*******************************/
void GetClientAddr(char *clientAddr)
{
  if(debug>1) err_msg("DEBUG:=>getClientAddr( )");
  getClientAddr(clientAddr);
  if(debug>1) err_msg("DEBUG:<=getClientAddr(%s)",clientAddr);
}


int GetPostData(char *userid, char *password, char *clientAddr4, int *durationPtr, int *durationEntered, char *language, char *redirectedUrl)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getPostData( )");
  ret=getPostData(userid,password,clientAddr4,durationPtr,durationEntered,language,redirectedUrl);
  if(debug>1) err_msg("DEBUG:%d<=getPostData(%s,%s,%d,%d,%s)",ret,userid,clientAddr4,*durationPtr,durationEntered,language,redirectedUrl);
  return ret;
}

int GetUserIdFromEnv(char *userid){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getUserIdFromEnv(%s)",userid);
  ret=getUserIdFromEnv(userid);
  if(debug>1) err_msg("DEBUG:%d<=getUserIdFromEnv(%s)",ret,userid);
  return ret;
}

void PutClientAccept(char *userid, char *sessionId, int port, int pid, char *clientAddr4, char *clientAddr6, int ipStatus, int duration, int durationEntered, char *language, char *cookie, int cookieAuth, char *redirectedUrl)
{
  if(debug>1) err_msg("DEBUG:=>putClientAccept(%s,%s,%d,%d,%s,%s,%d,%d,%d,%s,%s,%d,%s)",userid,sessionId,port,pid,clientAddr4,clientAddr6,ipStatus, duration, durationEntered, language, cookie, cookieAuth, redirectedUrl);
  putClientAccept(userid,sessionId,port,pid,clientAddr4,clientAddr6,ipStatus,duration, durationEntered, language, cookie, cookieAuth,redirectedUrl);
  if(debug>1) err_msg("DEBUG:<=putClientAccept( )");
}

void PutClientDeny(char *clientAddr4, char *language)
{
  if(debug>1) err_msg("DEBUG:=>putClientDeny(%s,%s)",clientAddr4,language);
  putClientDeny(clientAddr4,language);
  if(debug>1) err_msg("DEBUG:<=putClientDeny( )");
}

void PutClientRetry(char *lang)
{
  if(debug>1) err_msg("DEBUG:=>putClientRetry(%s)",lang);
  putClientRetry(lang);
  if(debug>1) err_msg("DEBUG:<=putClientRetry( )");
}

void PutClientMsg(char *message)
{
  if(debug>1) err_msg("DEBUG:=>putClientMsg( %s )",message);
  putClientMsg(message);
  if(debug>1) err_msg("DEBUG:<=putClientMsg( )");
}

int CheckReferer(void)
{
  int ret;
  if(debug>1) err_msg("DEBUG:=>checkReferer( )");
  ret = checkReferer();
  if(debug>1) err_msg("DEBUG:(%d)<=checkReferer( )",ret);
  return ret;
}

int GetAuthCookie(char *cookie, char *userid){
  int ret;

  if(debug>1) err_msg("DEBUG:=>getAuthCookie( )");
  ret=getAuthCookie(cookie, userid);
  if(debug>1) err_msg("DEBUG:%d<=getAuthCookie(%s,%s)",
		      ret,cookie,userid);
  return ret;
}

int GetCookieData(char *userid, char *clientAddr4, int *duration, int *durationEntered, char *language, char* closeTime){
  int ret;

  if(debug>1) err_msg("DEBUG:=>getCookieData( )");
  ret=getCookieData(userid,clientAddr4,duration,durationEntered,
		    language,closeTime);
  if(debug>1) err_msg("DEBUG:%d<=getCookieData(%s,passwd,%s,%d,%d,%s,%s)",ret,userid,clientAddr4,*duration,*durationEntered,language,closeTime);
  return ret;
}

int TrimDuration(int duration, int durationEntered){
  int ret;

  if(debug>1) err_msg("DEBUG:=>trimDuration(%d,%d)", 
		      duration,durationEntered);
  ret=trimDuration(duration,durationEntered);
  if(debug>1) err_msg("DEBUG:%d<=trimDuration( )",ret);
  return ret;
}
