/*
 * FM3 On-Chip Flash memory driver
 * Copyright (C) 2012 Yoshinori Sato <ysato@users.sourceforge.jp>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

/* The DEBUG define must be before common to enable debugging */
/* #define DEBUG	*/

#include <common.h>
#include <environment.h>
#include <flash.h>
#include <malloc.h>

extern int erase_sector_low(unsigned long sector_top);
extern int write_sector_low(unsigned long flash, uchar *src, int size);
static int flash_verbose = 0;

#define FLASH_FM3 0x464d3320

flash_info_t flash_info[] = {
	{
		.size = 0x4000,
		.sector_count = 2,
		.flash_id = FLASH_FM3,
		.start[0] = 0x00000000,
		.protect[0] = 0,
		.start[1] = 0x00004000,
		.protect[1] = 0,
	},
	{
		.size = 0x18000,
		.sector_count = 1,
		.flash_id = FLASH_FM3,
		.start[0] = 0x00008000,
		.protect[0] = 0,
	},
	{
		.size = 0x20000,
		.sector_count = 7,
		.flash_id = FLASH_FM3,
		.start[0] = 0x00020000,
		.protect[0] = 0,
		.start[1] = 0x00040000,
		.protect[1] = 0,
		.start[2] = 0x00060000,
		.protect[2] = 0,
		.start[3] = 0x00080000,
		.protect[3] = 0,
		.start[4] = 0x000a0000,
		.protect[4] = 0,
		.start[5] = 0x000c0000,
		.protect[5] = 0,
		.start[6] = 0x000e0000,
		.protect[6] = 0,
	},
};

/*-----------------------------------------------------------------------
 */
#if defined(CONFIG_ENV_IS_IN_FLASH) || defined(CONFIG_ENV_ADDR_REDUND) || (CONFIG_SYS_MONITOR_BASE >= CONFIG_SYS_FLASH_BASE)
flash_info_t *flash_get_info(ulong base)
{
	int i, j;
	flash_info_t *info = NULL;

	for (i = 0; i < CONFIG_SYS_MAX_FLASH_BANKS; i++) {
		info = & flash_info[i];
		for (j = 0; j < info->sector_count; j++) {
			if (info->size && info->start[j] <= base &&
			    base <= info->start[j] + info->size - 1)
				return info;
		}
	}

	return NULL;
}
#endif

unsigned long flash_sector_size(flash_info_t *info, flash_sect_t sect)
{
	return info->size;
}

/*-----------------------------------------------------------------------
 */
int flash_erase (flash_info_t * info, int s_first, int s_last)
{
	int ret;
	int i;
	for (i = s_first; i <= s_last; i++) {
		ret = erase_sector_low(info->start[i]);
		if (ret)
			return 2;
		if (flash_verbose)
			putc ('.');
	}
	if (flash_verbose)
		puts (" done\n");

	return 0;
}

void flash_print_info (flash_info_t * info)
{
	int i;

	printf("FM3 On-chip flash\n");
	for (i = 0; i < info->sector_count; i++)
		     printf("Sector %d: %ldKB (%08lx - %08lx)\n",
			    i, info->size / 1024,
			    info->start[i], info->start[i] + info->size -1);
}

/*-----------------------------------------------------------------------
 * Copy memory to flash, returns:
 * 0 - OK
 * 1 - write timeout
 * 2 - Flash not erased
 */
int write_buff (flash_info_t * info, uchar * src, ulong addr, ulong cnt)
{
	return write_sector_low(addr, src, cnt);
}

/*-----------------------------------------------------------------------
 */

unsigned long flash_init(void)
{
	return CONFIG_SYS_FLASH_SIZE;
}
