/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/Moenizer/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.moenizer;

import groovy.util.ConfigObject;
import groovy.util.ConfigSlurper;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import jp.sourceforge.users.yutang.omegat.plugin.moenizer.MoeUI.Parts;
import jp.sourceforge.users.yutang.omegat.plugin.moenizer.effect.Effect;
import jp.sourceforge.users.yutang.omegat.plugin.moenizer.effect.Effect.Type;
import org.omegat.util.FileUtil;
import org.omegat.util.Log;

/**
 * Manage config tree
 * 
 * @author Yu-Tang
 */
public class MoeConfig {

    static final String FILE_NAME = "moenizer.conf";

    private final String KEY_INDEX = "index";
    private final String KEY_ISACTIVE = "isActive";

    private final File file;
    private ConfigObject config = null;
    private final TreeSet<ConfigObject> sortedConfigs
            = new TreeSet<ConfigObject>
        (new Comparator<ConfigObject>() {
        @Override
        public int compare(ConfigObject conf1, ConfigObject conf2) {
            int index1 = (Integer) conf1.get(KEY_INDEX);
            int index2 = (Integer) conf2.get(KEY_INDEX);
            return index1 - index2;
        }
    });

    public MoeConfig(File file) throws IOException {
        this.file = file;
        if (file.isFile()) {
            String script = FileUtil.readTextFile(file); // as UTF-8
            config = createBindedConfigSlurper().parse(script);
            addAll(sortedConfigs, config);
        }
    }

    private ConfigSlurper createBindedConfigSlurper() {
        ConfigSlurper configSlurper = new ConfigSlurper();
        Map map = new HashMap<String, Parts>();

        // MoeUI.Parts enum
        for (Parts part: Parts.configullables()) {
            map.put(part.name(), part);
        }

        // Effect.Type enum
        for (Type type: Type.values()) {
            map.put(type.name(), type);
        }

        configSlurper.setBinding(map);
        return configSlurper;
    }
    
    public Object get(String key) {
        if (config != null) {
            Map flatten = config.flatten();
            if (flatten.containsKey(key)) {
                return flatten.get(key);
            }
        }
        return null;
    }

    public boolean get(String key, boolean defaultValue) {
        Object ret = get(key);
        return ret != null ? (Boolean) ret : defaultValue;
    }

    public Set<ConfigObject> getSortedConfigs() {
        return sortedConfigs;
    }

    public ConfigObject getCurrentConfig() {
        for (ConfigObject entry : sortedConfigs) {
            if (entry.containsKey(KEY_ISACTIVE)) {
                boolean isActive = (Boolean) entry.get(KEY_ISACTIVE);
                if (isActive) {
                    return entry;
                }                
            }
        }

        // No active config found. Fallback to default config.
        ConfigObject defaultConfig = getDefaultConfig();
        boolean isActive = (Boolean) defaultConfig.get(KEY_ISACTIVE);
        return isActive ? defaultConfig : null;
    }

    public void save() throws IOException {
        Writer writer = null;
        try {
            writer = new BufferedWriter(new FileWriter(file));
            config.writeTo(writer);
        } finally {
            try {
                if (writer != null) {
                    writer.close();
                }
            } catch(IOException ex) { /* ignore */ }
        }
    }

    private void addAll(Set<ConfigObject> sortedConfigs, ConfigObject config) {
        for (Object entry : config.entrySet()) {
            Map.Entry<String, ConfigObject> e = (Map.Entry<String, ConfigObject>) entry;
            //String name = e.getKey();
            ConfigObject conf = e.getValue();
            sortedConfigs.add(conf);
        }
    }

    public static ConfigObject getDefaultConfig() {
        // search image files from PrimaryMoeConfigDir
        File parentDir = MoeUtil.getPrimaryMoeConfigDir();
        String imagePath = getFirstImagePath(parentDir);

        // If image file not found, search image files from SecondaryMoeConfigDir
        if (imagePath.isEmpty()) {
            parentDir = MoeUtil.getSecondaryMoeConfigDir();
            imagePath = getFirstImagePath(parentDir);
        }

        // If image file still not found, search image files from PluginJarDir
        // for backward compatibility.
        if (imagePath.isEmpty()) {
            try {
                parentDir = MoeUtil.getPluginJarDir();
                imagePath = getFirstImagePath(parentDir);
            } catch (URISyntaxException ex) {
                Log.log(ex.getMessage());
            }
        }

        ConfigObject effect = new ConfigObject();
        effect.put("type", Effect.Type.Basic);
        Map image = new LinkedHashMap<String, Object>();
        image.put("path", imagePath);
        effect.put("image", image);
        
        List<ConfigObject> effects = new ArrayList<ConfigObject>(1);
        effects.add(effect);

        ConfigObject node = new ConfigObject();
        node.put("index", -1);
        node.put("isActive", !image.isEmpty());
        node.put("effects", effects);

        return node;
    }

    private static String getFirstImagePath(File parentDir) {
        try {
            List<String> imagePaths = MoeUtil.collectImagePaths(parentDir);
            if (!imagePaths.isEmpty()) {
                return imagePaths.get(0);
            }
        } catch (IOException ex) {
            Log.log(ex.getMessage());
        }
        return "";
    }

}
