/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/Moenizer/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.moenizer.effect;

import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.imageio.ImageIO;
import jp.sourceforge.users.yutang.omegat.plugin.moenizer.MoeUI;
import jp.sourceforge.users.yutang.omegat.plugin.moenizer.MoeUtil;
import org.omegat.util.Log;
import org.omegat.util.gui.UIThreadsUtil;

/**
 *
 * @author Yu-Tang
 */
public class SlideEffect extends Effect implements Runnable {

    private final String KEY_TARGET = "target";
    private final String KEY_OPACITY = "opacity";
    private final String KEY_IMAGE = "image";
    private final String KEY_PATHS = "paths";
    private final String KEY_BGCOLOR = "bgColor";
    private final String KEY_EXCLUDE = "exclude";
    private final String KEY_INTERVAL = "interval"; // in minutes

    private MoeUI.Parts target = MoeUI.Parts.MainWindow;
    private float opacity = 0.5f;
    private List<String> images = null;
    private Color bgColor = null;
    private EnumSet<MoeUI.Parts> exclude = null;
    private int intervalInMinutes = 1;
    private int currentImageIndex = 0;
    private MoeUI ui = null;

    public SlideEffect(Map<String, Object> config) {
        target = get(KEY_TARGET, config, target);
        if (config.containsKey(KEY_IMAGE)) {
            Map conf = (Map) config.get(KEY_IMAGE);
            images = getImages(get(KEY_PATHS, conf, images));
            opacity = get(KEY_OPACITY, conf, opacity);
        }
        Integer rgb = get(KEY_BGCOLOR, config, null);
        if (rgb != null) {
            bgColor = new Color(rgb);
        }
        List<MoeUI.Parts> excludeList = new ArrayList<MoeUI.Parts>();
        exclude = getExcludeParts(get(KEY_EXCLUDE, config, excludeList));
        intervalInMinutes = get(KEY_INTERVAL, config, intervalInMinutes);
    }

    @Override
    public void invoke(MoeUI ui, ScheduledThreadPoolExecutor service) {
        if (images.isEmpty()) {
            return;
        }

        try {
            ui.setBackground(target, getImage(images.get(0)));
        } catch (MalformedURLException ex) {
            Log.log(ex.getMessage());
        } catch (IOException ex) {
            Log.log(ex.getMessage());
        }

        ui.setOpacity(target, opacity);
        ui.setBackground(target, bgColor);
        ui.transparent(target, exclude);

        // schedule next execution
        if (images.size() > 1) {
            this.ui = ui;
            service.scheduleAtFixedRate(
                    this,
                    intervalInMinutes,
                    intervalInMinutes,
                    TimeUnit.MINUTES);
        }
    }

    @Override
    public void run() {
        // increment index
        currentImageIndex ++;
        if (currentImageIndex >= images.size()) {
            currentImageIndex = 0;
        }

        UIThreadsUtil.executeInSwingThread(new Runnable() {

            @Override
            public void run() {
                try {
                    ui.setBackground(target, getImage(images.get(currentImageIndex)));
                    ui.repaint();
                } catch (MalformedURLException ex) {
                    Log.log(ex.getMessage());
                } catch (IOException ex) {
                    Log.log(ex.getMessage());
                }
            }
        });
    }

    private List<String> getImages(List<String> paths) {
        List<String> ret = new ArrayList<String>();
        for (String path: paths) {
            String lcased = path.toLowerCase();
            if (lcased.startsWith("http://") || lcased.startsWith("https://")) {
                ret.add(path);
            } else {
                File file = new File(path);
                if (file.isFile()) {
                    ret.add(path);   
                } else if (file.isDirectory()) {
                    try {
                        List<String> imagePaths = MoeUtil.collectImagePaths(file);
                        ret.addAll(imagePaths);
                    } catch (IOException ex) {
                        // ignore
                    }
                }
            }
        }
        return ret;
    }

    private EnumSet<MoeUI.Parts> getExcludeParts(List<MoeUI.Parts> exclude) {
        EnumSet<MoeUI.Parts> excludeParts = EnumSet.noneOf(MoeUI.Parts.class);
        for (MoeUI.Parts part: EnumSet.allOf(MoeUI.Parts.class)) {
            if (exclude.contains(part)) {
                excludeParts.add(part);
            }
        }
        return excludeParts;
    }

    private BufferedImage getImage(String path) throws IOException {
        String lcased = path.toLowerCase();
        return lcased.startsWith("http://")|| lcased.startsWith("https://")
                ? ImageIO.read(new URL(path))
                : ImageIO.read(new File(path));
    }
}
