/**************************************************************************
 UIFontChanger - Allow to change UI font for OmegaT.
 
 Copyright (C) 2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/Moenizer/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.uifontchanger;

import groovy.lang.Closure;
import java.awt.Font;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import javax.swing.UIDefaults;
import javax.swing.UIDefaults.ActiveValue;
import javax.swing.UIManager;
import javax.swing.plaf.FontUIResource;
import org.omegat.core.Core;
import org.omegat.core.CoreEvents;
import org.omegat.core.events.IApplicationEventListener;
import org.omegat.util.Log;
import org.omegat.util.StaticUtils;

/**
 * Allow to change UI font for OmegaT
 * 
 * @author Yu-Tang
 */
public class UIFontChanger implements IApplicationEventListener {

    private static UIFontChanger instance = null;

    private final String VERSION_STRING = "0.1.0";
    private final Config config;
    //private MenuHandler menu;

    public UIFontChanger(Config config) throws IOException {
        this.config = config;
        Log.log("UIFontChanger: " + VERSION_STRING);
    }

    public static void loadPlugins() {
        try {
            if (instance == null) {
                instance = new UIFontChanger(new Config(Config.getConfigFile()));

                // フォントの変更は、GUI がインスタンス化されてからでは面倒に
                // なるので、即座にここで行う。
                instance.initUIFont();

                //CoreEvents.registerApplicationEventListener(instance);
            } else {
                throw new RuntimeException("UIFontChanger plugin could be instantiated only once.");
            }
        } catch (Throwable ex) {
            String msg = ex.getMessage();
            Log.logErrorRB("LD_ERROR", msg);
            Core.pluginLoadingError(msg);
        }
   }

    public static void unloadPlugins() {
        // do nothing
    }

    @Override
    public void onApplicationStartup() {
        // not used.
    }

    @Override
    public void onApplicationShutdown() {
        // not used.
    }

    private void initUIFont() {
        // includes font family names
        final List<String> availableFonts = Arrays.asList(StaticUtils.getFontNames());
        UIDefaults defUI = UIManager.getDefaults();
        Object universalFont = config.get("*");

        for (Entry<Object, Object> e : defUI.entrySet()) {
            String key = e.getKey().toString();
            if (key.toLowerCase().endsWith("font")) {
                Object val = e.getValue();

                // Windows 環境の JRE7 で確認する限りでは、value は以下の 3 種類
                // の型に分類できる
                //
                // 1. com.sun.java.swing.plaf.windows.WindowsLookAndFeel$WindowsFontProperty
                // 2. javax.swing.plaf.FontUIResource
                // 3. sun.swing.SwingLazyValue
                //
                // 1 と 3 は ActiveValue の派生。2 は Font の派生。
                // 1 と 3 の場合、Font を取得して、型を Font に統一する。
                
                Object oldVal = val;
                if (val instanceof ActiveValue) {
                    oldVal = ((ActiveValue) val).createValue(defUI);
                }
                if (oldVal instanceof Font) {
                    Object newVal = config.get(key);
                    // 設定ファイルに指定が無い場合、全称フォントにフォールバックしてみる
                    if (newVal == null) {
                        newVal = universalFont;
                    }
                    if (newVal != null) {
                        // newVal が Closure の場合は、実行結果 (String or Font) を取得する
                        if (newVal instanceof Closure) {
                            newVal = ((Closure) newVal).call((Font) oldVal, availableFonts);
                        }

                        // この時点で、newVal は String (font name) か Font のはず
                        boolean fontAvalable = newVal instanceof Font 
                                ? availableFonts.contains(((Font) newVal).getName()) 
                                : availableFonts.contains((String) newVal);

                        if (fontAvalable) {
                            if (val instanceof ActiveValue) {
                                defUI.put(key, getActiveFont((Font) oldVal, newVal));
                            } else if (val instanceof FontUIResource) {
                                defUI.put(key, getFontUIResource((Font) oldVal, newVal));
                            }
                        }                        
                    }
                }
            }
        }
    }

    private ActiveFont getActiveFont(Font fontOld, Object fontNew) {
        if (fontNew instanceof Font) {
            return ActiveFont.getInstance((Font) fontNew);
        } else if (fontNew instanceof String) {
            return ActiveFont.getInstance((String) fontNew, fontOld.getStyle(), fontOld.getSize());
        }
        throw new IllegalArgumentException("fontNew argument must be Font type or String.");
    }

    private FontUIResource getFontUIResource(Font fontOld, Object fontNew) {
        if (fontNew instanceof Font) {
            return new FontUIResource((Font) fontNew);
        } else if (fontNew instanceof String) {
            return new FontUIResource((String) fontNew, fontOld.getStyle(), fontOld.getSize());
        }
        throw new IllegalArgumentException("fontNew argument must be Font type or String.");
    }

    private static class ActiveFont implements ActiveValue {
        private static final Map<FontArgs, ActiveFont> cache = new HashMap<FontArgs, ActiveFont>(); 

        private final String fontName;
        private final int fontStyle;
        private final int fontSize;

        public static ActiveFont getInstance(Font font) {
            return getInstance(font.getFontName(), font.getStyle(), font.getSize());
        }

        public static ActiveFont getInstance(String fontName, int fontStyle, int fontSize) {
            // 同じフォント情報の場合は、ActiveFont のキャッシュを再利用する
            FontArgs key = new FontArgs(fontName, fontStyle, fontSize);
            if (cache.containsKey(key)) {
                return cache.get(key);
            } else {
                ActiveFont activeFont = new ActiveFont(fontName, fontStyle, fontSize);
                cache.put(key, activeFont);
                return activeFont;
            }
        }

        private ActiveFont(String fontName, int fontStyle, int fontSize) {
            this.fontName = fontName;
            this.fontStyle = fontStyle;
            this.fontSize = fontSize;
        }

        @Override
        public Object createValue(UIDefaults table) {
            return new FontUIResource(this.fontName, this.fontStyle, this.fontSize);
        }
        
        private static class FontArgs {
            private final String fontName;
            private final int fontStyle;
            private final int fontSize;

            public FontArgs(String fontName, int fontStyle, int fontSize) {
                this.fontName = fontName;
                this.fontStyle = fontStyle;
                this.fontSize = fontSize;
            }

            @Override
            public int hashCode() {
                int hash = 7;
                hash = 11 * hash + (this.fontName != null ? this.fontName.hashCode() : 0);
                hash = 11 * hash + this.fontStyle;
                hash = 11 * hash + this.fontSize;
                return hash;
            }

            @Override
            public boolean equals(Object obj) {
                if (obj == null) {
                    return false;
                }
                if (getClass() != obj.getClass()) {
                    return false;
                }
                final FontArgs other = (FontArgs) obj;
                if ((this.fontName == null) ? (other.fontName != null) : !this.fontName.equals(other.fontName)) {
                    return false;
                }
                if (this.fontStyle != other.fontStyle) {
                    return false;
                }
                if (this.fontSize != other.fontSize) {
                    return false;
                }
                return true;
            }
        }
    }

}
