/* Copyright (c) 2020-2022 The Creators of Simphone

   See the file COPYING.LESSER.txt for copying permission.
*/

#ifndef QTFIX_H
#define QTFIX_H

#include <QCoreApplication>
#include <QStatusBar>
#include <QTableView>
#include <QCheckBox>
#include <QRadioButton>
#include <QComboBox>
#include <QPushButton>
#include <QSplitter>
#include <QInputDialog>
#include <QMainWindow>
#include <QMessageBox>
#include <QProxyStyle>
#include <QAction>
#include <QElapsedTimer>

class fixMessageBox : public QMessageBox
{
  Q_OBJECT

  typedef QMessageBox Parent;

public:
  explicit fixMessageBox(Icon icon, const QString & title, const QString & text, StandardButtons buttons = NoButton,
                         QWidget * parent = 0, Qt::WindowFlags f = Qt::Dialog | Qt::MSWindowsFixedSizeDialogHint);

  QPushButton * addNocloseButton(const QString & text, ButtonRole role);

public slots:
  void done(int r) Q_DECL_OVERRIDE;

private:
  QList<QAbstractButton *> m_buttons;
};

class fixProxyStyle : public QProxyStyle
{
  int pixelMetric(PixelMetric metric, const QStyleOption * option, const QWidget * widget) const Q_DECL_OVERRIDE;
};

class qtfix
{
  Q_DECLARE_TR_FUNCTIONS(Main)

public:
  static bool hasAltModifier(QKeyEvent * keyEvent, Qt::KeyboardModifiers modifiers);
  static int hasSelectedRows(const QItemSelectionModel * model);

  static QString getStrings(const char * nick) { return QString(nick).replace('\t', ' '); }
  static QString getString(const char * nick) { return getStrings(nick).replace('\n', ' '); }
  static QString getString(QString text, const char * regexp) { return text.replace(QRegExp(regexp), " "); }

  static QString getVersion();
  static char ** getArguments() { return mc_arguments; }
  static const char * getStyleArgument() { return mc_styleArgument; }
  static QString getStyleSheetName() { return mc_styleSheetName; }

  static const char * getLanguage() { return mc_language; }
  static bool setLanguage(const QString & language);

  static qint64 getOverrideCursor() { return mc_elapsed.elapsed(); }
  static qint64 setOverrideCursor(bool start);

  static bool setGeometry(QWidget * window, QRect geometry, bool proper);

  static QStringList fixArguments(int & argc, char **& argv);
  static const char * fixApplication(const QString & language);

  static void fixStatusBar(QStatusBar * statusBar);
  static void fixTableView(QTableView * tableView);

  static QFont * createPasswordFont(int pointSize /* = 0*/, int * fontId);
  static void removePasswordFont(int fontId);

  static QFont fixFontSize(QFont oldFont, const char * newFontClass, bool monospace = false);
  static void fixPointSize(QWidget * widget, int pointSize, int maxPointSize = 0);
  static QPixmap fixPixmapSize(const QPixmap & pixmap, int fontH, bool nodownscale);

  static void fixCheckBoxIcon(QCheckBox * checkBox, const QFont & font);
  static void fixRadioButtonIcon(QRadioButton * radioButton, const QFont & font);
  static void fixComboBoxIcon(QComboBox * comboBox, const QFont & font);

  static void fixPushButtonIcon(QPushButton * pushButton, const QFont & font);
  static void fixMenuIndicator(QAction * item, QMenu * menu);

  static QAction * addMenuItem(QMenu * menu, const QString & text, bool checked);
  static QAction * addMenuItem(QMenu * menu, const QString & text, QActionGroup * group = 0);
  static QAction * addMenuItem(QMenu * menu, const QString & text, const QPixmap & icon);

  static void fixIconSize(QWidget * widget, const QFont & font);
  static void fixSplitterHandle(QSplitter * splitter, const QFont & font, char enable);

  static QObject * findChild(QObject * parent, const char * type);
  static int execInputDialog(QInputDialog * input);

  static bool execMessageBox(QMessageBox::Icon icon, const char * title, const QString & message,
                             QWidget * parent, const QString & checkBox); // no title on mac os
  static bool execMessageBox(QCheckBox * checkBox, const QString & title, const QString & message,
                             QWidget * parent, bool yes);
  static bool execMessageBox(bool critical, const QString & title, const QString & message,
                             QWidget * parent, const QString & checkBox = QString());
  static void updateMessageBox(QMessageBox * messageBox, const QString & text);

  static void resizeMainWindow(QMainWindow * window, int width, int height);
  static bool showActivateWindow(QWidget * window, bool spontaneous);
  static void showMaximizedWindow(QWidget * window);
  static void showMinimizedWindow(QWidget * window, int delay);
  static Qt::WindowStates hideMinimizedWindow(QWidget * window);

private:
  static char ** mc_arguments;
  static char * mc_styleArgument;
  static char mc_style;
  static bool mc_styleSheet;
  static QString mc_styleName;
  static QString mc_styleSheetName;
  static char mc_language[64];
  static QLocale mc_locale;
  static QElapsedTimer mc_elapsed;
};

#endif
