#version 330 core

layout(location = 0) in vec3 vertexPos;

out vec3 position;
out vec3 worldNormal;

uniform mat4 MVP;

uniform float waterlevel;
uniform float mytime;



const float pi = 3.14159;

const int numWaves=4;

// adjustments of 31aug15 to improve appearance using circsurf(100)
// rather than a bigger number, due to need to reduce graphical burden:

//const float amplitude[4] = float[4]( 0.0004, 0.0006, 0.0002, 0.0008 ); //orig
const float amplitude[4] = float[4]( 0.004, 0.006, 0.002, 0.008 );

//const float wavelength[4]= float[4]( 0.20, 0.24, 0.16, 0.12 ); //orig
const float wavelength[4]= float[4]( 0.80, 0.96, 0.64, 0.48 );

//const float speed[4]     = float[4]( 0.018, 0.026, 0.030, 0.034 ); //orig
const float speed[4]     = float[4]( 0.072, 0.105, 0.120, 0.136 );



const float dir[4] = float[4]( 0, 2*pi/3, pi/3, -pi/2 );
const float diry[4] = float[4]( sin(dir[0]), sin(dir[1]), sin(dir[2]), sin(dir[3]) );
const float dirx[4] = float[4]( cos(dir[0]), cos(dir[1]), cos(dir[2]), cos(dir[3]) );



float wave(int i, float x, float y) {
	float rr=sqrt(x*x+y*y);
	float r=rr+5; // was rr+1 (31aug15)
    float frequency = 2*pi/wavelength[i];
    float phase = speed[i] * frequency;
    float theta = dot( vec2( dirx[i], diry[i] ), vec2(x, y));
    return amplitude[i]/r * sin(theta*frequency+mytime*phase);

	 // W = A*f/r
	 // so quotient rule says
	 // W' = A*f'/r - A*f*r'/r^2
	 // where dr/dx = x/sqrt(x*x+y*y) = x/rr ~= x/r
	 // and similarly dr/dy = y/rr ~= y/r
	 // thus
	 // dW/dx ~= A/r * df/dx - A/r/r * f * x/r
	 // dW/dy ~= A/r * df/dy - A/r/r * f * y/r
}

float waveHeight(float x, float y) {
    float height = 0.0;
    for (int i = 0; i < numWaves; ++i)
        height += wave(i, x, y);
    return height;
}

float dWavedx(int i, float x, float y) {
	float rr=sqrt(x*x+y*y);
	float r=rr+5;
    float frequency = 2*pi/wavelength[i];
    float phase = speed[i] * frequency;
    float theta = dot( vec2( dirx[i], diry[i]), vec2(x, y));
    //float A = amplitude[i] * dirx[i] * frequency;
    float A = amplitude[i];

    // use x/r to approximate the singularity:  x/sqrt(x*x+y*y)
	 return
	 	A/r * dirx[i]*frequency * cos(theta*frequency+mytime*phase) -
		A/r/r * x/r             * sin(theta*frequency+mytime*phase);
}

float dWavedy(int i, float x, float y) {
	float rr=sqrt(x*x+y*y);
	float r=rr+5;
    float frequency = 2*pi/wavelength[i];
    float phase = speed[i] * frequency;
    float theta = dot( vec2( dirx[i], diry[i]), vec2(x, y));
    //float A = amplitude[i] * diry[i] * frequency;
    float A = amplitude[i];

	 // use y/r to approximate the singularity:  y/sqrt(x*x+y*y)
    return  
	 	A/r * diry[i]*frequency * cos(theta*frequency+mytime*phase) -
		A/r/r * y/r             * sin(theta*frequency+mytime*phase);
}


vec3 waveNormal(float x, float y) {
    float dx = 0.0;
    float dy = 0.0;
    for (int i = 0; i < numWaves; ++i) {
        dx += dWavedx(i, x, y);
        dy += dWavedy(i, x, y);
    }
    //vec3 n = vec3(-dx, -dy, 1.0); --Conrod:  Z is up
    vec3 n = vec3(-dx, 1.0, -dy); //my Y plays roll of Conrod's Z
    return normalize(n);
}


void main(){
	vec4 pos = vec4(vertexPos,1);

  	pos.y = waterlevel + waveHeight(pos.x, pos.z);
  	worldNormal = waveNormal(pos.x, pos.z);
		//worldNormal = vec3(0,1,0); //good

   position = pos.xyz/pos.w;
   gl_Position = MVP * pos;

}


//--
//-- Copyright (C) 2018  <fastrgv@gmail.com>
//--
//-- This program is free software: you can redistribute it and/or modify
//-- it under the terms of the GNU General Public License as published by
//-- the Free Software Foundation, either version 3 of the License, or
//-- (at your option) any later version.
//--
//-- This program is distributed in the hope that it will be useful,
//-- but WITHOUT ANY WARRANTY; without even the implied warranty of
//-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//-- GNU General Public License for more details.
//--
//-- You may read the full text of the GNU General Public License
//-- at <http://www.gnu.org/licenses/>.
//--

