package pcsweb

import (
	"context"
	"encoding/hex"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"os"
	"path/filepath"
	"regexp"
	"strconv"
	"strings"
	"sync/atomic"
	"time"

	"github.com/Erope/BaiduPCS-Go/baidupcs"
	"github.com/Erope/BaiduPCS-Go/baidupcs/pcserror"
	"github.com/Erope/BaiduPCS-Go/internal/pcscommand"
	"github.com/Erope/BaiduPCS-Go/internal/pcsconfig"
	"github.com/Erope/BaiduPCS-Go/internal/pcsfunctions/pcsdownload"
	"github.com/Erope/BaiduPCS-Go/pcstable"
	"github.com/Erope/BaiduPCS-Go/pcsutil/checksum"
	"github.com/Erope/BaiduPCS-Go/pcsutil/converter"
	"github.com/Erope/BaiduPCS-Go/pcsutil/waitgroup"
	"github.com/Erope/BaiduPCS-Go/requester"
	"github.com/Erope/BaiduPCS-Go/requester/downloader"
	"github.com/Erope/BaiduPCS-Go/requester/transfer"
	"github.com/oleiade/lane"
	"github.com/zyxar/argo/rpc"
	"golang.org/x/net/websocket"
)

const (
	//DownloadSuffix 文件下载后缀
	DownloadSuffix = ".BaiduPCS-Go-downloading"
	//StrDownloadInitError 初始化下载发生错误
	StrDownloadInitError = "初始化下载发生错误"
	// StrDownloadFailed 下载文件错误
	StrDownloadFailed = "下载文件错误"
	// DefaultDownloadMaxRetry 默认下载失败最大重试次数
	DefaultDownloadMaxRetry = 3
)

var (
	// ErrDownloadNotSupportChecksum 文件不支持校验
	ErrDownloadNotSupportChecksum = errors.New("该文件不支持校验")
	// ErrDownloadChecksumFailed 文件校验失败
	ErrDownloadChecksumFailed = errors.New("该文件校验失败, 文件md5值与服务器记录的不匹配")
	// ErrDownloadFileBanned 违规文件
	ErrDownloadFileBanned = errors.New("该文件可能是违规文件, 不支持校验")
	// ErrDlinkNotFound 未取得下载链接
	ErrDlinkNotFound = errors.New("未取得下载链接")
	MsgBody          string
	DownloaderMap    = make(map[int]*downloader.Downloader)
)

// ListTask 队列状态 (基类)
type ListTask struct {
	ID       int // 任务id
	MaxRetry int // 最大重试次数
	retry    int // 任务失败的重试次数
}

type (
	// dtask 下载任务
	dtask struct {
		ListTask
		path         string                  // 下载的路径
		savePath     string                  // 保存的路径
		downloadInfo *baidupcs.FileDirectory // 文件或目录详情
	}

	//DownloadOptions 下载可选参数
	DownloadOptions struct {
		IsTest                 bool
		IsPrintStatus          bool
		IsExecutedPermission   bool
		IsOverwrite            bool
		IsShareDownload        bool
		IsLocateDownload       bool
		IsLocatePanAPIDownload bool
		IsStreaming            bool
		SaveTo                 string
		Parallel               int
		Load                   int
		MaxRetry               int
		NoCheck                bool
		Out                    io.Writer
	}

	// LocateDownloadOption 获取下载链接可选参数
	LocateDownloadOption struct {
		FromPan bool
	}
)

func downloadPrintFormat(load int) string {
	if load <= 1 {
		return "\r[%d] ↓ %s/%s %s/s in %s, left %s ............"
	}
	return "[%d] ↓ %s/%s %s/s in %s, left %s ...\n"
}

func download(conn *websocket.Conn, id int, fileInfo *baidupcs.FileDirectory, downloadURL, savePath string, loadBalansers []string, client *requester.HTTPClient, newCfg downloader.Config, downloadOptions *DownloadOptions) error {
	var (
		writer downloader.Writer
		file   *os.File
		err    error
	)

	if !newCfg.IsTest {
		newCfg.InstanceStatePath = savePath + DownloadSuffix

		// 创建下载的目录
		dir := filepath.Dir(savePath)
		fileInfo, err := os.Stat(dir)
		if err != nil {
			err = os.MkdirAll(dir, 0777)
			if err != nil {
				return err
			}
		} else if !fileInfo.IsDir() {
			return fmt.Errorf("%s, path %s: not a directory", StrDownloadInitError, dir)
		}

		// 打开文件
		writer, file, err = downloader.NewDownloaderWriterByFilename(savePath, os.O_CREATE|os.O_WRONLY, 0666)
		if err != nil {
			sendResponse(conn, 2, -4, "初始化下载发生错误", "", true, true)
		}
		defer file.Close()
	}

	download := downloader.NewDownloader(downloadURL, writer, &newCfg)
	download.SetClient(client)
	download.SetDURLCheckFunc(pcsdownload.BaiduPCSURLCheckFunc)
	download.AddLoadBalanceServer(loadBalansers...)
	download.SetStatusCodeBodyCheckFunc(func(respBody io.Reader) error {
		return pcserror.DecodePCSJSONError(baidupcs.OperationDownloadFile, respBody)
	})

	var (
		format = downloadPrintFormat(downloadOptions.Load)
	)
	download.OnDownloadStatusEvent(func(status transfer.DownloadStatuser, workersCallback func(downloader.RangeWorkerFunc)) {
		DownloaderMap[id] = download
		if downloadOptions.IsPrintStatus {
			// 输出所有的worker状态
			var (
				builder = &strings.Builder{}
				tb      = pcstable.NewTable(builder)
			)
			tb.SetHeader([]string{"#", "status", "range", "left", "speeds", "error"})
			workersCallback(func(key int, worker *downloader.Worker) bool {
				wrange := worker.GetRange()
				tb.Append([]string{fmt.Sprint(worker.ID()), worker.GetStatus().StatusText(), wrange.ShowDetails(), strconv.FormatInt(wrange.Len(), 10), strconv.FormatInt(worker.GetSpeedsPerSecond(), 10), fmt.Sprint(worker.Err())})
				return true
			})
			tb.Render()
			fmt.Fprintf(downloadOptions.Out, "\n\n"+builder.String())
		}
		var leftStr string
		downloaded, totalSize, speeds := status.Downloaded(), status.TotalSize(), status.SpeedsPerSecond()
		if speeds <= 0 {
			leftStr = "-"
		} else {
			leftStr = (time.Duration((totalSize-downloaded)/(speeds)) * time.Second).String()
		}

		var avgSpeed int64 = 0
		timeUsed := status.TimeElapsed() / 1e7 * 1e7
		timeSecond := status.TimeElapsed().Seconds()
		if int64(timeSecond) > 0 {
			avgSpeed = downloaded / int64(timeSecond)
		}
		//fmt.Println(timeUsed, timeSecond, avgSpeed)

		fmt.Fprintf(downloadOptions.Out, format, id,
			converter.ConvertFileSize(downloaded, 2),
			converter.ConvertFileSize(totalSize, 2),
			converter.ConvertFileSize(speeds, 2),
			timeUsed, leftStr,
		)

		MsgBody = fmt.Sprintf("{\"LastID\": %d, \"download_size\": \"%s\", \"total_size\": \"%s\", \"percent\": %.2f, \"speed\": \"%s\", \"avg_speed\": \"%s\", \"time_used\": \"%s\", \"time_left\": \"%s\"}", id,
			converter.ConvertFileSize(downloaded, 2),
			converter.ConvertFileSize(totalSize, 2),
			float64(downloaded)/float64(totalSize)*100,
			converter.ConvertFileSize(speeds, 2),
			converter.ConvertFileSize(avgSpeed, 2),
			timeUsed, leftStr)
		sendResponse(conn, 2, 5, "下载中", MsgBody, false, true)
	})
	download.OnPause(func() {
		MsgBody = fmt.Sprintf("{\"LastID\": %d}", id)
		sendResponse(conn, 2, 6, "任务暂停", MsgBody, true, true)
		fmt.Println("\n任务暂停, ID:", id)
	})
	download.OnResume(func() {
		MsgBody = fmt.Sprintf("{\"LastID\": %d}", id)
		sendResponse(conn, 2, 7, "任务恢复", MsgBody, true, true)
		fmt.Println("\n任务恢复, ID:", id)
	})
	download.OnCancel(func() {
		MsgBody = fmt.Sprintf("{\"LastID\": %d}", id)
		sendResponse(conn, 2, 8, "任务取消", MsgBody, true, true)
		fmt.Println("\n任务取消, ID:", id)
	})
	download.OnFinish(func() {
		DownloaderMap[id] = nil
		fmt.Println("\n任务完成, ID:", id)
	})

	err = download.Execute()
	fmt.Fprintf(downloadOptions.Out, "\n")
	if err != nil {
		// 下载失败, 删去空文件
		if info, infoErr := file.Stat(); infoErr == nil {
			if info.Size() == 0 {
				pcsCommandVerbose.Infof("[%d] remove empty file: %s\n", id, savePath)
				removeErr := os.Remove(savePath)
				if removeErr != nil {
					pcsCommandVerbose.Infof("[%d] remove file error: %s\n", id, removeErr)
				}
			}
		}

		return err
	}

	if downloadOptions.IsExecutedPermission {
		err = file.Chmod(0766)
		if err != nil {
			sendResponse(conn, 2, -5, "警告, 加执行权限错误", "", true, true)
			fmt.Fprintf(downloadOptions.Out, "[%d] 警告, 加执行权限错误: %s\n", id, err)
		}
	}

	if !newCfg.IsTest {
		totalSize := fileSize(savePath)
		MsgBody = fmt.Sprintf("{\"LastID\": %d, \"download_size\": \"%s\", \"total_size\": \"%s\", \"percent\": %.2f, \"speed\": \"%s\", \"avg_speed\": \"%s\", \"time_used\": \"%s\", \"time_left\": \"%s\"}", id,
			converter.ConvertFileSize(totalSize, 2),
			converter.ConvertFileSize(totalSize, 2),
			float64(totalSize)/float64(totalSize)*100,
			converter.ConvertFileSize(0, 2),
			converter.ConvertFileSize(0, 2),
			"0", "0")
		sendResponse(conn, 2, 5, "下载中", MsgBody, true, true)
		MsgBody = fmt.Sprintf("{\"LastID\": %d, \"savePath\": \"%s\"}", id, savePath)
		sendResponse(conn, 2, 9, "下载完成", MsgBody, true, true)
		fmt.Fprintf(downloadOptions.Out, "[%d] 下载完成, 保存位置: %s\n", id, savePath)
	} else {
		fmt.Fprintf(downloadOptions.Out, "[%d] 测试下载结束\n", id)
	}

	return nil
}

// checkFileValid 检测文件有效性
func checkFileValid(filePath string, fileInfo *baidupcs.FileDirectory) error {
	if len(fileInfo.BlockList) != 1 {
		return ErrDownloadNotSupportChecksum
	}

	f := checksum.NewLocalFileChecksum(filePath, int(baidupcs.SliceMD5Size))
	err := f.OpenPath()
	if err != nil {
		return err
	}

	defer f.Close()

	err = f.Sum(checksum.CHECKSUM_MD5)
	if err != nil {
		return err
	}
	md5Str := hex.EncodeToString(f.MD5)

	if md5Str != fileInfo.MD5 { // md5不一致
		// 检测是否为违规文件
		if pcsdownload.IsSkipMd5Checksum(f.Length, md5Str) {
			return ErrDownloadFileBanned
		}
		return ErrDownloadChecksumFailed
	}
	return nil
}

// RunDownload 执行下载网盘内文件
func RunDownload(conn *websocket.Conn, paths []string, options *DownloadOptions) {
	if Aria2 {
		//发送下载链接到Aria2
		opts := make(map[string]interface{})
		opts["user-agent"] = pcsconfig.Config.PanUA
		opts["stream-piece-selector"] = "inorder"
		if pcsconfig.Config.MaxParallel > 16 {
			opts["max-connection-per-server"] = 16
		} else {
			opts["max-connection-per-server"] = pcsconfig.Config.MaxParallel
		}
		rpcc, err := rpc.New(context.Background(), Aria2_Url, Aria2_Secret, time.Second, nil)
		if err != nil {
			fmt.Fprintln(os.Stderr, err)
			return
		}
		for k := range paths {
			rawDlinks, err := getLocateDownloadLinks(paths[k])
			if len(PD_Url) >= 4 {
				opts["user-agent"] = "LogStatistic"
			} else {
				opts["user-agent"] = pcsconfig.Config.PanUA
			}
			if err == nil {
				handleHTTPLinkURL(rawDlinks[0])
				d_url := []string{Aria2_prefix + rawDlinks[0].String()}
				gid, err := rpcc.AddURI(d_url, opts)
				if err == nil {
					fmt.Printf("成功将 %s 送入Aria2下载列表，并发数: %d，gid: %s\n", paths[k], opts["max-connection-per-server"], gid)
				} else {
					fmt.Printf("添加任务到aria2时出错: %s 请检查aria2配置是否正确\n", err)
				}
			} else {
				fmt.Printf("出错: %s\n", err)
			}
		}
		rpcc.Close()
		return
	}
	if options == nil {
		options = &DownloadOptions{}
	}

	if options.Out == nil {
		options.Out = os.Stdout
	}

	if options.Load <= 0 {
		options.Load = pcsconfig.Config.MaxDownloadLoad
	}

	if options.MaxRetry < 0 {
		options.MaxRetry = DefaultDownloadMaxRetry
	}

	// 设置下载配置
	cfg := &downloader.Config{
		Mode:                       transfer.RangeGenMode_BlockSize,
		CacheSize:                  pcsconfig.Config.CacheSize,
		BlockSize:                  baidupcs.MaxDownloadRangeSize,
		MaxRate:                    pcsconfig.Config.MaxDownloadRate,
		InstanceStateStorageFormat: downloader.InstanceStateStorageFormatProto3,
		IsTest:                     options.IsTest,
		TryHTTP:                    !pcsconfig.Config.EnableHTTPS,
	}

	// 设置下载最大并发量
	if options.Parallel < 1 {
		options.Parallel = pcsconfig.Config.MaxParallel
	}

	paths, err := matchPathByShellPattern(paths...)
	if err != nil {
		fmt.Println(err)
		return
	}

	fmt.Fprintf(options.Out, "\n")
	fmt.Fprintf(options.Out, "[0] 提示: 当前下载最大并发量为: %d, 下载缓存为: %d\n", options.Parallel, cfg.CacheSize)

	var (
		pcs       = pcscommand.GetBaiduPCS()
		dlist     = lane.NewDeque()
		lastID    = 0
		loadCount = 0
	)

	// 预测要下载的文件数量
	// TODO: pcscache
	for k := range paths {
		pcs.FilesDirectoriesRecurseList(paths[k], baidupcs.DefaultOrderOptions, func(depth int, _ string, fd *baidupcs.FileDirectory, pcsError pcserror.Error) bool {
			if pcsError != nil {
				pcsCommandVerbose.Warnf("%s\n", pcsError)
				return true
			}

			if !fd.Isdir {
				loadCount++
				if loadCount >= options.Load {
					return false
				}
			}
			return true
		})

		if loadCount >= options.Load {
			break
		}
	}
	// 修改Load, 设置MaxParallel
	if loadCount > 0 {
		options.Load = loadCount
		// 取平均值
		cfg.MaxParallel = pcsconfig.AverageParallel(options.Parallel, loadCount)
	} else {
		cfg.MaxParallel = options.Parallel
	}

	// 处理队列
	for k := range paths {
		lastID++
		ptask := &dtask{
			ListTask: ListTask{
				ID:       lastID,
				MaxRetry: options.MaxRetry,
			},
			path: paths[k],
		}
		if options.SaveTo != "" {
			ptask.savePath = filepath.Join(options.SaveTo, filepath.Base(paths[k]))
		} else {
			ptask.savePath = pcscommand.GetActiveUser().GetSavePath(paths[k])
		}
		dlist.Append(ptask)
		fmt.Fprintf(options.Out, "[%d] 加入下载队列: %s\n", lastID, paths[k])
		MsgBody = fmt.Sprintf("{\"LastID\": %d, \"path\": \"%s\"}", lastID, paths[k])

		sendResponse(conn, 2, 1, "添加进任务队列", MsgBody, true, true)
	}

	var (
		totalSize     int64
		failedList    []string
		handleTaskErr = func(task *dtask, errManifest string, err error) {
			if task == nil {
				panic("task is nil")
			}

			if err == nil {
				return
			}

			// 不重试的情况
			switch {
			case err == ErrDownloadNotSupportChecksum:
				fallthrough
			case errManifest == StrDownloadFailed && strings.Contains(err.Error(), StrDownloadInitError):
				fmt.Fprintf(options.Out, "[%d] %s, %s\n", task.ID, errManifest, err)
				MsgBody = fmt.Sprintf("{\"LastID\": %d, \"errManifest\": \"%s\", \"error\": \"%s\"}", task.ID, errManifest, err)
				err = sendResponse(conn, 2, -1, "下载文件错误", MsgBody, true, true)
				return
			}

			// 未达到失败重试最大次数, 将任务推送到队列末尾
			if task.retry < task.MaxRetry {
				task.retry++
				MsgBody = fmt.Sprintf("{\"LastID\": %d, \"errManifest\": \"%s\", \"error\": \"%s\", \"retry\": %d, \"max_retry\": %d}", task.ID, errManifest, err, task.retry, task.MaxRetry)
				sendResponse(conn, 2, -2, "重试", MsgBody, true, true)
				fmt.Fprintf(options.Out, "[%d] %s, %s, 重试 %d/%d\n", task.ID, errManifest, err, task.retry, task.MaxRetry)
				dlist.Append(task)
				time.Sleep(3 * time.Duration(task.retry) * time.Second)
			} else {
				fmt.Fprintf(options.Out, "[%d] %s, %s\n", task.ID, errManifest, err)
				failedList = append(failedList, task.path)
			}

			switch err {
			case ErrDownloadChecksumFailed:
				// 删去旧的文件, 重新下载
				rerr := os.Remove(task.savePath)
				if rerr != nil {
					fmt.Fprintf(options.Out, "[%d] 删除校验失败的文件出错, %s\n", task.ID, rerr)
					failedList = append(failedList, task.path)
					return
				}

				fmt.Fprintf(options.Out, "[%d] 已删除校验失败的文件\n", task.ID)
			}
		}
		startTime = time.Now()
	)

	for {
		// Wait之后不能再add了，重建一个wg
		wg := waitgroup.NewWaitGroup(options.Load)
		for {
			e := dlist.Shift()
			if e == nil { // 任务为空
				break
			}

			task := e.(*dtask)
			wg.AddDelta()
			go func() {
				defer wg.Done()

				if task.downloadInfo == nil {
					task.downloadInfo, err = pcs.FilesDirectoriesMeta(task.path)
					if err != nil {
						// 不重试
						fmt.Printf("[%d] 获取路径信息错误, %s\n", task.ID, err)
						MsgBody = fmt.Sprintf("{\"LastID\": %d, \"error\": \"%s\"}", task.ID, err)
						sendResponse(conn, 2, -3, "获取路径信息错误", MsgBody, true, true)
						return
					}
				}

				fmt.Fprintf(options.Out, "\n")
				fmt.Fprintf(options.Out, "[%d] ----\n%s\n", task.ID, task.downloadInfo.String())

				// 如果是一个目录, 将子文件和子目录加入队列
				if task.downloadInfo.Isdir {
					if !options.IsTest { // 测试下载, 不建立空目录
						os.MkdirAll(task.savePath, 0777) // 首先在本地创建目录, 保证空目录也能被保存
					}

					fileList, err := pcs.FilesDirectoriesList(task.path, baidupcs.DefaultOrderOptions)
					if err != nil {
						// 不重试
						MsgBody = fmt.Sprintf("{\"LastID\": %d, \"error\": \"%s\"}", task.ID, err)
						sendResponse(conn, 2, -3, "获取目录信息错误", MsgBody, true, true)
						fmt.Fprintf(options.Out, "[%d] 获取目录信息错误, %s\n", task.ID, err)
						return
					}

					MsgBody = fmt.Sprintf("{\"LastID\": %d}", task.ID)
					sendResponse(conn, 2, 8, "删除文件夹任务", MsgBody, true, true)

					for k := range fileList {
						lastID++
						subTask := &dtask{
							ListTask: ListTask{
								ID:       lastID,
								MaxRetry: options.MaxRetry,
							},
							path:         fileList[k].Path,
							downloadInfo: fileList[k],
						}

						if options.SaveTo != "" {
							subTask.savePath = filepath.Join(task.savePath, fileList[k].Filename)
						} else {
							subTask.savePath = pcscommand.GetActiveUser().GetSavePath(subTask.path)
						}

						dlist.Append(subTask)
						fmt.Fprintf(options.Out, "[%d] 加入下载队列: %s\n", lastID, fileList[k].Path)
						MsgBody = fmt.Sprintf("{\"LastID\": %d, \"path\": \"%s\"}", lastID, fileList[k].Path)
						sendResponse(conn, 2, 1, "添加进任务队列", MsgBody, true, true)
					}
					return
				}

				fmt.Fprintf(options.Out, "[%d] 准备下载: %s\n", task.ID, task.path)
				MsgBody = fmt.Sprintf("{\"LastID\": %d, \"path\": \"%s\"}", task.ID, task.path)
				sendResponse(conn, 2, 3, "准备下载", MsgBody, true, true)

				if !options.IsTest && !options.IsOverwrite && fileExist(task.savePath) {
					fmt.Fprintf(options.Out, "[%d] 文件已经存在: %s, 跳过...\n", task.ID, task.savePath)
					MsgBody = fmt.Sprintf("{\"LastID\": %d, \"savePath\": \"%s\"}", task.ID, task.savePath)
					sendResponse(conn, 2, -4, "文件已经存在, 跳过...", MsgBody, true, true)
					return
				}

				if !options.IsTest {
					fmt.Fprintf(options.Out, "[%d] 将会下载到路径: %s\n\n", task.ID, task.savePath)
					MsgBody = fmt.Sprintf("{\"LastID\": %d, \"savePath\": \"%s\"}", task.ID, task.savePath)
					sendResponse(conn, 2, 4, "将会下载到路径", MsgBody, true, true)
				}

				// 获取直链, 或者以分享文件的方式获取下载链接来下载
				var (
					dlink  string
					dlinks []string
				)

				switch {
				case options.IsLocateDownload:
					// 获取直链下载
					var rawDlinks []*url.URL
					rawDlinks, err = getLocateDownloadLinks(task.path)
					if err == nil {
						handleHTTPLinkURL(rawDlinks[0])
						dlink = rawDlinks[0].String()
						dlinks = make([]string, 0, len(rawDlinks)-1)
						for _, rawDlink := range rawDlinks[1:len(rawDlinks)] {
							handleHTTPLinkURL(rawDlink)
							dlinks = append(dlinks, rawDlink.String())
						}
					}
				case options.IsShareDownload: // 分享下载
					dlink, err = pcscommand.GetShareDLink(task.path)
					switch err {
					case nil, pcscommand.ErrShareInfoNotFound: // 未分享, 采用默认下载方式
					default:
						handleTaskErr(task, StrDownloadFailed, err)
						return
					}
				case options.IsLocatePanAPIDownload: // 由第三方服务器处理
					dlink, err = getLocatePanLink(pcs, task.downloadInfo.FsID)
					if err != nil {
						handleTaskErr(task, StrDownloadFailed, err)
						return
					}
				}

				if (options.IsShareDownload || options.IsLocateDownload || options.IsLocatePanAPIDownload) && err == nil {
					pcsCommandVerbose.Infof("[%d] 获取到下载链接: %s\n", task.ID, dlink)
					client := pcsconfig.Config.PanHTTPClient()
					client.CheckRedirect = func(req *http.Request, via []*http.Request) error {
						// 去掉 Referer
						if !pcsconfig.Config.EnableHTTPS {
							req.Header.Del("Referer")
						}
						if len(via) >= 10 {
							return errors.New("stopped after 10 redirects")
						}
						return nil
					}
					client.SetTimeout(20 * time.Minute)
					client.SetKeepAlive(true)
					err = download(conn, task.ID, task.downloadInfo, dlink, task.savePath, dlinks, client, *cfg, options)
				} else {
					if options.IsShareDownload || options.IsLocateDownload || options.IsLocatePanAPIDownload {
						fmt.Fprintf(options.Out, "[%d] 错误: %s, 将使用默认的下载方式\n", task.ID, err)
					}

					dfunc := func(downloadURL string, jar http.CookieJar) error {
						h := pcsconfig.Config.PCSHTTPClient()
						h.SetCookiejar(jar)
						h.SetKeepAlive(true)
						h.SetTimeout(10 * time.Minute)

						err := download(conn, task.ID, task.downloadInfo, downloadURL, task.savePath, dlinks, h, *cfg, options)
						if err != nil {
							return err
						}

						return nil
					}
					if options.IsStreaming {
						err = pcs.DownloadStreamFile(task.path, dfunc)
					} else {
						err = pcs.DownloadFile(task.path, dfunc)
					}
				}
				if err != nil {
					handleTaskErr(task, StrDownloadFailed, err)
					return
				}

				// 检验文件有效性
				if !cfg.IsTest && !options.NoCheck {
					if task.downloadInfo.Size >= 128*converter.MB {
						fmt.Fprintf(options.Out, "[%d] 开始检验文件有效性, 请稍候...\n", task.ID)
					}
					err = checkFileValid(task.savePath, task.downloadInfo)
					if err != nil {
						switch err {
						case ErrDownloadFileBanned:
							fmt.Fprintf(options.Out, "[%d] 检验文件有效性: %s\n", task.ID, err)
							return
						default:
							handleTaskErr(task, "检验文件有效性出错", err)
							return
						}
					}

					fmt.Fprintf(options.Out, "[%d] 检验文件有效性成功\n", task.ID)
				}

				atomic.AddInt64(&totalSize, task.downloadInfo.Size)
			}()
		}
		wg.Wait()

		// 没有任务了
		if dlist.Size() == 0 {
			break
		}
	}

	fmt.Fprintf(options.Out, "\n任务结束, 时间: %s, 数据总量: %s\n", time.Since(startTime)/1e6*1e6, converter.ConvertFileSize(totalSize))
	if len(failedList) != 0 {
		fmt.Printf("以下文件下载失败: \n")
		tb := pcstable.NewTable(os.Stdout)
		for k := range failedList {
			tb.Append([]string{strconv.Itoa(k), failedList[k]})
		}
		tb.Render()
	}
}

// RunLocateDownload 执行获取直链
func RunLocateDownload(pcspaths []string, opt *LocateDownloadOption) {
	if opt == nil {
		opt = &LocateDownloadOption{}
	}

	absPaths, err := matchPathByShellPattern(pcspaths...)
	if err != nil {
		fmt.Println(err)
		return
	}

	pcs := pcscommand.GetBaiduPCS()

	if opt.FromPan {
		fds, err := pcs.FilesDirectoriesBatchMeta(absPaths...)
		if err != nil {
			fmt.Printf("%s\n", err)
			return
		}

		fidList := make([]int64, 0, len(fds))
		for i := range fds {
			fidList = append(fidList, fds[i].FsID)
		}

		list, err := pcs.LocatePanAPIDownload(fidList...)
		if err != nil {
			fmt.Printf("%s\n", err)
			return
		}

		tb := pcstable.NewTable(os.Stdout)
		tb.SetHeader([]string{"#", "fs_id", "路径", "链接"})

		var (
			i          int
			fidStrList = converter.SliceInt64ToString(fidList)
		)
		for k := range fidStrList {
			for i = range list {
				if fidStrList[k] == list[i].FsID {
					tb.Append([]string{strconv.Itoa(k), list[i].FsID, fds[k].Path, list[i].Dlink})
					list = append(list[:i], list[i+1:]...)
					break
				}
			}
		}
		tb.Render()
		fmt.Printf("\n注意: 以上链接不能直接访问, 需要登录百度帐号才可以下载\n")
		return
	}

	for i, pcspath := range absPaths {
		info, err := pcs.LocateDownload(pcspath)
		if err != nil {
			fmt.Printf("[%d] %s, 路径: %s\n", i, err, pcspath)
			continue
		}

		fmt.Printf("[%d] %s: \n", i, pcspath)
		tb := pcstable.NewTable(os.Stdout)
		tb.SetHeader([]string{"#", "链接"})
		for k, u := range info.URLStrings(pcsconfig.Config.EnableHTTPS) {
			tb.Append([]string{strconv.Itoa(k), u.String()})
		}
		tb.Render()
		fmt.Println()
	}
	fmt.Printf("提示: 访问下载链接, 需将下载器的 User-Agent 设置为: %s\n", pcsconfig.Config.PanUA)
}

// RunFixMD5 执行修复md5
func RunFixMD5(pcspaths ...string) {
	absPaths, err := matchPathByShellPattern(pcspaths...)
	if err != nil {
		fmt.Println(err)
		return
	}

	pcs := pcscommand.GetBaiduPCS()
	finfoList, err := pcs.FilesDirectoriesBatchMeta(absPaths...)
	if err != nil {
		fmt.Println(err)
		return
	}

	for k, finfo := range finfoList {
		err := pcs.FixMD5ByFileInfo(finfo)
		if err == nil {
			fmt.Printf("[%d] - [%s] 修复md5成功\n", k, finfo.Path)
			continue
		}

		if err.GetError() == baidupcs.ErrFixMD5Failed {
			fmt.Printf("[%d] - [%s] 修复md5失败, 可能是服务器未刷新\n", k, finfo.Path)
			continue
		}
		fmt.Printf("[%d] - [%s] 修复md5失败, 错误信息: %s\n", k, finfo.Path, err)
	}
}

func getLocateDownloadLinks(pcspath string) (dlinks []*url.URL, err error) {
	if len(PD_Url) < 4 {
		pcs := pcscommand.GetBaiduPCS()
		dInfo, pcsError := pcs.LocateDownload(pcspath)
		if pcsError != nil {
			return nil, pcsError
		}

		us := dInfo.URLStrings(pcsconfig.Config.EnableHTTPS)
		if len(us) == 0 {
			return nil, ErrDlinkNotFound
		}
		config := pcsconfig.Config
		config.SetPanUA("netdisk;2.2.51.6;netdisk;10.0.63;PC;android-android")
		return us, nil
	}
	// 获取Pandownload网页版下载链接
	// 先分享拿到分享链接
	paths := make([]string, 0, 10)
	fmt.Printf("检测到开启了Pandownload网页版加速功能，正在为%s 获取分享链接\n", pcspath)
	paths = append(paths, pcspath)
	shared, err := pcsconfig.Config.ActiveUserBaiduPCS().ShareSet(paths, nil)
	if err != nil {
		return nil, ErrDlinkNotFound
	}
	fmt.Printf("获取的分享链接为: %s, 密码为pass\n", shared.Link)
	surl := strings.TrimPrefix(shared.Link, "https://pan.baidu.com/s/")

	// 再请求Pandownload网页版
	fmt.Printf("正在请求PD网页版获取文件列表: %s\n", PD_Url+"list")
	resp, err := http.Post(PD_Url+"list",
		"application/x-www-form-urlencoded",
		strings.NewReader("surl="+surl+"&pwd=pass"))
	if err != nil {
		return nil, ErrDlinkNotFound
	}

	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, ErrDlinkNotFound
	}
	flysnowRegexp := regexp.MustCompile(`dl\('(.*)',(.*),'(.*)','(.*)','(.*)','(.*)'\)`)
	params := flysnowRegexp.FindStringSubmatch(string(body))

	if len(params) < 6 {
		fmt.Printf("文件列表请求失败!\n")
		return nil, ErrDlinkNotFound
	}
	// 最终获得下载link
	fmt.Printf("正在请求PD网页版获取下载链接: %s\n", PD_Url+"download")
	resp, err = http.Post(PD_Url+"download",
		"application/x-www-form-urlencoded",
		strings.NewReader("fs_id="+params[1]+"&time="+params[2]+"&sign="+params[3]+"&randsk="+params[4]+"&share_id="+params[5]+"&uk="+params[6]))
	if err != nil {
		return nil, ErrDlinkNotFound
	}
	defer resp.Body.Close()
	body, err = ioutil.ReadAll(resp.Body)
	if err != nil {
		fmt.Printf("网页未回复或出错...\n")
		return nil, ErrDlinkNotFound
	}
	// 获取http下载link
	if pcsconfig.Config.EnableHTTPS {
		flysnowRegexp = regexp.MustCompile(`(?U)id="https" href="(.*)"`)
	} else {
		flysnowRegexp = regexp.MustCompile(`(?U)id="http" href="(.*)"`)
	}
	params = flysnowRegexp.FindStringSubmatch(string(body))

	if len(params) < 1 {
		fmt.Printf("无法获取下载链接...\n")
		return nil, ErrDlinkNotFound
	}
	urls := make([]*url.URL, 0, 10)
	u, err := url.Parse(params[1])
	urls = append(urls, u)
	config := pcsconfig.Config
	config.SetPanUA("LogStatistic")
	fmt.Printf("成功获取下载链接: %s\n", params[1])
	return urls, nil
}

func getLocatePanLink(pcs *baidupcs.BaiduPCS, fsID int64) (dlink string, err error) {
	list, err := pcs.LocatePanAPIDownload(fsID)
	if err != nil {
		return
	}

	var link string
	for k := range list {
		if strconv.FormatInt(fsID, 10) == list[k].FsID {
			link = list[k].Dlink
		}
	}

	if link == "" {
		return "", ErrDlinkNotFound
	}

	dc := pcsconfig.Config.DlinkClient()
	dlink, err = dc.CacheLinkRedirectPr(link)
	return
}

func handleHTTPLinkURL(linkURL *url.URL) {
	if pcsconfig.Config.EnableHTTPS {
		if linkURL.Scheme == "http" {
			linkURL.Scheme = "https"
		}
	}
}

// fileExist 检查文件是否存在,
// 只有当文件存在, 文件大小不为0或断点续传文件不存在时, 才判断为存在
func fileExist(path string) bool {
	if info, err := os.Stat(path); err == nil {
		if info.Size() == 0 {
			return false
		}
		if _, err = os.Stat(path + DownloadSuffix); err != nil {
			return true
		}
	}

	return false
}

func fileSize(path string) int64 {
	if info, err := os.Stat(path); err == nil {
		return info.Size()
	}
	return 0
}
