"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const yargs_1 = require("yargs");
const helpers_1 = require("yargs/helpers");
const yargs_2 = __importDefault(require("yargs/yargs"));
const constants_1 = require("../constants");
const analyzeCommand_1 = require("./commands/analyzeCommand");
const cleanCommand_1 = require("./commands/cleanCommand");
const SCRIPT_NAME = 'clean-modules';
const CLEAN_DESCRIPTION = 'Removes unnecessary files to reduce the size of your node_modules directory.';
const ANALYZE_DESCRIPTION = 'Helps determining why a file is included by the clean command without removing any files.';
function handleError(error) {
    console.error(error);
    process.exit(1);
}
const printWidth = Math.min(Math.max((0, yargs_1.terminalWidth)(), 50), 150);
(0, yargs_2.default)((0, helpers_1.hideBin)(process.argv))
    .scriptName(SCRIPT_NAME)
    .wrap(printWidth)
    .usage(`${SCRIPT_NAME} 🧹\n`)
    .alias('v', 'version')
    .alias('h', 'help')
    .option('include', {
    alias: 'i',
    description: 'Custom glob patterns for files to include',
    type: 'string',
    array: true,
})
    .option('exclude', {
    alias: 'e',
    description: 'Custom glob patterns for files to exclude',
    type: 'string',
    array: true,
})
    .option('directory', {
    alias: 'D',
    description: 'Path to node_modules.',
    type: 'string',
    default: 'node_modules',
})
    .option('glob-file', {
    alias: 'f',
    description: 'Path to a custom globs file.',
    type: 'string',
    default: constants_1.DEFAULT_USER_GLOBS_FILE_NAME,
})
    .option('no-defaults', {
    alias: 'n',
    description: 'Only includes/excludes globs specified by a custom glob file or CLI arguments.',
    type: 'boolean',
})
    .option('json', {
    alias: 'j',
    description: 'Only logs a final JSON dump at the end of the script.',
    type: 'boolean',
})
    .command('*', CLEAN_DESCRIPTION, yargsArgv => {
    return yargsArgv
        .usage('Usage:\n $ $0 [command] [options]')
        .option('keep-empty', {
        alias: 'k',
        description: 'Skips removing empty folders after removing contents.',
        type: 'boolean',
        default: false,
    })
        .option('dry-run', {
        alias: 'd',
        description: 'Runs the script and prints results without removing any files.',
        type: 'boolean',
    })
        .option('silent', {
        alias: 's',
        description: 'Does not log anything to console (unless --json is enabled).',
        type: 'boolean',
    })
        .option('yes', {
        alias: 'y',
        description: 'Skips the confirmation prompt at the start of the script.',
        type: 'boolean',
    });
}, async (args) => {
    await (0, cleanCommand_1.cleanCommand)({
        argGlobs: { included: args.include || [], excluded: args.exclude || [] },
        nodeModulesPath: path_1.default.resolve(process.cwd(), args.directory),
        useDefaultGlobs: args['defaults'] !== false,
        userGlobsFilePath: args['glob-file'],
        keepEmpty: args['keep-empty'],
        dryRun: !!args['dry-run'],
        json: !!args.json,
        silent: !!args.silent,
        yes: !!args.yes,
    }).catch(handleError);
})
    .command('analyze', ANALYZE_DESCRIPTION, yargsArgv => {
    return yargsArgv.usage(`${ANALYZE_DESCRIPTION}\n\nUsage:\n $ $0 [options]`);
}, async (args) => {
    await (0, analyzeCommand_1.analyzeCommand)({
        argGlobs: { included: args.include || [], excluded: args.exclude || [] },
        nodeModulesPath: path_1.default.resolve(process.cwd(), args.directory),
        useDefaultGlobs: args['defaults'] !== false,
        userGlobsFilePath: args['glob-file'],
    }).catch(handleError);
}).argv;
