/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_ioctl.c
   This file contains the implementation of the TAPI ioctl handling.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_api.h"
#include "drv_tapi.h"
#include "drv_tapi_ll_interface.h"
#include "drv_tapi_event.h"
#ifdef TAPI_POLL
#include "drv_tapi_polling.h"
#endif /* TAPI_POLL */
#include "drv_tapi_fxo_ll_interface.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_ioctl.h"

#ifdef TAPI_CID
#include "drv_tapi_cid.h"
#endif
#ifdef KPI_SUPPORT
#include "drv_tapi_kpi.h"
#endif
#ifdef QOS_SUPPORT
#include "drv_tapi_qos.h"
#include "drv_tapi_qos_io.h"
#endif
#ifdef TAPI_ANNOUNCEMENTS
#include "drv_tapi_announcements.h"
#endif /* TAPI_ANNOUNCEMENTS */

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */
/*lint -save
-esym(750, ON_IOCTL_LLFUNC, ON_IOCTLW3, ON_IOCTLW2, ON_IOCTLR1, ON_IOCTLW1)
*/
#define ON_IOCTL_LLFUNC(msg,func,ioarg)                    \
           case msg:                                        \
              if (func == IFX_NULL)                         \
                  ret = TAPI_statusLLNotSupp;            \
              else                                          \
                  ret = func(pChannel->pLLChannel, ioarg);  \
              break

/* 3 parameter, call like
   IFX_TAPI_Ring_SetConfig (pChannel, (IFX_TAPI_RING_CFG_t *) ioarg);
*/
#define ON_IOCTLW3(msg,func,arg1,arg2,type)          \
           case msg:                                   \
              ret = func(arg1,arg2,(type)ioarg);       \
              break

#define ON_IOCTLW2(msg,func,arg,type)          \
           case msg:                                  \
              ret = func(arg,(type)ioarg);              \
              break

#define ON_IOCTLR1(msg,func,arg,type)          \
           case msg:                                  \
              ret = func(arg,(type)ioarg);     \
              break

#define ON_IOCTLW1 (msg,func, arg)               \
           case msg:                           \
              ret = func(arg);                   \
              break

/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Local function declaration    */
/* ============================= */

static IFX_int32_t TAPI_Dev_Spec_Ioctl (
                        IFX_TAPI_ioctlCtx_t *pCtx,
                        IFX_uint32_t nCmd,
                        IFX_ulong_t nArgument);
static IFX_int32_t TAPI_Spec_Ioctl (
                        IFX_TAPI_ioctlCtx_t *pCtx,
                        IFX_uint32_t iocmd,
                        IFX_ulong_t ioarg);
static IFX_int32_t TAPI_IoctlDrv (
                        IFX_uint32_t iocmd,
                        IFX_ulong_t ioarg);
static IFX_int32_t TAPI_IoctlDev (
                        IFX_TAPI_DRV_CTX_t* pDrvCtx,
                        IFX_TAPI_ioctlCtx_t* pCtx,
                        IFX_uint32_t iocmd,
                        IFX_ulong_t ioarg);
static IFX_int32_t  TAPI_IoctlCh (
                        IFX_TAPI_DRV_CTX_t* pDrvCtx,
                        IFX_TAPI_ioctlCtx_t* pCtx,
                        IFX_uint32_t iocmd,
                        IFX_ulong_t ioarg);
#ifdef TAPI_AUDIO_CHANNEL
static IFX_int32_t  TAPI_IoctlAudioCh (
                        IFX_TAPI_DRV_CTX_t* pDrvCtx,
                        IFX_TAPI_ioctlCtx_t* pCtx,
                        IFX_uint32_t iocmd,
                        IFX_ulong_t ioarg);
#endif /* TAPI_AUDIO_CHANNEL */

/* ============================= */
/* Local variable definition     */
/* ============================= */

/* ============================= */
/* Local function definition     */
/* ============================= */

/* ============================= */
/* Global function definition    */
/* ============================= */

/**
   TAPI global IOCTL handling

   \param  pCtx         Pointer to IOCTL context.
   \param  iocmd        IOCTL command.
   \param  ioarg        IOCTL argument.

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErr in case of error
   - TAPI_statusCtxErr
   - TAPI_statusParam

   \remarks
   This function does the following functions:
      - If the ioctl command is device specific, low-level driver's ioctl function
      - If the ioctl command is TAPI specific, it is handled at this level
*/
IFX_int32_t  TAPI_Ioctl (IFX_TAPI_ioctlCtx_t *pCtx,
                         IFX_uint32_t iocmd,
                         IFX_ulong_t ioarg)
{
   IFX_int32_t         ret = TAPI_statusErr;

#if 0
   TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("IOCTL %08X   DEV:%08X FDS:%03d ENTER\n",
          iocmd, (IFX_uint32_t)pCtx->pTapiDev, pCtx->nFds));
#endif

   switch (IFXOX_IO_GETMAGIC (iocmd))
   {
      case VMMC_IOC_MAGIC:
      case VINETIC_IOC_MAGIC:
      case DUS_IOC_MAGIC:
      case SVIP_IOC_MAGIC:
         /* Device specific ioctl command */
         ret = TAPI_Dev_Spec_Ioctl (pCtx, iocmd, ioarg);
         break;

#ifdef QOS_SUPPORT
      case QOS_IOC_MAGIC:
         /* This group contains extra TAPI IOCTLs that are handled just like
            all other TAPI IOCTLs below. */
         /*lint -fallthrough */
#endif /* QOS_SUPPORT */
      case IFX_TAPI_IOC_MAGIC:
         /* TAPI specific ioctl */
         ret = TAPI_Spec_Ioctl     (pCtx, iocmd, ioarg);
         break;

      default:
         /* magic is not known - return error */
         ret = TAPI_statusErr;
   }

   return ret;
}


/**
   Device Specific IOCTL handling

   \param  pCtx         Pointer to IOCTL context.
   \param  nCmd         IOCTL command
   \param  ioarg        IOCTL argument

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErr in case of error
*/
IFX_int32_t TAPI_Dev_Spec_Ioctl (IFX_TAPI_ioctlCtx_t* pCtx,
                                 IFX_uint32_t nCmd,
                                 IFX_ulong_t ioarg)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx  = pCtx->pTapiDev->pDevDrvCtx;
   TAPI_DEV            *pTapiDev = pCtx->pTapiDev;
   IFX_int32_t          ret      = TAPI_statusOk,
                        retLL    = TAPI_statusOk;

   /* make sure that the LL supports IOCTLs */
   if (pDrvCtx->Ioctl == IFX_NULL)
   {
      ret = TAPI_statusLLNotSupp;
   }
   else
   {
      /* dispatch to low level IOCTL handler */

      /* Provide either a low-level device or channel context to the low-level
         IOCTL handler. The low level IOCTL handler has to check the correctness
         of the context. */
      if (pCtx->bSingleFd || pCtx->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
      {
         retLL = pDrvCtx->Ioctl (pTapiDev->pLLDev, nCmd, ioarg);
      }
      else
      {
         TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[pCtx->nFds]);
         retLL = pDrvCtx->Ioctl (pTapiCh->pLLChannel, nCmd, ioarg);
      }

      if (retLL != TAPI_statusOk)
      {
         /* if LL failed add a generic HL error code */
         ret = TAPI_statusLLFailed;
      }
   }

   if (ret != TAPI_statusOk)
   {
      /* store on the error stack and set error variable */
      TAPI_ErrorStatus (pTapiDev, ret, retLL, __LINE__, __FILE__);

      /* store channel detail */
      pTapiDev->error.nCh = pCtx->nFds;

      /* return value just indicates an error - details are on the stack */
      ret = TAPI_statusErr;
   }

   return ret;
}


/**
   TAPI specific IOCTL handling

   \param  pCtx         Pointer to IOCTL context.
   \param  iocmd        IOCTL command.
   \param  ioarg        IOCTL argument.

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErr in case of error
   - TAPI_statusCtxErr
   - TAPI_statusParam

   \remark
   Only TAPI IOCTLs with appropriate TAPI magic number are handled within
   this function.
*/
IFX_int32_t  TAPI_Spec_Ioctl (IFX_TAPI_ioctlCtx_t *pCtx,
                              IFX_uint32_t iocmd,
                              IFX_ulong_t ioarg)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pCtx->pTapiDev->pDevDrvCtx;
   TAPI_DEV            *pTapiDev  = pCtx->pTapiDev;
   IFX_int32_t          ret       = TAPI_statusOk;
   IFX_boolean_t        bProcessed = IFX_TRUE;

   /* These IOCTLs can be used at all times. Even before the TAPI is started
      for this device. The list can be extended by just adding a new case. */
   switch (iocmd)
   {
      /* Driver IOCTLs */
      case IFX_TAPI_DEBUG_REPORT_SET:
      case IFX_TAPI_VERSION_GET:
      case IFX_TAPI_VERSION_CHECK:
      case IFX_TAPI_TONE_TABLE_CFG_SET:
      case IFX_TAPI_KPI_GRP_CFG_SET:
         ret = TAPI_IoctlDrv (iocmd, ioarg);
         break;

      /* Device IOCTLs */
      case IFX_TAPI_DEV_START:
      case IFX_TAPI_DEV_STOP:
      case IFX_TAPI_CAP_NR:
      case IFX_TAPI_CAP_LIST:
      case IFX_TAPI_CAP_CHECK:
      case IFX_TAPI_LASTERR:
      case IFX_TAPI_CH_INIT:  /* kept for compatiblity but executed on device */
         TAPI_OS_MutexGet (&pTapiDev->semTapiDevSingleIoctlAccess);
         ret = TAPI_IoctlDev (pDrvCtx, pCtx, iocmd, ioarg);
         TAPI_OS_MutexRelease (&pTapiDev->semTapiDevSingleIoctlAccess);
         break;

      default:
         /* All other IOCTLs are handled below */
         bProcessed = IFX_FALSE;
         break;
   } /* switch */

   /* If the device is not yet started no other IOCTLs are allowed. */
   if ((bProcessed == IFX_FALSE) && (pTapiDev->bInitialized == IFX_FALSE))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_LOW,
            ("IOCTL %08X DEV:%08X blocked\n", iocmd, (IFX_uint32_t)pTapiDev));
      /* errmsg: IOCTL is blocked until the device is started */
      ret = TAPI_statusIoctlBlocked;
      TAPI_ErrorStatus (pTapiDev, ret, 0, __LINE__, __FILE__);
      ret = BUILD_HL_ERROR(ret);
   }
   else if (bProcessed == IFX_FALSE)
   {
      /* These IOCTL can only be used after TAPI was started for this device. */
      switch(iocmd)
      {
         /* Device IOCTLs */
         case IFX_TAPI_POLL_CONFIG:
         case IFX_TAPI_POLL_DEV_ADD:
         case IFX_TAPI_POLL_DEV_REM:
         case IFX_TAPI_POLL_WRITE:
         case IFX_TAPI_POLL_READ:
         case IFX_TAPI_POLL_EVENTS:
         case IFX_TAPI_POLL_TEST:
         case IFX_TAPI_PCM_IF_CFG_SET:
         case IFX_TAPI_EVENT_ENABLE:
         case IFX_TAPI_EVENT_DISABLE:
         case IFX_TAPI_EVENT_GET:
         case IFX_TAPI_T38_CAP_GET:
            TAPI_OS_MutexGet (&pTapiDev->semTapiDevSingleIoctlAccess);
            ret = TAPI_IoctlDev (pDrvCtx, pCtx, iocmd, ioarg);
            TAPI_OS_MutexRelease (&pTapiDev->semTapiDevSingleIoctlAccess);
            break;


         default:
#ifdef TAPI_ONE_DEVNODE
            if (pCtx->bSingleFd && (ioarg >= 0UL && ioarg <= 255UL))
            {
               /* case (5): return error if single dev node and
                  no structure passed */
               ret = TAPI_statusParam;
               TAPI_ErrorStatus (pTapiDev, ret, 0, __LINE__, __FILE__);
               ret = BUILD_HL_ERROR(ret);
            }
            else
#endif /* TAPI_ONE_DEVNODE */
            {
               /* process with channel fd */
               ret = TAPI_IoctlCh (pDrvCtx, pCtx, iocmd, ioarg);
            }
            break;
      } /* switch */
   }

   /* Process return codes */
   if (ret != TAPI_statusOk)
   {
      /* The following IOCTLs have individual return values that will
         be passed transparently to the application. For all other IOCTLs
         the return value is handled as an error code and stored onto the
         error stack. */
      if ((iocmd != IFX_TAPI_CAP_CHECK) &&
          (iocmd != IFX_TAPI_RING))
      {
         /* store error details */
         pTapiDev->error.nCh = pCtx->nFds;
         pTapiDev->error.nCode = (IFX_uint32_t)ret;

         /* return value just indicates an error - details are on the stack */
         ret = TAPI_statusErr;
      }
   }

   return ret;
}


/**
   TAPI Driver specific IOCTL handling

   These IOCTLs affect the entire driver. No device or channel context is
   needed. They can be used at all times.

   \param  iocmd        IOCTL command.
   \param  ioarg        IOCTL argument.

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErrKernCpy error copying data to userspace
   - TAPI_statusNoMem no memory to allocate some buffer memory
   - TAPI_statusErr in case of error
*/
static IFX_int32_t TAPI_IoctlDrv (IFX_uint32_t iocmd,
                                  IFX_ulong_t ioarg)
{
   /* Within this function the error codes are not aligned to the correct
      position of the error field. This will be corrected upon return. */
   IFX_int32_t ret = TAPI_statusOk;

   switch(iocmd)
   {
      case IFX_TAPI_DEBUG_REPORT_SET:
         SetTraceLevel (TAPI_DRV, (IFX_uint32_t)ioarg);
         break;

      case IFX_TAPI_VERSION_GET:
#ifdef LINUX
         {
            IFX_char_t *p_tmp = (IFX_char_t *)TAPI_OS_Malloc (80);

            if (p_tmp != IFX_NULL)
            {
               ret = TAPI_Phone_Get_Version(p_tmp);
               /* check for ret (string length) > 0 */
               if( ret > 0 )
               {
                  if( copy_to_user((IFX_char_t *)ioarg, p_tmp, 80) != 0)
                  {
                     ret = TAPI_statusErrKernCpy;
                  }
                  else
                  {
                     ret = TAPI_statusOk;
                  }
               }
               else
               {
                  ret = TAPI_statusErr;
               }
               TAPI_OS_Free (p_tmp);
            }
            else
            {
               ret = TAPI_statusNoMem;
            }
         }
#else
         ret = TAPI_Phone_Get_Version((IFX_char_t*)ioarg);
#endif /* LINUX */
         break;

      case IFX_TAPI_VERSION_CHECK:
         ret = TAPI_Phone_Check_Version ((IFX_TAPI_VERSION_t *)ioarg);
         break;

      case IFX_TAPI_TONE_TABLE_CFG_SET:
         {
            IFX_TAPI_TONE_t *p_tmp;
#ifdef LINUX
            IFX_TAPI_TONE_t tone;

            p_tmp = &tone;
            if (copy_from_user (p_tmp, (IFX_TAPI_TONE_t*)ioarg,
                                sizeof(IFX_TAPI_TONE_t)) > 0 )
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
#else
            {
               p_tmp = (IFX_TAPI_TONE_t *) ioarg;
            }
#endif /* LINUX */
            {
               ret = TAPI_Phone_Tone_TableConf (p_tmp);
            }
         }
         break;

#ifdef KPI_SUPPORT
      /* Kernel Packet Interface group configuration */
      case IFX_TAPI_KPI_GRP_CFG_SET:
         {
            IFX_TAPI_KPI_GRP_CFG_t *p_tmp;
#ifdef LINUX
             p_tmp = (IFX_TAPI_KPI_GRP_CFG_t *)
                     TAPI_OS_Malloc (sizeof(IFX_TAPI_KPI_GRP_CFG_t));

            if (TAPI_OS_CpyUsr2Kern (p_tmp, (IFX_void_t*)ioarg,
                                     sizeof(IFX_TAPI_KPI_GRP_CFG_t)) == 0)
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
#endif /* LINUX */
            {
#ifndef LINUX
               p_tmp = (IFX_TAPI_KPI_GRP_CFG_t *)ioarg;
#endif /* LINUX */
               ret = IFX_TAPI_KPI_GrpCfgSet (p_tmp);
            }
#ifdef LINUX
            TAPI_OS_Free (p_tmp);
#endif /* LINUX */
         }
         break;
#endif /* KPI_SUPPORT */

      default:
         /* Coding error. This function is called for an unlisted IOCTL. */
         ret = TAPI_statusErr;
         break;
   }

   /* Up to there the error code was not aligned to the high-word part.
      Now move the value into the correct position of the error code field. */
   return BUILD_HL_ERROR(ret);
}


/**
   TAPI Device specific IOCTL handling

   These IOCTL affect one TAPI device. Some of them can be used even if TAPI
   was not yet initialised. But most of them must not be used until TAPI is
   started. Blocking of these IOCTLs must be done in the function levels above.

   \param  pDrvCtx      Pointer to device driver context.
   \param  pCtx         Pointer to IOCTL context.
   \param  iocmd        IOCTL command.
   \param  ioarg        IOCTL argument.

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErr in case of error
*/
static IFX_int32_t TAPI_IoctlDev (IFX_TAPI_DRV_CTX_t* pDrvCtx,
                                  IFX_TAPI_ioctlCtx_t* pCtx,
                                  IFX_uint32_t iocmd,
                                  IFX_ulong_t ioarg)
{
   TAPI_DEV            *pTapiDev   = pCtx->pTapiDev;
   IFX_int32_t          ret        = TAPI_statusOk;

   /* context verification */
#ifdef TAPI_ONE_DEVNODE
   if ((pCtx->bSingleFd) && (pCtx->nFds != IFX_TAPI_DEVICE_CH_NUMBER))
   {
      RETURN_DEVSTATUS (TAPI_statusCtxErr, 0);
   }
#endif /* TAPI_ONE_DEVNODE */

   if (pCtx->nFds != IFX_TAPI_DEVICE_CH_NUMBER)
   {
      /* Warn about channel context. */
      switch (iocmd)
      {
         case IFX_TAPI_CH_INIT:
         case IFX_TAPI_LASTERR:
            /* Do not warn about these - they are allowed on dev and channel. */
            break;
         default:
            TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("Warning: device IOCTL 0x%08X was issued on channel fd (ch%d) - "
            "tolerated\n", iocmd, pCtx->nFds));
            break;
      }
   }

   switch(iocmd)
   {
      /* ----------------------------------------------------------------------
         These IOCTLs can be used at all times. Even before the TAPI is started
         for this device. */

      case IFX_TAPI_DEV_START:
         if (ioarg == 0UL)
         {
            ret = IFX_TAPI_DeviceStart (pTapiDev, IFX_NULL);
         }
         else
         {
            IFX_TAPI_DEV_START_CFG_t nDevStartCfg;

            if (TAPI_OS_CpyUsr2Kern (&nDevStartCfg, (IFX_void_t*)ioarg,
                                     sizeof(nDevStartCfg)) == IFX_NULL)
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
            {
               ret = IFX_TAPI_DeviceStart (pTapiDev, &nDevStartCfg);
            }
         }
         break;

      case IFX_TAPI_DEV_STOP:
         ret = IFX_TAPI_DeviceStop(pTapiDev);
         break;

      case IFX_TAPI_LASTERR:
         TAPI_OS_CpyKern2Usr ((IFX_void_t*)ioarg, &(pTapiDev->error),
                              sizeof(IFX_TAPI_Error_t));
         /* clear the error now that we read it. */
         pTapiDev->error.nCnt = 0;
         pTapiDev->error.nCode = 0;
         break;

      case IFX_TAPI_CH_INIT:
         /* Support for the legacy ioctl IFX_TAPI_CH_INIT.
            Despite the name this is now executed on the device. */
         if (ioarg == 0UL)
         {
            ret = IFX_TAPI_Phone_Init(pTapiDev, IFX_NULL);
         }
         else
         {
            IFX_TAPI_CH_INIT_t  chInit;

            if (TAPI_OS_CpyUsr2Kern (&chInit, (IFX_void_t*)ioarg,
                                     sizeof(chInit)) == IFX_NULL)
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
            {
               ret = IFX_TAPI_Phone_Init(pTapiDev, &chInit);
            }
         }
         break;

      /* --- Capability reporting --- */

      case IFX_TAPI_CAP_NR:
         if (pDrvCtx->CAP_Number_Get == IFX_NULL)
         {
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         }
         *((IFX_int32_t*)ioarg) = pDrvCtx->CAP_Number_Get (pTapiDev->pLLDev);
         break;

      case IFX_TAPI_CAP_LIST:
         if (pDrvCtx->CAP_List_Get == IFX_NULL)
         {
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         }
#ifdef LINUX
         {
            IFX_TAPI_CAP_t *p_tmp = IFX_NULL;
            IFX_uint32_t   tmpcapsize;

            if (pDrvCtx->CAP_Number_Get == IFX_NULL)
            {
               RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
            }
            else
            {
               tmpcapsize = pDrvCtx->CAP_Number_Get (pTapiDev->pLLDev);
               p_tmp = TAPI_OS_Malloc (tmpcapsize * sizeof(IFX_TAPI_CAP_t));
               if (p_tmp == IFX_NULL)
               {
                  RETURN_DEVSTATUS (TAPI_statusNoMem, 0);
               }

               ret = pDrvCtx->CAP_List_Get (pTapiDev->pLLDev, p_tmp);
               if ((ret == TAPI_statusOk) &&
                   (copy_to_user((IFX_void_t*)ioarg, p_tmp,
                                 sizeof(IFX_TAPI_CAP_t) * tmpcapsize) > 0))
               {
                  ret = TAPI_statusErrKernCpy;
               }
               TAPI_OS_Free (p_tmp);
            }
         }
#else
         ret = pDrvCtx->CAP_List_Get (pTapiDev->pLLDev, (IFX_TAPI_CAP_t *)ioarg);
#endif /* LINUX */
         break;

      case IFX_TAPI_CAP_CHECK:
         if (pDrvCtx->CAP_Check == IFX_NULL)
         {
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         }
#ifdef LINUX
         {
            IFX_TAPI_CAP_t *p_tmp;
            p_tmp = (IFX_TAPI_CAP_t *) TAPI_OS_Malloc (sizeof(IFX_TAPI_CAP_t));

            if (copy_from_user (p_tmp, (IFX_void_t *)ioarg,
                                sizeof(IFX_TAPI_CAP_t)) > 0 )
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
            {
               ret = pDrvCtx->CAP_Check (pTapiDev->pLLDev, p_tmp);

               if (ret == TAPI_statusOk || ret == 1)
               {
                  copy_to_user ((IFX_void_t*)ioarg, p_tmp, sizeof(IFX_TAPI_CAP_t));
               }
            }
            TAPI_OS_Free (p_tmp);
         }
#else
         ret = pDrvCtx->CAP_Check (pTapiDev->pLLDev, (IFX_TAPI_CAP_t*)ioarg);
#endif /* LINUX */
         break;

      /* --------------------------------------------------------------------
         These IOCTL can only be used after TAPI was started for this device.
         Checking for this condition is done in the function one level above. */

      /* --- PCM Services --- */

      case IFX_TAPI_PCM_IF_CFG_SET:
         {
            IFX_TAPI_PCM_IF_CFG_t *p_tmp;
#ifdef LINUX
            p_tmp = (IFX_TAPI_PCM_IF_CFG_t *)
                    TAPI_OS_Malloc (sizeof(IFX_TAPI_PCM_IF_CFG_t));

            if (TAPI_OS_CpyUsr2Kern (p_tmp, (IFX_void_t*)ioarg,
                                     sizeof(IFX_TAPI_PCM_IF_CFG_t)) == 0)
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
#else
            {
               p_tmp = (IFX_TAPI_PCM_IF_CFG_t *)ioarg;
            }
#endif /* LINUX */
            {
               ret = TAPI_Phone_PCM_IF_Set_Config(pTapiDev, p_tmp);
            }
#ifdef LINUX
            TAPI_OS_Free (p_tmp);
#endif /* LINUX */
         }
         break;

      /* --- TAPI event handling --- */

      case IFX_TAPI_EVENT_GET:
#ifdef LINUX
         {
            IFX_TAPI_EVENT_t *p_tmp = IFX_NULL;

            if (ioarg == 0)
            {
               RETURN_DEVSTATUS (TAPI_statusInvalidIoctl, 0);
            }
            p_tmp = TAPI_OS_Malloc (sizeof(IFX_TAPI_EVENT_t));
            if (p_tmp == IFX_NULL)
            {
               RETURN_DEVSTATUS (TAPI_statusNoMem, 0);
            }
            if (copy_from_user(p_tmp, (IFX_void_t *) ioarg, sizeof(IFX_TAPI_EVENT_t)) > 0)
            {
               ret = TAPI_statusErrKernCpy;
            }
            if (TAPI_SUCCESS (ret))
            {
               ret = TAPI_Phone_GetEvent(pTapiDev, p_tmp);
               if (copy_to_user ((IFX_void_t *)ioarg, (IFX_void_t *) p_tmp, sizeof(IFX_TAPI_EVENT_t))> 0)
               {
                  ret = TAPI_statusErrKernCpy;
               }
            }
            TAPI_OS_Free (p_tmp);
         }
#else
         ret = TAPI_Phone_GetEvent(pTapiDev, (IFX_TAPI_EVENT_t*)ioarg);
#endif /* LINUX */
         break;

      case IFX_TAPI_EVENT_ENABLE:
      case IFX_TAPI_EVENT_DISABLE:
         {
            /* Enables or disables a event on a given channel. The parameter
               specifies which channel and also the details of the event.
               The command can only be sent on a device file descriptor to make
               clear that the channel used is the one given in the parameter. */
            IFX_TAPI_EVENT_t *pEvent;

#ifdef LINUX
            pEvent =
               (IFX_TAPI_EVENT_t *) TAPI_OS_Malloc (sizeof(IFX_TAPI_EVENT_t));

            if (TAPI_OS_CpyUsr2Kern (pEvent, (IFX_void_t*)ioarg,
                                     sizeof(IFX_TAPI_EVENT_t)) == 0)
            {
               ret = TAPI_statusErrKernCpy;
               break;
            }
            else
#else
            {
               pEvent = (IFX_TAPI_EVENT_t *)ioarg;
            }
#endif /* LINUX */

            /* Make sure the given channel parameter is within limits.
               The channel is passed in the struct of the parameter and needs to
               be verified to prevent out of range access of the channel array. */
            if ((pEvent->ch > pTapiDev->nMaxChannel) &&
                (pEvent->ch != IFX_TAPI_EVENT_ALL_CHANNELS))
            {
               TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
                    ("TAPI: IFX_TAPI_EVENT_EN-/DISABLE called with channel "
                     "parameter out of range\n"));
               ret = TAPI_statusParam;
               break;
            }
            /* call function to set event reporting mask according to the ioctl
               on either one or all channels of a device */
            if (pEvent->ch != IFX_TAPI_EVENT_ALL_CHANNELS)
            {
               /* only one specific channel is addressed */
               ret = IFX_TAPI_Event_SetMask(
                        (pTapiDev->pTapiChanelArray + pEvent->ch),
                        pEvent,
                        (iocmd == IFX_TAPI_EVENT_ENABLE) ?
                                          IFX_EVENT_ENABLE : IFX_EVENT_DISABLE);
            }
            else
            {
               /* all channels of the device are addressed */
               IFX_uint8_t i;

               for (i = 0; (ret == IFX_SUCCESS) &&
                           (i < pTapiDev->nMaxChannel); i++)
               {
                  ret = IFX_TAPI_Event_SetMask(
                           (pTapiDev->pTapiChanelArray + i), pEvent,
                           (iocmd == IFX_TAPI_EVENT_ENABLE) ?
                                          IFX_EVENT_ENABLE : IFX_EVENT_DISABLE);
               }
            }
#ifdef LINUX
            TAPI_OS_Free (pEvent);
#endif /* LINUX */
         }
         break;

#ifdef TAPI_POLL
         /* --- Polling services ---*/

         case IFX_TAPI_POLL_CONFIG:
         {
            IFX_TAPI_POLL_CONFIG_t* pTapiPollCfg = IFX_NULL;
#ifdef LINUX
             ret = IFX_ERROR;
#else
            {
                pTapiPollCfg = (IFX_TAPI_POLL_CONFIG_t *)ioarg;
            }
#endif /* LINUX */
            if (pTapiPollCfg != IFX_NULL)
            {
               ret = TAPI_IrqPollConf(pTapiPollCfg);
            }
            else
            {
               RETURN_DEVSTATUS (TAPI_statusParam, 0);
            }
            break;
         }

         case IFX_TAPI_POLL_DEV_ADD:
         {
             ret = TAPI_AddDev_PollPkts(pDrvCtx, pTapiDev);
             if (TAPI_SUCCESS (ret))
             {
                ret = TAPI_AddDev_PollEvts(pDrvCtx, pTapiDev);
             }
             break;
         }

         case IFX_TAPI_POLL_DEV_REM:
         {
            ret = TAPI_RemDev_PollPkts(pDrvCtx, pTapiDev);
            if (TAPI_SUCCESS (ret))
            {
               ret = TAPI_RemDev_PollEvts(pDrvCtx, pTapiDev);
            }
            break;
         }

         /* Test TAPI DEVICE list handling with simulation. */
         case IFX_TAPI_POLL_TEST:
         {
            ret = TAPI_Poll_Test();
            break;
         }

#if 0
         case IFX_TAPI_POLL_WRITE:
         {
            IFX_void_t* pPktsArray[IFX_TAPI_POLL_QUEUE];
            IFX_void_t** ppPkts;
            IFX_int32_t pPktsNum = pPollDown->pPktsNum;
            IFX_TAPI_POLL_DATA_t *pPollDown = (IFX_TAPI_POLL_DATA_t *) parg;
            IFX_int32_t i = 0;


            ppPkts = pPollDown->ppPkts;
            for (i = 0; i < pPktsNum; i++)
            {
               /* skip any NULL pointers in the array */
               while (*ppPkts == IFX_NULL)
               {
                  *ppPkts++;

                  pPkt = *ppPkts++;

                  if (copy_from_user (pPollDown, parg, sizeof(IFX_TAPI_POLL_DATA_t)) > 0 )
                  {
                     RETURN_DEVSTATUS (TAPI_statusErrKernCpy, 0);
                  }
               }
               ret = TAPI_Poll_Down (pPollDown->ppPkts, &pPollDown->pPktsNum);
               break;
            }
         }

         case IFX_TAPI_POLL_READ:
            /* to be completed... */
            ret = TAPI_Poll_Up (pPollDown->ppPkts, &pPollDown->pPktsNum);
         break;
#endif

#endif /* TAPI_POLL */

#ifdef TAPI_FAX_T38_FW
         case IFX_TAPI_T38_CAP_GET:
         {
            IFX_TAPI_T38_CAP_t *p_tmp;

            if (pDrvCtx->COD.FAX_Cap_Get == IFX_NULL)
            {
               RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
            }
#ifdef LINUX
            p_tmp = (IFX_TAPI_T38_CAP_t* )
                    TAPI_OS_Malloc (sizeof(IFX_TAPI_T38_CAP_t));

            ret = pDrvCtx->COD.FAX_Cap_Get(pTapiDev->pLLDev, p_tmp);

            if (TAPI_SUCCESS (ret))
            {
               if (TAPI_OS_CpyKern2Usr ((IFX_void_t*)ioarg, p_tmp,
                                        sizeof(IFX_TAPI_T38_CAP_t)) == 0)
               {
                  ret = TAPI_statusErrKernCpy;
                  TAPI_OS_Free (p_tmp);
                  break;
               }
            }
            TAPI_OS_Free (p_tmp);
#else
            p_tmp = (IFX_TAPI_T38_CAP_t* )ioarg;
            ret = pDrvCtx->COD.FAX_Cap_Get(pTapiDev->pLLDev, p_tmp);
#endif /*LINUX*/
         }
         break;
#endif /*TAPI_FAX_T38_FW*/

      default:
         ret = TAPI_statusNoIoctl;
   }

   RETURN_DEVSTATUS(ret, 0);
}


/**
   TAPI Channel specific IOCTL handling

   \param  pDrvCtx      Pointer to device driver context.
   \param  pCtx         Pointer to the ioctl context structure.
   \param  iocmd        IOCTL command
   \param  ioarg        IOCTL argument

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErr in case of error
*/
static IFX_int32_t  TAPI_IoctlCh (IFX_TAPI_DRV_CTX_t* pDrvCtx,
                                  IFX_TAPI_ioctlCtx_t* pCtx,
                                  IFX_uint32_t iocmd,
                                  IFX_ulong_t ioarg)
{
   TAPI_DEV   *pTapiDev  = pCtx->pTapiDev;
   IFX_int32_t ret = TAPI_statusOk,
               retLL = TAPI_statusOk;
   /* param is used for simple int arguments in one direction */
   IFX_int32_t param = 0;
   TAPI_CHANNEL* pChannel;
#ifdef TAPI_ONE_DEVNODE
   IFX_TAPI_IOCTL_t tmpAdr;
#endif /* TAPI_ONE_DEVNODE */
   IFX_boolean_t bHandled;

   pChannel = &(pTapiDev->pTapiChanelArray[0]);

   /* context verification */
   if (pCtx->nFds == IFX_TAPI_DEVICE_CH_NUMBER)
   {
      /* TAPI ch ioctl issued on dev fd */
      RETURN_STATUS (TAPI_statusCtxErr, 0);
   }

   if (pCtx->nFds >= pDrvCtx->maxChannels)
   {
      /*errmsg: Service not supported on called channel context */
      RETURN_STATUS (TAPI_statusInvalidCh, 0);
   }

   /* If we support the only one fd, find out the device and channel from the
      given structure. Distinguish between already ported structures,
      applications and ioctls that are global or device global.
      Following is supported:
      - (1) any ioctl using new structures with dev and ch on the only dev node
      - (2) any ioctl using new structurs with dev and ch on channel fds
      - (3) channel ioctls, when used on channel fds with new structures
      - (4) channel ioctls using simple integers, when used on channel fds
      - (5) return error if single dev node and no structure passed, handled in
            calling function
   */
#ifdef TAPI_ONE_DEVNODE
   if (pCtx->bSingleFd == IFX_FALSE)
#endif /* TAPI_ONE_DEVNODE */
   {
      /* handling case (2), (3) and (4) */
      pChannel = &(pTapiDev->pTapiChanelArray[pCtx->nFds]);
#ifdef TAPI_ONE_DEVNODE
      if ((ioarg >= 0UL) && (ioarg <= 255UL))
         /* case (4) */
#endif
         param = (IFX_int32_t)ioarg;
#ifdef TAPI_ONE_DEVNODE
      else
      {
         /* case (2) and (3) using a new strucutre, get the context and pass
            again the one argument. For case (2) we ignore the dev and ch */
         TAPI_OS_CpyUsr2Kern ((IFX_char_t*)&tmpAdr, (IFX_uint8_t*)ioarg,
                              sizeof (IFX_TAPI_IOCTL_t));
         /* copy the one parameter */
         param = tmpAdr.param;
      }
   }
   else
   {
      /* case (1) using a new strucutre, get the context and pass again
         the one argument */
      TAPI_OS_CpyUsr2Kern ((IFX_char_t*)&tmpAdr, (IFX_uint8_t*)ioarg,
                           sizeof (IFX_TAPI_IOCTL_t));
      /* In single dev node always a pointer to a structure must be given */
      if (tmpAdr.dev >= pDrvCtx->maxDevs)
      {
         RETURN_STATUS (TAPI_statusParam, 0);
      }
      if (tmpAdr.ch == IFX_TAPI_EVENT_ALL_CHANNELS)
      {
         pChannel = &((pDrvCtx->pTapiDev[tmpAdr.dev]).pTapiChanelArray[0]);
      }
      else
      {
         if (tmpAdr.ch >= pDrvCtx->maxChannels)
         {
            RETURN_STATUS (TAPI_statusParam, 0);
         }
         pChannel = &((pDrvCtx->pTapiDev[tmpAdr.dev]).
                       pTapiChanelArray[tmpAdr.ch]);
      }
      /* copy the one parameter */
      param = tmpAdr.param;
#endif /* TAPI_ONE_DEVNODE */
   }

   TAPI_OS_MutexGet (&pChannel->semTapiChSingleIoctlAccess);
   /* copy back one parameter if applicable. This is just temporarily till
      all applications and structures are ported. */
   bHandled = IFX_TRUE;
   /* simple calls with one or none parameter -> no copying from and to user
      space (linux). And calls which copy the parameters themself. */
   switch (iocmd)
   {
      /* FXO services */
      case IFX_TAPI_FXO_DIAL_CFG_SET:
      case IFX_TAPI_FXO_FLASH_CFG_SET:
      case IFX_TAPI_FXO_OSI_CFG_SET:
      case IFX_TAPI_FXO_DIAL_START:
      case IFX_TAPI_FXO_DIAL_STOP:
      case IFX_TAPI_FXO_HOOK_SET:
      case IFX_TAPI_FXO_FLASH_SET:
      case IFX_TAPI_FXO_BAT_GET:
      case IFX_TAPI_FXO_HOOK_GET:
      case IFX_TAPI_FXO_APOH_GET:
      case IFX_TAPI_FXO_RING_GET:
      case IFX_TAPI_FXO_POLARITY_GET:
      case IFX_TAPI_FXO_LINE_MODE_SET:
         ret = TAPI_FXO_Ioctl (pChannel, iocmd, ioarg);
         break;

#ifdef QOS_SUPPORT
      /* QOS services */
      case FIO_QOS_START:
      case FIO_QOS_ACTIVATE:
      case FIO_QOS_STOP:
      case FIO_QOS_CLEAN:
         ret = IFX_TAPI_Qos_Ctrl ((IFX_void_t *)pChannel, iocmd, ioarg);
         break;
#endif /* QOS_SUPPORT */

#ifdef TAPI_AUDIO_CHANNEL
      case IFX_TAPI_AUDIO_VOLUME_SET:
      case IFX_TAPI_AUDIO_ROOM_TYPE_SET:
      case IFX_TAPI_AUDIO_MUTE_SET:
      case IFX_TAPI_AUDIO_MODE_SET:
      case IFX_TAPI_AUDIO_RING_START:
      case IFX_TAPI_AUDIO_RING_STOP:
      case IFX_TAPI_AUDIO_RING_VOLUME_SET:
      case IFX_TAPI_AUDIO_ICA_SET:
         ret = TAPI_IoctlAudioCh (pDrvCtx, pCtx, iocmd, ioarg);
         break;
#endif /* TAPI_AUDIO_CHANNEL */

      /* Operation Control Services */
      case IFX_TAPI_LINE_FEED_SET:
         ret = TAPI_Phone_Set_Linefeed(pChannel, param);
         break;
      /* Operation Control Services */
      case IFX_TAPI_LINE_HOOK_STATUS_GET:
#ifdef TAPI_ONE_DEVNODE
         if (pCtx->bSingleFd)
            ret = TAPI_Phone_Hookstate (pChannel,
                     (IFX_int32_t *) &tmpAdr.param);
         else
#endif /* TAPI_ONE_DEVNODE */
            ret = TAPI_Phone_Hookstate (pChannel, (IFX_int32_t *) ioarg);
         break;
      /* PCM Services */
      case IFX_TAPI_PCM_ACTIVATION_GET:
#ifdef TAPI_ONE_DEVNODE
         if (pCtx->bSingleFd)
            ret = TAPI_Phone_PCM_Get_Activation (pChannel,
                     (IFX_boolean_t*) &tmpAdr.param);
         else
#endif /* TAPI_ONE_DEVNODE */
            ret = TAPI_Phone_PCM_Get_Activation (pChannel,
                     (IFX_boolean_t*) ioarg);
         break;

      case IFX_TAPI_PCM_ACTIVATION_SET:
         ret = TAPI_Phone_PCM_Set_Activation(pChannel, (IFX_uint32_t)param);
         break;

      case IFX_TAPI_PCM_DEC_HP_SET:
         if (ptr_chk(pDrvCtx->COD.DEC_HP_Set, "pDrvCtx->PCM.DEC_HP_Set"))
            retLL = pDrvCtx->PCM.DEC_HP_Set(pChannel->pLLChannel, (IFX_boolean_t)param);
         else
            ret = TAPI_statusLLNotSupp;
         break;

#ifdef TAPI_PACKET
      case IFX_TAPI_PKT_FLUSH:
         {
            IFX_uint8_t fifo_idx = (pCtx->bLinChFd == IFX_TRUE) ?
                                     IFX_TAPI_STREAM_LIN : IFX_TAPI_STREAM_COD;

            ret = IFX_TAPI_UpStreamFifo_Reset(pChannel, fifo_idx);
            break;
         }
#endif /* TAPI_PACKET */
#ifdef TAPI_VOICE
      case IFX_TAPI_PKT_RTCP_STATISTICS_RESET:
         if (pDrvCtx->COD.RTCP_Reset == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.RTCP_Reset (pChannel->pLLChannel);
         break;

      case IFX_TAPI_JB_STATISTICS_RESET:
         if (pDrvCtx->COD.JB_Stat_Reset == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.JB_Stat_Reset (pChannel->pLLChannel);
         break;

      /* Recording Services */
      case IFX_TAPI_ENC_START:
         if (pDrvCtx->COD.ENC_Start == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.ENC_Start (pChannel->pLLChannel);
         break;

      case IFX_TAPI_ENC_STOP:
         if (pDrvCtx->COD.ENC_Stop == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.ENC_Stop (pChannel->pLLChannel);
         break;

      case IFX_TAPI_ENC_HOLD:
         if (pDrvCtx->COD.ENC_Hold == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.ENC_Hold(pChannel->pLLChannel, param);
         break;

      case IFX_TAPI_ENC_VAD_CFG_SET:
         if (pDrvCtx->COD.VAD_Cfg == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.VAD_Cfg (pChannel->pLLChannel, param);
         break;

#ifdef LIN_SUPPORT
      case IFX_TAPI_LIN_ENC_START:
         if (pDrvCtx->LIN.ENC_Start == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
         {
            retLL = pDrvCtx->LIN.ENC_Start (pChannel->pLLChannel);
            if (IFX_SUCCESS == retLL)
               pChannel->bLinChannelActive = IFX_TRUE;
         }

         break;

      case IFX_TAPI_LIN_ENC_STOP:
         if (pDrvCtx->LIN.ENC_Stop == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
         {
            retLL = pDrvCtx->LIN.ENC_Stop (pChannel->pLLChannel);
            if (IFX_SUCCESS == retLL)
               pChannel->bLinChannelActive = IFX_FALSE;
         }
         break;

      case IFX_TAPI_LIN_DEC_START:
         if (pDrvCtx->LIN.DEC_Start == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
         {
            retLL = pDrvCtx->LIN.DEC_Start (pChannel->pLLChannel);
            if (IFX_SUCCESS == retLL)
               pChannel->bLinChannelActive = IFX_TRUE;
         }
         break;

      case IFX_TAPI_LIN_DEC_STOP:
         if (pDrvCtx->LIN.DEC_Stop == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
         {
            retLL = pDrvCtx->LIN.DEC_Stop (pChannel->pLLChannel);
            if (IFX_SUCCESS == retLL)
            {
               pChannel->bLinChannelActive = IFX_FALSE;
               /* clear pending data write request (if any) */
               pChannel->bLinDataRequest = IFX_FALSE;
            }
         }
         break;
#endif /* LIN_SUPPORT */

      case IFX_TAPI_DEC_START:
         if (pDrvCtx->COD.DEC_Start == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.DEC_Start (pChannel->pLLChannel);
         break;

      case IFX_TAPI_DEC_STOP:
         if (pDrvCtx->COD.DEC_Stop == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.DEC_Stop (pChannel->pLLChannel);
         break;

      case IFX_TAPI_COD_DEC_HP_SET:
         if (ptr_chk(pDrvCtx->COD.DEC_HP_Set, "pDrvCtx->COD.DEC_HP_Set"))
            retLL = pDrvCtx->COD.DEC_HP_Set(pChannel->pLLChannel,
                                            (IFX_boolean_t)param);
         else
            ret = TAPI_statusLLNotSupp;
         break;
#endif /* TAPI_VOICE */

      case IFX_TAPI_TEST_HOOKGEN:
         if (ptr_chk(pDrvCtx->ALM.TestHookGen, "ALM.TestHookGen"))
            retLL = pDrvCtx->ALM.TestHookGen (pChannel->pLLChannel,
                                              (IFX_boolean_t) param);
         else
            ret = TAPI_statusLLNotSupp;
         break;

      /* Ring services */
      case IFX_TAPI_RING_START:
         ret = IFX_TAPI_Ring_Start(pChannel);
         break;
      case IFX_TAPI_RING_STOP:
         ret = IFX_TAPI_Ring_Stop(pChannel);
         break;
      case IFX_TAPI_RING_CADENCE_SET:
         ret = IFX_TAPI_Ring_SetCadence(pChannel, (IFX_uint32_t)param);
         break;
      case IFX_TAPI_RING_MAX_SET:
         ret = IFX_TAPI_Ring_SetMaxRings(pChannel, (IFX_uint32_t)param);
         break;
      case IFX_TAPI_RING:
         ret = IFX_TAPI_Ring_DoBlocking(pChannel);
         break;


      case IFX_TAPI_LINE_LEVEL_SET:
         if (pDrvCtx->ALM.Volume_High_Level == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->ALM.Volume_High_Level (pChannel->pLLChannel,
                       param);
         break;

      /* Message Waiting Lamp Services */
      case IFX_TAPI_MWL_ACTIVATION_GET:
         if (ptr_chk(pDrvCtx->ALM.WML_Activation_Get, "ALM.WML_Activation_Get"))
         {
            IFX_TAPI_MWL_ACTIVATION_t *p_tmp;
#ifdef LINUX
            p_tmp = (IFX_TAPI_MWL_ACTIVATION_t *) TAPI_OS_Malloc (sizeof(*p_tmp));
#else
            p_tmp = (IFX_TAPI_MWL_ACTIVATION_t *)ioarg;
#endif /* LINUX */
            retLL = pDrvCtx->ALM.WML_Activation_Get (pChannel->pLLChannel,
                                                     p_tmp);
#ifdef LINUX
            TAPI_OS_CpyKern2Usr ((IFX_void_t*)ioarg, p_tmp, sizeof(*p_tmp));
            TAPI_OS_Free (p_tmp);
#endif /* LINUX */
         }
         else
            ret = TAPI_statusLLNotSupp;
         break;

      case IFX_TAPI_MWL_ACTIVATION_SET:
         if (ptr_chk(pDrvCtx->ALM.WML_Activation_Set, "ALM.WML_Activation_Set"))
         {
            IFX_TAPI_MWL_ACTIVATION_t *p_tmp;
#ifdef LINUX
            p_tmp = (IFX_TAPI_MWL_ACTIVATION_t *) TAPI_OS_Malloc (sizeof(*p_tmp));

            if (TAPI_OS_CpyUsr2Kern (p_tmp, (IFX_void_t*)ioarg,
                                     sizeof(*p_tmp)) == 0)
            {
               ret = TAPI_statusErrKernCpy;
            }
            else
#endif /* LINUX */
            {
#ifndef LINUX
               p_tmp = (IFX_TAPI_MWL_ACTIVATION_t *)ioarg;
#endif /* LINUX */
               retLL = pDrvCtx->ALM.WML_Activation_Set (pChannel->pLLChannel,
                                                     p_tmp);
            }
#ifdef LINUX
            TAPI_OS_Free (p_tmp);
#endif /* LINUX */
         }
         else
            ret = TAPI_statusLLNotSupp;
         break;

       /* Tone Services */
      case IFX_TAPI_TONE_DIALTONE_PLAY:
         ret = TAPI_Phone_Tone_Dial(pChannel);
         break;

      case IFX_TAPI_TONE_RINGBACK_PLAY:
         ret = TAPI_Phone_Tone_Ringback(pChannel);
         break;

      case IFX_TAPI_TONE_BUSY_PLAY:
         ret = TAPI_Phone_Tone_Busy(pChannel);
         break;

      case IFX_TAPI_TONE_LOCAL_PLAY:
         ret = TAPI_Phone_Tone_Play (pChannel,
                                     param, TAPI_TONE_DST_LOCAL);
         break;

      case IFX_TAPI_TONE_NET_PLAY:
         ret = TAPI_Phone_Tone_Play (pChannel,
                                     param, TAPI_TONE_DST_NET);
         break;

      case IFX_TAPI_TONE_LOCAL_STOP:
         ret = TAPI_Phone_Tone_Stop (pChannel,
                                     param, TAPI_TONE_DST_LOCAL);
         break;

      case IFX_TAPI_TONE_NET_STOP:
         ret = TAPI_Phone_Tone_Stop (pChannel,
                                     param, TAPI_TONE_DST_NET);
         break;

#ifdef DECT_SUPPORT
      case IFX_TAPI_TONE_DECT_PLAY:
         ret = TAPI_Phone_Tone_Play (pChannel,
                                     param | IFX_TAPI_TONE_SRC_DECT,
                                     TAPI_TONE_DST_LOCAL);
         break;

      case IFX_TAPI_TONE_DECT_STOP:
         ret = TAPI_DECT_Tone_Stop (pChannel);
         break;
#endif /* DECT_SUPPORT */

#ifdef TAPI_FAX_T38
      case IFX_TAPI_T38_STOP:
         if (pDrvCtx->COD.T38_Datapump_Disable == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.T38_Datapump_Disable (pChannel->pLLChannel);
         break;
#endif /* TAPI_FAX_T38 */

#ifdef TAPI_ANNOUNCEMENTS
      case IFX_TAPI_COD_ANNOUNCE_STOP:
         ret = IFX_TAPI_Ann_Stop(pChannel);
         break;
#endif /* TAPI_ANNOUNCEMENTS */

      case IFX_TAPI_PKT_EV_OOB_DTMF_SET:
#ifdef TAPI_EXT_KEYPAD
         if(pChannel->nChannel > 0 )
         {
            ret = TAPI_statusInvalidCh;
         }
         else
         {
           /* Overwrite lower 7 bits of status byte with passed argument
              and ensure that local play flag remains unchanged. */
            pChannel->nDtmfInfo = (pChannel->nDtmfInfo & DTMF_EV_LOCAL_PLAY) |
                                   ((IFX_uint8_t)ioarg & ~DTMF_EV_LOCAL_PLAY );
         }
#else
         if (pDrvCtx->SIG.DTMFD_OOB == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->SIG.DTMFD_OOB (pChannel->pLLChannel, param);
#endif
         break;

      case IFX_TAPI_PKT_EV_OOB_MFTD_SET:
         if (ptr_chk(pDrvCtx->SIG.MFTD_OOB, "pDrvCtx->SIG.MFTD_OOB"))
            retLL = pDrvCtx->SIG.MFTD_OOB (pChannel->pLLChannel,
                     ((IFX_TAPI_PKT_EV_OOB_MFTD_t *)ioarg)->nTransmitMode);
         else
            ret = TAPI_statusLLNotSupp;
         break;

#ifdef TAPI_CID
      /* Caller ID Reception service */
      case IFX_TAPI_CID_RX_START:
         ret = TAPI_Phone_CidRx_Start (pChannel,
                  (IFX_TAPI_CID_HOOK_MODE_t) param);
         break;

      case IFX_TAPI_CID_RX_STOP:
         ret = TAPI_Phone_CidRx_Stop (pChannel);
         break;

      case IFX_TAPI_CID_TX_INFO_STOP:
         if (ptr_chk(pDrvCtx->SIG.CID_TX_Stop, "pDrvCtx->SIG.CID_TX_Stop"))
         {
            ret = TAPI_Phone_CID_Stop_Tx(pChannel);
         }
         else
         {
            /* Abort CID sending not supported by LL driver */
            ret = TAPI_statusLLNotSupp;
         }
         break;
#endif /* TAPI_CID */

      case IFX_TAPI_TONE_CPTD_STOP:
         ret = TAPI_Phone_DetectToneStop(pChannel);
         break;

      case IFX_TAPI_ENC_AGC_ENABLE:
         if (pDrvCtx->COD.AGC_Enable == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->COD.AGC_Enable(pChannel->pLLChannel, param);
         break;

      case IFX_TAPI_ENC_ROOM_NOISE_DETECT_STOP:
         if (ptr_chk(pDrvCtx->COD.ENC_RoomNoise, "pDrvCtx->COD.ENC_RoomNoise"))
            retLL = pDrvCtx->COD.ENC_RoomNoise(pChannel->pLLChannel,
                                               IFX_FALSE, 0, 0, 0);
         else
            ret = TAPI_statusLLNotSupp;
         break;

#ifdef TAPI_CONT_MEASUREMENT
      case IFX_TAPI_CONTMEASUREMENT_REQ:
         if (pDrvCtx->ALM.ContMeasReq == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->ALM.ContMeasReq (pChannel->pLLChannel);
         break;
      case IFX_TAPI_CONTMEASUREMENT_RESET:
         if (pDrvCtx->ALM.ContMeasReset == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->ALM.ContMeasReset (pChannel->pLLChannel);
         break;
#endif /* TAPI_CONT_MEASUREMENT */

#ifdef TAPI_PHONE_DETECTION
      case IFX_TAPI_LINE_MEASURE_CAPACITANCE_START:
         if (pDrvCtx->ALM.CapMeasStart == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->ALM.CapMeasStart (pChannel->pLLChannel);
         break;
      case IFX_TAPI_LINE_MEASURE_CAPACITANCE_STOP:
         if (pDrvCtx->ALM.CapMeasStop == IFX_NULL)
            ret = TAPI_statusLLNotSupp;
         else
            retLL = pDrvCtx->ALM.CapMeasStop (pChannel->pLLChannel);
         break;
#endif /* TAPI_PHONE_DETECTION */

      /* Tone Services */
      case IFX_TAPI_TONE_ON_TIME_GET:
      case IFX_TAPI_TONE_OFF_TIME_GET:
      case IFX_TAPI_TONE_ON_TIME_SET:
      case IFX_TAPI_TONE_OFF_TIME_SET:
      /* Dial Services */
         ret = TAPI_statusNotSupported;
         /*lint -fallthrough*/
      default:
         bHandled = IFX_FALSE;
         break;
   }
   if (GET_HL_ERROR(ret))
   {
      /* The HL error is already reported in the correct position. */
      TAPI_OS_MutexRelease (&pChannel->semTapiChSingleIoctlAccess);
      return ret;
   }
   if (!TAPI_SUCCESS(ret))
   {
      /* The errorcode needs to be built from HL and LL errorcode. */
      TAPI_OS_MutexRelease (&pChannel->semTapiChSingleIoctlAccess);
      RETURN_STATUS(ret, retLL);
   }
   if (!TAPI_SUCCESS(retLL))
   {
      /* Just the LL reported an error so add an generic HL errorcode. */
      TAPI_OS_MutexRelease (&pChannel->semTapiChSingleIoctlAccess);
      /* errmsg: LL driver returned an error */
      RETURN_STATUS (TAPI_statusLLFailed, retLL);
   }
   if (bHandled)
   {
      TAPI_OS_MutexRelease (&pChannel->semTapiChSingleIoctlAccess);
#ifdef TAPI_ONE_DEVNODE
      if (pCtx->bSingleFd)
      {
         /* using a new strucutre, get the context and pass again
            the one argument */
         TAPI_OS_CpyKern2Usr ((IFX_char_t*)ioarg, (IFX_uint8_t*)&tmpAdr,
                               sizeof (IFX_TAPI_IOCTL_t));
      }
#endif /* TAPI_ONE_DEVNODE */
      return ret;
   }

#ifdef LINUX
   ret = TAPI_OS_IoctlCh (pDrvCtx, pCtx, pChannel, iocmd, ioarg, &retLL);
#else

   switch(iocmd)
   {
   /* Ringing Services */
      ON_IOCTLR1 (IFX_TAPI_RING_CFG_GET, IFX_TAPI_Ring_GetConfig, pChannel, IFX_TAPI_RING_CFG_t*);
/*
   case IFX_TAPI_RING_CFG_GET:
      return IFX_TAPI_Ring_GetConfig(pChannel, (IFX_TAPI_RING_CFG_t*)ioarg);
*/
      ON_IOCTLW2 (IFX_TAPI_RING_CFG_SET, IFX_TAPI_Ring_SetConfig, pChannel, IFX_TAPI_RING_CFG_t *);
/*
   case IFX_TAPI_RING_CFG_SET:
      return IFX_TAPI_Ring_SetConfig(pChannel, (IFX_TAPI_RING_CFG_t *) ioarg);
*/
      /* simple parameter without result */
/*
      ON_IOCTLW2 (IFX_TAPI_RING_CADENCE_SET, IFX_TAPI_Ring_SetCadence, pChannel, IFX_uint32_t);
   case IFX_TAPI_RING_CADENCE_SET:
      return (IFX_TAPI_Ring_SetCadence (pChannel, param));
*/
   case IFX_TAPI_RING_CADENCE_HR_SET:
      {
         IFX_TAPI_RING_CADENCE_t *pCadence = (IFX_TAPI_RING_CADENCE_t*) ioarg;
         ret = (IFX_TAPI_Ring_SetCadenceHighRes (pChannel, pCadence));
      }
      break;
   case IFX_TAPI_RING_START:
      /* no parameter without result */
      ret = IFX_TAPI_Ring_Start (pChannel);
      break;

   /* Operation Control Services */
   case IFX_TAPI_LINE_FEED_SET:
      /* simple parameter without result */
      ret = TAPI_Phone_Set_Linefeed(pChannel, param);
      break;

   case IFX_TAPI_LINE_TYPE_SET:
      ret = TAPI_Phone_Set_LineType(pChannel, (IFX_TAPI_LINE_TYPE_CFG_t *)ioarg);
      break;

   /* PCM Services */
   case IFX_TAPI_PCM_IF_CFG_SET:
      ret = TAPI_Phone_PCM_IF_Set_Config(pChannel->pTapiDevice,
                                        (IFX_TAPI_PCM_IF_CFG_t *) ioarg);
      break;

   case IFX_TAPI_PCM_CFG_SET:
      {
         IFX_TAPI_PCM_CFG_t *pPCMConfig = (IFX_TAPI_PCM_CFG_t*) ioarg;
         ret = (TAPI_Phone_PCM_Set_Config(pChannel, pPCMConfig));
      }
      break;
   case IFX_TAPI_PCM_CFG_GET:
      {
         IFX_TAPI_PCM_CFG_t *pPCMConfig = (IFX_TAPI_PCM_CFG_t*) ioarg;
         TAPI_Phone_PCM_Get_Config (pChannel, pPCMConfig);
         ret = TAPI_statusOk;
      }
      break;

   case IFX_TAPI_PCM_ACTIVATION_SET:
      TAPI_Phone_PCM_Set_Activation(pChannel, (IFX_uint32_t)ioarg);
      ret = TAPI_statusOk;
      break;

#ifdef TAPI_HDLC
   case IFX_TAPI_PCM_D_CH_CFG_SET:
      {
         IFX_TAPI_PCM_D_CH_CFG_t *pPcmDChCfg = (IFX_TAPI_PCM_D_CH_CFG_t*) ioarg;
         ret = TAPI_Phone_PCM_DCh_Set(pChannel, pPcmDChCfg);
      }
      break;
#endif /* TAPI_HDLC */

   case IFX_TAPI_PCM_LOOP_CFG_SET:
      {
         IFX_TAPI_PCM_LOOP_CFG_t *pPcmLoop = (IFX_TAPI_PCM_LOOP_CFG_t*) ioarg;
         ret = TAPI_Phone_PCM_Loop_Set(pChannel, pPcmLoop);
      }
      break;

   case IFX_TAPI_TONE_LEVEL_SET:
      {
         IFX_TAPI_PREDEF_TONE_LEVEL_t *pToneLevel = (IFX_TAPI_PREDEF_TONE_LEVEL_t *) ioarg;
         ret = TAPI_Phone_Tone_Set_Level(pChannel, pToneLevel);
      }
      break;

   case IFX_TAPI_TONE_DIALTONE_PLAY:
      ret = TAPI_Phone_Tone_Dial(pChannel);
      break;

   case IFX_TAPI_TONE_RINGBACK_PLAY:
      ret = TAPI_Phone_Tone_Ringback(pChannel);
      break;

   case IFX_TAPI_TONE_BUSY_PLAY:
      ret = TAPI_Phone_Tone_Busy(pChannel);
      break;

   case IFX_TAPI_TONE_STOP:
      ret = TAPI_Phone_Tone_Stop (pChannel,
               (IFX_int32_t) ioarg, TAPI_TONE_DST_DEFAULT);
      break;

   /* Dial Services */
   case IFX_TAPI_LINE_HOOK_VT_SET:
      ret = IFX_TAPI_Dial_SetValidationTime(pChannel,
         (IFX_TAPI_LINE_HOOK_VT_t *)ioarg);
      break;

   case IFX_TAPI_PKT_EV_OOB_DTMF_SET:
      retLL = pDrvCtx->SIG.DTMFD_OOB (pChannel->pLLChannel,
         (IFX_TAPI_PKT_EV_OOB_t) ioarg);
      break;

   /* Miscellaneous Services */
   case IFX_TAPI_ENC_VAD_CFG_SET:
      /** \todo check if this is ok.
      ret = TAPI_LL_Phone_VAD(pDrvCtx, pChannel, ioarg);*/
      retLL = pDrvCtx->COD.VAD_Cfg (pChannel->pLLChannel, (IFX_int32_t)ioarg);
      break;

#ifdef TAPI_VOICE
   case IFX_TAPI_ENC_CFG_SET:
      if (pDrvCtx->COD.ENC_Cfg == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->COD.ENC_Cfg(pChannel->pLLChannel,
                     ((IFX_TAPI_ENC_CFG_SET_t*) ioarg)->nEncType,
                     ((IFX_TAPI_ENC_CFG_SET_t*) ioarg)->nFrameLen,
                     ((IFX_TAPI_ENC_CFG_SET_t*) ioarg)->AAL2BitPack);
      break;

   case IFX_TAPI_DEC_CFG_SET:
      if (pDrvCtx->COD.DEC_Cfg == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->COD.DEC_Cfg(pChannel->pLLChannel,
                     ((IFX_TAPI_DEC_CFG_t*) ioarg)->AAL2BitPack,
                     ((IFX_TAPI_DEC_CFG_t*) ioarg)->Plc);
      break;
#endif /* TAPI_VOICE */

#ifdef TAPI_METERING
   case IFX_TAPI_METER_BURST:
      ret = TAPI_Phone_Meter_Burst (pChannel, (IFX_TAPI_METER_BURST_t *) ioarg);
      break;
#endif /* TAPI_METERING */

   case IFX_TAPI_CALIBRATION_START:
      if (pDrvCtx->ALM.Calibration_Start == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->ALM.Calibration_Start (pChannel->pLLChannel);
      break;
   case IFX_TAPI_CALIBRATION_STOP:
      if (pDrvCtx->ALM.Calibration_Stop == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->ALM.Calibration_Stop (pChannel->pLLChannel);
      break;
   case IFX_TAPI_CALIBRATION_CFG_SET:
      if (pDrvCtx->ALM.Calibration_Set == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->ALM.Calibration_Set (pChannel->pLLChannel,
                                (IFX_TAPI_CALIBRATION_CFG_t*) ioarg);
      break;
   case IFX_TAPI_CALIBRATION_CFG_GET:
      if (pDrvCtx->ALM.Calibration_Get == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->ALM.Calibration_Get (pChannel->pLLChannel,
                                (IFX_TAPI_CALIBRATION_CFG_t*) ioarg);
      break;
   case IFX_TAPI_CALIBRATION_RESULTS_GET:
      if (pDrvCtx->ALM.Calibration_Results_Get == IFX_NULL)
         ret = TAPI_statusLLNotSupp;
      else
         retLL = pDrvCtx->ALM.Calibration_Results_Get (pChannel->pLLChannel,
                                (IFX_TAPI_CALIBRATION_CFG_t*) ioarg);
      break;
   /* Lec Configuration */
   case IFX_TAPI_WLEC_PHONE_CFG_SET:
      ret = TAPI_Phone_LecMode_Alm_Set (pChannel, (IFX_TAPI_WLEC_CFG_t *)ioarg);
      break;
   case IFX_TAPI_WLEC_PHONE_CFG_GET:
      ret = TAPI_Phone_LecMode_Alm_Get (pChannel, (IFX_TAPI_WLEC_CFG_t *)ioarg);
      break;
   case IFX_TAPI_WLEC_PCM_CFG_SET:
      ret = TAPI_Phone_LecMode_Pcm_Set (pChannel, (IFX_TAPI_WLEC_CFG_t *)ioarg);
      break;
   case IFX_TAPI_WLEC_PCM_CFG_GET:
      ret = TAPI_Phone_LecMode_Pcm_Get (pChannel, (IFX_TAPI_WLEC_CFG_t *)ioarg);
      break;
   case IFX_TAPI_DTMF_RX_CFG_SET:
      ret = TAPI_Phone_Dtmf_RxCoeff_Cfg (pChannel, IFX_FALSE,
                                               (IFX_TAPI_DTMF_RX_CFG_t *)ioarg);
      break;
   case IFX_TAPI_DTMF_RX_CFG_GET:
      ret = TAPI_Phone_Dtmf_RxCoeff_Cfg (pChannel, IFX_TRUE,
                                               (IFX_TAPI_DTMF_RX_CFG_t *)ioarg);
      break;
#ifdef TAPI_CID
   /* Caller ID Transmission service */
   case IFX_TAPI_CID_CFG_SET:
      ret = TAPI_Phone_CID_SetConfig (pChannel, (IFX_TAPI_CID_CFG_t *)ioarg);
      break;
   case IFX_TAPI_CID_TX_INFO_START:
      ret = TAPI_Phone_CID_Info_Tx (pChannel, (IFX_TAPI_CID_MSG_t *)ioarg);
      break;
   case IFX_TAPI_CID_TX_SEQ_START:
      ret = TAPI_Phone_CID_Seq_Tx (pChannel, (IFX_TAPI_CID_MSG_t *)ioarg);
      break;

   case IFX_TAPI_CID_TX_INFO_STOP:
      if (ptr_chk(pDrvCtx->SIG.CID_TX_Stop, "pDrvCtx->SIG.CID_TX_Stop"))
      {
         ret = TAPI_Phone_CID_Stop_Tx(pChannel);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

   /* Caller ID Reception service */
   case IFX_TAPI_CID_RX_STATUS_GET:
      ret = (TAPI_Phone_CidRx_Status (pChannel, (IFX_TAPI_CID_RX_STATUS_t *) ioarg));
      break;
   case IFX_TAPI_CID_RX_DATA_GET:
      ret = (TAPI_Phone_Get_CidRxData (pChannel, (IFX_TAPI_CID_RX_DATA_t *) ioarg));
      break;
#endif /* TAPI_CID */
#ifdef TAPI_VOICE
   /* Connection Services*/
   case IFX_TAPI_MAP_DATA_ADD:
      ret = TAPI_Data_Channel_Add (pChannel, (IFX_TAPI_MAP_DATA_t *)ioarg);
      break;
   case IFX_TAPI_MAP_DATA_REMOVE:
      ret = TAPI_Data_Channel_Remove (pChannel, (IFX_TAPI_MAP_DATA_t *)ioarg);
      break;

   case IFX_TAPI_MAP_PHONE_ADD:
      if (ptr_chk(pDrvCtx->CON.Module_Connect, ""))
      {
         retLL = pDrvCtx->CON.Module_Connect (pChannel->pLLChannel,
                                    IFX_TAPI_MAP_TYPE_PHONE,
                                    ((IFX_TAPI_MAP_PHONE_t *)ioarg)->nPhoneCh,
                                    ((IFX_TAPI_MAP_PHONE_t *)ioarg)->nChType);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_MAP_PHONE_REMOVE:
      if (ptr_chk(pDrvCtx->CON.Module_Disconnect, ""))
      {
         retLL = pDrvCtx->CON.Module_Disconnect (pChannel->pLLChannel,
                                    IFX_TAPI_MAP_TYPE_PHONE,
                                    ((IFX_TAPI_MAP_PHONE_t *)ioarg)->nPhoneCh,
                                    ((IFX_TAPI_MAP_PHONE_t *)ioarg)->nChType);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_MAP_PCM_ADD:
      if (ptr_chk(pDrvCtx->CON.Module_Connect, ""))
      {
         retLL = pDrvCtx->CON.Module_Connect (pChannel->pLLChannel,
                                    IFX_TAPI_MAP_TYPE_PCM,
                                    ((IFX_TAPI_MAP_PCM_t *)ioarg)->nDstCh,
                                    ((IFX_TAPI_MAP_PCM_t *)ioarg)->nChType);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_MAP_PCM_REMOVE:
      if (ptr_chk(pDrvCtx->CON.Module_Disconnect, ""))
      {
         retLL = pDrvCtx->CON.Module_Disconnect (pChannel->pLLChannel,
                                    IFX_TAPI_MAP_TYPE_PCM,
                                    ((IFX_TAPI_MAP_PCM_t *)ioarg)->nDstCh,
                                    ((IFX_TAPI_MAP_PCM_t *)ioarg)->nChType);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_PKT_RTP_CFG_SET:
      retLL = pDrvCtx->COD.RTP_Cfg (pChannel->pLLChannel, (IFX_TAPI_PKT_RTP_CFG_t*)ioarg);
      break;
   case IFX_TAPI_PKT_RTP_PT_CFG_SET:
      retLL = pDrvCtx->COD.RTP_PayloadTable_Cfg (pChannel->pLLChannel, (IFX_TAPI_PKT_RTP_PT_CFG_t *)ioarg);
      break;
   case IFX_TAPI_PKT_RTCP_STATISTICS_GET:
      retLL = pDrvCtx->COD.RTCP_Get (pChannel->pLLChannel, (IFX_TAPI_PKT_RTCP_STATISTICS_t *)ioarg);
      break;
   case IFX_TAPI_PKT_RTCP_STATISTICS_RESET:
      retLL = pDrvCtx->COD.RTCP_Reset (pChannel->pLLChannel);
      break;
   case IFX_TAPI_JB_CFG_SET:
      retLL = pDrvCtx->COD.JB_Cfg (pChannel->pLLChannel, (IFX_TAPI_JB_CFG_t *)ioarg);
      break;
   case IFX_TAPI_JB_STATISTICS_GET:
      retLL = pDrvCtx->COD.JB_Stat_Get (pChannel->pLLChannel, (IFX_TAPI_JB_STATISTICS_t *)ioarg);
      break;
   case IFX_TAPI_JB_STATISTICS_RESET:
      retLL = pDrvCtx->COD.JB_Stat_Reset (pChannel->pLLChannel);
      break;
#if (VIN_CFG_FEATURES & VIN_FEAT_PACKET_AAL)
   case IFX_TAPI_PKT_AAL_CFG_SET:
      ret = TAPI_LL_Phone_Aal_Set (pDrvCtx, pChannel, (IFX_TAPI_PCK_AAL_CFG_t *)ioarg);
      break;
   case IFX_TAPI_PKT_AAL_PROFILE_SET:
      ret = TAPI_LL_Phone_AalProfile (pDrvCtx, pChannel, (IFX_TAPI_PCK_AAL_PROFILE_t *)ioarg);
      break;
#endif /* (VIN_CFG_FEATURES & VIN_FEAT_PACKET_AAL) */
   /* Recording services */
   case IFX_TAPI_ENC_START:
      retLL = pDrvCtx->COD.ENC_Start (pChannel->pLLChannel);
      break;
   case IFX_TAPI_ENC_STOP:
      retLL = pDrvCtx->COD.ENC_Stop (pChannel->pLLChannel);
      break;
   case IFX_TAPI_ENC_HOLD:
   {
      if (pDrvCtx->COD.ENC_Hold == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.ENC_Hold(pChannel->pLLChannel,
                                     (IFX_int32_t) ioarg);
      }
      break;
   }
   /* Play Services*/
   /* not used */
   /*
   case IFX_TAPI_DEC_TYPE_SET:
      ret =(TAPI_Phone_Set_Play_Codec(pDrvCtx, pChannel, ioarg));
      break;
   */
   case IFX_TAPI_DEC_START:
      retLL = pDrvCtx->COD.DEC_Start (pChannel->pLLChannel);
      break;
   case IFX_TAPI_DEC_STOP:
      retLL = pDrvCtx->COD.DEC_Stop (pChannel->pLLChannel);
      break;
#endif /* TAPI_VOICE */

#ifdef TAPI_FAX_T38
      /* Fax Services */
      case IFX_TAPI_T38_MOD_START:
         if (pDrvCtx->COD.T38_Mod_Enable == IFX_NULL)
         {
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            retLL = pDrvCtx->COD.T38_Mod_Enable(pChannel->pLLChannel,
                                              (IFX_TAPI_T38_MOD_DATA_t *)ioarg);
         }
         break;

      case IFX_TAPI_T38_DEMOD_START:
         if (pDrvCtx->COD.T38_DeMod_Enable == IFX_NULL)
         {
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            retLL = pDrvCtx->COD.T38_DeMod_Enable(pChannel->pLLChannel,
                                            (IFX_TAPI_T38_DEMOD_DATA_t *)ioarg);
         }
         break;

      case IFX_TAPI_T38_STATUS_GET:
         if (pDrvCtx->COD.T38_Status_Get == IFX_NULL)
         {
            ret = TAPI_statusLLNotSupp;
         }
         else
         {
            retLL = pDrvCtx->COD.T38_Status_Get(pChannel->pLLChannel,
                                                (IFX_TAPI_T38_STATUS_t *)ioarg);
         }
         break;
#endif /* TAPI_FAX_T38 */

#ifdef TAPI_FAX_T38_FW
   /* Fax Services */
   case IFX_TAPI_T38_CFG_GET:
   {
      if (pDrvCtx->COD.FAX_Cfg_Get == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_Cfg_Get (pChannel->pLLChannel,
                                           (IFX_TAPI_T38_FAX_CFG_t *)ioarg);
      }
      break;
   }
   case IFX_TAPI_T38_CFG_SET:
   {
      if (pDrvCtx->COD.FAX_Cfg_Set == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_Cfg_Set (pChannel->pLLChannel,
                                           (IFX_TAPI_T38_FAX_CFG_t *)ioarg);
      }
      break;
   }
   case IFX_TAPI_T38_FDP_CFG_GET:
   {
      if (pDrvCtx->COD.FAX_FDP_Cfg_Get == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_FDP_Cfg_Get (pChannel->pLLChannel,
                                               (IFX_TAPI_T38_FDP_CFG_t *)ioarg);
      }
      break;
   }
   case IFX_TAPI_T38_FDP_CFG_SET:
   {
      if (pDrvCtx->COD.FAX_FDP_Cfg_Set == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_FDP_Cfg_Set (pChannel->pLLChannel,
                                               (IFX_TAPI_T38_FDP_CFG_t *)ioarg);
      }
      break;
   }
   case IFX_TAPI_T38_SESS_START:
   {
      if (pDrvCtx->COD.FAX_Start == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_Start (pChannel->pLLChannel,
                                         (IFX_TAPI_T38_SESS_CFG_t *)ioarg);
      }
      break;
   }
   case IFX_TAPI_T38_SESS_STATISTICS_GET:
   {
      if (pDrvCtx->COD.FAX_Stat_Get == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_Stat_Get (pChannel->pLLChannel,
                                       (IFX_TAPI_T38_SESS_STATISTICS_t *)ioarg);
      }
      break;
   }
   case IFX_TAPI_T38_SESS_STOP:
   {
      if (pDrvCtx->COD.FAX_Stop == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_Stop (pChannel->pLLChannel);
      }
      break;
   }
   case IFX_TAPI_T38_TRACE_SET:
   {
      if (pDrvCtx->COD.FAX_TraceSet == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.FAX_TraceSet (pChannel->pLLChannel,
                                            (IFX_TAPI_T38_TRACE_CFG_t *)ioarg);
      }
      break;
   }
#endif /* TAPI_FAX_T38_FW */

#ifdef TAPI_ANNOUNCEMENTS
   case IFX_TAPI_COD_ANNOUNCE_CFG_SET:
      ret = IFX_TAPI_Ann_Cfg (pChannel, (IFX_TAPI_COD_ANNOUNCE_CFG_t *)ioarg);
      break;
   case IFX_TAPI_COD_ANNOUNCE_START:
      ret = IFX_TAPI_Ann_Start (pChannel,
         (IFX_TAPI_COD_ANNOUNCE_START_t*)ioarg);
      break;
   case IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE:
      ret = IFX_TAPI_Ann_Free (pChannel,
         (IFX_TAPI_COD_ANNOUNCE_BUFFER_FREE_t*)ioarg);
      break;
#endif /* TAPI_ANNOUNCEMENTS */

   case IFX_TAPI_PHONE_VOLUME_SET:
      retLL = pDrvCtx->ALM.Volume_Set (pChannel->pLLChannel,
                                     (IFX_TAPI_LINE_VOLUME_t *)ioarg);
      break;

   case IFX_TAPI_PCM_VOLUME_SET:
      retLL = pDrvCtx->PCM.Volume_Set (pChannel->pLLChannel,
                                     (IFX_TAPI_LINE_VOLUME_t *)ioarg);
      break;
   case IFX_TAPI_COD_VOLUME_SET:
      retLL = pDrvCtx->COD.Volume_Set (pChannel->pLLChannel,
                                       (IFX_TAPI_PKT_VOLUME_t *)ioarg);
      break;

   case IFX_TAPI_SIG_DETECT_ENABLE:
      if (pDrvCtx->SIG.DetectEnable != IFX_NULL)
         retLL = pDrvCtx->SIG.DetectEnable (pChannel->pLLChannel,
                                            (IFX_TAPI_SIG_DETECTION_t *)ioarg);
      else
        ret = TAPI_statusLLNotSupp;
      break;
   case IFX_TAPI_SIG_DETECT_DISABLE:
      if (pDrvCtx->SIG.DetectDisable != IFX_NULL)
         retLL = pDrvCtx->SIG.DetectDisable (pChannel->pLLChannel,
                                             (IFX_TAPI_SIG_DETECTION_t *)ioarg);
      else
         ret = TAPI_statusLLNotSupp;
      break;
   case IFX_TAPI_TONE_CPTD_START:
      ret = TAPI_Phone_DetectToneStart(pChannel, (IFX_TAPI_TONE_CPTD_t*) ioarg);
   break;
   case IFX_TAPI_TONE_CPTD_STOP:
      ret = TAPI_Phone_DetectToneStop(pChannel);
      break;

   ON_IOCTL_LLFUNC (IFX_TAPI_ENC_AGC_CFG, pDrvCtx->COD.AGC_Cfg,
                    ((IFX_TAPI_ENC_AGC_CFG_t *) ioarg));

/*
   case IFX_TAPI_ENC_AGC_CFG:
   {
      if (pDrvCtx->COD.AGC_Cfg == IFX_NULL)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
               ("TAPI: pDrvCtx->COD.AGC_Cfg is NULL. "
               "(File: %s, line: %d)\n", __FILE__, __LINE__));

         ret = TAPI_statusErr;
      }
      else
      {
         retLL = pDrvCtx->COD.AGC_Cfg(pChannel->pLLChannel,
                                    (IFX_TAPI_ENC_AGC_CFG_t *) ioarg);
      }

      break;
   }
*/
   case IFX_TAPI_ENC_AGC_ENABLE:
   {
      if (pDrvCtx->COD.AGC_Enable == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->COD.AGC_Enable(pChannel->pLLChannel,
                                       (IFX_TAPI_ENC_AGC_MODE_t) ioarg);
      }
      break;
   }

   case IFX_TAPI_ENC_ROOM_NOISE_DETECT_START:
      if (ptr_chk(pDrvCtx->COD.ENC_RoomNoise, "pDrvCtx->COD.ENC_RoomNoise"))
      {
         retLL = pDrvCtx->COD.ENC_RoomNoise(pChannel->pLLChannel, IFX_TRUE,
                  ((IFX_TAPI_ENC_ROOM_NOISE_DETECT_t *)ioarg)->nThreshold,
                  ((IFX_TAPI_ENC_ROOM_NOISE_DETECT_t *)ioarg)->nVoicePktCnt,
                  ((IFX_TAPI_ENC_ROOM_NOISE_DETECT_t *)ioarg)->nSilencePktCnt);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

   case IFX_TAPI_ENC_ROOM_NOISE_DETECT_STOP:
      if (ptr_chk(pDrvCtx->COD.ENC_RoomNoise, "pDrvCtx->COD.ENC_RoomNoise"))
      {
         retLL = pDrvCtx->COD.ENC_RoomNoise(pChannel->pLLChannel,
                                          IFX_FALSE, 0, 0, 0);
      }
      else
      {
         ret = TAPI_statusLLNotSupp;
      }
      break;

#ifdef KPI_SUPPORT
   /* Kernel Packet Interface configuration */
   case IFX_TAPI_KPI_CH_CFG_SET:
      /* KPI is a TAPI functionality so it works on TAPI channels */
      ret = IFX_TAPI_KPI_ChCfgSet (pChannel, (IFX_TAPI_KPI_CH_CFG_t *) ioarg);
      break;
#endif /* KPI_SUPPORT */

#ifdef TAPI_GR909
   case IFX_TAPI_GR909_START:
      if (pDrvCtx->ALM.GR909_Start == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->ALM.GR909_Start(pChannel->pLLChannel,
                                              (IFX_TAPI_GR909_START_t* )ioarg);
      }
      break;
   case IFX_TAPI_GR909_STOP:
      if (pDrvCtx->ALM.GR909_Stop == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->ALM.GR909_Stop(pChannel->pLLChannel);
      }
      break;
   case IFX_TAPI_GR909_RESULT:
      if (pDrvCtx->ALM.GR909_Result == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->ALM.GR909_Result (pChannel->pLLChannel,
                                             (IFX_TAPI_GR909_RESULT_t* )ioarg);
      }
      break;
#endif /* TAPI_GR909 */

#ifdef TAPI_CONT_MEASUREMENT
   case IFX_TAPI_CONTMEASUREMENT_GET:
      if (pDrvCtx->ALM.ContMeasGet == IFX_NULL)
      {
         ret = TAPI_statusLLNotSupp;
      }
      else
      {
         retLL = pDrvCtx->ALM.ContMeasGet (pChannel->pLLChannel,
                                       (IFX_TAPI_CONTMEASUREMENT_GET_t* )ioarg);
      }
      break;
#endif /* TAPI_CONT_MEASUREMENT */

   /* TEST services */
   ON_IOCTL_LLFUNC (IFX_TAPI_TEST_LOOP, pDrvCtx->ALM.TestLoop,
                    ((IFX_TAPI_TEST_LOOP_t *) ioarg));

   default:
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,("TAPI: unknown ioctl 0x%x\n",iocmd));
      ret = TAPI_statusInvalidIoctl;
   }
#endif /* LINUX */

   TAPI_OS_MutexRelease (&pChannel->semTapiChSingleIoctlAccess);

   /* Handle return code */
   if (!TAPI_SUCCESS(ret))
   {
      /* The errorcode needs to be built from HL and LL errorcode. */
      RETURN_STATUS(ret, retLL);
   }
   if (!TAPI_SUCCESS(retLL))
   {
      /* Just the LL reported an error so add an generic HL errorcode. */
      /* errmsg: LL driver returned an error */
      RETURN_STATUS (TAPI_statusLLFailed, retLL);
   }
   /* no error occured so a return will do just fine */
   return ret;
}


#ifdef TAPI_AUDIO_CHANNEL
/**
   TAPI Channel audio specific IOCTL handling

   \param  pDrvCtx      Pointer to device driver context.
   \param  pCtx         Pointer to IOCTL context.
   \param  iocmd        IOCTL command.
   \param  ioarg        IOCTL argument.

   \return
   - TAPI_statusOk if successful
   - TAPI_statusErr in case of error
*/
static IFX_int32_t TAPI_IoctlAudioCh (IFX_TAPI_DRV_CTX_t*  pDrvCtx,
                                      IFX_TAPI_ioctlCtx_t* pCtx,
                                      IFX_uint32_t iocmd,
                                      IFX_ulong_t ioarg)
{
   TAPI_DEV     *pTapiDev = pCtx->pTapiDev;
   TAPI_CHANNEL *pTapiCh;
   IFX_int32_t    ret     = IFX_SUCCESS;

   /* channel context is taken from the first channel */
   pTapiCh = &(pTapiDev->pTapiChanelArray[0]);

   /* Audio chnl Services */
   switch (iocmd)
   {
      case IFX_TAPI_AUDIO_AFE_CFG_SET :
      {
         if (pDrvCtx->AUDIO.AFE_Cfg_Set == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.AFE_Cfg_Set (pTapiCh->pLLChannel,
                           (IFX_TAPI_AUDIO_AFE_CFG_SET_t*) ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_VOLUME_SET :
      {
         if (pDrvCtx->AUDIO.Volume_Set == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Volume_Set(pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_ROOM_TYPE_SET :
      {
         if (pDrvCtx->AUDIO.Room_Set == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Room_Set (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_MUTE_SET :
      {
         if (pDrvCtx->AUDIO.Mute_Set == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Mute_Set (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_MODE_SET :
      {
         if (pDrvCtx->AUDIO.Mode_Set == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Mode_Set (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_RING_START :
      {
         break;
      }
      case IFX_TAPI_AUDIO_RING_STOP :
      {
         if (pDrvCtx->AUDIO.Ring_Stop == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Ring_Stop (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_RING_VOLUME_SET :
      {
         if (pDrvCtx->AUDIO.Ring_Volume_Set == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Ring_Volume_Set (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_ICA_SET :
      {
         if (pDrvCtx->AUDIO.Incall_Anouncement == IFX_NULL)
            RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Incall_Anouncement (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
      case IFX_TAPI_AUDIO_TEST_SET:
      {
         if (pDrvCtx->AUDIO.Test_Mode_Set == IFX_NULL)
           RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
         ret = pDrvCtx->AUDIO.Test_Mode_Set (pTapiCh->pLLChannel, (IFX_uint32_t)ioarg);
         break;
      }
   }

   return ret;
}
#endif /* TAPI_AUDIO_CHANNEL */

/*lint -restore */
