/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_opcontrol.c
   TAPI Operation Control Services.

   \remarks
   All operations done by functions in this module are phone
   related and assumes a phone channel file descriptor.
   Caller of anyone of the functions must make sure that a phone
   channel is used. In case data channel functions are invoked here,
   an instance of the data channel must be passed.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_tapi.h"
#include "drv_tapi_ll_interface.h"
#include "drv_tapi_errno.h"


/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */
#define IFX_TAPI_PWR_SAVE_CHK_TIME_MS 5000 /* milliseconds */


/* ============================= */
/* Local variable definition     */
/* ============================= */
static Timer_ID   nPwrSaveTimerID = 0;


/* ============================= */
/* Local function declaration    */
/* ============================= */
static IFX_void_t ifx_tapi_PowerSave_OnTimer (Timer_ID Timer, IFX_ulong_t nArg);


/* ============================= */
/* Global function definition    */
/* ============================= */

/**

   Reads the hook status from the device

   \param pChannel        - handle to TAPI_CHANNEL structure
   \param pState          - storage variable for hook state

   \return
   TAPI_statusOk
*/
IFX_int32_t TAPI_Phone_Hookstate(TAPI_CHANNEL *pChannel, IFX_int32_t *pState)
{
   if (pChannel->TapiOpControlData.bHookState)
   {
      *pState = IFX_TRUE;
   }
   else
   {
      *pState = IFX_FALSE;
   }

   return (TAPI_statusOk);
}

/**
   Sets the linefeeding mode of the device

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param nMode         Line mode.

   \return
   \ref IFX_SUCCESS or \ref IFX_ERROR

   \remarks
   Line mode is ALWAYS set, also if it was set before.
*/
IFX_int32_t TAPI_Phone_Set_Linefeed(TAPI_CHANNEL *pChannel,
                                    IFX_int32_t nMode)
{
   IFX_TAPI_DRV_CTX_t* pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_boolean_t       bBatSw  = IFX_FALSE;
   IFX_boolean_t       bPol    = IFX_FALSE;
   IFX_int32_t         ret     = TAPI_statusOk;
   IFX_uint8_t         tmp_nLineMode = pChannel->TapiOpControlData.nLineMode;
   IFX_uint8_t         tmp_nBatterySw = pChannel->TapiOpControlData.nBatterySw;
   IFX_uint8_t         tmp_nPolarity = pChannel->TapiOpControlData.nPolarity;

#ifdef TAPI_METERING
   if (TAPI_Phone_Meter_IsActive (pChannel))
   {
      /* NOTE: Recursive call
               _Meter_Stop will call _Set_Linefeed to restore
               the line mode before metering */

      /* stop metering before line mode changing */
      TAPI_Phone_Meter_Stop (pChannel);
   }
#endif /* TAPI_METERING */

   /* save current linemode in tapi structure */
   pChannel->TapiOpControlData.nLineMode = (IFX_uint8_t) nMode;

   /* check if auto battery switch have to enabled */
   if ((nMode == IFX_TAPI_LINE_FEED_NORMAL_AUTO    ||
        nMode == IFX_TAPI_LINE_FEED_REVERSED_AUTO) &&
       !pChannel->TapiOpControlData.nBatterySw)
   {
      bBatSw = IFX_TRUE;
      pChannel->TapiOpControlData.nBatterySw = 0x01;
   }

   /* check if auto battery switch have to disabled */
   if (!(nMode == IFX_TAPI_LINE_FEED_NORMAL_AUTO    ||
         nMode == IFX_TAPI_LINE_FEED_REVERSED_AUTO) &&
       pChannel->TapiOpControlData.nBatterySw)
   {
      bBatSw = IFX_TRUE;
      pChannel->TapiOpControlData.nBatterySw = 0x00;
   }

   /* check if polarity has to change */
   if ((nMode == IFX_TAPI_LINE_FEED_ACTIVE_REV        ||
        nMode == IFX_TAPI_LINE_FEED_REVERSED_AUTO     ||
        nMode == IFX_TAPI_LINE_FEED_REVERSED_LOW      ||
        nMode == IFX_TAPI_LINE_FEED_ACTIVE_RES_REVERSED) &&
       !pChannel->TapiOpControlData.nPolarity)
   {
      bPol = IFX_TRUE;
      pChannel->TapiOpControlData.nPolarity = 0x01;
   }
   if (!(nMode == IFX_TAPI_LINE_FEED_ACTIVE_REV       ||
         nMode == IFX_TAPI_LINE_FEED_REVERSED_AUTO    ||
         nMode == IFX_TAPI_LINE_FEED_REVERSED_LOW     ||
         nMode == IFX_TAPI_LINE_FEED_ACTIVE_RES_REVERSED) &&
       pChannel->TapiOpControlData.nPolarity)
   {
      bPol = IFX_TRUE;
      pChannel->TapiOpControlData.nPolarity = 0x00;
   }

   /* call low level function to change operation mode */
   if (ptr_chk(pDrvCtx->ALM.Line_Mode_Set, "pDrvCtx->ALM.Line_Mode_Set"))
   {
       ret = pDrvCtx->ALM.Line_Mode_Set(pChannel->pLLChannel,
                              nMode, tmp_nLineMode);
   }

   if (!TAPI_SUCCESS(ret))
   {
      /* restore the old value, because the configuration on the driver failed */
      pChannel->TapiOpControlData.nLineMode = tmp_nLineMode;
      RETURN_STATUS (TAPI_statusLineModeFail, ret);
   }

   /* switch polarity only when it is necessary */
   if (bPol && (ret == TAPI_statusOk))
   {
      if (ptr_chk(pDrvCtx->ALM.Line_Polarity_Set,
                 "pDrvCtx->ALM.Line_Polarity_Set"))
         ret = pDrvCtx->ALM.Line_Polarity_Set(pChannel->pLLChannel);

      if (!TAPI_SUCCESS(ret))
      {
         /* restore the old value, because the configuration on the driver failed */
         pChannel->TapiOpControlData.nPolarity = tmp_nPolarity;
      }
   }

   /* enable / disable automatic battery switch */
   if (bBatSw && (ret == TAPI_statusOk))
   {
      if (ptr_chk(pDrvCtx->ALM.AutoBatterySwitch,
                 "pDrvCtx->ALM.AutoBatterySwitch"))
         ret = pDrvCtx->ALM.AutoBatterySwitch(pChannel->pLLChannel);

      if (!TAPI_SUCCESS(ret))
      {
         /* restore the old value, because the configuration on the driver failed */
         pChannel->TapiOpControlData.nBatterySw = tmp_nBatterySw;
      }
   }

   return ret;
}


/**
   Sets the line type mode of the specific analog channel

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pCfg          Pointer to struct IFX_TAPI_LINE_TYPE_CFG_t.

   \return TAPI_statusOk or TAPI_statusErr

   \remark Line mode is ALWAYS set, also if it was set before. By default
           FXS type is used.
*/
IFX_int32_t TAPI_Phone_Set_LineType(TAPI_CHANNEL *pChannel,
                                    IFX_TAPI_LINE_TYPE_CFG_t *pCfg)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = TAPI_statusOk;

   /* call low level function to change operation mode */
   if (ptr_chk(pDrvCtx->ALM.Line_Type_Set, "pDrvCtx->ALM.Line_Type_Set"))
   {
       ret = pDrvCtx->ALM.Line_Type_Set(pChannel->pLLChannel, pCfg->lineType);
   }

   if (TAPI_SUCCESS(ret))
   {
      /* Save new type and DAA number */
      pChannel->TapiOpControlData.nLineType = pCfg->lineType;
      pChannel->TapiOpControlData.nDAA      = pCfg->nDaaCh;
      /* initialize DAA only for non-SmartSLIC based systems */
      if (pCfg->lineType == IFX_TAPI_LINE_TYPE_FXO &&
          pChannel->pTapiDevice->bSmartSlicFxo != IFX_TRUE)
      {
         ret = TAPI_FXO_Register_DAA(pChannel, pChannel->TapiOpControlData.nDAA);
         /* init the DAA channel */
         if ((TAPI_SUCCESS(ret)) &&
             (! pChannel->TapiOpControlData.bDaaInitialized))
         {
            ret = TAPI_FXO_Init_DAA (pChannel);
            if (TAPI_SUCCESS(ret))
            {
               pChannel->TapiOpControlData.bDaaInitialized = IFX_TRUE;
            }
         }
      }
      else
      {
         ret = TAPI_FXO_Register_DAA (IFX_NULL, pChannel->TapiOpControlData.nDAA);
      }
   }
   return ret;
}


/**
   Sets the LEC operating mode

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pLecConf      Pointer to IFX_TAPI_WLEC_CFG_t structure.

   \return
   TAPI_statusOk or TAPI_statusErr
*/
IFX_int32_t TAPI_Phone_LecMode_Alm_Set (TAPI_CHANNEL *pChannel,
                                        IFX_TAPI_WLEC_CFG_t *pLecConf)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   TAPI_LEC_DATA_t oLecData;
   IFX_int32_t     ret;

   /* do not touch anything if LL does not support LEC */
   if (!ptr_chk(pDrvCtx->ALM.Lec_Cfg, "pDrvCtx->ALM.Lec_Cfg"))
   {
      /* LEC not supported by LL driver */
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);
   }

   /* make a copy of the current configuration */
   memcpy(&oLecData, &pChannel->TapiLecAlmData, sizeof(TAPI_LEC_DATA_t));

   /* assemble configuration data structure */
   oLecData.nOpMode     = pLecConf->nType;
   oLecData.bNlp        = pLecConf->bNlp;
   oLecData.nNBNEwindow = pLecConf->nNBNEwindow;
   oLecData.nNBFEwindow = pLecConf->nNBFEwindow;
   oLecData.nWBNEwindow = pLecConf->nWBNEwindow;

   /* call low level for settings */
   ret = pDrvCtx->ALM.Lec_Cfg (pChannel->pLLChannel, &oLecData);
   if (TAPI_SUCCESS (ret))
   {
      /* begin of protected area */
      TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);
      /* save configuration */
      memcpy(&pChannel->TapiLecAlmData, &oLecData, sizeof(TAPI_LEC_DATA_t));
      /* end of protected area */
      TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
   }

   return ret;
}


/**
   Get the current LEC operating mode

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pWLecConf     Pointer to IFX_TAPI_WLEC_CFG_t structure.

   \return
   TAPI_statusOk or TAPI_statusErr
*/
IFX_int32_t TAPI_Phone_LecMode_Alm_Get (TAPI_CHANNEL *pChannel,
                                        IFX_TAPI_WLEC_CFG_t *pWLecConf)
{
   TAPI_LEC_DATA_t *pLecData = &pChannel->TapiLecAlmData;

   /* begin of protected area */
   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   /* read config from channel structure */
   pWLecConf->nType = pLecData->nOpMode;
   pWLecConf->bNlp  = (IFX_TAPI_WLEC_NLP_t)pLecData->bNlp;
   pWLecConf->nNBNEwindow = pLecData->nNBNEwindow;
   pWLecConf->nNBFEwindow = pLecData->nNBFEwindow;
   pWLecConf->nWBNEwindow = pLecData->nWBNEwindow;

   /* end of protected area */
   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);

   return TAPI_statusOk;
}


/**
   Sets the LEC operating mode

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pLecConf      Pointer to IFX_TAPI_WLEC_CFG_t structure.

   \return
   TAPI_statusOk or TAPI_statusErr
*/
IFX_int32_t TAPI_Phone_LecMode_Pcm_Set (TAPI_CHANNEL *pChannel,
                                        IFX_TAPI_WLEC_CFG_t *pLecConf)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   TAPI_LEC_DATA_t oLecData;
   IFX_int32_t     ret;

   /* do not touch anything if LL does not support LEC */
   if (!ptr_chk(pDrvCtx->PCM.Lec_Cfg, "pDrvCtx->PCM.Lec_Cfg"))
   {
      /* LEC not supported by LL driver */
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);
   }

   /* make a copy of the current configuration */
   memcpy(&oLecData, &pChannel->TapiLecPcmData, sizeof(TAPI_LEC_DATA_t));

   /* assemble configuration data structure */
   oLecData.nOpMode     = pLecConf->nType;
   oLecData.bNlp        = pLecConf->bNlp;
   oLecData.nNBNEwindow = pLecConf->nNBNEwindow;
   oLecData.nNBFEwindow = pLecConf->nNBFEwindow;
   oLecData.nWBNEwindow = pLecConf->nWBNEwindow;

   /* call low level for settings */
   ret = pDrvCtx->PCM.Lec_Cfg (pChannel->pLLChannel, &oLecData);
   if (TAPI_SUCCESS (ret))
   {
      /* begin of protected area */
      TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);
      /* save configuration */
      memcpy(&pChannel->TapiLecPcmData, &oLecData, sizeof(TAPI_LEC_DATA_t));
      /* end of protected area */
      TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
   }

   return ret;
}


/**
   Get the current LEC operating mode

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pLecConf      Pointer to IFX_TAPI_WLEC_CFG_t structure.

   \return
   TAPI_statusOk or TAPI_statusErr
*/
IFX_int32_t TAPI_Phone_LecMode_Pcm_Get (TAPI_CHANNEL *pChannel,
                                        IFX_TAPI_WLEC_CFG_t *pLecConf)
{
   TAPI_LEC_DATA_t *pLecData = &pChannel->TapiLecPcmData;

   /* begin of protected area */
   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   /* read config from channel structure */
   pLecConf->bNlp  = (IFX_TAPI_WLEC_NLP_t)pLecData->bNlp;
   pLecConf->nType = pLecData->nOpMode;
   pLecConf->nNBNEwindow = pLecData->nNBNEwindow;
   pLecConf->nNBFEwindow = pLecData->nNBFEwindow;
   pLecConf->nWBNEwindow = pLecData->nWBNEwindow;

   /* end of protected area */
   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);

   return TAPI_statusOk;
}


/**
   Sets/Gets the DTMF Receiver Coefficients

   \param pChannel        - handle to TAPI_CHANNEL structure
   \param bRW
   \param pCoeff          - handle to IFX_TAPI_DTMF_RX_CFG_t structure

   \return
   TAPI_statusOk or TAPI_statusErr
*/
IFX_int32_t TAPI_Phone_Dtmf_RxCoeff_Cfg (TAPI_CHANNEL *pChannel,
                                         IFX_boolean_t bRW,
                                         IFX_TAPI_DTMF_RX_CFG_t *pCoeff)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret;

   /* do not touch anything if LL device DTMF Rx coefficients
      not configurable */
   if (!ptr_chk(pDrvCtx->SIG.DTMF_RxCoeff, "pDrvCtx->SIG.DTMF_RxCoeff"))
   {
      /* DTMF Rx coefficients not configurable on LL device */
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);
   }

   /* call low level routine to read/write the settings */
   ret = pDrvCtx->SIG.DTMF_RxCoeff (pChannel->pLLChannel, bRW, pCoeff);

   TRACE (TAPI_DRV, DBG_LEVEL_LOW,
         ("Ch%d: DTMF Rx coefficients %s: "
          "Level=%d [dB], Twist=%d [dB], Gain=%d [dB]\n",
          pChannel->nChannel, bRW == IFX_FALSE? "setting" : "getting",
          pCoeff->nLevel, pCoeff->nTwist, pCoeff->nGain));

   return ret;
}


/**
   Power save timer initialisation function

   Initialises the one global timer for the power save poll routine.
   Every time the timer expires all initialised low level devices of all
   registered drivers are polled to activate a power save mode.

   \return
   TAPI_statusOk or TAPI_statusErr
*/
IFX_int32_t IFX_TAPI_PowerSave_Init (IFX_void_t)
{
   IFX_int32_t ret = TAPI_statusOk;

   /* Create and set timer to check if LL power save feature can be activated,
      as we just need one timer for the whole TAPI, check if it already
      has been created. */
   if (nPwrSaveTimerID == 0)
   {
      nPwrSaveTimerID =
         TAPI_Create_Timer((TIMER_ENTRY)ifx_tapi_PowerSave_OnTimer, 0);
      if (nPwrSaveTimerID == 0)
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("TAPI_ERROR: Creating timer for power saving feature failed\n"));
         ret = TAPI_statusErr;
      }
      else
      {
         /* set the timer to an automatic reloading cycle of 5000ms */
         TAPI_SetTime_Timer (nPwrSaveTimerID, IFX_TAPI_PWR_SAVE_CHK_TIME_MS,
                             IFX_TRUE, IFX_FALSE);
      }
   }

   return ret;
}


/**
   Power save timer exit function

   Stop the one global timer for the power save poll routine.
*/
IFX_void_t IFX_TAPI_PowerSave_Exit (IFX_void_t)
{
   if (nPwrSaveTimerID != 0)
   {
      TAPI_Delete_Timer (nPwrSaveTimerID);
      nPwrSaveTimerID = 0;
   }
}


/**
   Power save timer callback function

   Every time the timer expires all initialised low level devices of all
   registered drivers are polled to activate a power save mode.

   \param  Timer        Timer ID (currently unused).
   \param  nArg         Argument (Currently unused).
*/
IFX_void_t ifx_tapi_PowerSave_OnTimer (Timer_ID Timer, IFX_ulong_t nArg)
{
   IFX_TAPI_DRV_CTX_t   *pDrvCtx;
   TAPI_DEV             *pTapiDev;
   IFX_uint16_t         i, j;
   IFX_int32_t          ret = TAPI_statusOk;

   IFX_UNUSED (nArg);
   IFX_UNUSED (Timer);

   /* Loop over all registered LL-drivers */
   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      pDrvCtx = gHLDrvCtx[i].pDrvCtx;

      /* If a LL-driver is registered in this slot and supports the power save
         feature all devices have to be polled. */
      if ((pDrvCtx != IFX_NULL) &&
          ptr_chk(pDrvCtx->Pwr_Save_Dev, "pDrvCtx->Pwr_Save_Dev"))
      {
         /* Loop over all devices of this LL-driver */
         for (j=0; j < pDrvCtx->maxDevs; j++)
         {
            pTapiDev = &(pDrvCtx->pTapiDev[j]);

            if (pTapiDev->bInitialized == IFX_TRUE)
            {
               ret = pDrvCtx->Pwr_Save_Dev(pTapiDev->pLLDev);
            }

            if (!TAPI_SUCCESS (ret))
            {
               TRACE (TAPI_DRV, DBG_LEVEL_LOW,
                     ("DRV_ERROR: Low level device failed "
                      "to set power save mode.\n"));
            }
         }
      }
   }
}
