/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/** \file drv_tapi_stream.c
    Data stream fifos and buffers for TAPI.
    This module provides management of the fifos and buffers for voice data
    transport in TAPI. */


/* ============================= */
/* Check if feature is enabled   */
/* ============================= */
#ifdef HAVE_CONFIG_H
#include <drv_config.h>
#endif

#ifdef TAPI_PACKET

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_tapi.h"
#include "drv_tapi_stream.h"

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */

/** Sizeof on VOICE PACKET. */
enum { PACKET_SIZE = 512 };

/** Sizeof of UpStream FIFO. */
enum { VOICE_UP_FIFO_SIZE = 25 };

#ifdef TAPI_POLL
/** Sizeof of DownStream FIFO. */
enum { VOICE_DOWN_FIFO_SIZE = 16 };
#endif /* TAPI_POLL */

/** Startup size of elements in bufferpool. */
enum { PKT_COUNT = 100 };

/** When bufferpool is full, increase it by this size. */
enum { PKT_COUNT_INC = 10 };

/** Limit bufferpool's automatic growth. */
enum { PKT_COUNT_LIMIT = 1000 };

/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Local function declaration    */
/* ============================= */
static IFX_return_t ifx_tapi_ClearFifo(FIFO_ID* pFifo);
#ifdef TAPI_POLL
static IFX_return_t ifx_tapi_DownStreamFifo_Reset(TAPI_DEV* pTapiDev);
#endif /* TAPI_POLL */

/* ============================= */
/* Local variable definition     */
/* ============================= */
/** Memory of free buffers for voice packets. */
static BUFFERPOOL* pVoicePktBuffer = IFX_NULL;
/** Memory of free buffers for voice packet wrappers. */
static BUFFERPOOL* pVoicePktWrappers = IFX_NULL;
/* Buffer pool access protection */
static TAPI_OS_mutex_t semVoiceBufferPoolAcc;

/* ============================= */
/* Function definitions          */
/* ============================= */

/**
   Reset FIFO, free elements in in fifo and clear it.

   \param  pFifo        Pointer to FIFO. Must not be NULL.

   \return
         Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
static IFX_return_t ifx_tapi_ClearFifo(FIFO_ID* pFifo)
{
   IFX_void_t* p_elem = IFX_NULL;
   IFX_int32_t fifo_cnt = 0;

   fifo_cnt = fifoSize(pFifo);
   if (fifo_cnt < 0)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("Wrong fifo size. (File: %s, line: %d)\n",
            __FILE__, __LINE__));
      return IFX_ERROR;
   }

   while ((fifoEmpty(pFifo)) != IFX_TRUE && (0 < fifo_cnt))
   {
      p_elem = (IFX_void_t *) fifoGet(pFifo, NULL);

      if (IFX_NULL != p_elem)
      {
         bufferPoolPut(p_elem);
      }
      else
      {
         /* Strange we have element in FIFO which is NULL? */
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Element NULL retrieved from fifo?\n"));
      }
      fifo_cnt--;
   }

   /* Now reset fifo */
   fifoReset(pFifo);

   return IFX_SUCCESS;
}


/**
   Initialize UpStream FIFO.

   \param  pChannel     Pointer to TAPI channel. Must not be NULL.

   \return
         Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_UpStreamFifo_Create (TAPI_CHANNEL* pChannel)
{
   IFX_TAPI_STREAM_t nStream;

   for (nStream = 0; nStream < IFX_TAPI_STREAM_MAX; nStream++)
   {

      if (pChannel->nChannel <
          pChannel->pTapiDevice->pDevDrvCtx->readChannels[nStream])
      {
         /* Initialize packet voice fifo. */
         if (IFX_NULL != pChannel->pUpStreamFifo[nStream])
         {
            /* Reset fifo */
            IFX_TAPI_UpStreamFifo_Reset(pChannel, nStream);
         }
         else
         {
            /* Create fifo */
            pChannel->pUpStreamFifo[nStream] = fifoInit(VOICE_UP_FIFO_SIZE);
            if (IFX_NULL == pChannel->pUpStreamFifo[nStream])
            {
               /* ERROR: Packet FIFO initialization failed, check fifoInit
                         description. */
               TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("ERR: Initializing "
                     "pUpStreamFifo. (File: %s, line: %d)\n",
                     __FILE__, __LINE__));
               return IFX_ERROR;
            }
         }
      }
   }

   return IFX_SUCCESS;
}


/**
   Reset UpStream FIFO.

   \param  pTapiCh      Pointer to TAPI channel. Must not be NULL.
   \param  nStream      Selects which FIFO to use from IFX_TAPI_STREAM_t.

   \return
         Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_UpStreamFifo_Reset(TAPI_CHANNEL* pTapiCh,
                                         IFX_TAPI_STREAM_t nStream)
{
   TAPI_DEV *pTapiDev = pTapiCh->pTapiDevice;
   IFX_TAPI_FIFO_ELEM_t *pWrapper;

   if (pTapiCh->pUpStreamFifo[nStream] == IFX_NULL)
      return IFX_ERROR;

   if (ptr_chk(pTapiDev->pDevDrvCtx->IRQ.LockDevice,
              "pTapiDev->pDevDrvCtx->IRQ.LockDevice"))
   {
      pTapiDev->pDevDrvCtx->IRQ.LockDevice(pTapiDev->pLLDev);
   }
   else
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
          ("TAPI IRQ.LockDevice not set in LL driver\n"));
   }

   while ( (pWrapper =
            fifoGet(pTapiCh->pUpStreamFifo[nStream], NULL)) != IFX_NULL )
   {
      bufferPoolPut(pWrapper->pBuf);
      bufferPoolPut(pWrapper);
   }
   fifoReset(pTapiCh->pUpStreamFifo[nStream]);

   if (ptr_chk(pTapiDev->pDevDrvCtx->IRQ.UnlockDevice,
              "pTapiDev->pDevDrvCtx->IRQ.UnlockDevice"))
   {
      pTapiDev->pDevDrvCtx->IRQ.UnlockDevice(pTapiDev->pLLDev);
   }
   else
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
          ("TAPI IRQ.UnlockDevice not set in LL driver\n"));
   }

   return IFX_SUCCESS;
}


/**
   Add a new element to the UpStream FIFO.

   \param  pTapiCh      Pointer to TAPI channel. Must not be NULL.
   \param  nStream      Selects which FIFO to use from IFX_TAPI_STREAM_t.
   \param  pData        Pointer that is stored in the fifo.
   \param  nLength      length information to be stored.
   \param  nOffset      offset information to be stored.

   \return
   SUCCESS or ERROR if no more element can be added to the FIFO
*/
IFX_return_t IFX_TAPI_UpStreamFifo_Put(TAPI_CHANNEL* pTapiCh,
                                       IFX_TAPI_STREAM_t nStream,
                                       const IFX_void_t * const pData,
                                       const IFX_uint32_t nLength,
                                       const IFX_uint32_t nOffset)
{
   IFX_return_t ret = IFX_ERROR;

   /* sanity check if this channel has a fifo */
   if ((nStream < IFX_TAPI_STREAM_MAX) &&
       (pTapiCh->pUpStreamFifo[nStream] != IFX_NULL))
   {
      /* allocate wrapper buffer */
      IFX_TAPI_FIFO_ELEM_t *pWrapper = bufferPoolGet(pVoicePktWrappers);
      if (pWrapper == IFX_NULL)
      {
         return IFX_ERROR;
      }
      /* fill wrapper buffer */
      pWrapper->pBuf = (IFX_void_t *)pData;
      pWrapper->nDataOffset = nOffset;
      /* put wrapper buffer into the upstream fifo */
      ret = fifoPut(pTapiCh->pUpStreamFifo[nStream], (void *)pWrapper, nLength);
      if (ret != IFX_SUCCESS)
      {
         /* return the wrapper buffer and ignore the return value */
         bufferPoolPut(pWrapper);
      }
   }

   return ret;
}


/**
   Get an element from the UpStream FIFO.

   \param  pTapiCh      Pointer to TAPI channel. Must not be NULL.
   \param  nStream      Selects which FIFO to use from IFX_TAPI_STREAM_t.
   \param  pLength      Pointer where length information is to be stored.
   \param  pOffset      Pointer where offset information is to be stored.

   \return
   pointer that was stored in the fifo -- IFX_NULL if empty or on error.
*/
IFX_void_t *IFX_TAPI_UpStreamFifo_Get(TAPI_CHANNEL* pTapiCh,
                                      IFX_TAPI_STREAM_t nStream,
                                      IFX_uint32_t *pLength,
                                      IFX_uint32_t *pOffset)
{
   IFX_void_t *ret = IFX_NULL;

   /* sanity check if this channel has a fifo */
   if ((nStream < IFX_TAPI_STREAM_MAX) &&
       (pTapiCh->pUpStreamFifo[nStream] != IFX_NULL))
   {
      IFX_TAPI_FIFO_ELEM_t *pWrapper;
      /* Get an element from the fifo */
      pWrapper = fifoGet(pTapiCh->pUpStreamFifo[nStream], pLength);
      if (pWrapper != IFX_NULL)
      {
         /* copy data from the wrapper */
         ret = pWrapper->pBuf;
         if (pOffset != IFX_NULL)
         {
            *pOffset = pWrapper->nDataOffset;
         }
         /* return wrapper buffer to the pool */
         bufferPoolPut(pWrapper);
      }
   }

   return ret;
}


/**
   Delete the UpStream FIFOs.

   \param  pTapiCh      Pointer to TAPI channel. Must not be NULL.

   \return
      Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_UpStreamFifo_Delete(TAPI_CHANNEL* pTapiCh)
{
   IFX_TAPI_STREAM_t nStream;
   IFX_return_t ret = IFX_SUCCESS;

   TAPI_ASSERT(pTapiCh != IFX_NULL);

   for (nStream = 0; nStream < IFX_TAPI_STREAM_MAX; nStream++)
   {
      if (pTapiCh->pUpStreamFifo[nStream] != IFX_NULL)
      {
         ret = ifx_tapi_ClearFifo(pTapiCh->pUpStreamFifo[nStream]);
         if (ret == IFX_SUCCESS)
         {
            ret = fifoFree (pTapiCh->pUpStreamFifo[nStream]);
            pTapiCh->pUpStreamFifo[nStream] = IFX_NULL;
         }
      }
   }

   return ret;
}


#ifdef TAPI_POLL
/* The downstream FIFO is only needed for the polling implementation */

/**
   Initialize DownStream FIFO.

   \param  pTapiDev     Pointer to TAPI device. Must not be NULL.

   \return
         Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_DownStreamFifo_Create(TAPI_DEV* pTapiDev)
{
   /* Initialize packet voice fifo. */
   if (IFX_NULL != pTapiDev->pDownStreamFifo)
   {
      /* Reset fifo and return */
      return ifx_tapi_DownStreamFifo_Reset(pTapiDev);
   }

   pTapiDev->pDownStreamFifo = fifoInit(VOICE_DOWN_FIFO_SIZE /* elem count */);
   if (IFX_NULL == pTapiDev->pDownStreamFifo)
   {
      /* ERROR: Packet FIFO initialization failed, check fifoInit
                description. */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("ERR: Initializing "
            "pDownStreamFifo. (File: %s, line: %d)\n",
            __FILE__, __LINE__));
      return IFX_ERROR;
   }

   return IFX_SUCCESS;
}


/**
   Reset DownStream FIFO.

   \param  pTapiDev     Pointer to TAPI device. Must not be NULL.

   \return
         Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t ifx_tapi_DownStreamFifo_Reset(TAPI_DEV* pTapiDev)
{

   if (IFX_NULL != pTapiDev->pDownStreamFifo)
   {
      return ifx_tapi_ClearFifo(pTapiDev->pDownStreamFifo);
   }

   return IFX_ERROR;
}


/**
   Retrieve handle to DownStream FIFO.

   \param  pTapiDev     Pointer to TAPI device. Must not be NULL.

   \return
         Returns IFX_NULL in case of an error, otherwise returns fifo handle.
*/
FIFO_ID* IFX_TAPI_DownStreamFifo_Handle_Get(TAPI_DEV* pTapiDev)
{
   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("Return downstream fifo handle.\n"));

   if (IFX_NULL == pTapiDev)
   {
      /* Wrong input arguments */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("Invalid input argument(s). (File: %s, line: %d)\n",
            __FILE__, __LINE__));
      return IFX_NULL;
   }

   return pTapiDev->pDownStreamFifo;
}


/**
   Delete DownStream FIFO.

   \param  pTapiDev     Pointer to TAPI device. Must not be NULL.

   \return
         Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_DownStreamFifo_Delete(TAPI_DEV* pTapiDev)
{
   IFX_return_t ret = IFX_SUCCESS;

   if (pTapiDev->pDownStreamFifo != IFX_NULL)
   {
      ret = fifoReset(pTapiDev->pDownStreamFifo);
      if (ret == IFX_SUCCESS)
      {
         ret = fifoFree(pTapiDev->pDownStreamFifo);
      }
      pTapiDev->pDownStreamFifo = IFX_NULL;
   }

   return ret;
}
#endif /* TAPI_POLL */


/**
   Prepare bufferpool for voice packets.

   There is just one global pool for the entire tapi. It will be initialised
   with the first call and subsequent calls just return and create nothing new.

   \return
      Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_VoiceBufferPool_Create(IFX_void_t)
{
   if (IFX_NULL != pVoicePktBuffer)
   {
      /* pVoicePktBuffer already initialized */
      return IFX_SUCCESS;
   }

   pVoicePktBuffer = bufferPoolInit(PACKET_SIZE, PKT_COUNT, PKT_COUNT_INC);
   if (IFX_NULL == pVoicePktBuffer)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Failed to init bufferpool handle "
            "pVoicePktBuffer.\n"));
      return IFX_ERROR;
   }
   bufferPoolGrowthLimitSet(pVoicePktBuffer, PKT_COUNT_LIMIT);
   bufferPoolIDSet (pVoicePktBuffer, 20);


   pVoicePktWrappers = bufferPoolInit(sizeof(IFX_TAPI_FIFO_ELEM_t),
                                      PKT_COUNT, PKT_COUNT_INC);
   if (IFX_NULL == pVoicePktBuffer)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Failed to init bufferpool handle "
            "pVoicePktWrappers.\n"));
      bufferPoolFree(pVoicePktBuffer);
      pVoicePktBuffer = IFX_NULL;
      return IFX_ERROR;
   }
   bufferPoolGrowthLimitSet(pVoicePktWrappers, PKT_COUNT_LIMIT);
   bufferPoolIDSet (pVoicePktWrappers, 21);

   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("pVoicePktBuffer initialized "
         "%d buffers free.\n", bufferPoolAvail(pVoicePktBuffer)));

   /* initialize buffer pool access protection semaphore */
   TAPI_OS_MutexInit (&semVoiceBufferPoolAcc);

   return IFX_SUCCESS;
}


/**
   Retrieve a buffer for a voice packet.

   \return
   Returns IFX_NULL in case of an error, otherwise returns packet buffer.

   \remarks
   This function uses a global irq lock - multiple drivers may be loaded and
   all may try to retrieve buffers from this shared pool in irq context. So we
   lock the interrupt during access to the shared buffer pool.
*/
IFX_void_t* IFX_TAPI_VoiceBufferGet(IFX_void_t)
{
   return IFX_TAPI_VoiceBufferGetWithOwnerId (IFX_TAPI_BUFFER_OWNER_UNKNOWN);
}


/**
   Retrieve a buffer for a voice packet and set the ID of the caller.

   \param  ownerId      ID value provided by the caller.

   \return
   Returns IFX_NULL in case of an error, otherwise returns packet buffer.

   \remarks
   This function uses a global irq lock - multiple drivers may be loaded and
   all may try to retrieve buffers from this shared pool in irq context. So we
   lock the interrupt during access to the shared buffer pool.
*/
IFX_void_t* IFX_TAPI_VoiceBufferGetWithOwnerId (IFX_uint32_t ownerId)
{
   IFX_void_t* buf = IFX_NULL;
   TAPI_OS_INTSTAT lock;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);
   buf = (IFX_void_t *) bufferPoolGetWithOwnerId(pVoicePktBuffer, ownerId);
   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }

   return buf;
}

#ifdef TAPI_PACKET_OWNID
/**
   Update owner ID for voice packet buffer.

   \param  pData     Pointer to a buffer that was retrieved through a call of
                     \ref IFX_TAPI_VoiceBufferGet().
   \param  ownerId   ID value provided by the caller.

   \return IFX_SUCCESS or IFX_ERROR

   \remarks
   This function uses a global irq lock - multiple drivers may be loaded and
   all may try to retrieve buffers from this shared pool in irq context. So we
   lock the interrupt during access to the shared buffer pool.
*/
IFX_int32_t  IFX_TAPI_VoiceBufferChOwn(IFX_void_t *pb, IFX_uint32_t ownerID)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t ret;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);
   ret = bufferPoolChOwn(pb, ownerID);
   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }

   return ret;
}
#endif /* TAPI_PACKET_OWNID */

/**
   Discard a voice packet buffer.

   \param  pData  Pointer to a buffer that was retrieved through a call of
                  \ref IFX_TAPI_VoiceBufferGet().

   \return IFX_SUCCESS or IFX_ERROR

   \remarks
   This function uses a global irq lock - multiple drivers may be loaded and
   all may try to retrieve buffers from this shared pool in irq context. So we
   lock the interrupt during access to the shared buffer pool.
*/
IFX_int32_t IFX_TAPI_VoiceBufferPut(IFX_void_t *pData)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t  ret;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);
   /* bufferPool has it's own return values, but they match IFX_SUCCESS,
       _ERROR */
   ret = bufferPoolPut((void *)pData);
   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }
   return ret;
}


/**
   Frees all buffers in the pool that are allocated and carry the given ID.

   \param  ownerId      ID value provided by the caller.

   \return IFX_SUCCESS or IFX_ERROR

   \remarks
   This function uses a global irq lock - multiple drivers may be loaded and
   all may try to retrieve buffers from this shared pool in irq context. So we
   lock the interrupt during access to the shared buffer pool.
*/
extern IFX_int32_t   IFX_TAPI_VoiceBufferFreeAllOwnerId (IFX_uint32_t ownerId)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t  ret;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);
   ret = bufferPoolFreeAllOwnerId(pVoicePktBuffer, ownerId);
   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }
   return ret;
}


/**
   Retrieve the size of one element in the voice-packet bufferpool.

   \return Size of one element in bytes.
*/
IFX_uint32_t IFX_TAPI_VoiceBufferPool_ElementSizeGet(IFX_void_t)
{
   return PACKET_SIZE;
}

/**
   Retrieve the overall number of elements of the voice-packet bufferpool.

   \return the overall number of elements
*/
IFX_int32_t IFX_TAPI_VoiceBufferPool_ElementCountGet(IFX_void_t)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t  elements = 0;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);
   elements = bufferPoolSize( pVoicePktBuffer );
   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }

   return elements;
}

/**
   Retrieve the available (free) number of elements of the
   voice-packet bufferpool.

   \return the number of available elements
*/
IFX_int32_t IFX_TAPI_VoiceBufferPool_ElementAvailCountGet(IFX_void_t)
{
   TAPI_OS_INTSTAT lock;
   IFX_int32_t  elements = 0;

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);
   elements = bufferPoolAvail( pVoicePktBuffer );
   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }

   return elements;
}

#ifdef TAPI_PACKET_OWNID
/**
   Print buffer information.

   \return 0 - continue enumeration
*/
static IFX_uint32_t ifx_tapi_VoiceBufferStatusShow (IFX_void_t *pArgs,
   IFX_void_t *pHandle,  IFX_uint32_t nOwnerID, IFX_uint32_t nState)
{
   printk (KERN_CRIT "%p 0x%-8x %-5u\n", pHandle, nOwnerID, nState);

   return 0;
}

/**
   Print out voice buffer information
*/
IFX_void_t IFX_TAPI_VoiceBufferPoolStatusShow (IFX_void_t)
{
   TAPI_OS_INTSTAT lock;

   if (IFX_NULL == pVoicePktBuffer)
   {
      printk (KERN_CRIT "Voice bufferpool has not been initialized yet.\n");
      return;
   }

   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexGet (&semVoiceBufferPoolAcc);
   }
   TAPI_OS_LOCKINT (lock);

   printk (KERN_CRIT "%-8s %-10s %-5s\n", "pointer", "owner_id", "state");
   bufferPoolEnumerate (pVoicePktBuffer, ifx_tapi_VoiceBufferStatusShow, NULL);

   TAPI_OS_UNLOCKINT (lock);
   if (!TAPI_OS_IN_INTERRUPT())
   {
      TAPI_OS_MutexRelease (&semVoiceBufferPoolAcc);
   }
}
#endif /* TAPI_PACKET_OWNID */

/**
   Destruct bufferpool for voice packets.

   \return
      Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_return_t IFX_TAPI_VoiceBufferPool_Delete(IFX_void_t)
{
   IFX_return_t ret = IFX_SUCCESS;

   if (pVoicePktBuffer != IFX_NULL)
   {
      if (bufferPoolFree(pVoicePktBuffer) != BUFFERPOOL_SUCCESS)
      {
         /* record the error but go with the other bufferpool anyhow */
         ret = IFX_ERROR;
      }
      pVoicePktBuffer = IFX_NULL;
   }

   if (pVoicePktWrappers != IFX_NULL)
   {
      if (bufferPoolFree(pVoicePktWrappers) != BUFFERPOOL_SUCCESS)
      {
         ret = IFX_ERROR;
      }
      pVoicePktWrappers = IFX_NULL;
   }

   /* delete the buffer pool access protection semaphore */
   TAPI_OS_MutexDelete (&semVoiceBufferPoolAcc);

   return ret;
}

#endif /* TAPI_PACKET */
