/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_vmmc_alm.c Implementation of the ALM module.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_api.h"
#include "drv_vmmc_alm_priv.h"
#include "drv_vmmc_fw_commands.h"
#include "drv_vmmc_api.h"
#include "drv_vmmc_alm.h"
#include "drv_vmmc_con.h"
#include "drv_mps_vmmc.h"
#if (VMMC_CFG_FEATURES & VMMC_FEAT_GR909)
#include "drv_vmmc_gr909.h"
#endif

#include "ifxos_time.h"

/* ============================= */
/* Global variables              */
/* ============================= */
#ifdef SYSTEM_DANUBE
/* default values for the ALM coefficient patch for narrowband */
static const VMMC_ALM_NB_WB_COEF_t VMMC_AlmNbCoeff =
{
   { 0x0200, 0x3708, 0x76D7, 0x871D, 0x7FBB, 0x7FF4 },
   { 0x0200, 0x5E02, 0x7E00, 0x0000 },
   { 0x0200, 0x6C02, 0x7E00, 0x0000 }
};

/* default values for the ALM coefficient patch for wideband */
static const VMMC_ALM_NB_WB_COEF_t VMMC_AlmWbCoeff =
{
   { 0x0200, 0x3708, 0x7CE0, 0x826A, 0x7FF8, 0x7FF4 },
   { 0x0200, 0x5E02, 0x7F00, 0x0000 },
   { 0x0200, 0x6C02, 0x7F00, 0x0000 }
};
#endif /* SYSTEM_DANUBE */

/* default line types for ALM module */
static VMMC_ALM_LINE_TYPE_t vmmc_alm_line[MAX_ALM_NUM] =
   {VMMC_ALM_LINE_FXS, VMMC_ALM_LINE_FXS, VMMC_ALM_LINE_FXO};

#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
/* mapping from DCCTRL Opmode number to SDD Opmode number */
static const IFX_uint8_t dc_opmode2sdd_opmode[] =
       {VMMC_SDD_OPMODE_DISABLED, VMMC_SDD_OPMODE_STANDBY,
        VMMC_SDD_OPMODE_ACTIVE, VMMC_SDD_OPMODE_RING_BURST,
        VMMC_SDD_OPMODE_GR909, VMMC_SDD_OPMODE_FXO, VMMC_SDD_OPMODE_CALIBRATE};
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION)
/* Limits for useful values returned by the calibration process. */
#define VMMC_SDD_CALIBRATE_TXOFFSET_MAX (+983)
#define VMMC_SDD_CALIBRATE_TXOFFSET_MIN (-983)
#define VMMC_SDD_CALIBRATE_MEOFFSET_MAX (+1311)
#define VMMC_SDD_CALIBRATE_MEOFFSET_MIN (-1311)
#define VMMC_SDD_CALIBRATE_ULIMOFFSET30_MAX (+683)
#define VMMC_SDD_CALIBRATE_ULIMOFFSET30_MIN (-683)
#define VMMC_SDD_CALIBRATE_ULIMOFFSET60_MAX (+683)
#define VMMC_SDD_CALIBRATE_ULIMOFFSET60_MIN (-683)
#define VMMC_SDD_CALIBRATE_IDACGAIN_MAX (+32183)
#define VMMC_SDD_CALIBRATE_IDACGAIN_MIN (+26331)
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION) */

/* ============================= */
/* Local variables and types     */
/* ============================= */

/* ============================= */
/* Local function declaration    */
/* ============================= */
#ifdef SYSTEM_DANUBE
static IFX_int32_t vmmc_alm_SamplingModeCoefSet (VMMC_CHANNEL *pCh,
                                                 OPMODE_SMPL   mode);
#endif /* SYSTEM_DANUBE */
#if (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION)
static IFX_int32_t vmmc_alm_Calibration_Get (VMMC_CHANNEL *pCh,
                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);
#if 0
static IFX_int32_t vmmc_alm_Calibration_Set (VMMC_CHANNEL *pCh,
                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);
#endif
static IFX_int32_t vmmc_alm_Calibration_Start (VMMC_CHANNEL *pCh);

static IFX_int32_t vmmc_alm_Calibration_Stop (VMMC_CHANNEL *pCh);

static IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Results (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);

static IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Get (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig);

static IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Start (
                        IFX_TAPI_LL_CH_t *pLLChannel);

static IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Stop (
                        IFX_TAPI_LL_CH_t *pLLChannel);

static IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Finish (
                        IFX_TAPI_LL_CH_t *pLLChannel);
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION) */

static IFX_int32_t VMMC_TAPI_LL_ALM_Line_Mode_Set (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_int32_t nMode,
                        IFX_uint8_t nTapiLineMode);

static IFX_int32_t VMMC_TAPI_LL_ALM_Line_Type_Set (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_TAPI_LINE_TYPE_t nType);

static IFX_int32_t VMMC_TAPI_LL_ALM_VMMC_Test_HookGen (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_boolean_t bHook);

static IFX_int32_t VMMC_TAPI_LL_ALM_VMMC_Test_Loop (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_TAPI_TEST_LOOP_t* pLoop);

static IFX_int32_t VMMC_TAPI_LL_ALM_Volume_Set (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_TAPI_LINE_VOLUME_t const *pVol);

static IFX_int32_t VMMC_TAPI_LL_ALM_Volume_High_Level (
                        IFX_TAPI_LL_CH_t *pLLCh,
                        IFX_int32_t bEnable);

static IFX_int32_t VMMC_TAPI_LL_ALM_LEC_Set (
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        TAPI_LEC_DATA_t *pLecConf);

static IFX_int32_t vmmc_alm_Echo_Suppressor_Set (
                        IFX_TAPI_LL_CH_t *pLLCh,
                        IFX_enDis_t nEnable,
                        IFX_enDis_t nActiveNLP);

#if (VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT)
static IFX_int32_t VMMC_TAPI_LL_ALM_ContMeas_Req (
                        IFX_TAPI_LL_CH_t *pLLChannel);

static IFX_int32_t VMMC_TAPI_LL_ALM_ContMeas_Reset(
                        IFX_TAPI_LL_CH_t *pLLChannel);

static IFX_int32_t VMMC_TAPI_LL_ALM_ContMeas_Get(
                        IFX_TAPI_LL_CH_t *pLLChannel,
                        IFX_TAPI_CONTMEASUREMENT_GET_t *pContMeas);
#endif /* VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasStart(
                        IFX_TAPI_LL_CH_t *pLLChannel);
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasStop(
                        IFX_TAPI_LL_CH_t *pLLChannel);
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasResult(
                        IFX_TAPI_LL_CH_t *pLLChannel);
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasStartInt(
                        IFX_TAPI_LL_CH_t *pLLChannel);
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */

/* ============================= */
/* Global function declaration   */
/* ============================= */
extern IFX_void_t VMMC_Ad1_Callback(VMMC_DEVICE *pDev, MbxEventRegs_s *pEvent);

/* ============================= */
/* Function definitions          */
/* ============================= */

/**
   Disables the line in case of an emergency.

   This function is used directly from interrupt contect to disable the line
   feeding in case of ground fault or SLIC overtemperature.

   \param  pCh             Pointer to the VMMC channel structure.

   \remarks
   This function is called from interrupt context and is thus not protected.
*/
IFX_void_t irq_VMMC_ALM_LineDisable (VMMC_CHANNEL *pCh)
{
   OPMODE_CMD_t* pOpmod = &pCh->pALM->dc_opmode;

   pOpmod->OP_MODE = VMMC_OPMOD_PDH;
   pOpmod->REV_POL = VMMC_OPMOD_POL_NORMAL;
   pOpmod->HOWLER  = VMMC_OPMOD_HOWLER_OFF;

   CmdWriteIsr (pCh->pParent, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
}


/**
   Configure or check ALM module for given sampling mode.

   \param  pCh             Pointer to the VMMC channel structure.
   \param  action          Action to be executed (set or check).
   \param  sigarray_mode   Signalling array operation mode (16kHz or 8 kHz).
   \param  module_mode     Operation mode of the module (16kHz or 8 kHz).

   \return
   If action is SM_SET: IFX_SUCCESS or IFX_ERROR.
   If action is SM_CHECK: IFX_TRUE when module would do a switch or IFX_FALSE
                          if nothing needs to be done.

*/
IFX_int32_t  VMMC_ALM_SamplingMode (VMMC_CHANNEL *pCh,
                                    SM_ACTION action,
                                    OPMODE_SMPL sigarray_mode,
                                    OPMODE_SMPL module_mode)
{
   VMMC_DEVICE       *pDev = pCh->pParent;
   ALI_CHAN_t        *p_fw_ali_ch;
   IFX_uint32_t      new_ISR, new_UD;
   IFX_int32_t       ret = IFX_SUCCESS;

   /* calling function should ensure valid parameters */
   VMMC_ASSERT(pCh != IFX_NULL);

   /* Signal array 8 kHz AND module 16 kHz is disallowed */
   VMMC_ASSERT((sigarray_mode != NB_8_KHZ) || (module_mode != WB_16_KHZ));
   /* Determine the desired value of the ISR bit from the parameters */
   new_ISR = (sigarray_mode == WB_16_KHZ) ? 1 : 0;
   /* Determine the desired value of the up-/down-sampling bit */
   /* UD is set when signal array runs in other mode than the module */
   new_UD  = (sigarray_mode != module_mode) ? 1 : 0;

   /* protect fw msg */
   VMMC_OS_MutexGet (&pCh->chAcc);

   /* get pointer to cached fw message */
   p_fw_ali_ch = &pCh->pALM->fw_ali_ch;

   /* check if the ALM module already operates in requested mode */
   if( (p_fw_ali_ch->ISR != new_ISR) ||
       (p_fw_ali_ch->UD  != new_UD) )
   {
      /* If action is execute do changes otherwise report need for execution. */
      if (action == SM_SET)
      {
         /* change the ISR bit of the ALM module */
         /* before ISR bit can be changed the ALM module must be deactivated */
         /* before the ALM module can be deactivated both LEC and ES must be
            deactivated */

         /* LEC deactivation is always needed when LEC is enabled */
         /* Coding ensures that LEC is enabled when the resource id is valid. */
         if (VMMC_RES_ID_VALID (pCh->pALM->nLecResId))
         {
            ret = VMMC_RES_LEC_Enable (pCh->pALM->nLecResId, IFX_DISABLE);
         }
         /* Echo Suppressor deactivation is mandatory when enabled */
         /* Coding ensures that ES is enabled when the resource id is valid. */
         if ((ret == VMMC_statusOk) && VMMC_RES_ID_VALID (pCh->pALM->nEsResId))
         {
            ret = VMMC_RES_ES_Enable (pCh->pALM->nEsResId, IFX_DISABLE);
         }
         /* ALM module deactivation is needed if the module is enabled */
         if ((p_fw_ali_ch->EN != ALI_CHAN_DISABLE) && (ret == IFX_SUCCESS))
         {
            /* deactivate the ALM module */
            p_fw_ali_ch->EN = ALI_CHAN_DISABLE;
            /* write ALM channel command */
            ret = CmdWrite(pDev, (IFX_uint32_t*)p_fw_ali_ch,
                           sizeof(ALI_CHAN_t)- CMD_HDR_CNT);
            /* state when entering here was enabled so restore the setting */
            p_fw_ali_ch->EN = ALI_CHAN_ENABLE;
         }

         /* Here the LEC and ALM module are turned off - change settings now */

         if (ret == IFX_SUCCESS)
         {
            /* Set the ISR bit in the cached message to the new value */
            p_fw_ali_ch->ISR = new_ISR;
            /* Set the UD bit in the cached message to the new value */
            p_fw_ali_ch->UD = new_UD;

            TRACE(VMMC, DBG_LEVEL_LOW,
                  ("Set ALM channel %u ISR = %d UD = %d\n",
                   pCh->nChannel - 1, p_fw_ali_ch->ISR, p_fw_ali_ch->UD));
         }

         if ((ret == IFX_SUCCESS) && (module_mode != pCh->pALM->module_mode))
         {
#ifdef SYSTEM_DANUBE
            /* set ALI coefficients according to sampling rate (mode) */
            ret = vmmc_alm_SamplingModeCoefSet(pCh, module_mode);
#endif /* SYSTEM_DANUBE */
            if ((ret == VMMC_statusOk) &&
                VMMC_RES_ID_VALID (pCh->pALM->nLecResId))
            {
               /* set the LEC to the current sampling rate (mode) */
               VMMC_RES_LEC_SamplingModeSet (pCh->pALM->nLecResId, module_mode);
            }
         }
         /* write cached ALM channel command to the chip if enabled */
         if ((p_fw_ali_ch->EN != ALI_CHAN_DISABLE)  && (ret == IFX_SUCCESS))
         {
            /* write ALM channel command */
            ret = CmdWrite(pDev, (IFX_uint32_t*)p_fw_ali_ch,
                           sizeof(ALI_CHAN_t)- CMD_HDR_CNT);
         }
         /* enable the ES if it was running before */
         if ((ret == VMMC_statusOk) && VMMC_RES_ID_VALID (pCh->pALM->nEsResId))
         {
            ret = VMMC_RES_ES_Enable (pCh->pALM->nEsResId, IFX_ENABLE);
         }
         /* enable the LEC if it was running before */
         if ((ret == VMMC_statusOk) && VMMC_RES_ID_VALID (pCh->pALM->nLecResId))
         {
            ret = VMMC_RES_LEC_Enable (pCh->pALM->nLecResId, IFX_ENABLE);
         }
         /* remember the module_mode so that we can avoid loading ALM coeffs
            if the module modes does not change from this one */
         pCh->pALM->module_mode = module_mode;
      }
      else
      {
         /* action is check: return that this module would do a switch */
         ret = IFX_TRUE;
      }
   }
   else
   {
      if (action == SM_SET)
      {
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("Sampling rate of ALM on channel %u already matching\n",
                pCh->nChannel - 1));
      }
      else
      {
         /* action is check: return that this module does not need a switch */
         ret = IFX_FALSE;
      }
   }

   VMMC_OS_MutexRelease (&pCh->chAcc);

   return ret;
}


#ifdef SYSTEM_DANUBE
/**
   Store sampling rate coefficients for the ALM channel inside the channel
   context strucutre to allow dynamic coefficient swapping at runtime.

   \param  pCh             Pointer to the VMMC channel structure.
   \param  mode            Choice of either NB_8_KHZ or WB_16_KHZ specifing
                           the mode for which the following coefficients
                           shall be used.
   \param  pSampl          Pointer to the sampling rate coefficients.

   \return VMMC_statusOk as this function cannot fail.
*/
IFX_int32_t VMMC_ALM_SamplingModeCoefficientStore(VMMC_CHANNEL *pCh,
                                                   OPMODE_SMPL   mode,
                                      VMMC_ALM_SAMPLING_COEFS_t *pSampl)
{
   VMMC_ALM_NB_WB_COEF_t *pCoefs = &pCh->pALM->smpl_coef[mode];

   pCoefs->block0[0] = (0x0200) | (pCh->nChannel - 1);
   pCoefs->block0[1] = (IFX_uint16_t)(pSampl->blk0_offset << 8);
   pCoefs->block0[2] = pSampl->blk0_data0;
   pCoefs->block0[3] = pSampl->blk0_data1;
   pCoefs->block0[4] = pSampl->blk0_data2;
   pCoefs->block0[5] = pSampl->blk0_data3;

   /* construct second coefficient block */
   pCoefs->block1[0] = (0x0200) | (pCh->nChannel - 1);
   pCoefs->block1[1] = (IFX_uint16_t)(pSampl->blk1_offset << 8);
   pCoefs->block1[2] = pSampl->blk1_data0;

   /* construct third coefficient block */
   pCoefs->block2[0] = (0x0200) | (pCh->nChannel - 1);
   pCoefs->block2[1] = (IFX_uint16_t)(pSampl->blk2_offset << 8);
   pCoefs->block2[2] = pSampl->blk2_data0;

   return VMMC_statusOk;
}


/**
   Configure the ALM channel for either NB or WB mode by switching the
   corresponding coefficients. As the NB/WB switch cannot be exectuted
   on the fly, this function shall be called only from
   VMMC_ALM_SamplingModeSet which stores/restores all related settings.

   \param  pCh             Pointer to the VMMC channel structure.
   \param  mode            Choice of either NB_8_KHZ or WB_16_KHZ.

   \return
   - VMMC_statusCmdWr Writing the command has failed
   - VMMC_statusOk if successful

*/
static IFX_int32_t vmmc_alm_SamplingModeCoefSet(VMMC_CHANNEL *pCh,
                                                OPMODE_SMPL   mode)
{
   VMMC_DEVICE           *pDev   = pCh->pParent;
   VMMC_ALM_NB_WB_COEF_t *pCoefs = &pCh->pALM->smpl_coef[mode];
   IFX_int32_t           ret;

   /* write the three coefficient blocks */
   ret = CmdWrite (pDev, (IFX_uint32_t*) &pCoefs->block0[0], 8);
   if (VMMC_SUCCESS(ret))
   {
      ret = CmdWrite (pDev, (IFX_uint32_t*) &pCoefs->block1[0], 2);
   }
   if (VMMC_SUCCESS(ret))
   {
      ret = CmdWrite (pDev, (IFX_uint32_t*) &pCoefs->block2[0], 2);
   }

   RETURN_STATUS (ret);
}
#endif /* SYSTEM_DANUBE */


/**
   base ALM configuration

   Use this function where needed to set the base configuration
   of Analog Line Module.

   \param  pCh             Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk         If successful
   - VMMC_statusCmdWr      Writing the command has failed
   - VMMC_statusAlmInit    Analog line intialization failed
*/
IFX_int32_t VMMC_ALM_baseConf (VMMC_CHANNEL *pCh)
{

   VMMC_DEVICE      *pDev  = pCh->pParent;
   IFX_int32_t       ret;

   /* configure ALI channel */
   pCh->pALM->fw_ali_ch.EN  = ALI_CHAN_ENABLE;

   pCh->pALM->fw_ali_ch.DG1 = VMMC_GAIN_0DB;
   pCh->pALM->fw_ali_ch.DG2 = VMMC_GAIN_0DB;

   if (pDev->caps.bEventMailboxSupported)
   {
      pCh->pALM->fw_ali_ch.EH  = ALI_CHAN_EH_ON;
   }

   /* Set the inputs in the cached message and write it */
   ret = VMMC_ALM_Set_Inputs(pCh);

   if (!VMMC_SUCCESS(ret))
   {
      /* errmsg: Analog line intialization failed, due to command write error */
      RETURN_STATUS (VMMC_statusAlmInit);
   }

   /* activate FXO channel in firmware (SmartSLIC only) */
   if (pCh->pALM->line_type_fxs == IFX_FALSE &&
       pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE) &&
       pDev->bSmartSlic == IFX_TRUE)
   {
      OPMODE_CMD_t *pOpmod = &pCh->pALM->dc_opmode;

      pOpmod->OP_MODE = VMMC_OPMOD_PDH;
      ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
      if (VMMC_SUCCESS(ret))
      {
         pOpmod->OP_MODE = VMMC_OPMOD_FXO;
         ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
      }
   }

   return ret;
}


/**
   Initalize the analog module and the cached firmware messages.

   \param  pCh             Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_ALM_InitCh (VMMC_CHANNEL *pCh)
{
   VMMC_ALMCH_t        *pALM = pCh->pALM;
   ALI_CHAN_t          *pAliCh  = &pALM->fw_ali_ch;
   OPMODE_CMD_t        *pOpmode = &pALM->dc_opmode;
   VMMC_SDD_BasicConfig_t *pSDD_BasicConfig = &pALM->fw_sdd_basic_config;
   IFX_uint8_t         ch = pCh->nChannel - 1;

   /* ALI ch message */
   memset (pAliCh, 0, sizeof (*pAliCh));
   pAliCh->CHAN  = ch;
   pAliCh->CMD   = ALI_CHAN_CMD;
   pAliCh->MOD   = MOD_ALI;
   pAliCh->ECMD  = ALI_CHAN_ECMD;

   /* DC control message for operating mode */
   memset (pOpmode, 0, sizeof (*pOpmode));
   pOpmode->CHAN = ch;
   pOpmode->CMD  = CMD_OPMODE;
   pOpmode->MOD  = MOD_DCCTL;
   pOpmode->ECMD = ECMD_OPMODE;

   /* Reading calibration data */
   memset(&pALM->fw_sdd_calibrate, 0, sizeof(pALM->fw_sdd_calibrate));
   pALM->fw_sdd_calibrate.CHAN = ch;
   pALM->fw_sdd_calibrate.CMD  = CMD_SDD;
   pALM->fw_sdd_calibrate.MOD  = SDD_Calibrate_MOD;
   pALM->fw_sdd_calibrate.ECMD = SDD_Calibrate_ECMD;

   /* no calibration was done yet */
   pALM->nCalibrationState = IFX_TAPI_CALIBRATION_STATE_NO;
   /* currently calibration is not running */
   pALM->bCalibrationRunning = IFX_FALSE;

   /* Clear the cache for the last calibration results */
   memset(&pALM->calibrationLastResults, 0,
          sizeof(pALM->calibrationLastResults));

   /* setup linetype */
   pALM->line_type_fxs = (vmmc_alm_line[ch] == VMMC_ALM_LINE_FXS) ?
                                     IFX_TRUE : IFX_FALSE;
   /* initialize FXO bits - all 0, polarity 1 (normal polarity) */
   pALM->fxo_flags = 0 | (1 << FXO_POLARITY);
#ifdef VMMC_ACT_FXO
   /* initialize default FXO line mode */
   pALM->fxo_flags |= (1 << FXO_LINE_MODE);
#endif /*VMMC_ACT_FXO*/

#ifdef SYSTEM_DANUBE
   /* use these ALM coefficient patches as a default for NB / WB mode
      normally they are overwritten with values loaded from BBD */
   pALM->smpl_coef[NB_8_KHZ] = VMMC_AlmNbCoeff;
   pALM->smpl_coef[NB_8_KHZ].block0[0] |= ch;
   pALM->smpl_coef[NB_8_KHZ].block1[0] |= ch;
   pALM->smpl_coef[NB_8_KHZ].block2[0] |= ch;
   pALM->smpl_coef[WB_16_KHZ] = VMMC_AlmWbCoeff;
   pALM->smpl_coef[WB_16_KHZ].block0[0] |= ch;
   pALM->smpl_coef[WB_16_KHZ].block1[0] |= ch;
   pALM->smpl_coef[WB_16_KHZ].block2[0] |= ch;
#endif /* SYSTEM_DANUBE */

   /* start in narrowband operation mode */
   pALM->module_mode = NB_8_KHZ;

   /* echo suppressor resource id */
   pALM->nEsResId = VMMC_RES_ID_NULL;

   /* line echo canceller resource id */
   pALM->nLecResId = VMMC_RES_ID_NULL;

   /* SDD_BasicConfig */
   memset (pSDD_BasicConfig, 0, sizeof(VMMC_SDD_BasicConfig_t));
   pSDD_BasicConfig->CMD = SDD_BasicConfig_CMD;
   pSDD_BasicConfig->CHAN = ch;
   pSDD_BasicConfig->MOD = SDD_BasicConfig_MOD;
   pSDD_BasicConfig->ECMD = SDD_BasicConfig_ECMD;
   pALM->bIdleExtNeedRestore = IFX_FALSE;

#if (VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT)
   /* Read continuous measurement from SDD */
   memset(&pALM->fw_sdd_contMeasRead, 0, sizeof(pALM->fw_sdd_contMeasRead));
   pALM->fw_sdd_contMeasRead.CHAN = ch;
   pALM->fw_sdd_contMeasRead.CMD  = CMD_SDD;
   pALM->fw_sdd_contMeasRead.ECMD = SDD_ContMeasRead_ECMD;

   /* Clear continuous measurement values in SDD */
   memset(&pALM->fw_sdd_contMearClear, 0, sizeof(pALM->fw_sdd_contMearClear));
   pALM->fw_sdd_contMearClear.CHAN = ch;
   pALM->fw_sdd_contMearClear.CMD  = CMD_SDD;
   pALM->fw_sdd_contMearClear.ECMD = SDD_ContMeasClear_ECMD;
#endif /* VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT */

   /* SDD_FxoHookSwitch */
   memset(&pALM->fw_sdd_fxoHookSwitch, 0, sizeof(pALM->fw_sdd_fxoHookSwitch));
   pALM->fw_sdd_fxoHookSwitch.CHAN = ch;
   pALM->fw_sdd_fxoHookSwitch.CMD  = CMD_SDD;
   pALM->fw_sdd_fxoHookSwitch.ECMD = SDD_FxoHookSwitch_ECMD;

   /* SDD_FxoConfig */
   memset(&pALM->fw_sdd_fxoConfig, 0, sizeof(pALM->fw_sdd_fxoConfig));
   pALM->fw_sdd_fxoConfig.CHAN   = ch;
   pALM->fw_sdd_fxoConfig.CMD    = CMD_SDD;
   pALM->fw_sdd_fxoConfig.ECMD   = SDD_FxoConfig_ECMD;
   pALM->fw_sdd_fxoConfig.TACT   = 10;
   pALM->fw_sdd_fxoConfig.DEB2   = 70;
   pALM->fw_sdd_fxoConfig.OSIT   = 100;
   pALM->fw_sdd_fxoConfig.FLASHT = 50;

#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
   /* SDD_GR909Config */
   memset(&pALM->fw_sdd_GR909Config, 0, sizeof(pALM->fw_sdd_GR909Config));
   pALM->fw_sdd_GR909Config.CHAN = ch;
   pALM->fw_sdd_GR909Config.CMD  = CMD_SDD;
   pALM->fw_sdd_GR909Config.ECMD = SDD_GR909Config_ECMD;
   pALM->fw_sdd_GR909Config.HptW2gAcLim  = 0x24EE;
   pALM->fw_sdd_GR909Config.HptW2wAcLim  = 0x24EE;
   pALM->fw_sdd_GR909Config.HptW2gDcLim  = 0x4738;
   pALM->fw_sdd_GR909Config.HptW2wDcLim  = 0x4738;
   pALM->fw_sdd_GR909Config.FemfW2gAcLim = 0x0762;
   pALM->fw_sdd_GR909Config.FemfW2wAcLim = 0x0762;
   pALM->fw_sdd_GR909Config.FemfW2gDcLim = 0x032A;
   pALM->fw_sdd_GR909Config.FemfW2wDcLim = 0x032A;
   pALM->fw_sdd_GR909Config.RftResLim    = 0x1DF5;
   pALM->fw_sdd_GR909Config.RohLinLim    = 0x000F;
   pALM->fw_sdd_GR909Config.RitLowLim    = 0x0047;
   pALM->fw_sdd_GR909Config.RitHighLim   = 0x07FD;
   /* SDD_Opmode */
   memset(&pALM->fw_sdd_opmode, 0, sizeof(pALM->fw_sdd_opmode));
   pALM->fw_sdd_opmode.CHAN = ch;
   pALM->fw_sdd_opmode.CMD  = CMD_SDD;
   pALM->fw_sdd_opmode.ECMD = SDD_Opmode_ECMD;
   /* SDD_GR909PhoneDetection */
   memset(&pALM->fw_sdd_phone_detection, 0,
          sizeof(pALM->fw_sdd_phone_detection));
   pALM->fw_sdd_phone_detection.CHAN = ch;
   pALM->fw_sdd_phone_detection.CMD  = CMD_SDD;
   pALM->fw_sdd_phone_detection.MOD  = MOD_SDD1;
   /* Initial state of capacitance measurement */
   pALM->bCapMeasInProgress = IFX_FALSE;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */

   /* initialise the structures used for connecting modules */
   VMMC_CON_Init_AlmCh (pCh);
}


/**
   Set the signal inputs of the cached fw message for the given channel.

   \param  pCh             Pointer to the VMMC channel structure.

  \return
   - VMMC_statusCmdWr Writing the command has failed
   - VMMC_statusOk if successful
*/
IFX_int32_t VMMC_ALM_Set_Inputs (VMMC_CHANNEL *pCh)
{
   ALI_CHAN_t        *p_fw_ali_ch;
   IFX_int32_t       ret;

   /* update the signal inputs of this cached msg */
   p_fw_ali_ch = &pCh->pALM->fw_ali_ch;

   VMMC_OS_MutexGet (&pCh->chAcc);
   p_fw_ali_ch->I1 = VMMC_CON_Get_ALM_SignalInput (pCh, 0);
   p_fw_ali_ch->I2 = VMMC_CON_Get_ALM_SignalInput (pCh, 1);
   p_fw_ali_ch->I3 = VMMC_CON_Get_ALM_SignalInput (pCh, 2);
   p_fw_ali_ch->I4 = VMMC_CON_Get_ALM_SignalInput (pCh, 3);
   p_fw_ali_ch->I5 = VMMC_CON_Get_ALM_SignalInput (pCh, 4);

   /* Turn on the ALM channel although it should already run by default */
   p_fw_ali_ch->EN = ALI_CHAN_ENABLE;

   /* Write the updated cached message to fw. */
   ret = CmdWrite (pCh->pParent, (IFX_uint32_t *)p_fw_ali_ch,
                   sizeof(ALI_CHAN_t) - CMD_HDR_CNT);

   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS(ret);
}


/**
   Stop ALM on this channel

   \param  pCh             Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk         If successful
   - VMMC_statusCmdWr      Writing the command has failed
*/
IFX_int32_t VMMC_ALM_ChStop (VMMC_CHANNEL *pCh)
{
   ALI_CHAN_t       *p_fw_ali_ch;
   VMMC_DEVICE      *pDev  = pCh->pParent;
   IFX_int32_t       ret   = VMMC_statusOk;

   /* calling function should ensure valid parameters */
   VMMC_ASSERT(pCh != IFX_NULL);

   if (pCh->pALM != IFX_NULL)
   {
      /* protect fw msg */
      VMMC_OS_MutexGet (&pCh->chAcc);

      /* set the line to disabled */
      ret = VMMC_TAPI_LL_ALM_Line_Mode_Set (pCh, IFX_TAPI_LINE_FEED_DISABLED, 0);

      /* before the ALM module can be deactivated both LEC and ES must be
         deactivated */

      /* LEC deactivation is always needed when LEC is enabled */
      /* Coding ensures that LEC is enabled when the resource id is valid. */
      if ((ret == VMMC_statusOk) && VMMC_RES_ID_VALID (pCh->pALM->nLecResId))
      {
         ret = VMMC_RES_LEC_Enable (pCh->pALM->nLecResId, IFX_DISABLE);
      }

      /* Echo Suppressor deactivation is mandatory when enabled */
      /* Coding ensures that ES is enabled when the resource id is valid. */
      if ((ret == VMMC_statusOk) && VMMC_RES_ID_VALID (pCh->pALM->nEsResId))
      {
         ret = VMMC_RES_ES_Enable (pCh->pALM->nEsResId, IFX_DISABLE);
      }

      /* get pointer to cached fw message */
      p_fw_ali_ch = &pCh->pALM->fw_ali_ch;

      /* ALM module deactivation is done if the module is enabled */
      if ((ret == VMMC_statusOk) && (p_fw_ali_ch->EN != ALI_CHAN_DISABLE))
      {
         /* deactivate the ALM module */
         p_fw_ali_ch->EN = ALI_CHAN_DISABLE;
         /* write ALM channel command */
         ret = CmdWrite(pDev, (IFX_uint32_t *)p_fw_ali_ch,
                        sizeof(*p_fw_ali_ch)- CMD_HDR_CNT);
      }

      VMMC_OS_MutexRelease (&pCh->chAcc);
   }

   RETURN_STATUS(ret);
}


/**
   Allocate data structure of the ALM module in the given channel.

   \param  pCh             Pointer to the VMMC channel structure.

   \return
      - VMMC_statusOk if ok
      - VMMC_statusNoMem Memory allocation failed -> struct not created

   \remarks The channel parameter is no longer checked because the calling
   function assures correct values.
*/
IFX_int32_t VMMC_ALM_Allocate_Ch_Structures (VMMC_CHANNEL *pCh)
{
   VMMC_ALM_Free_Ch_Structures (pCh);

   pCh->pALM = VMMC_OS_Malloc (sizeof(VMMC_ALMCH_t));
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Memory allocation failed */
      RETURN_STATUS (VMMC_statusNoMem);
   }
   memset(pCh->pALM, 0, sizeof(VMMC_ALMCH_t));
#ifdef EVALUATION
   if (VMMC_Eval_ALM_Allocate_Ch_Structures (pCh) != VMMC_statusOk)
   {
      RETURN_STATUS(VMMC_statusDevInitFail);
   }
#endif /* #ifdef EVALUATION */

   return VMMC_statusOk;
}


/**
   Free data structure of the ALM module in the given channel.

   \param  pCh             Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_ALM_Free_Ch_Structures (VMMC_CHANNEL *pCh)
{
   if (pCh->pALM != IFX_NULL)
   {
#ifdef EVALUATION
      VMMC_Eval_ALM_Free_Ch_Structures (pCh);
#endif /* #ifdef EVALUATION */
      VMMC_OS_Free (pCh->pALM);
      pCh->pALM = IFX_NULL;
   }
}


#if defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
/**
   Check whether SmartSLIC is connected

   \param  pDev         Pointer to the VMMC device structure.

   \return
   - IFX_TRUE           SmartSLIC is connected
   - IFX_FALSE          SmartSLIC is NOT connected
*/
IFX_boolean_t VMMC_ALM_SmartSLIC_IsConnected(VMMC_DEVICE *pDev)
{
   VMMC_SYS_SLIC_TEST_t   sysSlic;
   IFX_int32_t            ret;

   memset((IFX_void_t *) &sysSlic, 0, sizeof(VMMC_SYS_SLIC_TEST_t));

   sysSlic.CMD    = CMD_EOP;
   sysSlic.MOD    = MOD_SYSTEM;
   sysSlic.ECMD   = ECMD_SYS_SLIC_STAT;
   ret = CmdRead(pDev, (IFX_uint32_t *) &sysSlic,
                       (IFX_uint32_t *) &sysSlic,
                        SYS_SLIC_STAT_LENGTH);
   if (VMMC_SUCCESS(ret))
   {
      if (sysSlic.SmartSLICStatus & SYS_SLIC_STAT_CONNECTED)
      {
         return IFX_TRUE;
      }
   }

   return IFX_FALSE;
}
#endif /*SYSTEM_AR9 || SYSTEM_VR9*/


#if defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
/**
   Read the number of channels on the SmartSLIC.

   \param  pDev          Pointer to the VMMC device structure.
   \param  nChannels     Returns the number of analog channels.
                         Only valid if return is VMMC_statusOk.
   \param  nFXOChannels  Returns the number of natively supported FXO channels.
                         Only valid if return is VMMC_statusOk.

   \return
   - VMMC_statusOk      if successful
   - VMMC_statusErr     if no SmartSLIC is connected or revision read failed or
                        total ALM channel count for FW is greater than MAX_ALM_NUM
*/
IFX_int32_t VMMC_ALM_SmartSLIC_ChGet (VMMC_DEVICE *pDev,
                                      IFX_uint8_t *nChannels,
                                      IFX_uint8_t *nFXOChannels)
{
   VMMC_SDD_REVISION_READ_t SDDVersCmd;
   IFX_int32_t ret = VMMC_statusErr;

   if (VMMC_ALM_SmartSLIC_IsConnected(pDev))
   {
      IFX_uint8_t ch;

      memset((IFX_void_t *)&SDDVersCmd, 0, sizeof(SDDVersCmd));
      SDDVersCmd.CMD = CMD_SDD;
      SDDVersCmd.MOD = MOD_SDD;
      SDDVersCmd.ECMD = ECMD_SDD_REV;
      ret = CmdRead(pDev, (IFX_uint32_t *)&SDDVersCmd,
                          (IFX_uint32_t *)&SDDVersCmd, 8);

      if (VMMC_SUCCESS(ret))
      {
         /* sanity check */
         if (SDDVersCmd.SmartSLIC_CH > MAX_ALM_NUM)
         {
            TRACE (VMMC, DBG_LEVEL_HIGH,
            ("VMMC: ALM count in FW is greater than supported by VMMC (%d>%d)\n",
                                         SDDVersCmd.SmartSLIC_CH, MAX_ALM_NUM));
            return VMMC_statusErr;
         }
         pDev->bSmartSlic = IFX_TRUE;
         *nChannels = SDDVersCmd.SmartSLIC_CH;
         TRACE (VMMC, DBG_LEVEL_LOW,
                  ("VMMC: SDD version %d.%d\n",
                  SDDVersCmd.DCCtrlVers, SDDVersCmd.DCCtrlVersStep));
         if ((SDDVersCmd.DCCtrlVers==1 && SDDVersCmd.DCCtrlVersStep>=18) ||
              SDDVersCmd.DCCtrlVers>1)
            pDev->bSlicSupportsIdleMode = IFX_TRUE;
         /* update FXO settings */
         vmmc_alm_line[0] = !SDDVersCmd.SmartSLIC_FXO0 ?
                                          VMMC_ALM_LINE_FXS : VMMC_ALM_LINE_FXO;
         vmmc_alm_line[1] = !SDDVersCmd.SmartSLIC_FXO1 ?
                                          VMMC_ALM_LINE_FXS : VMMC_ALM_LINE_FXO;
         /* return the count of FXO channels to calling function */
         for (ch=0, *nFXOChannels=0; ch<SDDVersCmd.SmartSLIC_CH; ch++)
            if (vmmc_alm_line[ch] == VMMC_ALM_LINE_FXO)
               (*nFXOChannels)++;
         /* update HL TAPI SmartSLIC flag - required for Clare FXO support on AR9 */
         if (*nFXOChannels &&
             ((SDDVersCmd.DCCtrlVers==1 && SDDVersCmd.DCCtrlVersStep>=37) ||
              SDDVersCmd.DCCtrlVers>1))
         {
            TAPI_DEV *pTapiDev = (pDev->pChannel[0].pTapiCh)->pTapiDevice;

            IFX_TAPI_Update_SlicFxo(pTapiDev, IFX_TRUE);
         }
      }
   }

   return ret;
}
#endif /*SYSTEM_AR9 || SYSTEM_VR9*/

#if (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION)
/**
   Retrieve calibration data

   \param  pCh          Pointer to the VMMC channel structure.
   \param  pClbConfig   Result as current calibration data.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusNoRes                no resource on given channel
   - VMMC_statusCalInProgress        Calibration in progress
   - VMMC_statusOpModeWrErr          Writing the command has failed
*/
static IFX_int32_t vmmc_alm_Calibration_Get (VMMC_CHANNEL *pCh,
                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig)
{
   VMMC_DEVICE *pDev = pCh->pParent;
   VMMC_SDD_Calibrate_t *pCalibrate = &pCh->pALM->fw_sdd_calibrate;
   OPMODE_CMD_t *pOpmode = &pCh->pALM->dc_opmode;
   IFX_int32_t  ret = VMMC_statusOk;

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   /* Sanity check */
   if (pClbConfig == IFX_NULL)
   {
      RETURN_STATUS (VMMC_statusParam);
   }

   /* Set return structure to defined values. */
   memset(pClbConfig, 0x00, sizeof(*pClbConfig));
   pClbConfig->dev = pDev->nDevNr;
   pClbConfig->ch = pCh->nChannel - 1;

   /* While calibration is running no values can be read - return with error */
   if (pOpmode->OP_MODE == VMMC_OPMOD_CALIBRATE)
   {
      /* All returned values are invalid */
      pClbConfig->nState = IFX_TAPI_CALIBRATION_STATE_NO;
      /* errmsg: Current line mode is CALIBRATE */
      RETURN_STATUS (VMMC_statusCalInProgress);
   }

   /* Read calibration values from FW */
   ret = CmdRead (pDev, (IFX_uint32_t *)pCalibrate,
                  (IFX_uint32_t *)pCalibrate, SDD_Calibrate_LEN);

   if (VMMC_SUCCESS (ret))
   {
      pClbConfig->nState = pCh->pALM->nCalibrationState;
      pClbConfig->nITransOffset = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pCalibrate->TxOffset) * 5000) / 32768);
      pClbConfig->nMeOffset     = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pCalibrate->MeOffset) * 25000) / 32768);
      pClbConfig->nUlimOffset30 = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pCalibrate->UlimOffset30) * 7200) / 32768);
      pClbConfig->nUlimOffset60 = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pCalibrate->UlimOffset60) * 14400) / 32768);
      pClbConfig->nIdacGain     = (IFX_int16_t)((((IFX_int32_t)((IFX_int16_t)pCalibrate->IdacGain) * 1000) / 29257) - 1000);
      pClbConfig->nILongOffset  = IFX_TAPI_CALIBRATION_UNUSED;
      pClbConfig->nIRingOffset  = IFX_TAPI_CALIBRATION_UNUSED;
   }
   else
   {
      /* All returned values are invalid */
      pClbConfig->nState = IFX_TAPI_CALIBRATION_STATE_NO;
   }

   RETURN_STATUS(ret);
}


#if 0
/**
   Set the calibration data

   \param  pCh          Pointer to the VMMC channel structure.
   \param  pClbConfig   New calibration data.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusCalInProgress        Calibration in progress
   - VMMC_statusOpModeWrErr          Writing the command has failed
*/
static IFX_int32_t vmmc_alm_Calibration_Set (VMMC_CHANNEL *pCh,
                        IFX_TAPI_CALIBRATION_CFG_t *pClbConfig)
{
   VMMC_DEVICE *pDev = pCh->pParent;
   VMMC_SDD_Calibrate_t *pCalibrate = &pCh->pALM->fw_sdd_calibrate;
   OPMODE_CMD_t *pOpmode = &pCh->pALM->dc_opmode;
   IFX_int32_t ret;

   /* check if transition is valid, prepare command */
   if (pOpmode->OP_MODE == VMMC_OPMOD_CALIBRATE)
   {
      /* errmsg: Current line mode is CALIBRATE */
      RETURN_STATUS (VMMC_statusCalInProgress);
   }

   /* prepare command */
   /* todo: calculation not yet done */
   pCalibrate->TxOffset       = pClbConfig->nITransOffset;
   pCalibrate->MeOffset       = pClbConfig->nMeOffset;
   pCalibrate->UlimOffset30   = pClbConfig->nUlimOffset30;
   pCalibrate->UlimOffset60   = pClbConfig->nUlimOffset60;
   pCalibrate->IdacGain       = pClbConfig->nIdacGain;
   ret = CmdWrite (pDev, (IFX_uint32_t *)pCalibrate, SDD_Calibrate_LEN);

   RETURN_STATUS (ret);
}
#endif


/**
   Start calibration process for analog channel.

   \param  pCh          Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusNoRes                no resource on given channel
   - VMMC_statusCalLineNotDisabled   Invalid current line mode
   - VMMC_statusOpModeWrErr          Writing the command has failed
*/
static IFX_int32_t vmmc_alm_Calibration_Start (VMMC_CHANNEL *pCh)
{
   VMMC_DEVICE *pDev = pCh->pParent;
   OPMODE_CMD_t *pOpmode = &pCh->pALM->dc_opmode;
   IFX_int32_t ret = VMMC_statusOk;

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   /* If Calibration is already active - do nothing and return immediately. */
   if (pOpmode->OP_MODE == VMMC_OPMOD_CALIBRATE)
   {
      return VMMC_statusOk;
   }
   /* Check if transition is valid */
   if (pOpmode->OP_MODE != VMMC_OPMOD_PDH)
   {
      /* errmsg: Current line mode is not DISABLED */
      RETURN_STATUS (VMMC_statusCalLineNotDisabled);
   }

   /* Start calibration */

   if (pDev->caps.bEventMailboxSupported)
   {
      /* Enable the OPC event needed for end notification. */
      pCh->pALM->fw_ali_ch.EO = ALI_CHAN_EO_ON;
      ret = CmdWrite (pDev, (IFX_uint32_t *) &(pCh->pALM->fw_ali_ch),
                      sizeof(ALI_CHAN_t) - CMD_HDR_CNT);
   }

   if (VMMC_SUCCESS(ret))
   {
      /* Changing the operating mode does the actual start. */
      pOpmode->OP_MODE = VMMC_OPMOD_CALIBRATE;
      ret = CmdWrite (pDev, (IFX_uint32_t *)pOpmode, OPMODE_CMD_LEN);
   }

   if (!VMMC_SUCCESS (ret))
   {
      /* Writing the operation mode failed. Restore the last OP_MODE.
         The only way we could have got here was when line mode was PDH. */
      pOpmode->OP_MODE = VMMC_OPMOD_PDH;
   }

   RETURN_STATUS(ret);
}


/**
   Stop calibration process for analog channel.

   \param  pCh          Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusCalLineNotCalibrating Invalid current line mode
   - VMMC_statusOpModeWrErr          Writing the command has failed
*/
static IFX_int32_t vmmc_alm_Calibration_Stop (VMMC_CHANNEL *pCh)
{
   VMMC_DEVICE *pDev = pCh->pParent;
   OPMODE_CMD_t *pOpmode = &pCh->pALM->dc_opmode;
   IFX_int32_t ret;

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   /* check if transition is valid */
   if (pOpmode->OP_MODE != VMMC_OPMOD_CALIBRATE)
   {
      /* We are not in calibration state. So stopping is not needed.
         This is what was intended by the user and so not an error. */
      return VMMC_statusOk;
   }

   /* Stop calibration by changing operating mode to PDH */
   pOpmode->OP_MODE = VMMC_OPMOD_PDH;
   ret = CmdWrite (pDev,
                   (IFX_uint32_t *)pOpmode, sizeof(*pOpmode)-CMD_HDR_CNT);

   if (!VMMC_SUCCESS (ret))
   {
      /* Writing the operation mode failed. Restore the last OP_MODE.
         The only way we could have got here was when line mode was CALIBRATE.*/
      pOpmode->OP_MODE = VMMC_OPMOD_CALIBRATE;
   }

   RETURN_STATUS (ret);
}


/**
   Retrieve the results of the last calibration process

   \param  pLLChannel   Pointer to the VMMC channel structure.
   \param  pClbConfig   Result as current calibration data.

   \return
   - VMMC_statusOk             if successful
   - VMMC_statusInvalCh        Channel number out of range
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Results (IFX_TAPI_LL_CH_t *pLLChannel,
                                         IFX_TAPI_CALIBRATION_CFG_t *pClbConfig)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_SDD_Calibrate_t *pLastResults = &pCh->pALM->calibrationLastResults;

   /* Sanity checks */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }
   if (pClbConfig == IFX_NULL)
   {
      RETURN_STATUS (VMMC_statusParam);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   /* Set return structure to defined values. */
   memset(pClbConfig, 0x00, sizeof(*pClbConfig));
   pClbConfig->dev = pCh->pParent->nDevNr;
   pClbConfig->ch = pCh->nChannel - 1;
   pClbConfig->nState = pCh->pALM->nCalibrationState;
   pClbConfig->nITransOffset = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pLastResults->TxOffset) * 5000) / 32768);
   pClbConfig->nMeOffset     = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pLastResults->MeOffset) * 25000) / 32768);
   pClbConfig->nUlimOffset30 = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pLastResults->UlimOffset30) * 7200) / 32768);
   pClbConfig->nUlimOffset60 = (IFX_int16_t)(((IFX_int32_t)((IFX_int16_t)pLastResults->UlimOffset60) * 14400) / 32768);
   pClbConfig->nIdacGain     = (IFX_int16_t)((((IFX_int32_t)((IFX_int16_t)pLastResults->IdacGain) * 1000) / 29257) - 1000);
   pClbConfig->nILongOffset  = IFX_TAPI_CALIBRATION_UNUSED;
   pClbConfig->nIRingOffset  = IFX_TAPI_CALIBRATION_UNUSED;

   /* release protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   return VMMC_statusOk;
}


/**
   Retrieve calibration data from device

   \param  pLLChannel   Pointer to the VMMC channel structure.
   \param  pClbConfig   Result as current calibration data.

   \return
   - VMMC_statusOk             if successful
   - VMMC_statusInvalCh        Channel number out of range
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Get (IFX_TAPI_LL_CH_t *pLLChannel,
                                   IFX_TAPI_CALIBRATION_CFG_t *pClbConfig)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t ret;

   /* sanity check */
   VMMC_ASSERT(pClbConfig != IFX_NULL);
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   ret = vmmc_alm_Calibration_Get (pCh, pClbConfig);

   /* release protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}


/** Set the calibration data.

   \param  pLLChannel   Pointer to the VMMC channel structure.
   \param  pClbConfig   New calibration data.

   \return
   - VMMC_statusOk             if successful
   - VMMC_statusInvalCh        Channel number out of range

   \todo Adapt code to AR9
*/
#if 0
IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Set (IFX_TAPI_LL_CH_t *pLLChannel,
                                   IFX_TAPI_CALIBRATION_CFG_t *pClbConfig)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE *pDev = pCh->pParent;
   IFX_int32_t ret;


   /* sanity check */
   VMMC_ASSERT(pClbConfig != IFX_NULL);
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   ret = vmmc_alm_Calibration_Set (pCh, pClbConfig);

   /* release protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}
#endif


/**
   Start calibration on the given analog line.

   Wrapper to protect the actual start function that runs calibration on
   the analog line. Calibration takes about a second and ends automatically.
   The end is signalled with the event IFX_TAPI_EVENT_CALIBRATION_END.

   \param  pLLChannel   Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusInvalCh              Channel number out of range
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Start (IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   ret = vmmc_alm_Calibration_Start (pCh);

   /* release protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}


/**
   Stop a started calibration immediately.

   \param  pLLChannel   Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusInvalCh              Channel number out of range
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Stop (IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   ret = vmmc_alm_Calibration_Stop (pCh);

   /* release protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}


/**
   Finish calibration process on the analog line.

   This function is called after the firmware indicated that calibration has
   finished. It reads the calibration values and does a plausibility check.
   If values are out of range a reset to defaults is done. In the end the
   line is made available again an a TAPI event is sent to the application
   to inform about the result of the calibration process.

   \param  pLLChannel   Pointer to the VMMC channel structure.

   \return
   - VMMC_statusOk                   if successful
   - VMMC_statusInvalCh              Channel number out of range
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Calibration_Finish (IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev = pCh->pParent;
   VMMC_SDD_Calibrate_t *pCalibrate = &pCh->pALM->fw_sdd_calibrate,
                        *pLastResults = &pCh->pALM->calibrationLastResults;

   IFX_TAPI_EVENT_t tapiEvent;
   IFX_int32_t ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* This should actually never occur since this function is only called
         within TAPI HL where the channel is already checked. */
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* protect fw messages from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   if (pDev->caps.bEventMailboxSupported)
   {
      /* Disable again the OPC event that was used for end notification. */
      pCh->pALM->fw_ali_ch.EO = ALI_CHAN_EO_OFF;
      CmdWrite (pDev, (IFX_uint32_t *) &(pCh->pALM->fw_ali_ch),
                sizeof(ALI_CHAN_t) - CMD_HDR_CNT);
      /* Ignore return value because clearing this bit will not hurt us. */
   }

   /* Read the calibrated values for verification below. */
   ret = CmdRead (pDev, (IFX_uint32_t *)pCalibrate,
                  (IFX_uint32_t *)pCalibrate, SDD_Calibrate_LEN);

   if (VMMC_SUCCESS (ret))
   {
      /* Remember the just read results for later reading. */
      *pLastResults = *pCalibrate;

      TRACE(VMMC, DBG_LEVEL_LOW,
            ("Calibration results on channel %d:\n"
             "  TX path offset %i\n"
             "  Measurement equipment offset %i\n"
             "  RX DC path offset %i\n"
             "  RX DC path offset %i\n"
             "  IDAC gain correction %i\n",
             pCh->nChannel - 1,
             (IFX_int16_t)pCalibrate->TxOffset,
             (IFX_int16_t)pCalibrate->MeOffset,
             (IFX_int16_t)pCalibrate->UlimOffset30,
             (IFX_int16_t)pCalibrate->UlimOffset60,
             (IFX_int16_t)pCalibrate->IdacGain));
   }

   /* If reading the values failed or range check results that values are
      out of range reset to zero offset values to prevent that implausible
      values are used. */
   if (!VMMC_SUCCESS(ret) ||
       ((IFX_int16_t)pCalibrate->TxOffset > VMMC_SDD_CALIBRATE_TXOFFSET_MAX) ||
       ((IFX_int16_t)pCalibrate->TxOffset < VMMC_SDD_CALIBRATE_TXOFFSET_MIN) ||
       ((IFX_int16_t)pCalibrate->MeOffset > VMMC_SDD_CALIBRATE_MEOFFSET_MAX) ||
       ((IFX_int16_t)pCalibrate->MeOffset < VMMC_SDD_CALIBRATE_MEOFFSET_MIN) ||
       ((IFX_int16_t)pCalibrate->UlimOffset30 > VMMC_SDD_CALIBRATE_ULIMOFFSET30_MAX) ||
       ((IFX_int16_t)pCalibrate->UlimOffset30 < VMMC_SDD_CALIBRATE_ULIMOFFSET30_MIN) ||
       ((IFX_int16_t)pCalibrate->UlimOffset60 > VMMC_SDD_CALIBRATE_ULIMOFFSET60_MAX) ||
       ((IFX_int16_t)pCalibrate->UlimOffset60 < VMMC_SDD_CALIBRATE_ULIMOFFSET60_MIN) ||
       ((IFX_int16_t)pCalibrate->IdacGain > VMMC_SDD_CALIBRATE_IDACGAIN_MAX) ||
       ((IFX_int16_t)pCalibrate->IdacGain < VMMC_SDD_CALIBRATE_IDACGAIN_MIN)
      )
   {
      /* At least one of the values is out of range -> reset to defaults. */
      pCalibrate->TxOffset       = 0;
      pCalibrate->MeOffset       = 0;
      pCalibrate->UlimOffset30   = 0;
      pCalibrate->UlimOffset60   = 0;
      pCalibrate->IdacGain       = 0;
      ret = CmdWrite (pDev, (IFX_uint32_t *)pCalibrate, SDD_Calibrate_LEN);

      /* Set flag that calibration failed */
      pCh->pALM->nCalibrationState = IFX_TAPI_CALIBRATION_STATE_FAILED;
   }
   else
   {
      pCh->pALM->nCalibrationState = IFX_TAPI_CALIBRATION_STATE_DONE;
   }

   /* Calibration automatically returns to PDH. Update the FW-message.
      This field is also used to lock other linemode settings while calibration
      is running. */
   pCh->pALM->dc_opmode.OP_MODE = VMMC_OPMOD_PDH;

   /* release protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   /* Send the calibration-end event for the application */
   memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
   tapiEvent.ch = pCh->nChannel - 1;
   tapiEvent.id = IFX_TAPI_EVENT_CALIBRATION_END;
   if (pCh->pALM->nCalibrationState == IFX_TAPI_CALIBRATION_STATE_DONE)
   {
      tapiEvent.data.calibration = IFX_TAPI_EVENT_CALIBRATION_SUCCESS;
   }
   else
   {
      tapiEvent.data.calibration = IFX_TAPI_EVENT_CALIBRATION_ERROR_RANGE;
   }
   IFX_TAPI_Event_Dispatch(pCh->pTapiCh, &tapiEvent);

   RETURN_STATUS(ret);
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION) */


/**
   Prepare parameters and call the target configuration function to switch the
   line mode.

   A check is done, whether the channel supports a analog line. If not
   VMMC_statusInvalCh is returned.
   This function maybe called from timer and interrupt context for switching
   ringing, metering, CID.

   \param  pLLChannel      Pointer to the VMMC channel structure.
   \param  nMode           Linefeed mode.
   \param  nTapiLineMode   The currently set line mode. Usage in discussion.

   \return
   - VMMC_statusInvalCh the channel does not support a analog line
   - VMMC_statusParam The parameters are wrong
   - VMMC_statusCmdWr Writing the command has failed
   - VMMC_statusOk if successful
*/
/*lint -esym(715, nTapiLineMode) */
IFX_int32_t VMMC_TAPI_LL_ALM_Line_Mode_Set (IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_int32_t nMode,
                                           IFX_uint8_t nTapiLineMode)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev = pCh->pParent;
   IFX_int32_t ret = VMMC_statusOk;
   OPMODE_CMD_t*  pOpmod;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE &&
          nMode != IFX_TAPI_LINE_FEED_DISABLED)
   {
      /* errmsg: Protection from calling on FXO line;
         however PDH is allowed, e.g. for IFX_TAPI_DEV_STOP */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
   if (IFX_TRUE == pCh->pALM->bCapMeasInProgress)
   {
      /* errmsg: Action not possible when
         capacitance measurement is active */
      RETURN_STATUS (VMMC_statusFailCapMeasActive);
   }
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */

   pOpmod = &pCh->pALM->dc_opmode;

   /* check if transition is valid, prepare command */
   switch (nMode)
   {
      case IFX_TAPI_LINE_FEED_ACTIVE:
      case IFX_TAPI_LINE_FEED_RING_PAUSE:
      /* for backward compatibility only */
      case IFX_TAPI_LINE_FEED_NORMAL_AUTO:
      case IFX_TAPI_LINE_FEED_ACTIVE_LOW:
      case IFX_TAPI_LINE_FEED_ACTIVE_BOOSTED:
         switch (pOpmod->OP_MODE)
         {
            case VMMC_OPMOD_ACT:
            case VMMC_OPMOD_RING:
            case VMMC_OPMOD_ONHOOK:
               pOpmod->OP_MODE = VMMC_OPMOD_ACT;
               /* in case of ring pause, leave reverse
                  polarity bit as is */
               if (nMode != IFX_TAPI_LINE_FEED_RING_PAUSE)
                  pOpmod->REV_POL = VMMC_OPMOD_POL_NORMAL;
               break;
            default:
               /* errmsg: Line mode switch is invalid. Not every transition
                  is valid. */
               RETURN_INFO_STATUS (VMMC_statusInvalLMSwitch,
                                   &nMode, sizeof(nMode));
         }
         break;
      case IFX_TAPI_LINE_FEED_ACTIVE_REV:
      /* for backward compatibility only */
      case IFX_TAPI_LINE_FEED_REVERSED_AUTO:
         switch (pOpmod->OP_MODE)
         {
            case VMMC_OPMOD_ACT:
            case VMMC_OPMOD_RING:
            case VMMC_OPMOD_ONHOOK:
               pOpmod->OP_MODE = VMMC_OPMOD_ACT;
               pOpmod->REV_POL = VMMC_OPMOD_POL_REVERSE;
               break;
            default:
               /** Line mode switch is invalid. Not every transition is valid. */
               RETURN_INFO_STATUS (VMMC_statusInvalLMSwitch,
                                   &nMode, sizeof(nMode));
         }
         break;
      case IFX_TAPI_LINE_FEED_PARKED_REVERSED:
      case IFX_TAPI_LINE_FEED_STANDBY:
         switch (pOpmod->OP_MODE)
         {
            case VMMC_OPMOD_ACT:
            case VMMC_OPMOD_RING:
            case VMMC_OPMOD_ONHOOK:
            case VMMC_OPMOD_PDH:
               pOpmod->OP_MODE = VMMC_OPMOD_ONHOOK;
               pOpmod->REV_POL = VMMC_OPMOD_POL_NORMAL;
               break;
            default:
               /** Line mode switch is invalid. Not every transition is valid. */
               RETURN_INFO_STATUS (VMMC_statusInvalLMSwitch,
                                   &nMode, sizeof(nMode));
         }
         break;
      case IFX_TAPI_LINE_FEED_DISABLED:
         pOpmod->OP_MODE = VMMC_OPMOD_PDH;
         pOpmod->REV_POL = VMMC_OPMOD_POL_NORMAL;
         break;
      case IFX_TAPI_LINE_FEED_RING_BURST:
         switch (pOpmod->OP_MODE)
         {
            case VMMC_OPMOD_ACT:        /* check for hook in case of active? */
            case VMMC_OPMOD_RING:
            case VMMC_OPMOD_ONHOOK:
               /* leave reversal bit as is */
               pOpmod->OP_MODE = VMMC_OPMOD_RING;
               break;
            default:
               /* Line mode switch is invalid. Not every transition is valid. */
               RETURN_INFO_STATUS (VMMC_statusInvalLMSwitch,
                                   &nMode, sizeof(nMode));
         }
         break;
      /* unsupported linemodes */
      case IFX_TAPI_LINE_FEED_HIGH_IMPEDANCE:
      case IFX_TAPI_LINE_FEED_METER:
      case IFX_TAPI_LINE_FEED_ACT_TEST:
      case IFX_TAPI_LINE_FEED_ACT_TESTIN:
      case IFX_TAPI_LINE_FEED_DISABLED_RESISTIVE_SWITCH:
      default:
         RETURN_INFO_STATUS (VMMC_statusNotSupported,
                             &nMode, sizeof(nMode));
   }

   ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
   return ret;
}
/*lint +esym(715, nTapiLineMode) */


/**
   Set line type to FXS or FXO and the sampling mode operation.

   In case switching to FXS line type, the line mode used is Power Down High
   Impedance (PDH). For FXO the line type FXO is set.
   The sampling rate for FXS can be narrowband, wideband or automatic switching.
   For FXO the sampling rate is always narrowband.
   A check is done, whether the channel number is not higher then the supported
   analog lines. If not VMMC_statusInvalCh is returned.

   \param  pLLChannel      Pointer to the VMMC channel structure.
   \param  nType           New line type, can be IFX_TAPI_LINE_TYPE_FXS or
                           IFX_TAPI_LINE_TYPE_FXO.

   \return
   - VMMC_statusInvalCh    Channel number out of range
   - VMMC_statusParam      The parameters are wrong
   - VMMC_statusCmdWr      Writing the command has failed
   - VMMC_statusOk         If successful
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Line_Type_Set(IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_TAPI_LINE_TYPE_t nType)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   OPMODE_CMD_t *pOpmod;
   IFX_int32_t   ret = VMMC_statusOk;

   /* parameter check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   pDev   = pCh->pParent;
   pOpmod = &pCh->pALM->dc_opmode;

   /* check if setting is valid, set opmode upon changes between FXS and FXO */
   switch (nType)
   {
      case IFX_TAPI_LINE_TYPE_FXS_NB:
      case IFX_TAPI_LINE_TYPE_FXS_WB:
      case IFX_TAPI_LINE_TYPE_FXS_AUTO:
         if (pCh->pALM->line_type_fxs == IFX_FALSE &&
             pDev->bSmartSlic == IFX_TRUE)
         {
            /* SmartSLIC has fixed channel types.
               errmsg: Line type change not allowed */
            RETURN_STATUS (VMMC_statusLineTypChNotAll);
         }
         /* Change to FXS only if not already in FXS mode.
            This prevents switch to PDH when already in FXS mode. */
         if (pCh->pALM->line_type_fxs != IFX_TRUE)
         {
            pCh->pALM->line_type_fxs = IFX_TRUE;
            TRACE(VMMC, DBG_LEVEL_NORMAL,
                  ("INFO: VMMC_TAPI_LL_ALM_Line_Type_Set: Dev%d,Ch%d: "
                   "switch to FXS line type\n",
                   pDev->nDevNr, pCh->nChannel - 1));

            pOpmod->OP_MODE = VMMC_OPMOD_PDH;
            ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
         }
         break;

      case IFX_TAPI_LINE_TYPE_FXO:
         if (pCh->pALM->line_type_fxs != IFX_FALSE &&
             pDev->bSmartSlic == IFX_TRUE)
         {
            /* SmartSLIC has fixed channel types.
               errmsg: Line type change not allowed */
            RETURN_STATUS (VMMC_statusLineTypChNotAll);
         }
         /* do nothing if already in FXO mode on non-SmartSLIC system */
         if (pDev->bSmartSlic != IFX_TRUE &&
             pCh->pALM->line_type_fxs == IFX_FALSE)
         {
            break;
         }
         pCh->pALM->line_type_fxs = IFX_FALSE;
         TRACE(VMMC, DBG_LEVEL_NORMAL,
                  ("INFO: VMMC_TAPI_LL_ALM_Line_Type_Set: Dev%d,Ch%d: "
                   "switch to FXO line type\n",
                   pDev->nDevNr, pCh->nChannel - 1));

         /* switching to linemode FXO requires to switch to PDH first... */
         pOpmod->OP_MODE = VMMC_OPMOD_PDH;
         ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
         pOpmod->OP_MODE = VMMC_OPMOD_FXO;
         ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
         break;

      default:
         TRACE(VMMC, DBG_LEVEL_NORMAL,
               ("Invalid line type / sampling mode %d for analog line\n", nType));
         /* errmsg: Invalid line type or sampling mode for analog line */
         RETURN_STATUS (VMMC_statusInvalAlmTypeSmpl);
   }

   /* Set the module's ability to switch between narrowband and wideband
      operation mode. */
   switch (nType)
   {
      default:
         /* just to make the compiler happy; case is already blocked above */
      case IFX_TAPI_LINE_TYPE_FXO:
      case IFX_TAPI_LINE_TYPE_FXS_NB:
         VMMC_CON_ModuleSamplingModeSet (pCh, VMMCDSP_MT_ALM, VMMC_CON_SMPL_NB);
         break;
      case IFX_TAPI_LINE_TYPE_FXS_WB:
         VMMC_CON_ModuleSamplingModeSet (pCh, VMMCDSP_MT_ALM, VMMC_CON_SMPL_WB);
         break;
      case IFX_TAPI_LINE_TYPE_FXS_AUTO:
         VMMC_CON_ModuleSamplingModeSet (pCh, VMMCDSP_MT_ALM, VMMC_CON_SMPL_AUTO);
         break;
   }

   if (VMMC_SUCCESS (ret))
   {
      /* reevaluate sampling in the conference that this module belongs to */
      ret = VMMC_CON_MatchConfSmplRate(pCh, VMMCDSP_MT_ALM);
   }

   return ret;
}


/**
   This test service generates hook events.

   \param  pLLChannel      Pointer to the VMMC channel structure.
   \param  bHook           IFX_TRUE generate off-hook,
                           IFX_FALSE generates on-hook.

   \return
   - VMMC_statusInvalCh the channel does not support a analog line
   - VMMC_statusCmdWr Writing the command has failed
   - VMMC_statusOk if successful

   \remarks
   The hook event is triggered by calling the irq handler directly from
   this function. The hook event then gets to the hook state machine for
   validation. Depending on the timing of calling this interface
   also hook flash and pulse dialing can be verified.
*/
IFX_int32_t VMMC_TAPI_LL_ALM_VMMC_Test_HookGen(IFX_TAPI_LL_CH_t *pLLChannel,
                                               IFX_boolean_t bHook)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   MbxEventRegs_s nHookEvent;

   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   memset (&nHookEvent, 0, sizeof(nHookEvent));
   /* set required event */
   if(pCh->nChannel == 1)
   {
      if(bHook == IFX_FALSE)
         nHookEvent.MPS_Ad1Reg.fld.onhook0 = 1;
      else
         nHookEvent.MPS_Ad1Reg.fld.offhook0 = 1;
   }
   else
   {
      if(bHook == IFX_FALSE)
         nHookEvent.MPS_Ad1Reg.fld.onhook1 = 1;
      else
         nHookEvent.MPS_Ad1Reg.fld.offhook1 = 1;
   }

   VMMC_Ad1_Callback(pCh->pParent, &nHookEvent);

   return VMMC_statusOk;
}


/**
   This service controls an 8 kHz loop in the device or the driver for testing.
   Additionally the signals will pass through to the line undisturbed.

   \param  pLLChannel      Pointer to the VMMC channel structure.
   \param  pLoop           if IFX_TRUE a loop is switched on
                           if IFX_FALSE a loop is switched off

   \return
   - VMMC_statusInvalCh    Channel number out of range
   - VMMC_statusParam      The parameters are wrong
   - VMMC_statusCmdWr      Writing the command has failed
   - VMMC_statusOk         If successful
*/
IFX_int32_t VMMC_TAPI_LL_ALM_VMMC_Test_Loop(IFX_TAPI_LL_CH_t *pLLChannel,
                                            IFX_TAPI_TEST_LOOP_t* pLoop)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE *pDev = pCh->pParent;
   IFX_int32_t ret;

   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   {
#ifdef SYSTEM_DANUBE
      DCCTL_DBG_CMD_t debugCfg;

      memset (&debugCfg, 0, sizeof (DCCTL_DBG_CMD_t));
      debugCfg.CMD = CMD_ALI;
      debugCfg.MOD = MOD_DCCTL;
      debugCfg.ECMD = ECMD_DCCTL_DEBUG;
      debugCfg.CHAN = pCh->nChannel - 1;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)&debugCfg),
                           (IFX_uint32_t *)((IFX_void_t *)&debugCfg),
                           DCCTL_CMD_LEN);

      if (!VMMC_SUCCESS(ret))
      {
         RETURN_STATUS (ret);
      }

      /* depending on the request enable or disable testloop */
      if (pLoop->bAnalog)
      {
         /* enable debug operation */
         debugCfg.EN_DEBUG = 1;
         /* make sure that the AC path is enabled */
         debugCfg.AC_EN = 1;
         /* close digital loop at 8kHz */
         debugCfg.AC_DLB8K_EN = 1;
      }
      else
      {
         /* enable normal operation */
         debugCfg.EN_DEBUG = 0;
         /* normal operation, digital loop at 8 kHz open */
         debugCfg.AC_DLB8K_EN = 0;
      }

      /* write updated message contents */
      ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)&debugCfg),
                      DCCTL_CMD_LEN);
#elif defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
      IFX_uint32_t dcctrlLoop[2];
      IFX_uint32_t ch = (IFX_uint32_t)(pCh->nChannel - 1);

      memset (&dcctrlLoop, 0, sizeof (dcctrlLoop));
      dcctrlLoop[0] = 0x04001E04L | (ch & 0x0000000FL) << 16;

      /* depending on the request enable or disable testloop */
      if (pLoop->bAnalog)
      {
         /* enable debug operation */
         dcctrlLoop[1] = 0x00000001L;
      }
      else
      {
         /* enable normal operation */
         dcctrlLoop[1] = 0x00000000L;
      }

      /* write the message */
      ret = CmdWrite (pDev, dcctrlLoop, 4);
#endif /* SYSTEM_... */
   }

   RETURN_STATUS (ret);
}


/**
   Set the phone volume

   Gain Parameter are given in 'dB'. The range is -24dB ... 12dB.
   This function enables the ALM in firmware. There will be
   no error reported if it is not enabled before (IFX_TAPI_CH_INIT done).

   \param  pLLChannel      Pointer to the VMMC channel structure.
   \param  pVol            Pointer to IFX_TAPI_LINE_VOLUME_t structure.

   \return
   - VMMC_statusInvalCh    Channel number out of range
   - VMMC_statusCmdWr Writing the command has failed
   - VMMC_statusParamRange Wrong parameters passed. This code is returned
     when any gain parameter is lower than -24 dB or higher than 12 dB
   - VMMC_statusOk if successful
*/
IFX_int32_t VMMC_TAPI_LL_ALM_Volume_Set (IFX_TAPI_LL_CH_t *pLLChannel,
                                         IFX_TAPI_LINE_VOLUME_t const *pVol)
{
   VMMC_CHANNEL   *pCh = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t     ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* range check, because gain var is integer */
   if ((pVol->nGainTx < VMMC_VOLUME_GAIN_MIN) ||
       (pVol->nGainTx > VMMC_VOLUME_GAIN_MAX) ||
       (pVol->nGainRx < VMMC_VOLUME_GAIN_MIN) ||
       (pVol->nGainRx > VMMC_VOLUME_GAIN_MAX))
   {
      /* errmsg: parameters are out of the supported range */
      RETURN_STATUS (VMMC_statusParamRange);
   }

   /* protect fw msg */
   VMMC_OS_MutexGet (&pCh->chAcc);

   pCh->pALM->fw_ali_ch.DG1 = VMMC_Gaintable[pVol->nGainTx +
                                             (-VMMC_VOLUME_GAIN_MIN)];
   pCh->pALM->fw_ali_ch.DG2 = VMMC_Gaintable[pVol->nGainRx +
                                             (-VMMC_VOLUME_GAIN_MIN)];
   pCh->pALM->fw_ali_ch.EN  = ALI_CHAN_ENABLE;

   ret = CmdWrite (pCh->pParent,
                   (IFX_uint32_t *) &(pCh->pALM->fw_ali_ch), ALI_CHAN_LEN);

   /* release fw protection */
   VMMC_OS_MutexRelease (&pCh->chAcc);

   return ret;
}


/**
   This service enables or disables a high level path of a phone channel.

   It is intended for phone channels only and must be used in
   combination with IFX_TAPI_PHONE_VOLUME_SET or IFX_TAPI_PCM_VOLUME_SET
   to set the max. level (IFX_TAPI_LINE_VOLUME_HIGH) or to restore level

   \param  pLLChannel      Pointer to the VMMC channel structure.
   \param  bEnable         The parameter represent a boolean value of
                           IFX_TAPI_LINE_LEVEL_t.
                           - 0: IFX_TAPI_LINE_LEVEL_DISABLE,
                                disable the high level path.
                           - 1: IFX_TAPI_LINE_LEVEL_ENABLE,
                                enable the high level path.

   \return
   - VMMC_statusInvalCh    Channel number out of range
   - VMMC_statusCmdWr Writing the command failed
   - VMMC_statusOk if successful

*/
IFX_int32_t VMMC_TAPI_LL_ALM_Volume_High_Level  (IFX_TAPI_LL_CH_t *pLLChannel,
                                                 IFX_int32_t bEnable)
{
   VMMC_CHANNEL   *pCh = (VMMC_CHANNEL *) pLLChannel;
   OPMODE_CMD_t*  pOpmod;
   IFX_int32_t    ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   pOpmod = &pCh->pALM->dc_opmode;

   /* Update the cached message with the latest values from the FW because
      the FW may have autonomously changed the opmode due to an off-hook. */
   ret = CmdRead(pCh->pParent, (IFX_uint32_t *)pOpmod,
                               (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);

   if (VMMC_SUCCESS (ret))
   {
      pOpmod->HOWLER = bEnable ? 1 : 0;

      ret = CmdWrite (pCh->pParent, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
   }

   return ret;
}


#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
/**
   Sets the LEC configuration on the analog line.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  pLecConf     Handle to IFX_TAPI_LEC_CFG_t structure.

   \return
   - VMMC_statusInvalCh    Channel number out of range
   - VMMC_statusFuncParm Wrong parameters passed. This code is returned
     when the nOpMode parameter has an invalid value.
   - VMMC_statusNotSupported Requested action is not supported. This code
     is returned when a WLEC is requested but not supported by the VoFW.
   - VMMC_statusNoRes No free LEC resources is available
   - VMMC_statusCmdWr Writing the command has failed
   - VMMC_statusOk if successful
*/
IFX_int32_t VMMC_TAPI_LL_ALM_LEC_Set (IFX_TAPI_LL_CH_t *pLLChannel,
                                     TAPI_LEC_DATA_t *pLecConf)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t   ret  = VMMC_statusOk;
   IFX_enDis_t   bEnES,
                 bEnNLP = IFX_DISABLE;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* Needed for LEC and ES activation. */
   bEnES = ((pLecConf->nOpMode == IFX_TAPI_WLEC_TYPE_ES) ||
            (pLecConf->nOpMode == IFX_TAPI_WLEC_TYPE_NE_ES) ||
            (pLecConf->nOpMode == IFX_TAPI_WLEC_TYPE_NFE_ES)) ?
           IFX_ENABLE : IFX_DISABLE;

   /* protect channel variables */
   VMMC_OS_MutexGet (&pCh->chAcc);

   if ((pLecConf->nOpMode == IFX_TAPI_WLEC_TYPE_OFF) ||
       (pLecConf->nOpMode == IFX_TAPI_WLEC_TYPE_ES)    )
   {
      /* Disable the line echo canceller. */

      if (VMMC_RES_ID_VALID(pCh->pALM->nLecResId))
      {
         /* Disable the LEC, free resource and forget the resource ID. */
         ret = VMMC_RES_LEC_Enable (pCh->pALM->nLecResId, IFX_DISABLE);
         if (VMMC_SUCCESS (ret))
         {
            ret = VMMC_RES_LEC_Release (pCh->pALM->nLecResId);
         }
         if (VMMC_SUCCESS (ret))
         {
            pCh->pALM->nLecResId = VMMC_RES_ID_NULL;
         }
      }
   }
   else
   {
      /* Enable the line echo canceller. */

      VMMC_RES_LEC_MODE_t nOperatingMode;

      /* Note: ALM is always enabled so we do not need to check for this and
         can start right away to enable the LEC and/or the ES. */

      /* Translate the operating mode parameter */
      switch (pLecConf->nOpMode)
      {
         case  IFX_TAPI_WLEC_TYPE_NE:
         case  IFX_TAPI_WLEC_TYPE_NE_ES:
            nOperatingMode = VMMC_RES_LEC_MODE_NLEC;
            break;
         case  IFX_TAPI_WLEC_TYPE_NFE:
         case  IFX_TAPI_WLEC_TYPE_NFE_ES:
            nOperatingMode = VMMC_RES_LEC_MODE_WLEC;
            break;
         default:
            VMMC_OS_MutexRelease (&pCh->chAcc);
            RETURN_STATUS (VMMC_statusParam);
      }

      /* Translate the NLP parameter */
      bEnNLP = (pLecConf->bNlp == IFX_TAPI_LEC_NLP_OFF) ?
               IFX_DISABLE : IFX_ENABLE;

      /* Do parameter checks and corrections on the window size parameters */
      ret = VMMC_RES_LEC_CoefWinValidate (VMMC_RES_MOD_ALM,
                                          nOperatingMode, pLecConf);
      if (!VMMC_SUCCESS (ret))
      {
         VMMC_OS_MutexRelease (&pCh->chAcc);
         RETURN_STATUS (ret);
      }

      if (!VMMC_RES_ID_VALID(pCh->pALM->nLecResId))
      {
         /* allocate a LEC resource */
         pCh->pALM->nLecResId = VMMC_RES_LEC_Allocate (pCh, VMMC_RES_MOD_ALM);

         if (!VMMC_RES_ID_VALID(pCh->pALM->nLecResId))
         {
            VMMC_OS_MutexRelease (&pCh->chAcc);
            RETURN_STATUS (VMMC_statusNoRes);
         }

         /* set the current sampling mode */
         VMMC_RES_LEC_SamplingModeSet (pCh->pALM->nLecResId,
                                       pCh->pALM->module_mode);
      }

      VMMC_RES_LEC_OperatingModeSet (pCh->pALM->nLecResId,
                                     nOperatingMode, bEnNLP);

      /* Set window sizes for 8kHz sampling mode */
      switch (nOperatingMode)
      {
         default:
         case VMMC_RES_LEC_MODE_NLEC:
            /* NLEC operating mode */
            VMMC_RES_LEC_CoefWinSet (pCh->pALM->nLecResId,
                                     NB_8_KHZ, nOperatingMode,
                                     (IFX_uint8_t)pLecConf->nNBNEwindow, 0);
            break;
         case VMMC_RES_LEC_MODE_WLEC:
            /* WLEC operating mode */
            VMMC_RES_LEC_CoefWinSet (pCh->pALM->nLecResId,
                                     NB_8_KHZ, nOperatingMode,
                                     (IFX_uint8_t)pLecConf->nNBNEwindow +
                                     (IFX_uint8_t)pLecConf->nNBFEwindow,
                                     (IFX_uint8_t)pLecConf->nNBFEwindow);
            break;
      }

      /* Set window sizes for 16kHz sampling mode */
      VMMC_RES_LEC_CoefWinSet (pCh->pALM->nLecResId,
                               WB_16_KHZ, nOperatingMode,
                               (IFX_uint8_t)pLecConf->nWBNEwindow, 0);

      /* Set information if the ES is also working on this channel */
      VMMC_RES_LEC_ParameterSelect (pCh->pALM->nLecResId, bEnES);

      /* Finally enable the echo suppressor resource */
      ret = VMMC_RES_LEC_Enable (pCh->pALM->nLecResId, IFX_ENABLE);
   }

   VMMC_OS_MutexRelease (&pCh->chAcc);

   if (VMMC_SUCCESS(ret))
   {
      /* Turn the echo suppressor on or off as requested. */
      if (bEnES == IFX_ENABLE)
      {
         ret = vmmc_alm_Echo_Suppressor_Set (pLLChannel, IFX_ENABLE, bEnNLP);
      }
      else
      {
         ret = vmmc_alm_Echo_Suppressor_Set (pLLChannel, IFX_DISABLE, bEnNLP);
      }
   }

   RETURN_STATUS (ret);
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */


/**
   Enable or disable the echo suppressor.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  nEnable      Enable (<>0) or disable (=0) the echo suppressor.
   \param  nActiveNLP   Flag that indicates if LEC+NLP is active. This is used
                        to select a different parameter set.

   \return
      - VMMC_statusInvalCh Channel number is out of range
      - VMMC_statusNoRes No echo suppressor resource available
      - VMMC_statusCmdWr Writing the command has failed
      - VMMC_statusOk if successful
*/
static IFX_int32_t vmmc_alm_Echo_Suppressor_Set (IFX_TAPI_LL_CH_t *pLLChannel,
                                                 IFX_enDis_t nEnable,
                                                 IFX_enDis_t nActiveNLP)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t   ret  = VMMC_statusOk;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   if (nEnable == IFX_DISABLE)
   {
      /* Disable the echo suppressor. */

      if (VMMC_RES_ID_VALID(pCh->pALM->nEsResId))
      {
         /* Disable the ES, free resource and forget the resource ID. */
         ret = VMMC_RES_ES_Enable (pCh->pALM->nEsResId, IFX_DISABLE);
         if (VMMC_SUCCESS (ret))
         {
            ret = VMMC_RES_ES_Release (pCh->pALM->nEsResId);
         }
         if (VMMC_SUCCESS (ret))
         {
            pCh->pALM->nEsResId = VMMC_RES_ID_NULL;
         }
      }
   }
   else
   {
      /* Enable the echo suppressor. */

      if (!VMMC_RES_ID_VALID(pCh->pALM->nEsResId))
      {
         /* allocate an ES resource */
         pCh->pALM->nEsResId = VMMC_RES_ES_Allocate (pCh, VMMC_RES_MOD_ALM);
      }

      if (!VMMC_RES_ID_VALID(pCh->pALM->nEsResId))
      {
         RETURN_STATUS (VMMC_statusNoRes);
      }

      /* In the ES_Enable() we need to know if the LEC+NLP is active on this
         channel. */
      ret = VMMC_RES_ES_ParameterSelect (pCh->pALM->nEsResId, nActiveNLP);

      if (VMMC_SUCCESS (ret))
      {
         ret = VMMC_RES_ES_Enable (pCh->pALM->nEsResId, IFX_ENABLE);
      }
   }

   RETURN_STATUS (ret);
}


/**
   Correct the cached linemode inside the low level device driver.

   \param  pCh          Pointer to the VMMC channel structure.
   \param  lm           New linemode.
*/
IFX_void_t VMMC_ALM_CorrectLinemodeCache (VMMC_CHANNEL *pCh,
                                          IFX_uint16_t lm)
{
   VMMC_ASSERT ((pCh != IFX_NULL) && (pCh->pALM != IFX_NULL));

   pCh->pALM->dc_opmode.OP_MODE = lm;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
   pCh->pALM->fw_sdd_opmode.OperatingMode = dc_opmode2sdd_opmode[lm];
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */
}

#if (VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT)
/**
   Request continuous measurement results.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return
      - VMMC_statusInvalCh
      - VMMC_statusFXSCallOnFXO
      - VMMC_statusOk

   \remarks
      This function does nothing on firmware interface. It spawns an
      event IFX_TAPI_EVENT_CONTMEASUREMENT in a task context. Added for
      compatibility with TAPI_V4 interface.
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_ContMeas_Req(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   TAPI_CHANNEL    *pChannel = pCh->pTapiCh;
   IFX_TAPI_EVENT_t tapiEvent;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   memset (&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
   tapiEvent.id = IFX_TAPI_EVENT_CONTMEASUREMENT;
   IFX_TAPI_Event_Dispatch(pChannel, &tapiEvent);

   RETURN_STATUS (VMMC_statusOk);
}

/**
   Reset continuous measurement results.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return
      - VMMC_statusInvalCh
      - VMMC_statusFXSCallOnFXO
      - VMMC_statusCmdWr Writing the command has failed
      - VMMC_statusOk if successful
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_ContMeas_Reset(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev = pCh->pParent;
   VMMC_SDD_ContMeasClear_t *pContMeasClear = &pCh->pALM->fw_sdd_contMearClear;
   IFX_int32_t   ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   pContMeasClear->ClearCMD = 1;
   ret = CmdWrite(pDev, (IFX_uint32_t*)pContMeasClear, SDD_ContMeasClear_LEN);

   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}

/**
   Return the stored continuous measure results.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return
      - VMMC_statusInvalCh
      - VMMC_statusFXSCallOnFXO
      - IFX_ERROR CmdRead error
      - VMMC_statusOk if successful
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_ContMeas_Get(IFX_TAPI_LL_CH_t *pLLChannel,
                                      IFX_TAPI_CONTMEASUREMENT_GET_t *pContMeas)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev = pCh->pParent;
   VMMC_SDD_ContMeasRead_t *pContMeasRead = &pCh->pALM->fw_sdd_contMeasRead;
   IFX_int32_t   ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   ret = CmdRead(pDev, (IFX_uint32_t *)pContMeasRead,
                       (IFX_uint32_t *)pContMeasRead, SDD_ContMeasRead_LEN);

   if (VMMC_SUCCESS(ret))
   {
      /* typecasts are mandatory because of signed values */
      pContMeas->nVLineWireRing =
                        ((IFX_int16_t)pContMeasRead->VlineRing * 14400) / 32768;
      pContMeas->nVLineWireTip =
                         ((IFX_int16_t)pContMeasRead->VlineTip * 14400) / 32768;
      pContMeas->nVLineDesired = (pContMeasRead->Vlim * 14400) >> 15;
      pContMeas->nILine = ((IFX_int16_t)pContMeasRead->Itrans * 10000) / 32768;
      pContMeas->nILineLong =
                            ((IFX_int16_t)pContMeasRead->Ilong * 10000) / 32768;
      pContMeas->nILineRingPeak = (pContMeasRead->Iring * 10000) >> 15;
      pContMeas->nVLineRingPeak = (pContMeasRead->Vring * 14400) >> 15;
      pContMeas->nTtxMeterReal = pContMeasRead->TtxReal;
      pContMeas->nTtxMeterImag = pContMeasRead->TtxImag;
      pContMeas->nTtxMeterLen = pContMeasRead->TtxLen;
      pContMeas->nITtxMeter = pContMeasRead->Ittx;
      pContMeas->nVTtxMeter = pContMeasRead->Vttx;
      pContMeas->nVBat = ((IFX_int16_t)pContMeasRead->Vbat * 14400) / 32768;
   }

   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}
#endif /* VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT */

/**
   Set the new hook state on FXO line.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  hook         Enum type for TAPI FXO hook

   \return
      - VMMC_statusInvalCh if no ALM resource
      - VMMC_statusParam if hook parameter is ambiguous
      - VMMC_statusLineNotFXO if line is not FXO
      - IFX_ERROR CmdWrite error
      - VMMC_statusOk if successful
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoHookSet(IFX_TAPI_LL_CH_t *pLLChannel,
                                               IFX_TAPI_FXO_HOOK_t hook)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_FxoHookSwitch_t *pFxoHookSwitch;
   IFX_int32_t   ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs != IFX_FALSE)
   {
      /* errmsg: Protection from calling on non-FXO line */
      RETURN_STATUS (VMMC_statusLineNotFXO);
   }

   if (!(pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE)))
   {
      /* errmsg: Protection from calling on disabled FXO line */
      RETURN_STATUS (VMMC_statusFXOLineDisabled);
   }
   pDev = pCh->pParent;
   pFxoHookSwitch = &pCh->pALM->fw_sdd_fxoHookSwitch;

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);
   switch (hook)
   {
      case IFX_TAPI_FXO_HOOK_ONHOOK:
         pFxoHookSwitch->HF = 0;
         break;
      case IFX_TAPI_FXO_HOOK_OFFHOOK:
         pFxoHookSwitch->HF = 1;
         break;
      default:
         ret = VMMC_statusParam;
         goto error;
   }

   ret = CmdWrite(pDev, (IFX_uint32_t*)pFxoHookSwitch, SDD_FxoHookSwitch_LEN);

error:
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}

/**
   Issue the hook flash on FXO line.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return
      - VMMC_statusInvalCh if no ALM resource
      - VMMC_statusLineNotFXO if line is not FXO
      - IFX_ERROR CmdWrite error
      - VMMC_statusOk if successful
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoFlashSet(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_FxoHookSwitch_t *pFxoHookSwitch;
   IFX_int32_t   ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs != IFX_FALSE)
   {
      /* errmsg: Protection from calling on non-FXO line */
      RETURN_STATUS (VMMC_statusLineNotFXO);
   }

   if (!(pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE)))
   {
      /* errmsg: Protection from calling on disabled FXO line */
      RETURN_STATUS (VMMC_statusFXOLineDisabled);
   }
   pDev = pCh->pParent;
   pFxoHookSwitch = &pCh->pALM->fw_sdd_fxoHookSwitch;

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);

   pFxoHookSwitch->HF = 2;
   ret = CmdWrite(pDev, (IFX_uint32_t*)pFxoHookSwitch, SDD_FxoHookSwitch_LEN);

   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}

/**
   Configure the hook flash time.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  p_fhCfg      Pointer to IFX_TAPI_FXO_FLASH_CFG_t structure.

   \return
      - VMMC_statusInvalCh if no ALM resource
      - IFX_ERROR CmdWrite error
      - VMMC_statusOk if successful

   \remarks
      No protection from calling on non-FXO line or disabled FXO line
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoFlashCfg(IFX_TAPI_LL_CH_t *pLLChannel,
                                             IFX_TAPI_FXO_FLASH_CFG_t *p_fhCfg)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_FxoConfig_t *pFxoConfig;
   IFX_int32_t   ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }
   pDev = pCh->pParent;
   pFxoConfig = &pCh->pALM->fw_sdd_fxoConfig;

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);
   pFxoConfig->FLASHT = p_fhCfg->nFlashTime >> 1;
   ret = CmdWrite(pDev, (IFX_uint32_t*)pFxoConfig, SDD_FxoConfig_LEN);
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS (ret);
}

/**
   Configure the maximum detection time for an open switching interval (OSI).

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  p_osiCfg     Pointer to IFX_TAPI_FXO_OSI_CFG_t structure.

   \return
      - VMMC_statusInvalCh if no ALM resource
      - VMMC_statusParam if OSI max time <= 70 ms
      - IFX_ERROR CmdWrite error
      - VMMC_statusOk if successful

   \remarks
      No protection from calling on non-FXO line or disabled FXO line
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoOsiCfg(IFX_TAPI_LL_CH_t *pLLChannel,
                                              IFX_TAPI_FXO_OSI_CFG_t *p_osiCfg)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_FxoConfig_t *pFxoConfig;
   IFX_int32_t   ret;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   /* time must be > 70 ms */
   if (p_osiCfg->nOSIMax <= 70)
   {
      RETURN_STATUS (VMMC_statusParam);
   }
   pDev = pCh->pParent;
   pFxoConfig = &pCh->pALM->fw_sdd_fxoConfig;

   /* protect channel from mutual access */
   VMMC_OS_MutexGet (&pCh->chAcc);
   pFxoConfig->OSIT = p_osiCfg->nOSIMax >> 1;
   ret = CmdWrite(pDev, (IFX_uint32_t*)pFxoConfig, SDD_FxoConfig_LEN);
   VMMC_OS_MutexRelease (&pCh->chAcc);
   RETURN_STATUS (ret);
}

/**
   Get current FXO hook state.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  p_hook       Pointer to IFX_TAPI_FXO_HOOK_t enum.

   \return
      - VMMC_statusInvalCh if no ALM resource
      - VMMC_statusLineNotFXO if line is not FXO
      - IFX_SUCCESS

   \remarks
      The hook status cannot be read from firmware as SDD_FxoHookSwitch
      command is write-only. The hook state is obtained from SDD_FxoHookSwitch
      command cache in ALM context.
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoHookGet(IFX_TAPI_LL_CH_t *pLLChannel,
                                               IFX_TAPI_FXO_HOOK_t *p_hook)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_SDD_FxoHookSwitch_t *pFxoHookSwitch;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs != IFX_FALSE)
   {
      /* errmsg: Protection from calling on non-FXO line */
      RETURN_STATUS (VMMC_statusLineNotFXO);
   }

   if (!(pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE)))
   {
      /* errmsg: Protection from calling on disabled FXO line */
      RETURN_STATUS (VMMC_statusFXOLineDisabled);
   }
   pFxoHookSwitch = &pCh->pALM->fw_sdd_fxoHookSwitch;

   switch (pFxoHookSwitch->HF)
   {
      case 1:
         *p_hook = IFX_TAPI_FXO_HOOK_OFFHOOK;
         break;
      default:
         *p_hook = IFX_TAPI_FXO_HOOK_ONHOOK;
         break;
   }
   return IFX_SUCCESS;
}

/**
   Get FXO status bit.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  bit_pos      Bit position defined in drv_tapi_ll_interface.h.
                        Can be one of FXO_BATTERY,FXO_APOH,FXO_POLARITY,
                        FXO_RING.
   \param  enable       Pointer to IFX_enDis_t; output value.

   \return
      - IFX_SUCCESS

   \remarks
      Just a trivial converter from bit value of fxo_flags to
      IFX_enDis_t enum.
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoStatGet(IFX_TAPI_LL_CH_t *pLLChannel,
                                               IFX_uint32_t bit_pos,
                                               IFX_enDis_t *enable)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs != IFX_FALSE)
   {
      /* errmsg: Protection from calling on non-FXO line */
      RETURN_STATUS (VMMC_statusLineNotFXO);
   }

   if (!(pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE)))
   {
      /* errmsg: Protection from calling on disabled FXO line */
      RETURN_STATUS (VMMC_statusFXOLineDisabled);
   }

   *enable = (pCh->pALM->fxo_flags >> bit_pos & 1) ? IFX_ENABLE : IFX_DISABLE;

   return IFX_SUCCESS;
}

/**
   FXO line mode set.

   \param  pLLChannel   Handle to TAPI low level channel structure.
   \param  pMode        Pointer to IFX_TAPI_FXO_LINE_MODE_t type

   \return

*/
static IFX_int32_t VMMC_TAPI_LL_ALM_FxoLineModeSet(IFX_TAPI_LL_CH_t *pLLChannel,
                                                IFX_TAPI_FXO_LINE_MODE_t *pMode)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   OPMODE_CMD_t *pOpmod;
   IFX_int32_t  ret = IFX_SUCCESS;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs != IFX_FALSE)
   {
      /* errmsg: Protection from calling on non-FXO line */
      RETURN_STATUS (VMMC_statusLineNotFXO);
   }

   pDev = pCh->pParent;
   pOpmod = &pCh->pALM->dc_opmode;
   VMMC_OS_MutexGet (&pCh->chAcc);

   switch (pMode->mode)
   {
      case IFX_TAPI_FXO_LINE_MODE_DISABLED:
         {
            /* if already disabled - do nothing and exit */
            if (!(pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE)))
               goto exit;

            pOpmod->OP_MODE = VMMC_OPMOD_PDH;
         }
         break;

      case IFX_TAPI_FXO_LINE_MODE_ACTIVE:
         {
            /* if already active - do nothing and exit */
            if (pCh->pALM->fxo_flags & (1 << FXO_LINE_MODE))
               goto exit;

            /* switching to linemode FXO requires to switch to PDH first;
               if current mode is not PDH - switch to PDH */
            if (pOpmod->OP_MODE != VMMC_OPMOD_PDH)
            {
               pOpmod->OP_MODE = VMMC_OPMOD_PDH;
               ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);
               if (!VMMC_SUCCESS(ret))
               {
                  goto exit;
               }
            }
            pOpmod->OP_MODE = VMMC_OPMOD_FXO;
         }
         break;

      default:
         {
            RETURN_STATUS (VMMC_statusParam);
         }
         break;
   } /* switch */

   ret = CmdWrite(pDev, (IFX_uint32_t *)pOpmod, OPMODE_CMD_LEN);

   /* update FXO status bit */
   if (VMMC_SUCCESS(ret) &&
       pMode->mode == IFX_TAPI_FXO_LINE_MODE_ACTIVE)
   {
      pCh->pALM->fxo_flags |= (1 << FXO_LINE_MODE);
   }

   if (VMMC_SUCCESS(ret) &&
       pMode->mode == IFX_TAPI_FXO_LINE_MODE_DISABLED)
   {
      pCh->pALM->fxo_flags &= ~(1 << FXO_LINE_MODE);
   }

exit:
   VMMC_OS_MutexRelease (&pCh->chAcc);
   RETURN_STATUS(ret);
}

#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
/**
   Start an analog line capacitance measurement.

   \param  pCh pointer to VMMC_CHANNEL structure

   \return

   \remarks
      This function configures GR909 for phone detection
      and starts the GR909 measurement in FW.
*/
static IFX_int32_t vmmc_alm_Capacitance_Measurement_Start(VMMC_CHANNEL *pCh)
{
   VMMC_DEVICE            *pDev = pCh->pParent;
   VMMC_SDD_GR909Config_t *pSddGR909Config = &pCh->pALM->fw_sdd_GR909Config;
   VMMC_SDD_Opmode_t      *pSddOpmode = &pCh->pALM->fw_sdd_opmode;
   IFX_int32_t            ret;

   pCh->pALM->bCapMeasInProgress = IFX_TRUE;
   /* configure SDD to start line capacitance measurement */
   pSddGR909Config->Pdf = 1;
   ret = CmdWrite(pDev, (IFX_uint32_t *)pSddGR909Config,
                                        SDD_GR909Config_LEN);
   if (VMMC_SUCCESS(ret))
   {
      pSddOpmode->OperatingMode = VMMC_SDD_OPMODE_GR909;
      ret = CmdWrite(pDev, (IFX_uint32_t *)pSddOpmode,
                                           SDD_Opmode_LEN);
   }

   return ret;
}

/**
   Start an analog line capacitance measurement session.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return

   \remarks
      This function enables Opmode change events from FW,
      then sets the line Opmode to PDH. If the line mode already
      was PDH, the function starts the capacitance measurement.
      Otherwise the capacitance measurement start will be
      triggered by Opmode change event (PDH) from FW.
      The reason for this solution is that GR909 measurement
      can only be started in PDH line mode, so the driver has
      to be sure that FW has completed the switch to PDH.
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasStart(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_Opmode_t      *pSddOpmode;
   OPMODE_CMD_t           *pOpmod;
   IFX_int32_t  ret = VMMC_statusOk;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   if (IFX_TRUE == pCh->pALM->bCapMeasInProgress)
   {
      /* errmsg: Capacitance measurement already in progress */
      RETURN_STATUS (VMMC_statusCapMeasStartWhileActive);
   }
   pDev = pCh->pParent;
   pSddOpmode = &pCh->pALM->fw_sdd_opmode;
   pOpmod = &pCh->pALM->dc_opmode;

   /* workaround: let DCCTRL finish previous state transition, if any */
   IFXOS_MSecSleep(1);
   /* end of workaround */

   VMMC_OS_MutexGet (&pCh->chAcc);

   /* Enable the OPC event needed for end notification. */
   if (pDev->caps.bEventMailboxSupported &&
       pCh->pALM->fw_ali_ch.EO != ALI_CHAN_EO_ON)
   {
      pCh->pALM->fw_ali_ch.EO = ALI_CHAN_EO_ON;
      ret = CmdWrite (pDev, (IFX_uint32_t *) &pCh->pALM->fw_ali_ch,
                      sizeof(ALI_CHAN_t) - CMD_HDR_CNT);
   }

   if (VMMC_SUCCESS(ret))
   {
      if (pOpmod->OP_MODE == VMMC_OPMOD_PDH)
      {
         /* line mode is already PDH - proceed to measurement */
         ret = vmmc_alm_Capacitance_Measurement_Start(pCh);
      }
      else
      {
         /* set PDH line mode */
         pSddOpmode->OperatingMode = VMMC_SDD_OPMODE_DISABLED;
         ret = CmdWrite(pDev, (IFX_uint32_t *)pSddOpmode, SDD_Opmode_LEN);
      }
   }

   VMMC_OS_MutexRelease (&pCh->chAcc);
   RETURN_STATUS(ret);
}

/**
   Stop any running analog line capacitance measurement session.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return

*/
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasStop(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_Opmode_t      *pSddOpmode;
   OPMODE_CMD_t           *pOpmod;
   IFX_int32_t  ret = IFX_SUCCESS;

   /* sanity check */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }
   pDev = pCh->pParent;
   pSddOpmode = &pCh->pALM->fw_sdd_opmode;
   pOpmod = &pCh->pALM->dc_opmode;
   VMMC_OS_MutexGet (&pCh->chAcc);
   if (pDev->caps.bEventMailboxSupported &&
       pCh->pALM->fw_ali_ch.EO != ALI_CHAN_EO_OFF)
   {
      /* Disable the OPC event that was used for end notification. */
      pCh->pALM->fw_ali_ch.EO = ALI_CHAN_EO_OFF;
      CmdWrite (pDev, (IFX_uint32_t *) &(pCh->pALM->fw_ali_ch),
                sizeof(ALI_CHAN_t) - CMD_HDR_CNT);
   }
   pSddOpmode->OperatingMode = VMMC_SDD_OPMODE_DISABLED;
   ret = CmdWrite(pDev, (IFX_uint32_t *)pSddOpmode, SDD_Opmode_LEN);
   if (VMMC_SUCCESS(ret))
   {
      /* update DC_OPMODE cache */
      pOpmod->OP_MODE = VMMC_OPMOD_PDH;
      pCh->pALM->bCapMeasInProgress = IFX_FALSE;
   }
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS(ret);
}

/**
   Read capacitance measurement result.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return

*/
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasResult(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE  *pDev;
   VMMC_SDD_GR909PhoneDetection_t *pGR909PhoneDetection;
   IFX_TAPI_EVENT_t tapiEvent;
   IFX_int32_t  ret;

   /* next three sanity checks are paranoid, as this function is driver
      internal API and thus the channel resources should be correct.
      The checks may be hit only in case if FW reports wrong channel in
      a capacitance measurement ready event. */
   VMMC_ASSERT(pCh != IFX_NULL);
   if (pCh->pALM == IFX_NULL)
   {
      /* errmsg: Resource not valid. Channel number out of range */
      RETURN_STATUS (VMMC_statusInvalCh);
   }

   if (pCh->pALM->line_type_fxs == IFX_FALSE)
   {
      /* errmsg: Protection from calling on FXO line */
      RETURN_STATUS (VMMC_statusFXSCallOnFXO);
   }

   pDev = pCh->pParent;
   pGR909PhoneDetection = &pCh->pALM->fw_sdd_phone_detection;
   memset(&tapiEvent, 0, sizeof(tapiEvent));

   VMMC_OS_MutexGet (&pCh->chAcc);
   tapiEvent.ch = pCh->nChannel - 1;
   tapiEvent.id = IFX_TAPI_EVENT_LINE_MEASURE_CAPACITANCE_RDY;

   ret = CmdRead(pDev, (IFX_uint32_t *) pGR909PhoneDetection,
                       (IFX_uint32_t *) pGR909PhoneDetection,
                       SDD_GR909PhoneDetection_LEN);
   if (VMMC_SUCCESS(ret))
   {
      /* zero capacitance is an error */
      tapiEvent.data.lcap.nReturnCode =
                (pGR909PhoneDetection->Capacitance == 0) ? IFX_ERROR : ret;
      tapiEvent.data.lcap.nCapacitance = pGR909PhoneDetection->Capacitance;

      /* clear progress indicator */
      pCh->pALM->bCapMeasInProgress = IFX_FALSE;

      IFX_TAPI_Event_Dispatch(pCh->pTapiCh, &tapiEvent);
   }
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS(ret);
}

/**
   Start the capacitance measurement.

   \param  pLLChannel   Handle to TAPI low level channel structure.

   \return

   \remarks
      This is an internal API called by TAPI HL after having got
      the confirmation from FW (Opmode change event) that line
      mode has been switched to PDH and GR909 measurement is
      now safe to start.
*/
static IFX_int32_t VMMC_TAPI_LL_ALM_CapMeasStartInt(IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh  = (VMMC_CHANNEL *) pLLChannel;
   IFX_int32_t  ret;

   VMMC_OS_MutexGet (&pCh->chAcc);
   ret = vmmc_alm_Capacitance_Measurement_Start(pCh);
   VMMC_OS_MutexRelease (&pCh->chAcc);

   RETURN_STATUS(ret);
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */

/**
   Function that fills in the ALM module function pointers in the driver
   context structure which is passed to HL TAPI during registration.

   \param  pAlm         Pointer to the ALM part in the driver context struct.
*/
IFX_void_t VMMC_ALM_Func_Register (IFX_TAPI_DRV_CTX_ALM_t *pAlm)
{
   /* Fill the function pointers with the services provided */
#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
   pAlm->Lec_Cfg           = VMMC_TAPI_LL_ALM_LEC_Set;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */
   pAlm->Line_Mode_Set     = VMMC_TAPI_LL_ALM_Line_Mode_Set;
   pAlm->Line_Type_Set     = VMMC_TAPI_LL_ALM_Line_Type_Set;
   pAlm->Volume_Set        = VMMC_TAPI_LL_ALM_Volume_Set;
   pAlm->Volume_High_Level = VMMC_TAPI_LL_ALM_Volume_High_Level;
   pAlm->TestHookGen       = VMMC_TAPI_LL_ALM_VMMC_Test_HookGen;
   pAlm->TestLoop          = VMMC_TAPI_LL_ALM_VMMC_Test_Loop;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION)
   pAlm->Calibration_Get   = VMMC_TAPI_LL_ALM_Calibration_Get;
#if 0
   pAlm->Calibration_Set   = VMMC_TAPI_LL_ALM_Calibration_Set;
#endif
   pAlm->Calibration_Start = VMMC_TAPI_LL_ALM_Calibration_Start;
   pAlm->Calibration_Stop  = VMMC_TAPI_LL_ALM_Calibration_Stop;
   pAlm->Calibration_Finish = VMMC_TAPI_LL_ALM_Calibration_Finish;
   pAlm->Calibration_Results_Get = VMMC_TAPI_LL_ALM_Calibration_Results;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_CALIBRATION) */
#if (VMMC_CFG_FEATURES & VMMC_FEAT_GR909)
   pAlm->GR909_Start  = VMMC_TAPI_LL_ALM_GR909_Start;
   pAlm->GR909_Stop   = VMMC_TAPI_LL_ALM_GR909_Stop;
   pAlm->GR909_Result = VMMC_TAPI_LL_ALM_GR909_Result;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_GR909) */
#if (VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT)
   pAlm->ContMeasReq   = VMMC_TAPI_LL_ALM_ContMeas_Req;
   pAlm->ContMeasReset = VMMC_TAPI_LL_ALM_ContMeas_Reset;
   pAlm->ContMeasGet   = VMMC_TAPI_LL_ALM_ContMeas_Get;
#endif /* VMMC_CFG_FEATURES & VMMC_FEAT_CONT_MEASUREMENT */
   pAlm->FXO_HookSet   = VMMC_TAPI_LL_ALM_FxoHookSet;
   pAlm->FXO_HookGet   = VMMC_TAPI_LL_ALM_FxoHookGet;
   pAlm->FXO_FlashSet  = VMMC_TAPI_LL_ALM_FxoFlashSet;
   pAlm->FXO_FlashCfg  = VMMC_TAPI_LL_ALM_FxoFlashCfg;
   pAlm->FXO_OsiCfg    = VMMC_TAPI_LL_ALM_FxoOsiCfg;
   pAlm->FXO_StatGet   = VMMC_TAPI_LL_ALM_FxoStatGet;
   pAlm->FXO_LineModeSet = VMMC_TAPI_LL_ALM_FxoLineModeSet;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION)
   pAlm->CapMeasStart  = VMMC_TAPI_LL_ALM_CapMeasStart;
   pAlm->CapMeasStop   = VMMC_TAPI_LL_ALM_CapMeasStop;
   pAlm->CapMeasResult = VMMC_TAPI_LL_ALM_CapMeasResult;
   pAlm->CapMeasStartInt = VMMC_TAPI_LL_ALM_CapMeasStartInt;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_PHONE_DETECTION) */
}
