/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_vmmc_con.c VMMC DSP module interconnection managment module.
   This module provides functions to connect different DSP modules. It is
   used for all kinds of connections including multi party conferences.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_api.h"
#include "drv_vmmc_con_priv.h"
#include "drv_vmmc_api.h"
#include "drv_vmmc_con.h"
#include "drv_vmmc_alm.h"
#include "drv_vmmc_pcm.h"
#include "drv_vmmc_sig.h"
#include "drv_vmmc_cod.h"
#ifdef DECT_SUPPORT
#include "drv_vmmc_dect.h"
#endif /* DECT_SUPPORT */
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
#include "drv_vmmc_audio.h"
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
#ifdef LIN_SUPPORT
#include "drv_vmmc_lin.h"
#endif /* LIN_SUPPORT */

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */

#ifdef DEBUG
static const IFX_char_t signalName [64][15] = {
   "Null\0",
   "\0",
   "AUDIO0-Out\0",
   "AUDIO1-Out\0",
   "ALM0-Out\0",
   "ALM1-Out\0",
   "ALM2-Out\0",
   "\0",
   "PCM0-Out\0",
   "PCM1-Out\0",
   "PCM2-Out\0",
   "PCM3-Out\0",
   "PCM4-Out\0",
   "PCM5-Out\0",
   "PCM6-Out\0",
   "PCM7-Out\0",
   "PCM8-Out\0",
   "PCM9-Out\0",
   "PCM10-Out\0",
   "PCM11-Out\0",
   "PCM12-Out\0",
   "PCM13-Out\0",
   "PCM14-Out\0",
   "PCM15-Out\0",
   "COD0-Out\0",
   "COD1-Out\0",
   "COD2-Out\0",
   "COD3-Out\0",
   "COD4-Out\0",
   "\0",
   "SIG4-OutA\0",
   "SIG4-OutB\0",
   "SIG0-OutA\0",
   "SIG0-OutB\0",
   "SIG1-OutA\0",
   "SIG1-OutB\0",
   "SIG2-OutA\0",
   "SIG2-OutB\0",
   "SIG3-OutA\0",
   "SIG3-OutB\0",
   "LIN0-Out\0",
   "LIN1-Out\0",
   "Test_Channel0\0",
   "Test_Channel1\0",
   "Test_Channel2\0",
   "Test_Channel3\0",
   "Test_Channel4\0",
   "Test_Channel5\0",
   "Test_Channel6\0",
   "Test_Channel7\0",
   "DECT0-Out\0",
   "DECT1-Out\0",
   "DECT2-Out\0",
   "DECT3-Out\0",
   "DECT4-Out\0",
   "DECT5-Out\0",
   "\0",
   "\0",
   "\0",
   "\0",
   "\0",
   "\0",
   "\0",
   "\0"
};
#endif /* DEBUG */

typedef enum
{
   VMMC_CON_ACTION_CREATE = 0,
   VMMC_CON_ACTION_REMOVE = 1
} VMMC_CON_ACTION_t;


/* ============================= */
/* Local function declaration    */
/* ============================= */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
static
IFX_return_t VMMC_TAPI_LL_Data_Channel_Add (IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_TAPI_MAP_DATA_t const *pMap);

static
IFX_return_t VMMC_TAPI_LL_Data_Channel_Remove (IFX_TAPI_LL_CH_t *pLLChannel,
                                              IFX_TAPI_MAP_DATA_t const *pMap);
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */

static
IFX_return_t VMMC_TAPI_LL_ModuleConnect    (IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_TAPI_MAP_TYPE_t nType1,
                                           unsigned char nCh2,
                                           IFX_TAPI_MAP_TYPE_t nType2);

static
IFX_return_t VMMC_TAPI_LL_ModuleDisconnect (IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_TAPI_MAP_TYPE_t nType1,
                                           unsigned char nCh2,
                                           IFX_TAPI_MAP_TYPE_t nType2);

#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
#ifdef TAPI_CID
static
IFX_return_t VMMC_TAPI_LL_Data_Channel_Mute (IFX_TAPI_LL_CH_t *pLLChannel,
                                            IFX_boolean_t nMute);
#endif /* TAPI_CID */

static
IFX_return_t VMMC_TAPI_LL_Data_Channel_Find_Connected_Module (
                                          IFX_TAPI_LL_CH_t *pLLCh,
                                          TAPI_CHANNEL **pTapiCh,
                                          IFX_TAPI_MAP_TYPE_t *pModType);

static
IFX_return_t VMMC_TAPI_LL_Module_Find_Connected_Data_Channel (
                                          IFX_TAPI_LL_CH_t *pLLCh,
                                          IFX_TAPI_MAP_TYPE_t pModType,
                                          TAPI_CHANNEL **pTapiCh);
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */

static
IFX_return_t LOCAL_Module_Connect (VMMC_CON_ACTION_t nAction,
                                   VMMC_DEVICE *pDev,
                                   IFX_uint32_t nCh1,
                                   IFX_TAPI_MAP_TYPE_t nChType1,
                                   IFX_uint32_t nCh2,
                                   IFX_TAPI_MAP_TYPE_t nChType2);

static
IFX_return_t LOCAL_ConnectPrepare (VMMCDSP_MODULE_t* pSrc,
                                   VMMCDSP_MODULE_t* pDst,
                                   SIG_OUTPUT_SIDE nSide);
static
IFX_return_t LOCAL_DisconnectPrepare (VMMCDSP_MODULE_t* pSrc,
                                      VMMCDSP_MODULE_t* pDst,
                                      SIG_OUTPUT_SIDE nSide);

/* functions for wideband/narrowband detection and conference configuration */
static IFX_boolean_t vmmc_con_ConfMemList_ElemInList (CONF_LIST_ELEMENT_t
                                                      *plist_elem);
static IFX_return_t  vmmc_con_ConfMemList_AppendElem (CONF_LIST_ELEMENT_t
                                                      *plist_elem);
static IFX_return_t vmmc_con_GetInputMod (VMMCDSP_MODULE_t* pMod,
                                          IFX_uint8_t input_index,
                                          CONF_LIST_ELEMENT_t* pModConn);
static IFX_return_t  vmmc_con_TraverseModules (VMMCDSP_MODULE_t* pMod);

static CONF_LIST_ELEMENT_t* vmmc_con_ConfMemListCreate(VMMC_CHANNEL *pCh,
                                                       VMMCDSP_MT start_module);
static IFX_return_t  vmmc_con_ConfMemListDelete (IFX_void_t);
static IFX_int32_t   vmmc_con_ModSamplingMode (SM_ACTION        action,
                                               VMMCDSP_MODULE_t *pMod,
                                               OPMODE_SMPL      mode);
static
IFX_return_t vmmc_con_ConfMemList_MuteCtrlCoderChan (IFX_operation_t mute);

/*
 * The conference member list is a single linked list of firmware modules
 * that are connected to each other. It is able to host all basic
 * firmware modules like COD, SIG, ALM, AUDIO and PCM. The list is build
 * on demand and deleted after use. The list is used to determine and configure
 * the necessary operation (8 or 16 kHz sampling rate, ISR) mode for all
 * conference members.
 */
/* conference member list created flag */
static IFX_boolean_t        conf_list_created = IFX_FALSE;
/* conference member list start pointer */
static CONF_LIST_ELEMENT_t* conf_listStart    = IFX_NULL;
/* conference member list end pointer */
static CONF_LIST_ELEMENT_t* conf_listEnd      = IFX_NULL;
/* number of conference members */
static IFX_uint8_t          conf_listMembers  = 0;


#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
static IFX_boolean_t vmmc_con_ModuleIsAudioSubType( IFX_uint8_t mod_type );
#endif

/* ============================= */
/* Function definitions          */
/* ============================= */

/** \defgroup VMMC_CONNECTIONS VMMC Connection Module */

/**
   Allocate data structure of the CON module in the given channel.

   \param  pCh             Pointer to the VMMC channel structure.

   \return
      - VMMC_statusOk if ok
      - VMMC_statusNoMem Memory allocation failed -> struct not created

   \remarks The channel parameter is no longer checked because the calling
   function assures correct values.
*/
IFX_int32_t VMMC_CON_Allocate_Ch_Structures (VMMC_CHANNEL *pCh)
{
   VMMC_CON_Free_Ch_Structures (pCh);

   pCh->pCON = VMMC_OS_Malloc (sizeof(VMMC_CON_t));
   if (pCh->pCON == IFX_NULL)
   {
      /* errmsg: Memory allocation failed */
      RETURN_STATUS (VMMC_statusNoMem);
   }

   memset(pCh->pCON, 0, sizeof(VMMC_CON_t));

   return VMMC_statusOk;
}

/**
   Free data structure of the CON module in the given channel.

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Free_Ch_Structures (VMMC_CHANNEL *pCh)
{
   if (pCh->pCON != IFX_NULL)
   {
      VMMC_OS_Free (pCh->pCON);
      pCh->pCON = IFX_NULL;
   }
}

/**
   Function called during initialisation of channel, fills up
   the connection module related structure for ALM channel

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Init_AlmCh (VMMC_CHANNEL *pCh)
{
   VMMCDSP_MODULE_t *pModAlm = &(pCh->pCON->modAlm);
   IFX_uint8_t ch = pCh->nChannel - 1;
   IFX_uint8_t i;

   pModAlm->nModType = VMMCDSP_MT_ALM;
   pModAlm->nSignal  = ECMD_IX_ALM_OUT0 + ch;
   pModAlm->pCh      = pCh;
   pModAlm->sampling_mode = VMMC_CON_SMPL_NB;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
      pModAlm->in[i].pParent = pModAlm;
}

/**
   Function called during initialisation of channel, fills up
   the connection module related structure for PCM channel

   \param  pCh     Pointer to the VMMC channel structure.
   \param  pcmCh   PCM channel identifier.
*/
IFX_void_t VMMC_CON_Init_PcmCh (VMMC_CHANNEL *pCh, IFX_uint8_t pcmCh)
{
   VMMCDSP_MODULE_t *pModPcm = &(pCh->pCON->modPcm);
   IFX_uint8_t i;

   pModPcm->nModType = VMMCDSP_MT_PCM;
   pModPcm->nSignal  = ECMD_IX_PCM_OUT0 + pcmCh;
   pModPcm->pCh      = pCh;
   pModPcm->sampling_mode = VMMC_CON_SMPL_OFF;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
      pModPcm->in[i].pParent = pModPcm;
}

/**
   Function called during initialisation of channel, fills up
   the connection module related structure for coder channel

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Init_CodCh (VMMC_CHANNEL *pCh)
{
   VMMCDSP_MODULE_t *pModCod = &(pCh->pCON->modCod);
   IFX_uint8_t ch = pCh->nChannel - 1;
   IFX_uint8_t i;

   pModCod->nModType = VMMCDSP_MT_COD;
   pModCod->nSignal  = ECMD_IX_COD_OUT0 + ch;
   pModCod->pCh      = pCh;
   pModCod->sampling_mode = VMMC_CON_SMPL_OFF;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
      pModCod->in[i].pParent = pModCod;
}

/**
   Function called during initialisation of channel, fills up
   the connection module related structure for signaling channel

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Init_SigCh (VMMC_CHANNEL *pCh)
{
   VMMCDSP_MODULE_t *pModSig = &(pCh->pCON->modSig);
   IFX_uint8_t ch = pCh->nChannel - 1;
   IFX_uint8_t i;

   pModSig->nModType = VMMCDSP_MT_SIG;
   if (ch < 4)
   {
      pModSig->nSignal  = ECMD_IX_SIG_OUTA0 + (IFX_uint8_t)(2*ch);
      pModSig->nSignal2 = ECMD_IX_SIG_OUTB0 + (IFX_uint8_t)(2*ch);
   }
   else
   {
      pModSig->nSignal  = ECMD_IX_SIG_OUTA4;
      pModSig->nSignal2 = ECMD_IX_SIG_OUTB4;
   }
   pModSig->pCh      = pCh;
   pModSig->sampling_mode = VMMC_CON_SMPL_NB;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
      pModSig->in[i].pParent = pModSig;
}

#ifdef DECT_SUPPORT
/**
   Function called during initialisation of channel, fills up
   the connection module related structure for the DECT channel

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Init_DectCh (VMMC_CHANNEL *pCh)
{
   VMMCDSP_MODULE_t *pModDect = &(pCh->pCON->modDect);
   IFX_uint8_t ch = pCh->nChannel - 1;
   IFX_uint8_t i;

   pModDect->nModType = VMMCDSP_MT_DECT;
   pModDect->nSignal  = ECMD_IX_DECT_OUT0 + ch;
   pModDect->pCh      = pCh;
   pModDect->sampling_mode = VMMC_CON_SMPL_OFF;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
      pModDect->in[i].pParent = pModDect;
}
#endif /* DECT_SUPPORT */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
/**
   Function called during initialisation of channel, fills up
   the connection module related structure for signaling channel

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Init_AudioCh (VMMC_CHANNEL *pCh)
{
   VMMCDSP_MODULE_t *pModAudio      = &(pCh->pCON->modAudio);
   VMMCDSP_MODULE_t *pModAudioAux   = &(pCh->pCON->modAudioAux);
   VMMCDSP_MODULE_t *pmodAudioLoop0 = &(pCh->pCON->modAudioLoop0);
   VMMCDSP_MODULE_t *pmodAudioLoop1 = &(pCh->pCON->modAudioLoop1);
   VMMCDSP_MODULE_t *pmodDiag0In    = &(pCh->pCON->modDiag0In);
   VMMCDSP_MODULE_t *pmodDiag0Out   = &(pCh->pCON->modDiag0Out);
   VMMCDSP_MODULE_t *pmodDiag1In    = &(pCh->pCON->modDiag1In);
   VMMCDSP_MODULE_t *pmodDiag1Out   = &(pCh->pCON->modDiag1Out);
   IFX_uint8_t i;

   /* Initialize Audio Module Structure */
   pModAudio->nModType = VMMCDSP_MT_AUDIO;
   pModAudio->nSignal  = ECMD_IX_AUDIO_OUT0;
   pModAudio->nSignal2 = ECMD_IX_EMPTY;
   pModAudio->pCh      = pCh;
   pModAudio->sampling_mode = VMMC_CON_SMPL_AUTO;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pModAudio->in[i].i       = ECMD_IX_EMPTY;
      pModAudio->in[i].pParent = pModAudio;
   }

   /* Initialize Audio Aux Module Structure */
   pModAudioAux->nModType = VMMCDSP_MT_AUDIO_AUX;
   pModAudioAux->nSignal  = ECMD_IX_AUDIO_OUT1;
   pModAudioAux->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pModAudioAux->in[i].i       = ECMD_IX_EMPTY;
      pModAudioAux->in[i].pParent = pModAudioAux;
   }

   /* Initialize Audio Loop 0 Module Structure */
   pmodAudioLoop0->nModType = VMMCDSP_MT_AUDIO_LOOP0;
   pmodAudioLoop0->nSignal  = ECMD_IX_TEST_OUT3;
   pmodAudioLoop0->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pmodAudioLoop0->in[i].i       = ECMD_IX_EMPTY;
      pmodAudioLoop0->in[i].pParent = pmodAudioLoop0;
   }
   /* Initialize Audio Loop 1 Module Structure */
   pmodAudioLoop1->nModType = VMMCDSP_MT_AUDIO_LOOP1;
   pmodAudioLoop1->nSignal  = ECMD_IX_TEST_OUT1;
   pmodAudioLoop1->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pmodAudioLoop1->in[i].i       = ECMD_IX_EMPTY;
      pmodAudioLoop1->in[i].pParent = pmodAudioLoop1;
   }
   /* Initialize Audio Diag 0 IN Module Structure */
   pmodDiag0In->nModType = VMMCDSP_MT_AUDIO_DIAG0_IN;
   pmodDiag0In->nSignal  = ECMD_IX_EMPTY;
   pmodDiag0In->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pmodDiag0In->in[i].i       = ECMD_IX_EMPTY;
      pmodDiag0In->in[i].pParent = pmodDiag0In;
   }
   /* Initialize Audio Diag 0 OUT Module Structure */
   pmodDiag0Out->nModType = VMMCDSP_MT_AUDIO_DIAG0_OUT;
   pmodDiag0Out->nSignal  = ECMD_IX_EMPTY;
   pmodDiag0Out->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pmodDiag0Out->in[i].i       = ECMD_IX_EMPTY;
      pmodDiag0Out->in[i].pParent = pmodDiag0Out;
   }
   /* Initialize Audio Diag 1 IN Module Structure */
   pmodDiag1In->nModType = VMMCDSP_MT_AUDIO_DIAG1_IN;
   pmodDiag1In->nSignal  = ECMD_IX_EMPTY;
   pmodDiag1In->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pmodDiag1In->in[i].i       = ECMD_IX_EMPTY;
      pmodDiag1In->in[i].pParent = pmodDiag1In;
   }
   /* Initialize Audio Diag 1 OUT Module Structure */
   pmodDiag1Out->nModType = VMMCDSP_MT_AUDIO_DIAG1_OUT;
   pmodDiag1Out->nSignal  = ECMD_IX_EMPTY;
   pmodDiag1Out->nSignal2 = ECMD_IX_EMPTY;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
   {
      pmodDiag1Out->in[i].i       = ECMD_IX_EMPTY;
      pmodDiag1Out->in[i].pParent = pmodDiag1Out;
   }
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

#ifdef LIN_SUPPORT
/**
   Function called during initialisation of channel, fills up
   the connection module related structure for LIN channel

   \param  pCh  Pointer to the VMMC channel structure.
*/
IFX_void_t VMMC_CON_Init_LinCh (VMMC_CHANNEL *pCh)
{
   VMMCDSP_MODULE_t *pModLin = &(pCh->pCON->modLin);
   IFX_uint8_t ch = pCh->nChannel - 1;
   IFX_uint8_t i;

   pModLin->nModType = VMMCDSP_MT_LIN;
   pModLin->nSignal  = ECMD_IX_LIN_TEST_CHAN0 + ch;
   pModLin->pCh      = pCh;
   pModLin->sampling_mode = VMMC_CON_SMPL_NB;

   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; ++i)
      pModLin->in[i].pParent = pModLin;
}
#endif /* LIN_SUPPORT */

IFX_uint8_t VMMC_CON_Get_ALM_SignalInput (VMMC_CHANNEL *pCh, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pCh->pCON->modAlm.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

IFX_uint8_t VMMC_CON_Get_PCM_SignalInput (VMMC_CHANNEL *pCh, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pCh->pCON->modPcm.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

IFX_uint8_t VMMC_CON_Get_COD_SignalInput (VMMC_CHANNEL *pCh, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pCh->pCON->modCod.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

IFX_uint8_t VMMC_CON_Get_SIG_SignalInput (VMMC_CHANNEL *pCh, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pCh->pCON->modSig.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

#ifdef DECT_SUPPORT
IFX_uint8_t VMMC_CON_Get_DECT_SignalInput (VMMC_CHANNEL *pCh, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pCh->pCON->modDect.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}
#endif /* DECT_SUPPORT */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
IFX_uint8_t VMMC_CON_Get_AUDIO_SignalInput (VMMC_DEVICE *pDev,
                                            IFX_uint8_t ch, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pDev->pChannel[ch].pCON->modAudio.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Loop0_SignalInput (VMMC_DEVICE *pDev,
                                            IFX_uint8_t ch)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pDev->pChannel[ch].pCON->modAudioLoop0.in[0];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Loop1_SignalInput (VMMC_DEVICE *pDev,
                                            IFX_uint8_t ch)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pDev->pChannel[ch].pCON->modAudioLoop1.in[0];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}

IFX_uint8_t VMMC_CON_Get_AUDIO_SignalStatus (VMMC_DEVICE *pDev,
                                             IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modAudio.modified);
}

IFX_return_t VMMC_CON_Set_AUDIO_SignalStatus (VMMC_DEVICE *pDev,
                                              IFX_uint8_t ch, IFX_uint8_t status)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return IFX_ERROR;
   }
   pDev->pChannel[ch].pCON->modAudio.modified = status;
   return IFX_SUCCESS;
}

IFX_return_t VMMC_CON_Set_AUDIO_Loop0_SignalStatus (VMMC_DEVICE *pDev,
                                                    IFX_uint8_t ch, IFX_uint8_t status)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return IFX_ERROR;
   }
   pDev->pChannel[ch].pCON->modAudioLoop0.modified = status;
   return IFX_SUCCESS;
}

IFX_return_t VMMC_CON_Set_AUDIO_Loop1_SignalStatus (VMMC_DEVICE *pDev,
                                                    IFX_uint8_t ch, IFX_uint8_t status)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return IFX_ERROR;
   }
   pDev->pChannel[ch].pCON->modAudioLoop1.modified = status;
   return IFX_SUCCESS;
}

IFX_uint8_t VMMC_CON_Get_AUDIO_AUX_SignalInput( VMMC_DEVICE *pDev,
                                                IFX_uint8_t ch)
{
   VMMCDSP_MODULE_t *pMod = &pDev->pChannel[ch].pCON->modAudioAux;

   /* Return auxiliary signal index */
   return pMod->in[0].i;
}

IFX_uint8_t VMMC_CON_Get_AUDIO_AUX_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of Audio AUX signalling inputs  */
   return (pDev->pChannel[ch].pCON->modAudioAux.modified);
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Loop0_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modAudioLoop0.modified);
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Loop1_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modAudioLoop1.modified);
}

#if 0
IFX_uint8_t VMMC_CON_Get_AUDIO_Diag0In_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modDiag0In.modified);
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Diag1In_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modDiag1In.modified);
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Diag0Out_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modDiag0Out.modified);
}

IFX_uint8_t VMMC_CON_Get_AUDIO_Diag1Out_SignalStatus( VMMC_DEVICE *pDev,
                                                 IFX_uint8_t ch)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return 0;
   }
  /* Return the modification status of SIG signalling inputs  */
   return (pDev->pChannel[ch].pCON->modDiag1Out.modified);
}
#endif

IFX_return_t VMMC_CON_Set_AUDIO_AUX_SignalStatus( VMMC_DEVICE *pDev,
                                                  IFX_uint8_t ch, IFX_uint8_t status)
{
   if (pDev == IFX_NULL)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH, (__FILE__ ":%d: pDev NULL\n", __LINE__));
      return IFX_ERROR;
   }
   pDev->pChannel[ch].pCON->modAudioAux.modified = status;
   return IFX_SUCCESS;
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

#ifdef LIN_SUPPORT
IFX_uint8_t VMMC_CON_Get_LIN_SignalInput (VMMC_CHANNEL *pCh, IFX_uint8_t index)
{
   VMMCDSP_MODULE_SIGNAL_t *pInput = &pCh->pCON->modLin.in[index];

   /* Return either the normal or the muted signal index */
   return pInput->mute ? pInput->i_mute : pInput->i;
}
#endif /* LIN_SUPPORT */

/**
   Function called by init_module() of device, fills up CONnection module
   function pointers which are passed to HL TAPI during registration

   \param  pCON         Pointer to CON module.
*/
/**
   Function that fills in the connection function pointers in the driver
   context structure which is passed to HL TAPI during registration.

   \param  pCON         Pointer to the CON part in the driver context struct.
*/
IFX_void_t VMMC_CON_Func_Register (IFX_TAPI_DRV_CTX_CON_t *pCON)
{
#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
   pCON->Data_Channel_Add     = VMMC_TAPI_LL_Data_Channel_Add;
   pCON->Data_Channel_Remove  = VMMC_TAPI_LL_Data_Channel_Remove;
#ifdef TAPI_CID
   pCON->Data_Channel_Mute    = VMMC_TAPI_LL_Data_Channel_Mute;
#endif /* TAPI_CID */
   pCON->Data_Channel_Find_Connected_Module
                              = VMMC_TAPI_LL_Data_Channel_Find_Connected_Module;
   pCON->Module_Find_Connected_Data_Channel
                              = VMMC_TAPI_LL_Module_Find_Connected_Data_Channel;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */

   pCON->Module_Connect       = VMMC_TAPI_LL_ModuleConnect;
   pCON->Module_Disconnect    = VMMC_TAPI_LL_ModuleDisconnect;
}

#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
/**
   Connect a data channel to an analog phone device or to PCM

   By definition a data channel consists of the COD-module and the SIG-module
   of the same TAPI connection.
   This function is used for basic channel setup (analog line to coder channel)
   and as well for conferencing.
   A conference is given, when two data channels are connected to one phone
   channel.

   Every channels configuration is only written once, when they are changed at
   the end of the signal array update. Therefore the connection is cached
   inside the driver in addition to the VoFW messages.
   ConnectPrepare may be called even if the input is already connected.

   The following steps are done:
   - connect signaling channel output to the destination (ALM or PCM) input,
     if not already done
   - Connect the destination to the data channel, if not already done. If the
     upstream input (I1) of the signalling module is already in use by another
     ALM or PCM output connect the new output to the next free upstream input
     on the coder of the same data channel instead.
   - Check if the destination is now connected to more than one data channel.
     If this is the case then connect this data channel to the conference in
     which the destination channel is involved.

   \param  pLLChannel  Pointer to the VMMC channel structure.
   \param  pMap        Handle to IFX_TAPI_MAP_DATA_t structure.

   \return
   - VMMC_statusCmdWr Writing the command failed
   - VMMC_statusNotSupported Requested action is not supported. Only types
      phone and PCM are supported.
   - VMMC_statusFuncParm Wrong parameters passed. This could be a wrong
   destination channel or this channel does not support the signaling
   module.
   - VMMC_statusWrongChMode The source channel is not is a data channel.
      It is verified by making sure SIG is getting it's input from the
      COD-module of the same channel.
   - VMMC_statusNoFreeInput No free input signal found or available
   - VMMC_statusOk if successful
*/
IFX_return_t VMMC_TAPI_LL_Data_Channel_Add (IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_TAPI_MAP_DATA_t const *pMap)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *)pLLChannel,
             *pDstCh = IFX_NULL;
   VMMC_DEVICE *pDev = pCh->pParent;
   int ret, cnt = 0, i, j;
   VMMCDSP_MODULE_t *pDstMod = IFX_NULL,
                    *tmpSigs[MAX_MODULE_SIGNAL_INPUTS] = {IFX_NULL};
   VMMC_CON_t       *pThisChModules = pCh->pCON;
   VMMCDSP_MODULE_SIGNAL_t *pInput = IFX_NULL;

   /* Validate channel parameter in the map struct given to us from outside. */
   if (pMap->nDstCh > VMMC_MAX_CH_NR)
   {
      /* Channel given as destination does not exist. */
      SET_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }
   pDstCh = &pDev->pChannel[pMap->nDstCh];

   switch (pMap->nChType)
   {
   case IFX_TAPI_MAP_TYPE_PCM:
      if (pDstCh->pPCM != IFX_NULL)
         pDstMod = &pDstCh->pCON->modPcm;
      break;
   case IFX_TAPI_MAP_TYPE_PHONE:
   case IFX_TAPI_MAP_TYPE_DEFAULT:
      if (pDstCh->pALM != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAlm;
      break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   case IFX_TAPI_MAP_TYPE_AUDIO:
      if (pMap->nDstCh < pDev->caps.nAudioCnt)
         pDstMod = &pDstCh->pCON->modAudio;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_AUX:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAudioAux;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_DIAG0_IN:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modDiag0In;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_DIAG0_OUT:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modDiag0Out;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_DIAG1_IN:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modDiag1In;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_DIAG1_OUT:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modDiag1Out;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_LOOP0:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAudioLoop0;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_LOOP1:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAudioLoop1;
      break;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
#ifdef DECT_SUPPORT
   case IFX_TAPI_MAP_TYPE_DECT:
      if (pDstCh->pDECT != IFX_NULL)
         pDstMod = &pDstCh->pCON->modDect;
      break;
#endif /* DECT_SUPPORT */
   case IFX_TAPI_MAP_TYPE_CODER:
   default:
      SET_ERROR (VMMC_ERR_NOTSUPPORTED);
      return IFX_ERROR;
   }
   if ((pDstMod == IFX_NULL) ||
       (pCh->pSIG == IFX_NULL) || (pCh->pCOD == IFX_NULL))
   {
      SET_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }
   /* Here pDstMod points to a valid module which should be connected to the
      data channel. It can either be an ALM, PCM or an AUDIO module. */

   /* Verify that this is a data channel by making sure SIG is getting it's
      input from the COD-module of the same channel. */
   if (pThisChModules->modSig.in[REMOTE_SIG_IN].i !=
       pThisChModules->modCod.nSignal)
   {
      SET_ERROR (VMMC_ERR_WRONG_CHANNEL_MODE);
      return IFX_ERROR;
   }

   /* connect SIG output to Dst (ALM or PCM) input, if not already done */
   ret = LOCAL_ConnectPrepare (&pThisChModules->modSig, pDstMod, LOCAL_SIG_OUT);
   if (ret == IFX_SUCCESS)
   {
      /* Connect the Dst (ALM, PCM or AUDIO) to the data channel, if not already done.
         If the upstream input (I1) of the signalling module is already in
         use by another ALM or PCM output connect the new output to the next
         free upstream input on the coder of the same data channel instead. */
      if (pThisChModules->modSig.in[LOCAL_SIG_IN].i == ECMD_IX_EMPTY)
      {
         /* connect Dst (ALM or PCM) output to SIG input (I1) */
         ret = LOCAL_ConnectPrepare (pDstMod,
                                     &pThisChModules->modSig, 0);
      }
      else
      {
         /* Connect the Dst (ALM or PCM or AUDIO ) output to COD input because the
            SIG input (I1) is already in use. This happens when adding more
            than one phone/PCM to a data channel. But prevent connecting
            Dst to the coder when it is already connected to signalling. */
         if (pThisChModules->modSig.in[LOCAL_SIG_IN].i != pDstMod->nSignal)
            ret = LOCAL_ConnectPrepare (pDstMod,
                                        &pThisChModules->modCod, 0);
      }
   }

   if (ret == IFX_SUCCESS)
   {

      if( pMap->nChType != IFX_TAPI_MAP_TYPE_AUDIO_AUX )
      {
         /* Check if the Dst (ALM or PCM or AUDIO) is now connected to more than one
            data channel. If this is the case then connect these data channels
            so that all the data channels can talk to each other */

         /* Count to how many data channels the analog module is connected
            and store a pointer to each signalling module in a temporary list. */
         for (pInput = pDstMod->pInputs, cnt = 0;
              pInput != IFX_NULL; pInput = pInput->pNext )
         {
            if (pInput->pParent->nModType == VMMCDSP_MT_SIG)
               tmpSigs[cnt++] = pInput->pParent;
            if (pInput->pParent->nModType == VMMCDSP_MT_COD)
               tmpSigs[cnt++] = pInput->pParent->pInputs->pParent;
         }

         if (cnt > 1)
         {
            /* Conference with more than one data channel. So now connect each
               data channel to all the others that are stored in the list above.
               For this each signaling output is connected to one input on every
               coder taking part in the conference. ConnectPrepare may be called
               even if the input is already connected. */
            for (i = 0; i <  cnt; i++)
               for (j = 0; j < cnt; j++)
                  if (i != j)
                     ret = LOCAL_ConnectPrepare (tmpSigs[i],
                              tmpSigs[j]->pInputs->pParent, LOCAL_SIG_OUT);
            /* PS: The pointer is implicitly valid by a check done above */
         }
      }
   }

   /*
    * After the connections are prepared but before writing them to the FW
    * reevaluate the sampling mode of the new conference. If neccessary the
    * modules of the conference are configured to the other sampling rate.
    * Data channel's COD module is used as root module for conference traversal.
    */
   VMMC_CON_MatchConfSmplRate(pCh, VMMCDSP_MT_COD);

   /*
    * Finally write down all FW configuration messages for new module connections
    */
   if (ret == IFX_SUCCESS)
   {
      ret = VMMC_CON_ConnectConfigure (pDev);
   }
   else
   {
       SET_ERROR(VMMC_ERR_NO_FREE_INPUT_SLOT);
   }

   return (ret == IFX_SUCCESS) ? IFX_SUCCESS : IFX_ERROR;
}

/**
   Removes a data channel from an analog phone device

   Every channels configuration is only written once, when they are changed at
   the end of the signal array update. Therefore the connection is cached
   inside the driver in addition to the VoFW messages.

   The function performs the following steps:
   - disconnect signaling channel output from the destination (ALM or PCM) input
   - disconnect destination from either the coder channel or signaling channel
     depending on where the  output of the destination is connected to.
   - if this channel was involved in a conference, remove it from the conference
     and set the internal pointer to IFX_NULL
   - Finally all references inside the conferencing module to this channel are
     removed.

   \param  pLLChannel  Pointer to the VMMC channel structure.
   \param  pMap        Handle to IFX_TAPI_MAP_DATA_t structure.

   \return
   - VMMC_statusCmdWr Writing the command failed
   - VMMC_statusNotSupported Requested action is not supported. Only types
      phone and PCM are supported.
   - VMMC_statusFuncParm Wrong parameters passed. This could be a wrong
   destination channel or this channel does not support the signaling
   module.
   - VMMC_statusConInvalid The given connection is not valid
   - VMMC_statusOk if successful
*/
IFX_return_t VMMC_TAPI_LL_Data_Channel_Remove (IFX_TAPI_LL_CH_t *pLLChannel,
                                              IFX_TAPI_MAP_DATA_t const *pMap)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *)pLLChannel,
             *pDstCh = IFX_NULL;
   VMMC_DEVICE *pDev = pCh->pParent;
   IFX_int32_t ret = IFX_SUCCESS, i, cnt = 0, still_required;
   VMMCDSP_MODULE_t *pTmpMod;
   VMMCDSP_MODULE_SIGNAL_t *pInput = IFX_NULL, *pDstInput;
   VMMCDSP_MODULE_t *pDstMod = IFX_NULL,
                    *tmpSigs[MAX_MODULE_SIGNAL_INPUTS] = {IFX_NULL};
   VMMC_CON_t       *pThisChModules = pCh->pCON;

   /* Validate channel parameter in the map struct given to us from outside. */
   if (pMap->nDstCh > VMMC_MAX_CH_NR)
   {
      /* Channel given as destination does not exist. */
      SET_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }
   pDstCh = &pDev->pChannel[pMap->nDstCh];

   switch (pMap->nChType)
   {
   case IFX_TAPI_MAP_TYPE_PCM:
      if (pDstCh->pPCM != IFX_NULL)
         pDstMod = &pDstCh->pCON->modPcm;
      break;
   case IFX_TAPI_MAP_TYPE_PHONE:
   case IFX_TAPI_MAP_TYPE_DEFAULT:
      if (pDstCh->pALM != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAlm;
      break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   case IFX_TAPI_MAP_TYPE_AUDIO:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAudio;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_AUX:
      if (pDstCh->pAUDIO != IFX_NULL)
         pDstMod = &pDstCh->pCON->modAudioAux;
      break;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
#ifdef DECT_SUPPORT
   case IFX_TAPI_MAP_TYPE_DECT:
      if (pDstCh->pDECT != IFX_NULL)
         pDstMod = &pDstCh->pCON->modDect;
      break;
#endif /* DECT_SUPPORT */
   case IFX_TAPI_MAP_TYPE_CODER:
   default:
      SET_ERROR (VMMC_ERR_NOTSUPPORTED);
      return IFX_ERROR;
   }
   if ((pDstMod == IFX_NULL) ||
       (pCh->pSIG == IFX_NULL) || (pCh->pCOD == IFX_NULL))
   {
      SET_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }

#ifdef TAPI_CID
   /* Prevent disconnection of Dst from SIG while CID is running.
      All other connections can be disconnected. This minimises blocking.
      With this lockout we also prevent difficulties with the otherwise
      possible reroute of signals to the local signalling input below. */
   if (TAPI_Cid_IsActive(pCh->pTapiCh) &&
       (pThisChModules->modSig.in[LOCAL_SIG_IN].i == pDstMod->nSignal))
   {
      SET_ERROR (VMMC_ERR_CID_RUNNING);
      return IFX_ERROR;
   }
#endif /* TAPI_CID */


   /* disconnect SIG output from Dst (ALM or PCM) input */
   ret = LOCAL_DisconnectPrepare (&pThisChModules->modSig,
                                  pDstMod, LOCAL_SIG_OUT);
   if (ret == IFX_SUCCESS)
   {
      /* Disconnect Dst from either the COD or SIG module depending on where
         the output of the Dst is connected to. */
      if( pThisChModules->modSig.in[LOCAL_SIG_IN].i == pDstMod->nSignal )
      {
         /* disconnect Dst (ALM or PCM) output from SIG input */
         ret = LOCAL_DisconnectPrepare (pDstMod,
                                        &pThisChModules->modSig, 0);
      }
      else
      {
         /* disconnect Dst (ALM or PCM) output from COD input */
         ret = LOCAL_DisconnectPrepare (pDstMod,
                                        &pThisChModules->modCod, 0);
      }
   }

   /* Now the local input on the SIG on the data channel may be empty
      so reroute a signal from an input of the COD on the same channel when
      possible. Do not reroute signals coming from signalling modules */
   for (i = 0;
        ret == IFX_SUCCESS && i <  MAX_MODULE_SIGNAL_INPUTS &&
        pThisChModules->modSig.in[LOCAL_SIG_IN].i == ECMD_IX_EMPTY; i++)
   {
      pTmpMod = pThisChModules->modCod.in[i].pOut;
      if (pTmpMod != IFX_NULL && pTmpMod->nModType != VMMCDSP_MT_SIG)
      {
         /* disconnect Dst (ALM or PCM) output from COD input */
         ret = LOCAL_DisconnectPrepare (pTmpMod,
                                        &pThisChModules->modCod, 0);
         if (ret == IFX_SUCCESS)
            /* connect Dst (ALM or PCM) output to SIG input (I1) */
            ret = LOCAL_ConnectPrepare (pTmpMod,
                                        &pThisChModules->modSig, 0);
      }
   }

   /* Finally the output of our signalling module may still be connected to
      the coder on other data channels because the channels have been
      connected to the Dst (ALM or PCM) module that we just disconnected.
      So now check all data channels that we are connected to because of the
      Dst and if the connection is still needed because we are still in a
      conference where another ALM or PCM module connects our data channel
      with another one. */

   /* Store a pointer to each signalling module in a data channel that
      the Dst (ALM or PCM) still connects to in a temporary list. */
   for (pInput = pDstMod->pInputs, cnt = 0;
        pInput != IFX_NULL; pInput = pInput->pNext )
   {
      if (pInput->pParent->nModType == VMMCDSP_MT_SIG)
         tmpSigs[cnt++] = pInput->pParent;
      if (pInput->pParent->nModType == VMMCDSP_MT_COD)
         tmpSigs[cnt++] = pInput->pParent->pInputs->pParent;
   }
   /* Loop over the list just built */
   for (i = 0, still_required = 0;
        ret == IFX_SUCCESS && i <  cnt && !still_required; i++)
   {
      /* Loop over all modules connecting to the data channel */
      for (pInput = tmpSigs[i]->pInputs2;
           pInput != IFX_NULL; pInput = pInput->pNext)
      {
         /* Skip COD modules connecting to the data channel */
         if (pInput->pParent->nModType == VMMCDSP_MT_COD)
            continue;
         /* For remaining modules check if any of the inputs connecting to
            the output is the signalling module we started with. If so then
            mark the connection as still required. */
         for (pDstInput = pInput->pParent->pInputs;
              pDstInput != IFX_NULL; pDstInput = pDstInput->pNext)
            if (pDstInput->pParent == &pThisChModules->modSig ||
                pDstInput->pParent == &pThisChModules->modCod   )
            {
               still_required = 1;
               break;
            }
      }
      /* If no longer required disconnect the data channels */
      if (! still_required)
      {
         ret = LOCAL_DisconnectPrepare (&pThisChModules->modSig,
                                        tmpSigs[i]->pInputs->pParent,
                                        LOCAL_SIG_OUT);
         if (ret == IFX_SUCCESS)
            ret = LOCAL_DisconnectPrepare (tmpSigs[i],
                                           &pThisChModules->modCod,
                                           LOCAL_SIG_OUT);
      }
   }

   /* write the configuration to the chip */
   if (ret == IFX_SUCCESS)
   {
      ret = VMMC_CON_ConnectConfigure (pDev);
   }
   else
   {
      SET_ERROR (VMMC_ERR_CON_INVALID);
   }

   /*
    * After FW modules are disconnected match the sampling modes of the
    * modules in each of the remaining two conferences.
    */
   VMMC_CON_MatchConfSmplRate(pCh, VMMCDSP_MT_COD);
   VMMC_CON_MatchConfSmplRate(&pDev->pChannel[pMap->nDstCh],
                              pDstMod->nModType);

   return (ret == IFX_SUCCESS) ? IFX_SUCCESS : IFX_ERROR;
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */


/**
   Connect any two of the following modules: PCM, ALM, DECT, AUDIO.

   \param  pLLChannel   Pointer to the VMMC channel structure.
   \param  nType1       Module type in the first channel.
   \param  nCh2         Channel number of second channel
   \param  nType2       Module type in the second channel

   \return
   - VMMC_statusCmdWr Writing the command failed
   - VMMC_statusNotSupported Requested action is not supported. Only types
      phone and PCM are supported.
   - VMMC_statusFuncParm Wrong parameters passed. This could be a wrong
   destination channel or this channel does not support the signaling
   module.
   - VMMC_statusConInvalid The given connection is not valid
   - VMMC_statusOk if successful
*/
IFX_return_t VMMC_TAPI_LL_ModuleConnect (IFX_TAPI_LL_CH_t *pLLChannel,
                                        IFX_TAPI_MAP_TYPE_t nType1,
                                        unsigned char nCh2,
                                        IFX_TAPI_MAP_TYPE_t nType2)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *)pLLChannel;

   return LOCAL_Module_Connect (VMMC_CON_ACTION_CREATE,
                                pCh->pParent,
                                pCh->nChannel - 1, nType1,
                                nCh2,              nType2);
}

/**
   Disconnect any two of the following modules: PCM, ALM, DECT, AUDIO.

   \param  pLLChannel   Pointer to the VMMC channel structure.
   \param  nType1       Module type in the first channel.
   \param  nCh2         Channel number of second channel
   \param  nType2       Module type in the second channel

   \return
   - VMMC_statusCmdWr Writing the command failed
   - VMMC_statusNotSupported Requested action is not supported. Only types
      phone and PCM are supported.
   - VMMC_statusFuncParm Wrong parameters passed. This could be a wrong
   destination channel or this channel does not support the signaling
   module.
   - VMMC_statusConInvalid The given connection is not valid
   - VMMC_statusOk if successful
*/
IFX_return_t VMMC_TAPI_LL_ModuleDisconnect (IFX_TAPI_LL_CH_t *pLLChannel,
                                           IFX_TAPI_MAP_TYPE_t nType1,
                                           unsigned char nCh2,
                                           IFX_TAPI_MAP_TYPE_t nType2)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *)pLLChannel;

   return LOCAL_Module_Connect (VMMC_CON_ACTION_REMOVE,
                                pCh->pParent,
                                pCh->nChannel - 1, nType1,
                                nCh2,              nType2);
}

/**
   Create or remove a symetric connection between two local modules
   (PCM, ALM, DECT, AUDIO) on the same device.

   \param nAction     Action to be performed VMMC_CON_ACTION_t
   \param pDev        Pointer to the VMMC device structure
   \param nCh1        Resource number of the first module
   \param nChType1    Which module type to use from channel 1
   \param nCh2        Resource number of the second module
   \param nChType2    Which module type to use from channel 2

   \return Return value according to IFX_return_t
   - IFX_ERROR if an error occured
   - IFX_SUCCESS if successful
*/
static
IFX_return_t LOCAL_Module_Connect (VMMC_CON_ACTION_t nAction,
                                   VMMC_DEVICE *pDev,
                                   IFX_uint32_t nCh1,
                                   IFX_TAPI_MAP_TYPE_t nChType1,
                                   IFX_uint32_t nCh2,
                                   IFX_TAPI_MAP_TYPE_t nChType2)
{
   VMMCDSP_MODULE_t  *pMod1   = IFX_NULL,
                     *pMod2   = IFX_NULL;
   IFX_int32_t       ret      = IFX_SUCCESS;
   /* needed for SET_ERROR macro */
   VMMC_CHANNEL*     pCh      = IFX_NULL;

   if (nCh1 >= VMMC_MAX_CH_NR)
   {
      SET_DEV_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }

   /* assign Channel handle for SET_ERROR */
   pCh = &pDev->pChannel[nCh1];

   /* get pointers to the signal struct of
      the selected modules in the channels */
   switch (nChType1)
   {
   case IFX_TAPI_MAP_TYPE_PCM:
      if (nCh1 < pDev->caps.nPCM)
         pMod1 = &pDev->pChannel[nCh1].pCON->modPcm;
      break;
   case IFX_TAPI_MAP_TYPE_PHONE:
   case IFX_TAPI_MAP_TYPE_DEFAULT:
      if (nCh1 < pDev->caps.nALI)
         pMod1 = &pDev->pChannel[nCh1].pCON->modAlm;
      break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   case IFX_TAPI_MAP_TYPE_AUDIO:
      if (nCh1 < pDev->caps.nAudioCnt)
         pMod1 = &pDev->pChannel[nCh1].pCON->modAudio;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_AUX:
      TRACE(VMMC, DBG_LEVEL_NORMAL,
            ("STBD:drv_vmmc_con.c: 1 MAP_PCM_ADD/REMOVE_AUDIO/AUX \n"));
      SET_ERROR (VMMC_ERR_NOTSUPPORTED);
      return IFX_ERROR;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
#ifdef DECT_SUPPORT
   case IFX_TAPI_MAP_TYPE_DECT:
      if (nCh1 < pDev->caps.nDECT)
         pMod1 = &pDev->pChannel[nCh1].pCON->modDect;
      break;
#endif /* DECT_SUPPORT */
#ifdef LIN_SUPPORT
   case IFX_TAPI_MAP_TYPE_LIN:
      if (nCh1 < pDev->caps.nLIN)
         pMod1 = &pDev->pChannel[nCh1].pCON->modLin;
      break;
#endif /* LIN_SUPPORT */
   case IFX_TAPI_MAP_TYPE_CODER:
   default:
      SET_ERROR (VMMC_ERR_NOTSUPPORTED);
      return IFX_ERROR;
   }

   switch (nChType2)
   {
   case IFX_TAPI_MAP_TYPE_PCM:
      if (nCh2 < pDev->caps.nPCM)
         pMod2 = &pDev->pChannel[nCh2].pCON->modPcm;
      break;
   case IFX_TAPI_MAP_TYPE_PHONE:
   case IFX_TAPI_MAP_TYPE_DEFAULT:
      if (nCh2 < pDev->caps.nALI)
         pMod2 = &pDev->pChannel[nCh2].pCON->modAlm;
      break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   case IFX_TAPI_MAP_TYPE_AUDIO:
      if (nCh2 < pDev->caps.nAudioCnt)
         pMod2 = &pDev->pChannel[nCh2].pCON->modAudio;
      break;
   case IFX_TAPI_MAP_TYPE_AUDIO_AUX:
      TRACE(VMMC, DBG_LEVEL_NORMAL,
            ("STBD:drv_vmmc_con.c: 2 MAP_PCM_ADD_AUDIO \n"));
      SET_ERROR (VMMC_ERR_NOTSUPPORTED);
      return IFX_ERROR;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
#ifdef DECT_SUPPORT
   case IFX_TAPI_MAP_TYPE_DECT:
      if (nCh2 < pDev->caps.nDECT)
         pMod2 = &pDev->pChannel[nCh2].pCON->modDect;
      break;
#endif /* DECT_SUPPORT */
#ifdef LIN_SUPPORT
   case IFX_TAPI_MAP_TYPE_LIN:
      if (nCh2 < pDev->caps.nLIN)
         pMod2 = &pDev->pChannel[nCh2].pCON->modLin;
      break;
#endif /* LIN_SUPPORT */
   case IFX_TAPI_MAP_TYPE_CODER:
   default:
      SET_ERROR (VMMC_ERR_NOTSUPPORTED);
      return IFX_ERROR;
   }

   if ((pMod1 == IFX_NULL) || (pMod2 == IFX_NULL))
   {
      SET_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }

   /* now pMod1 and pMod2 point to the modules to be connected/disconnected */

   if (nAction == VMMC_CON_ACTION_CREATE)
   {
      /* connect the forward direction */
      ret = LOCAL_ConnectPrepare (pMod1, pMod2, 0);
      if (ret == IFX_SUCCESS)
      {
         /* connect the reverse direction */
         ret = LOCAL_ConnectPrepare (pMod2, pMod1, 0);
      }
      if (ret != IFX_SUCCESS)
      {
         SET_ERROR(VMMC_ERR_NO_FREE_INPUT_SLOT);
      }
   }
   if (nAction == VMMC_CON_ACTION_REMOVE)
   {
      /* disconnect the forward direction */
      ret = LOCAL_DisconnectPrepare (pMod1, pMod2, 0);
      if (ret == IFX_SUCCESS)
      {
         /* disconnect the reverse direction */
         ret = LOCAL_DisconnectPrepare (pMod2, pMod1, 0);
      }
      if (ret != IFX_SUCCESS)
      {
         SET_ERROR(VMMC_ERR_CON_INVALID);
      }
   }


   if (nAction == VMMC_CON_ACTION_CREATE)
   {
      /* After the connections are prepared but before writing them to the FW
       * reevaluate the sampling mode of the new conference. If neccessary the
       * modules of the conference are configured to the other sampling rate. */
      VMMC_CON_MatchConfSmplRate(&pDev->pChannel[nCh1], pMod1->nModType);
   }

   /* finally write down FW commands to configure new module interconnections
      and calculated operation modes */
   if (ret == IFX_SUCCESS)
      ret = VMMC_CON_ConnectConfigure (pDev);

   if (nAction == VMMC_CON_ACTION_REMOVE)
   {
      /* After FW modules are disconnected match the sampling modes of the
       * modules in each of the remaining two conferences. */
      VMMC_CON_MatchConfSmplRate(&pDev->pChannel[nCh1], pMod1->nModType);
      VMMC_CON_MatchConfSmplRate(&pDev->pChannel[nCh2], pMod2->nModType);
   }

   return (ret == IFX_SUCCESS) ? IFX_SUCCESS : IFX_ERROR;
}

#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
#ifdef TAPI_CID
/**
   Mute/Unmute all connections to modules which are attached to a data channel
   that is currently sending CID2 and should not listen to the CID.

   This function is used to temporarily mute connections in the signalling
   array. This function finds all connections which need to be muted and
   mutes/unmutes them so that modules taking part in a conference will not
   hear the CID signal which is not intended from them.

   \param  pLLChannel   Handle to IFX_TAPI_LL_CH_t structure.
   \param  nMute        IFX_TRUE: mute / IFX_FALSE: unmute.

   \return
   - VMMC_statusCmdWr Writing the command failed
   - VMMC_statusNotSupported Requested action is not supported. Only types
      phone and PCM are supported.
   - VMMC_statusFuncParm Wrong parameters passed. This could be a wrong
   destination channel or this channel does not support the signaling
   module.
   - VMMC_statusConInvalid The given connection is not valid
   - VMMC_statusOk if successful
*/
IFX_return_t VMMC_TAPI_LL_Data_Channel_Mute (IFX_TAPI_LL_CH_t *pLLChannel,
                                            IFX_boolean_t nMute)
{
   VMMC_CHANNEL   *pCh = (VMMC_CHANNEL *)pLLChannel;
   VMMC_DEVICE   *pDev = pCh->pParent;
   VMMCDSP_MODULE_t *pSigMod = IFX_NULL,
                    *pAnaMod = IFX_NULL;
   VMMCDSP_MODULE_SIGNAL_t *pTmpSig;
   IFX_int8_t mute = nMute ? 1 : -1;
   unsigned i;
   IFX_int32_t ret;

   pSigMod = &pCh->pCON->modSig;
   /* find the analog module that is the one providing the input to SIG */
   /*lint -e{722} */
   for (pTmpSig = pSigMod->pInputs2;
        pTmpSig != IFX_NULL && (pAnaMod = pTmpSig->pParent,
        pSigMod->in[LOCAL_SIG_IN].i != pAnaMod->nSignal);
        pTmpSig = pTmpSig->pNext, pAnaMod = IFX_NULL);
   if ( pAnaMod == IFX_NULL )
   {
      /* no analog module connected to this SIG module */
      return IFX_ERROR;
   }

   /* 1. Flag the outputs on SIG and Analog module as muted so that all
         connections created from now on are also created in muted mode. */
   pSigMod->nMute = /*lint --e(821)*/
   pSigMod->nMute2 = pAnaMod->nMute = nMute ? 1 : 0;
   pSigMod->modified = pAnaMod->modified = IFX_TRUE;

   /* 2. Find all inputs getting the local output of SIG and set them to muted
         mode but not the Analog module. Set then the  muted inputs to
         receive the input of the SIG module. */
   for (pTmpSig = pSigMod->pInputs2;
        pTmpSig != IFX_NULL;
        pTmpSig = pTmpSig->pNext)
   {
      if (pTmpSig->pParent != pAnaMod)
      {
         pTmpSig->mute = (IFX_uint8_t)(pTmpSig->mute + mute);
         if (pTmpSig->mute == 1 && pTmpSig->pParent->nMute == 0)
            pTmpSig->i_mute = pSigMod->in[REMOTE_SIG_IN].i;
         pTmpSig->pParent->modified = IFX_TRUE;
      }
   }

   /* 3. Mute the remote input of the SIG module */
   pSigMod->in[REMOTE_SIG_IN].mute += /*lint --e(732)*/ mute;

   /* 4. Mute all inputs of the Analog module except the one connected to
         the SIG module. */
   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; i++)
   {
      if (pAnaMod->in[i].i != pSigMod->nSignal2)
      {
         pAnaMod->in[i].mute += /*lint --e(732)*/ mute;
         pAnaMod->in[i].i_mute = ECMD_IX_EMPTY;
      }
   }

   /* 5. Mute all inputs connected to the output of the Analog module but
         not the SIG module itself */
   for (pTmpSig = pAnaMod->pInputs;
        pTmpSig != IFX_NULL;
        pTmpSig = pTmpSig->pNext)
   {
      if (pTmpSig->pParent != pSigMod)
      {
         pTmpSig->mute += /*lint --e(732) */ mute;
         pTmpSig->pParent->modified = IFX_TRUE;
      }
   }

   /* 6. Mute all inputs that receive their input from SIG out remote */
   for (pTmpSig = pSigMod->pInputs;
        pTmpSig != IFX_NULL;
        pTmpSig = pTmpSig->pNext)
   {
      pTmpSig->mute += /*lint --e(732)*/ mute;
      pTmpSig->pParent->modified = IFX_TRUE;
   }

   /* 7. For more comfort in the unmute case write the muted signal index on
         now only once muted inputs of analog modules that have been muted
         multiple times before. */
   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; i++)
   {
      if ((pAnaMod->in[i].i != ECMD_IX_EMPTY) &&
          (pAnaMod->in[i].mute == 1) &&
          (pAnaMod->in[i].i_mute == ECMD_IX_EMPTY) &&
          (pAnaMod->in[i].pOut->nModType == VMMCDSP_MT_SIG) &&
          (pAnaMod->in[i].pOut->in[REMOTE_SIG_IN].mute != 0))
      {
         pAnaMod->in[i].i_mute = pAnaMod->in[i].pOut->in[REMOTE_SIG_IN].i;
      }
   }

   /* write the changed configuration to the chip */
   ret = VMMC_CON_ConnectConfigure (pDev);

   return (ret == IFX_SUCCESS) ? IFX_SUCCESS : IFX_ERROR;
}
#endif /* TAPI_CID */

/**
   Find the module which provides it's input to the data channel main input
   (on the SIG module) and gets an input from this data channel.

   \param  pLLCh        Pointer to Low-level channel structure.
   \param  pTapiCh      Returns pointer to the TAPI channel the found module
                        belongs to or IFX_NULL if no module is connected.
   \param  pModType     Returns of which type the found module is.
                        If pTapiCh is IFX_NULL this is undefined.
   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
*/
IFX_return_t VMMC_TAPI_LL_Data_Channel_Find_Connected_Module (
                                          IFX_TAPI_LL_CH_t *pLLChannel,
                                          TAPI_CHANNEL **pTapiCh,
                                          IFX_TAPI_MAP_TYPE_t *pModType)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *)pLLChannel;
   VMMCDSP_MODULE_t *pSigMod = IFX_NULL,
                    *pConMod = IFX_NULL;
   VMMCDSP_MODULE_SIGNAL_t *pTmpSig;

   /* start with a return value of fail */
   *pTapiCh = IFX_NULL;
   /* if nothing is found return something defined but meaningless */
   *pModType = IFX_TAPI_MAP_TYPE_DEFAULT;

   /* abort if called on an uninitialised channel */
   if (pCh->pCON == IFX_NULL)
   {
      return IFX_ERROR;
   }

   pSigMod = &pCh->pCON->modSig;

   /* find the module that is the one providing the input to the SIG module */
   /*lint -e{722} */
   for (pTmpSig = pSigMod->pInputs2, pConMod = IFX_NULL;
        pTmpSig != IFX_NULL && (pConMod = pTmpSig->pParent,
        pSigMod->in[LOCAL_SIG_IN].i != pConMod->nSignal);
        pTmpSig = pTmpSig->pNext, pConMod = IFX_NULL);

   if ( pConMod != IFX_NULL )
   {
      /* check the reverse direction */
      /*lint -e{722} */
      for (pTmpSig = pConMod->pInputs;
           pTmpSig != IFX_NULL && (pTmpSig->pParent != pSigMod);
           pTmpSig = pTmpSig->pNext);

      if (pTmpSig != IFX_NULL)
      {
         /* found a match */
         *pTapiCh = pConMod->pCh->pTapiCh;
         switch (pConMod->nModType)
         {
         case VMMCDSP_MT_ALM:
            *pModType = IFX_TAPI_MAP_TYPE_PHONE;
            break;
         case VMMCDSP_MT_PCM:
            *pModType = IFX_TAPI_MAP_TYPE_PCM;
            break;
         case VMMCDSP_MT_AUDIO:
            *pModType = IFX_TAPI_MAP_TYPE_AUDIO;
            break;
         case VMMCDSP_MT_DECT:
            *pModType = IFX_TAPI_MAP_TYPE_DECT;
            break;
         default:
            /* this is an error case - return with the default set above */
            break;
         }
      }
   }

   return IFX_SUCCESS;
}

/**
   Find the data channel which gets it's main input from a given module
   and which provides it's input to the given module.

    There can be more than one data channel connected to a module. So the
    function acts as an iterator with pTapiCh as input and output parameter.

   \param  pLLCh        Pointer to Low-level channel structure.
   \param  nModType     Type of the module where to start search.
   \param  pTapiCh      On calling specifies which was the last channel found
                        by this function. For the first call use IFX_NULL.
                        For iteration calls use the last output.
                        Returns pointer to the TAPI channel the found module
                        belongs to or IFX_NULL if no module is connected.

   \return
   IFX_SUCCESS if successful
   IFX_ERROR if an error occured
*/
IFX_return_t VMMC_TAPI_LL_Module_Find_Connected_Data_Channel (
                                          IFX_TAPI_LL_CH_t *pLLCh,
                                          IFX_TAPI_MAP_TYPE_t nModType,
                                          TAPI_CHANNEL **pTapiCh)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *)pLLCh;
   VMMCDSP_MODULE_t *pSigMod = IFX_NULL,
                    *pConMod = IFX_NULL;
   VMMCDSP_MODULE_SIGNAL_t *pTmpSig;
   TAPI_CHANNEL *pLastTapiCh;
   IFX_boolean_t bSkip;

   switch (nModType)
   {
   default:
   case IFX_TAPI_MAP_TYPE_PHONE:
      pConMod = &pCh->pCON->modAlm;
      break;
   case IFX_TAPI_MAP_TYPE_PCM:
      pConMod = &pCh->pCON->modPcm;
      break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   case IFX_TAPI_MAP_TYPE_AUDIO:
      pConMod = &pCh->pCON->modAudio;
      break;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
#ifdef DECT_SUPPORT
   case IFX_TAPI_MAP_TYPE_DECT:
      pConMod = &pCh->pCON->modDect;
      break;
#endif /* DECT_SUPPORT */
   }

   /* Make sure that we do not start with a non existing module. */
   if (pConMod == IFX_NULL /*lint --e(774) */)
   {
      /* The specified module does not exist on the specified channel. */
      VMMC_ASSERT(pConMod != IFX_NULL);
      *pTapiCh = IFX_NULL;
      return IFX_ERROR;
   }

   /* save the input and determine the iteration abort criteria */
   pLastTapiCh = *pTapiCh;
   bSkip = pLastTapiCh != IFX_NULL ? IFX_TRUE : IFX_FALSE;

   /* start with a return value of fail */
   *pTapiCh = IFX_NULL;

   /* find a SIG module that gets it local input from the given module */
   for (pTmpSig = pConMod->pInputs, pSigMod = IFX_NULL;
        pTmpSig != IFX_NULL;
        pTmpSig = pTmpSig->pNext, pSigMod = IFX_NULL)
   {
      pSigMod = pTmpSig->pParent;

      if (pSigMod->nModType == VMMCDSP_MT_SIG &&
          pSigMod->in[LOCAL_SIG_IN].i == pConMod->nSignal)
      {
         if (bSkip == IFX_FALSE)
         {
            break;
         }
         else
         {
            if (pSigMod->pCh->pTapiCh == pLastTapiCh)
               bSkip = IFX_FALSE;
         }
      }
   }

   if ( pSigMod != IFX_NULL )
   {
      /* TAPI allows only symetric connections so do not check for the
         reverse direction here. */

      /* set the return value */
      *pTapiCh = pSigMod->pCh->pTapiCh;
   }

   return IFX_SUCCESS;
}
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */



/**
   Add a signal connection to the internal shadow connection structure.
   Call Con_ConnectConfigure() to write the shadow structure to the chip.
\param pSrc      - handle to the source module which contains the output signal
\param pDst      - handle to the destination module where the output
                   is connected to
\param nSide     - specifies which output to use when the source is a
   signaling module, don't care for other module types (set to 0).
   If set to LOCAL_SIG_OUT then the output towards the local side is
   used and with REMOTE_SIG_OUT the output towards the remote side.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   This function only connects two given modules if they are not already
   connected. The destination is set to modified if all was successful.
   The connection inherits a mute attribute from the module output that
   serves as the signal source for the connection. So when a connection
   is created on an output set to mute it will itself carry the mute
   attribute.
   As signalling modules have two sides (local/remote) to which other
   modules can be connected the side has to be determined first.
   When the signalling modules acts as an input rules are used to determine
   to which input side a module is connected:
   Local modules like ALM and PCM (not always) support auto suppression and
   get the event playout. They are connected to In 1.
   Remote modules like coder and PCM in case of advanced with DSP are
   connected to In 2.
   When the signalling module acts as an output nSignal is used to select
   which side of the signalling module is used for the connection. The value
   LOCAL_SIG_OUT is used for all analog channels. The value REMOTE_SIG_OUT
   is used for the coder only to connect to the Sig-OutA output.

   \verbatim
                          nSignal
                          pInputs
   Signalling module:      |----------------|
                       <-  | Out A     In 1 |  <-
                           |  upstream (tx) |
          remote <==       |  -  -  -  -  - |       ==> local
                           | downstream (rx)|
                       ->  | In 2     Out B |  ->
                           |----------------|
                                      nSignal2
                                      pInputs2
   \endverbatim
*/
static
IFX_return_t LOCAL_ConnectPrepare (VMMCDSP_MODULE_t* pSrc,
                                   VMMCDSP_MODULE_t* pDst,
                                   SIG_OUTPUT_SIDE nSide)
{
   int i, in = -1;
   VMMCDSP_MODULE_SIGNAL_t **pInput;
   IFX_uint8_t sig;
   IFX_uint8_t mute;
   VMMCDSP_MODULE_SIGNAL_t *pDstIn = IFX_NULL;

   /* get signal number and pointer to the output of the source module */
   if( nSide == LOCAL_SIG_OUT && pSrc->nModType == VMMCDSP_MT_SIG )
   {
      /* SIG module: use the local output (Out B) of the signalling module */
      /* Exception Audio AUX: use OUT B since OUT A is used as audio voice channel */
      pInput = &pSrc->pInputs2;
      sig    = pSrc->nSignal2;
      mute   = pSrc->nMute2;
   }
   else
   {
      /* PCM-, COD-, ALM-module: use the standard output of given module */
      /* SIG module: use the remote output (Out A) of the signalling module */
      pInput = &pSrc->pInputs;
      sig    = pSrc->nSignal;
      mute   = pSrc->nMute;
   }

   /* *pInput now points to the first input signal of the list of input signals
      connected to this output. sig contains the number of the signal in the
      signal array */

   /* return now if the output is already connected to one of the inputs */
   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; i++)
   {
      if (pDst->in[i].i == sig)
      {
         return IFX_SUCCESS;
      }
   }

   /* find the index of a free input on the destination module */
   switch(pDst->nModType)
   {
      case VMMCDSP_MT_SIG:
         /* exception 1: coder connects always to input 2 */
         if (pSrc->nModType == VMMCDSP_MT_COD)
         {
            in = REMOTE_SIG_IN;
         }
         else /* default input on signaling module is input 1 */
         {
            in = LOCAL_SIG_IN;
         }
         /* exception 2 for PCM: in advanced PCM it does not act as ALM, but as
            coder. This case is selected by using LOCAL_SIG_OUT on the PCM
            as it is connected to the SIG. This cannot be done when connecting
            PCM and SIG by user configuration. */
         if (pSrc->nModType == VMMCDSP_MT_PCM && nSide == LOCAL_SIG_OUT)
         {
            in = REMOTE_SIG_IN;
         }

         /* finally verify that the input selected is currently not in use */
         if (pDst->in[in].i != ECMD_IX_EMPTY)
         {
            TRACE(VMMC, DBG_LEVEL_NORMAL,("Input signal on SIG already in use\n"));
            return IFX_ERROR;
         }
         pDstIn = &(pDst->in[in]);
         break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
      case VMMCDSP_MT_AUDIO_AUX:
         /* Audio AUX module uses only I1==in[0] as input */
         if( pDst->in[0].i != ECMD_IX_EMPTY )
         {
            TRACE(VMMC, DBG_LEVEL_NORMAL,("AUX signal on AUDIO already in use\n"));
            return IFX_ERROR;
         }
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Set Pointer to AUX IN of Audio Channel\n"));
         in = 0;
         pDstIn = &(pDst->in[0]);
         break;
      case VMMCDSP_MT_AUDIO_LOOP0:
         if( pDst->in[0].i != ECMD_IX_EMPTY )
         {
            TRACE(VMMC, DBG_LEVEL_NORMAL,("LOOP0 signal on AUDIO already in use\n"));
            return IFX_ERROR;
         }
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Set Pointer to LOOP0 IN of Audio Channel\n"));
         in = 0;
         pDstIn = &(pDst->in[0]);
         break;
      case VMMCDSP_MT_AUDIO_LOOP1:
         if( pDst->in[0].i != ECMD_IX_EMPTY )
         {
            TRACE(VMMC, DBG_LEVEL_NORMAL,("LOOP1 signal on AUDIO already in use\n"));
            return IFX_ERROR;
         }
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Set Pointer to LOOP1 of Audio Channel\n"));
         in = 0;
         pDstIn = &(pDst->in[0]);
         break;
      case VMMCDSP_MT_AUDIO_DIAG0_IN:
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Nothing to connect since DIAG0_IN has no input.\n"));
         return IFX_SUCCESS;
      case VMMCDSP_MT_AUDIO_DIAG0_OUT:
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Nothing to connect since DIAG0_OUT has no input.\n"));
         return IFX_SUCCESS;
      case VMMCDSP_MT_AUDIO_DIAG1_IN:
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Nothing to connect since DIAG1_IN has no input.\n"));
         return IFX_SUCCESS;
      case VMMCDSP_MT_AUDIO_DIAG1_OUT:
         TRACE(VMMC, DBG_LEVEL_NORMAL,("Nothing to connect since DIAG1_OUT has no input.\n"));
         return IFX_SUCCESS;
#endif
#ifdef LIN_SUPPORT
      case VMMCDSP_MT_LIN:
         /* this module only has one input */
         in = 0;
         if (pDst->in[in].i == ECMD_IX_EMPTY)
         {
            pDstIn = &(pDst->in[in]);
         }
         break;
#endif /* LIN_SUPPORT */

      default:
         /* find free input on the destination module (COD, ALM, PCM or AUDIO) */
         for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; i++)
         {
            if (pDst->in[i].i == ECMD_IX_EMPTY)
            {
               in = i;
               pDstIn = &(pDst->in[i]);
               break;
            }
         }
         break;
   }   /* switch(pDst->nModType) */

#ifdef DEBUG
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("adding Signal %-10s to %-10s [input:%d] %s\n",
          signalName[sig], signalName[pDst->nSignal], in, mute ? "muted" : ""));
#endif /* DEBUG */

   if( (pDstIn == IFX_NULL) || (in == -1) )
   {
      /* placed after the trace to get details on the modules (debug only) */
      TRACE(VMMC, DBG_LEVEL_NORMAL,
         ("No free input on destination module - stop\n"));
      return IFX_ERROR;
   }

   /* Connect output to the input */
   if( (*pDstIn).i != sig )
   {
      pDst->modified = IFX_TRUE;
      (*pDstIn).i    = sig;
      (*pDstIn).mute += mute;
   }
   /* Set the pointer showing which output connects to this input */
   (*pDstIn).pOut  = pSrc;
   /* Add the input to the linked list of all inputs attached to one output.
      Always add the element to the head of the list.*/
   (*pDstIn).pNext = *pInput;
   *pInput = pDstIn;

   return IFX_SUCCESS;
}

/**
   Removes a signal connection from the internal shadow connection structure.
   Call Con_ConnectConfigure() to write the shadow structure to the chip.
\param pSrc      - handle to the source module which contains the output signal
\param pDst      - handle to the destination module where the output
                   is connected to
\param nSide     - specifies which output to use when the source is a
   signaling module, don't care for other module types (set to 0).
   If set to LOCAL_SIG_OUT then the output towards the local side is
   used and with REMOTE_SIG_OUT the output towards the remote side.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   To remove a signal connection the input in the destination module has to
   be unlinked from the list the source module keeps of all inputs connecting
   to it's output. Then the input struct in the destination can be blanked.
   If all was successful the destination is set to modified.
   Disconnecting a non existing connection returns also IFX_SUCCESS.
*/
static
IFX_return_t LOCAL_DisconnectPrepare (VMMCDSP_MODULE_t* pSrc,
                                      VMMCDSP_MODULE_t* pDst,
                                      SIG_OUTPUT_SIDE nSide)
{
   int i, in = -1;
   VMMCDSP_MODULE_SIGNAL_t **pBase,
                         *pTemp;
   IFX_uint8_t mute;

   /* find the input on the destination which is connected to the source */
   for (i = 0; i < MAX_MODULE_SIGNAL_INPUTS; i++)
   {
      if (pDst->in[i].pOut == pSrc)
      {
         in = i;
         break;
      }
   }

#ifdef DEBUG
   TRACE(VMMC, DBG_LEVEL_LOW,
         ("removing Signal %-10s from %-10s [input:%d]\n",
          in != -1 ? signalName[pDst->in[in].i] : "<empty>",
          signalName[pDst->nSignal], in));
#endif /* DEBUG */

   if (in == -1)
   {
     /* placed after the trace to get details on the modules (debug only) */
      TRACE(VMMC, DBG_LEVEL_NORMAL,("Given modules are not connected\n"));
      return IFX_SUCCESS;
   }
   /* Get a pointer to the basepointer of the list of inputs connected. */
   if (nSide == LOCAL_SIG_OUT && pSrc->nModType == VMMCDSP_MT_SIG)
   {
      /* SIG-module only: inputs connected on local-side */
      pBase = &pSrc->pInputs2;
      mute = pSrc->nMute2;
   }
   else
   {
      /* PCM-, COD-, ALM-module: standard input */
      /* SIG module: inputs connected on remote-side */
      pBase = &pSrc->pInputs;
      mute = pSrc->nMute;
   }
   /* Get a pointer to the first element in the linked list of inputs. */
   pTemp = *pBase;
   /* pTemp now points to the first node in the list of input signals
      connected to this output. Now find the node of the input found
      above and remove this node from the linked list. */
   if (*pBase == &pDst->in[in])
   {
      /* special case: it is the first node so move the base pointer */
      *pBase = pDst->in[in].pNext;
   }
   else
   {
      /* Walk the list and try to find the node to be deleted. */
      while (pTemp != IFX_NULL && pTemp->pNext != &pDst->in[in])
         pTemp = pTemp->pNext;
      if (pTemp == IFX_NULL)
      {
         /* not found! The data structure is corrupt - this should not happen*/
         return IFX_ERROR;
      }
      /* unlink the node from the list */
      pTemp->pNext = pTemp->pNext->pNext;
   }
   /* clear this input -> connection is now removed */
   pDst->in[in].pNext = IFX_NULL;
   pDst->in[in].i     = ECMD_IX_EMPTY;
   pDst->in[in].i_mute= ECMD_IX_EMPTY;
   pDst->in[in].mute -= mute;
   pDst->in[in].pOut  = IFX_NULL;
   pDst->modified     = IFX_TRUE;

   return IFX_SUCCESS;
}



/** \addtogroup VMMC_CONNECTIONS */
/* @{ */

/**
   Add a signal connection to the internal shadow connection structure.
   Call Con_ConnectConfigure() to write the shadow structure to the chip.

   \param  pSrcCh       Pointer to the VMMC channel structure.
   \param  src          Source module which contains the output signal.
   \param  pDstCh       Pointer to the VMMC channel structure.
   \param  dst          Destination module where the output is connected to.
   \param  nSide        Specifies which output to use when the source is a
                        signaling module, don't care for other module types
                        (set to 0).
                        If set to LOCAL_SIG_OUT then the output towards the
                        local side is used and with REMOTE_SIG_OUT the output
                        towards the remote side.

   \return
   IFX_SUCCESS or IFX_ERROR

   \remarks
   This function only connects two given modules if they are not already
   connected. The destination is set to modified if all was successful.
   The connection inherits a mute attribute from the module output that
   serves as the signal source for the connection. So when a connection
   is created on an output set to mute it will itself carry the mute
   attribute.
   As signalling modules have two sides (local/remote) to which other
   modules can be connected the side has to be determined first.
   When the signalling modules acts as an input rules are used to determine
   to which input side a module is connected:
   Local modules like ALM and PCM (not always) support auto suppression and
   get the event playout. They are connected to In 1.
   Remote modules like coder and PCM in case of advanced with DSP are
   connected to In 2.
   When the signalling module acts as an output nSide is used to select
   which side of the signalling module is used for the connection. The value
   LOCAL_SIG_OUT is used for all analog channels. The value REMOTE_SIG_OUT
   is used for the coder only to connect to the Sig-OutA output.

   \verbatim
                          nSignal
                          pInputs
   Signalling module:      |----------------|
                       <-  | Out A     In 1 |  <-
                           |  upstream (tx) |
          remote <==       |  -  -  -  -  - |       ==> local
                           | downstream (rx)|
                       ->  | In 2     Out B |  ->
                           |----------------|
                                      nSignal2
                                      pInputs2
   \endverbatim
*/
IFX_return_t VMMC_CON_ConnectPrepare (VMMC_CHANNEL *pSrcCh, VMMCDSP_MT src,
                                      VMMC_CHANNEL *pDstCh, VMMCDSP_MT dst,
                                      SIG_OUTPUT_SIDE nSide)
{
   VMMCDSP_MODULE_t *pSrc = IFX_NULL,
                    *pDst = IFX_NULL;

   switch (src)
   {
   case VMMCDSP_MT_ALM:
      pSrc = &pSrcCh->pCON->modAlm;
      break;
   case VMMCDSP_MT_SIG:
      pSrc = &pSrcCh->pCON->modSig;
      break;
   case VMMCDSP_MT_COD:
      pSrc = &pSrcCh->pCON->modCod;
      break;
   case VMMCDSP_MT_PCM:
      pSrc = &pSrcCh->pCON->modPcm;
      break;
#ifdef LIN_SUPPORT
   case VMMCDSP_MT_LIN:
      pSrc = &pSrcCh->pCON->modLin;
      break;
#endif /* LIN_SUPPORT */
   default:
      /* internal error: not an supported module type */
      return IFX_ERROR;
   }

   switch (dst)
   {
   case VMMCDSP_MT_ALM:
      pDst = &pDstCh->pCON->modAlm;
      break;
   case VMMCDSP_MT_SIG:
      pDst = &pDstCh->pCON->modSig;
      break;
   case VMMCDSP_MT_COD:
      pDst = &pDstCh->pCON->modCod;
      break;
   case VMMCDSP_MT_PCM:
      pDst = &pDstCh->pCON->modPcm;
      break;
#ifdef LIN_SUPPORT
   case VMMCDSP_MT_LIN:
      pDst = &pDstCh->pCON->modLin;
      break;
#endif /* LIN_SUPPORT */
   default:
      /* internal error: not an supported module type */
      return IFX_ERROR;
   }

   return LOCAL_ConnectPrepare (pSrc, pDst, nSide);
}

#if 0
/* This code is actually never called because internally we never disconnect
   modules. So this is just a working demo on how a disconnect looks like. */
/**
   Removes a signal connection from the internal shadow connection structure.
   Call Con_ConnectConfigure() to write the shadow structure to the chip.

   \param  pSrcCh       Pointer to the VMMC channel structure.
   \param  src          Source module which contains the output signal.
   \param  pDstCh       Pointer to the VMMC channel structure.
   \param  dst          Destination module where the output is connected to.
   \param  nSide        Specifies which output to use when the source is a
                        signaling module, don't care for other module types
                        (set to 0).
                        If set to LOCAL_SIG_OUT then the output towards the
                        local side is used and with REMOTE_SIG_OUT the output
                        towards the remote side.

   \return
   IFX_SUCCESS or IFX_ERROR

   \remarks
   To remove a signal connection the input in the destination module has to
   be unlinked from the list the source module keeps of all inputs connecting
   to it's output. Then the input struct in the destination can be blanked.
   If all was successful the destination is set to modified.
   Disconnecting a non existing connection returns also IFX_SUCCESS.
*/
IFX_return_t VMMC_CON_DisconnectPrepare (VMMC_CHANNEL *pSrcCh, VMMCDSP_MT src,
                                         VMMC_CHANNEL *pDstCh, VMMCDSP_MT dst,
                                         SIG_OUTPUT_SIDE nSide)
{

   VMMCDSP_MODULE_t *pSrc = IFX_NULL,
                    *pDst = IFX_NULL;

   switch (src)
   {
   case VMMCDSP_MT_ALM:
      pSrc = &pSrcCh->pCON->modAlm;
      break;
   case VMMCDSP_MT_SIG:
      pSrc = &pSrcCh->pCON->modSig;
      break;
   case VMMCDSP_MT_COD:
      pSrc = &pSrcCh->pCON->modCod;
      break;
   case VMMCDSP_MT_PCM:
      pSrc = &pSrcCh->pCON->modPcm;
      break;
#ifdef LIN_SUPPORT
   case VMMCDSP_MT_LIN:
      pSrc = &pSrcCh->pCON->modLin;
      break;
#endif /* LIN_SUPPORT */
   default:
      /* internal error: not an supported module type */
      return IFX_ERROR;
   }

   switch (dst)
   {
   case VMMCDSP_MT_ALM:
      pDst = &pDstCh->pCON->modAlm;
      break;
   case VMMCDSP_MT_SIG:
      pDst = &pDstCh->pCON->modSig;
      break;
   case VMMCDSP_MT_COD:
      pDst = &pDstCh->pCON->modCod;
      break;
   case VMMCDSP_MT_PCM:
      pDst = &pDstCh->pCON->modPcm;
      break;
#ifdef LIN_SUPPORT
   case VMMCDSP_MT_LIN:
      pDst = &pDstCh->pCON->modLin;
      break;
#endif /* LIN_SUPPORT */
   default:
      /* internal error: not an supported module type */
      return IFX_ERROR;
   }

   return LOCAL_DisconnectPrepare (pSrc, pDst, nSide);
}
#endif

/**
   Write the configuration from the internal shadow connection structure
   to the chip (firmware).

   \param  pDev         Pointer to the VMMC device structure.

   \return
   IFX_SUCCESS or IFX_ERROR
*/
IFX_return_t VMMC_CON_ConnectConfigure (VMMC_DEVICE *pDev)
{
   VMMC_CHANNEL *pCh;
   IFX_return_t ret = IFX_SUCCESS;
   IFX_uint8_t  ch;

   VMMC_OS_MutexGet (&pDev->mtxMemberAcc);

   /* write ALM module inputs */
   for (ch = 0; (ret == IFX_SUCCESS) && (ch < pDev->caps.nALI); ++ch)
   {
      pCh = &pDev->pChannel[ch];

      /* call the module specific set function only if something was modified */
      if (pCh->pCON->modAlm.modified != IFX_FALSE)
      {
         /* write ALM module inputs */
         ret = VMMC_ALM_Set_Inputs (pCh);
         /* clear the modified flag now that we updated the cached fw message */
         pCh->pCON->modAlm.modified = IFX_FALSE;
      }
   }

   /* write CODer module inputs */
   for (ch = 0; (ret == IFX_SUCCESS) && (ch < pDev->caps.nCOD); ++ch)
   {
      pCh = &pDev->pChannel[ch];

      /* call the module specific set function only if something was modified */
      if (pCh->pCON->modCod.modified != IFX_FALSE)
      {
         /* write CODer module inputs */
         ret = VMMC_COD_Set_Inputs (pCh);
         /* clear the modified flag now that we updated the cached fw message */
         pCh->pCON->modCod.modified = IFX_FALSE;
      }
   }

   /* write PCM module inputs */
   for (ch = 0; (ret == IFX_SUCCESS) && (ch < pDev->caps.nPCM); ++ch)
   {
      pCh = &pDev->pChannel[ch];

      /* call the module specific set function only if something was modified */
      if (pCh->pCON->modPcm.modified != IFX_FALSE)
      {
         /* write PCM module inputs */
         ret = VMMC_PCM_Set_Inputs (pCh);
         /* clear the modified flag now that we updated the cached fw message */
         pCh->pCON->modPcm.modified = IFX_FALSE;
      }
   }

   /* write SIGnalling module inputs */
   for (ch = 0; (ret == IFX_SUCCESS) && (ch < pDev->caps.nSIG); ++ch)
   {
      pCh = &pDev->pChannel[ch];

      /* call the module specific set function only if something was modified */
      if (pCh->pCON->modSig.modified != IFX_FALSE)
      {
         /* write SIG module inputs */
         ret = VMMC_SIG_Set_Inputs (pCh);
         /* clear the modified flag now that we updated the cached fw message */
         pCh->pCON->modSig.modified = IFX_FALSE;
      }
   }

#ifdef DECT_SUPPORT
   /* write DECT module inputs */
   for (ch = 0; (ret == IFX_SUCCESS) && (ch < pDev->caps.nDECT); ++ch)
   {
      pCh = &pDev->pChannel[ch];

      /* call the module specific set function only if something was modified */
      if (pCh->pCON->modDect.modified != IFX_FALSE)
      {
         /* write DECT module inputs */
         ret = VMMC_DECT_Set_Inputs (pCh);
         /* clear the modified flag now that we updated the cached fw message */
         pCh->pCON->modDect.modified = IFX_FALSE;
      }
   }
#endif /* DECT_SUPPORT */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   if (ret == IFX_SUCCESS)
      /* write AUDIO module inputs */
      ret = VMMC_TAPI_LL_AUDIO_Channel_Cfg (pDev);
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

#ifdef LIN_SUPPORT
   /* write LIN module inputs */
   for (ch = 0; (ret == IFX_SUCCESS) && (ch < pDev->caps.nLIN); ++ch)
   {
      pCh = &pDev->pChannel[ch];

      /* call the module specific set function only if something was modified */
      if (pCh->pCON->modLin.modified != IFX_FALSE)
      {
         /* write LIN module inputs */
         ret = VMMC_LIN_Set_Inputs (pCh);
         /* clear the modified flag now that we updated the cached fw message */
         pCh->pCON->modLin.modified = IFX_FALSE;
      }
   }
#endif /* LIN_SUPPORT */

   VMMC_OS_MutexRelease (&pDev->mtxMemberAcc);

   return ret;
}

/* @} */


/* ==================================================== */
/* Functions for wideband/narrowband conference support */
/* ==================================================== */

/* The feature "Wideband" means that all signals are sampled with the double
   sampling frequency so instead of 8 kHz which is now called narrowband
   16 kHz are used. This directly leads to an increase of the bandwidth of
   the voice path.
   Wideband is realised distributed on the single modules in the system. So
   when connecting modules the conferencing code makes sure that all modules
   run in the same operation mode. All modules have the ability to select the
   operation mode 8/16 kHz of sampling from the signalling array. This ISR
   (internal sampling rate) switch must be set to the same rate on all modules
   which are connected. Apart from this most modules have an extra setting
   that allows them to run in 8 kHz mode while sampling with 16 kHz from the
   signalling array. This switch is done with either setting the appropriate
   coder or the the UD-bit (up-/down-sampling).

   The code here keeps an extra attribute in each channel which tells the
   current operation mode of each module. There are four states:
    -off: the module is turned off and does not provide any output to the
          signalling array.
    -NB:  the modules samples with 8 kHz to/from the signalling array
    -WB:  the modules samples with 16 kHz to/from the signalling array
    -AUTO: the module is able to run in NB or WB mode. Current mode is
          determined by the other modules in the conference. Modules with
          AUTO prefer to run in wideband mode if there are other AUTO modules
          in the conference. ALM and AUDIO modules are marked as AUTO.
   The state is updated by each module whenever it's operation mode changes.
   The following events in the modules trigger a reevaluation of the operation
   mode of the conference the module is part of:
    -Decoder changes between NB and WB
    -Encoder sampling mode is changed between NB and WB
    -Encoder or Decoder are started or stopped
    -Sampling mode of ALM is changed by IOCTL
    -Connection between modules is created
    -Connection between modules is deleted
*/

/**
   Set the module's sampling operation mode.

   \param  pCh             Pointer to the VMMC channel structure.
   \param  src             Module type.
   \param  sampling_mode   Sampling mode as defined in VMMC_CON_SAMPLING.
*/
IFX_void_t VMMC_CON_ModuleSamplingModeSet (VMMC_CHANNEL *pCh,
                                           VMMCDSP_MT src,
                                           VMMC_CON_SAMPLING sampling_mode)
{
   /* no parameter check because this function is called only by
      VMMC internal interfaces */
   switch (src)
   {
   case VMMCDSP_MT_PCM:
      pCh->pCON->modPcm.sampling_mode = sampling_mode;
      break;
   case VMMCDSP_MT_ALM:
      pCh->pCON->modAlm.sampling_mode = sampling_mode;
      break;
   case VMMCDSP_MT_COD:
      pCh->pCON->modCod.sampling_mode = sampling_mode;
      break;
#ifdef DECT_SUPPORT
   case VMMCDSP_MT_DECT:
      pCh->pCON->modDect.sampling_mode = sampling_mode;
      break;
#endif /* DECT_SUPPORT */
   default:
      /* error: not an supported module type */
      /*lint -e(506, 774) */
      VMMC_ASSERT(0);
   }
}


#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
/**
   Returns whether the passed module type is a sub type of audio module.

\param mod_type - FW module type to be checked.

\return
   - IFX_TRUE if module type is an audio module sub type (aux, loop or diag).
   - IFX_FALSE if module type is not an audio module sub type.
\remarks
*/
static IFX_boolean_t vmmc_con_ModuleIsAudioSubType( IFX_uint8_t mod_type )
{
   if( mod_type != VMMCDSP_MT_AUDIO_AUX &&
       mod_type != VMMCDSP_MT_AUDIO_LOOP0 &&
       mod_type != VMMCDSP_MT_AUDIO_LOOP1 &&
       mod_type != VMMCDSP_MT_AUDIO_DIAG0_IN &&
       mod_type != VMMCDSP_MT_AUDIO_DIAG1_IN &&
       mod_type != VMMCDSP_MT_AUDIO_DIAG0_OUT &&
       mod_type != VMMCDSP_MT_AUDIO_DIAG1_OUT )
   {
      return IFX_FALSE;
   }
   return IFX_TRUE;

}
#endif /* #if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */


/**
   Traverses the passed module's inputs for connected modules (recursively).

\param pMod - Handle to firmware module structure the traversal starts with.

\return
   - IFX_SUCCESS if connected module was found at input input_index.
   - IFX_ERROR if no module was found at input or an invalid parameter was passed.
\remarks
   The function scans all valid module inputs for connected modules.
   In case a module is found it is checked whether the module is already
   comprised in the conference member list.
   If not the module is added to the conference member list and the function
   calls itself pointing to the connected module found.
*/
static IFX_return_t vmmc_con_TraverseModules( VMMCDSP_MODULE_t* pMod )
{
   IFX_uint32_t        input_index;
   IFX_uint32_t        inputs_to_check = 0;  /* number of valid mod. inputs */
   CONF_LIST_ELEMENT_t connected_module;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   VMMC_CON_t          *pCON;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

   if( pMod == IFX_NULL )
      return IFX_ERROR;

   memset( &connected_module, 0, sizeof(connected_module));

#if 0
   TRACE(VMMC, DBG_LEVEL_LOW,
      ("%s - pMod->nModType = %d\n", __FUNCTION__, pMod->nModType));
#endif
   /*lint -e{764}*/
   switch( pMod->nModType )
   {
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
      case VMMCDSP_MT_AUDIO_AUX:
      case VMMCDSP_MT_AUDIO_LOOP0:
      case VMMCDSP_MT_AUDIO_LOOP1:
         inputs_to_check = 1;
         break;

      case VMMCDSP_MT_AUDIO_DIAG0_IN:
      case VMMCDSP_MT_AUDIO_DIAG0_OUT:
      case VMMCDSP_MT_AUDIO_DIAG1_IN:
      case VMMCDSP_MT_AUDIO_DIAG1_OUT:
         inputs_to_check = 0;
         break;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
      default:
        inputs_to_check = MAX_MODULE_SIGNAL_INPUTS;
        break;
   }

   for (input_index = 0; input_index < inputs_to_check; ++input_index)
   {
      if( (pMod->in[input_index].i != ECMD_IX_EMPTY) &&
          (pMod->in[input_index].pOut != IFX_NULL) )
      {
         /* get module connected to this input,
            add module to conference member list and proceed traversal of
            connected modules (recursive call of vmmc_con_TraverseModules) */
         if( vmmc_con_GetInputMod( pMod, input_index, &connected_module )
             == IFX_SUCCESS )
         {
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
            if( !vmmc_con_ModuleIsAudioSubType( connected_module.pModule->nModType ) )
            {
#endif
               /* Connected Module found at input */
               if( vmmc_con_ConfMemList_ElemInList( &connected_module ) == IFX_FALSE )
               {
                  /* connected module is not in the conf. member list
                     -> append it and proceed module traversal with
                        the connected module found                   */
                  vmmc_con_ConfMemList_AppendElem( &connected_module );
                  vmmc_con_TraverseModules( connected_module.pModule );
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
                  if(connected_module.pModule->nModType == VMMCDSP_MT_AUDIO)
                  {
                     pCON = connected_module.pModule->pCh->pCON;
                     /*
                      * The connected module found is an audio module.
                      * In this case traverse audio sub modules of this channel
                      * because there is no link between the audio module
                      * and its sub modules.
                      */
                     TRACE(VMMC, DBG_LEVEL_LOW,
                           ("-> Rec. calls for audio sub modules\n"));
                     vmmc_con_TraverseModules( &pCON->modAudioAux );
                     vmmc_con_TraverseModules( &pCON->modAudioLoop0 );
                     vmmc_con_TraverseModules( &pCON->modAudioLoop1 );
                     vmmc_con_TraverseModules( &pCON->modDiag0In );
                     vmmc_con_TraverseModules( &pCON->modDiag0Out );
                     vmmc_con_TraverseModules( &pCON->modDiag1In );
                     vmmc_con_TraverseModules( &pCON->modDiag1Out );
                  }
#endif
               }
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
            }
            else
            {
               /* audio sub modules are not appended to the conf. member
                  list. Since there is no link between an audio sub module and
                  its parent audio module the module traversal is proceeded with
                  the parent audio module. */
               TRACE(VMMC, DBG_LEVEL_LOW,
                     ("-> Rec. call for audio module\n"));
               vmmc_con_TraverseModules(
                  &connected_module.pModule->pCh->pCON->modAudio );
            }
#endif
         }
      }
   }

   return IFX_SUCCESS;
}


/**
   Returns the module connected to the module's input defined by input_index.

\param pMod        - Handle to firmware module structure
\param input_index - Index of module input to be checked
\param pModConn    - pointer to connected module found (already transformed to
                     list element structure type)
\return
   - IFX_SUCCESS if connected module was found at input input_index.
   - IFX_ERROR if no module was found at input or an invalid parameter was passed.
*/
static IFX_return_t vmmc_con_GetInputMod( VMMCDSP_MODULE_t*    pMod,
                                          IFX_uint8_t          input_index,
                                          CONF_LIST_ELEMENT_t* pModConn )
{
   IFX_return_t retval=IFX_ERROR;

   if( input_index >= MAX_MODULE_SIGNAL_INPUTS ||
       pMod == IFX_NULL || pModConn == IFX_NULL)
   {
      return retval;
   }
   /* check if selected input is connected to other FW module's output */
   if( (pMod->in[input_index].i    != ECMD_IX_EMPTY) &&
       (pMod->in[input_index].pOut != IFX_NULL) )
   {
      /* fill the pModule pointer */
      pModConn->pModule = pMod->in[input_index].pOut;

#if 0
      TRACE(VMMC, DBG_LEVEL_LOW,
           ("Module 0x%08lX, type %u, input %u <- Module 0x%08lX, type %u\n",
             (IFX_uint32_t)pMod, pMod->nModType, input_index,
             (IFX_uint32_t)pMod->in[input_index].pOut, pModConn->pModule->nModType));
#endif
      retval = IFX_SUCCESS;
   }

   return retval;
}


/**
   Mutes or unmutes active coder channels in conference.

   \param  mute  Mute (IFX_ENABLE) or unmute (IFX_DISABLE) active conference.

   \return
   - IFX_SUCCESS if all activer coder channels were (un)muted.
   - IFX_ERROR if the conf member list does not exist or
               coder channel (un)muting failed.
*/
static
IFX_return_t vmmc_con_ConfMemList_MuteCtrlCoderChan( IFX_operation_t mute )
{
   CONF_LIST_ELEMENT_t *pList;

   if( conf_list_created == IFX_FALSE )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
         (__FILE__ ":%d: Conf. Member List not available\n", __LINE__));
      return IFX_ERROR;
   }

   /*
    * Beginning at the conf. member list start element the list is scanned
    * for active coder channel modules.
    * Every active coder channel found is then (un)muted according
    * to passed mute-parameter.
    */
   pList = conf_listStart;
   while( pList )
   {
      if((pList->pModule->nModType == VMMCDSP_MT_COD) &&
         (pList->pModule->sampling_mode != VMMC_CON_SMPL_OFF))
      {
         /*
          * active coder module found
          * mute or unmute this active coder module
          */
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("-> %s coder channel %d\n",
               mute == IFX_ENABLE ? "mute" : "unmute",
               pList->pModule->pCh->nChannel-1));

         if( VMMC_COD_ENC_Hold( pList->pModule->pCh, mute) != IFX_SUCCESS )
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
                  (__FILE__ ":%d: failed to %s coder channel %d\n",
                  __LINE__, mute == IFX_ENABLE ? "mute" : "unmute",
                  pList->pModule->pCh->nChannel-1));
            return IFX_ERROR;
         }
      }
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
      else
      {
         if(pList->pModule->nModType == VMMCDSP_MT_AUDIO)
         {
            if( VMMC_AUDIO_IsChannelActive(pList->pModule->pCh) )
            {
               /*
                * active audio module found
                * mute or unmute this active audio module
                */
               if( VMMC_AUDIO_MuteCfg( pList->pModule->pCh,
                                       mute )  != IFX_SUCCESS )
               {
                  TRACE(VMMC, DBG_LEVEL_HIGH,
                        (__FILE__ ":%d: failed to %s audio channel %d\n",
                        __LINE__, mute == IFX_ENABLE ? "mute" : "unmute",
                        pList->pModule->pCh->nChannel-1));
                  return IFX_ERROR;
               }
            }
         }
      }
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

      /* proceed with next list element */
      pList=pList->next;
   }
   return IFX_SUCCESS;
}


/**
   Configures the module pointed to by pMod to the requested operation mode
   or checks if an action needs to be done.

   \param action Select setting or checking of sampling rate.
   \param pMod   Pointer to firmware module to be configured.
   \param mode   Sampling mode selector (8 KHz NB or 16KHz WB).
   \return
      - IFX_SUCCESS if passed module has been successfully configured.
      - IFX_ERROR in case of module configuration error.
*/
static IFX_int32_t vmmc_con_ModSamplingMode (SM_ACTION        action,
                                             VMMCDSP_MODULE_t *pMod,
                                             OPMODE_SMPL      mode)
{
   IFX_int32_t ret = IFX_ERROR;

   VMMC_ASSERT(pMod != IFX_NULL);
   /*lint -esym(613, pMod)*/

   switch( pMod->nModType )
   {
      case VMMCDSP_MT_ALM:
         ret = VMMC_ALM_SamplingMode(pMod->pCh, action, mode,
                  (((pMod->sampling_mode == VMMC_CON_SMPL_AUTO) && (mode == WB_16_KHZ)) ||
                    (pMod->sampling_mode == VMMC_CON_SMPL_WB)) ? WB_16_KHZ : NB_8_KHZ);
         break;
      case VMMCDSP_MT_SIG:
         ret = VMMC_SIG_SamplingMode(pMod->pCh, action, mode);
         break;
      case VMMCDSP_MT_COD:
         ret = VMMC_COD_SamplingMode(pMod->pCh, action, mode);
         break;
#ifdef DECT_SUPPORT
      case VMMCDSP_MT_DECT:
         ret = VMMC_DECT_SamplingMode(pMod->pCh, action, mode);
         break;
#endif /* DECT_SUPPORT */
      case VMMCDSP_MT_PCM:
         /* PCM does not support automatic mode it only knows NB or WB */
         ret = VMMC_PCM_SamplingMode(pMod->pCh, action, mode,
                  (pMod->sampling_mode == VMMC_CON_SMPL_WB) ? WB_16_KHZ : NB_8_KHZ);
         break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
      case VMMCDSP_MT_AUDIO:
         ret = VMMC_AUDIO_SamplingMode(pMod->pCh, action, mode);
         break;
#endif  /* VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO */
#ifdef LIN_SUPPORT
      case VMMCDSP_MT_LIN:
         ret = VMMC_LIN_SamplingMode(pMod->pCh, action, mode);
         break;
#endif /* LIN_SUPPORT */
      default:
         TRACE(VMMC, DBG_LEVEL_HIGH,
            (__FILE__ ":%d: Invalid Mod. type %d\n", __LINE__, pMod->nModType));
         break;
   }

   return ret;
}


/**
   Match the internal sampling rate setting of all modules in the conference
   that is attached to the given module.

   First the conference members are looked up. Then the members are checked
   for their sampling mode. The conference is put into wideband mode when
   there are at least 2 ALM or AUDIO modules which are in automatic mode
   or one of the conference members is configured to work in wideband only.
   If wideband is not required the conference is put into narrowband mode.

   \param  pCh             VMMC Channel handle.
   \param  start_module    Channels FW module to start traversal with.

   \return
   - IFX_SUCCESS If conference has been successfully configured.
   - IFX_ERROR   In case of module configuration error.

*/
IFX_int32_t VMMC_CON_MatchConfSmplRate (VMMC_CHANNEL *pCh,
                                        VMMCDSP_MT start_module)
{
   IFX_int32_t wb_pref_cnt = 0,      /* WB prefered decisive modules counter */
               wb_only_cnt = 0;      /* wideband mode only counter */
   CONF_LIST_ELEMENT_t  *pList;      /* conference member list pointer */
   OPMODE_SMPL sampling_rate;
   IFX_int32_t retval=IFX_SUCCESS;
   IFX_boolean_t switching_needed;

   /* 1. create a list of conference members */
   vmmc_con_ConfMemListCreate(pCh, start_module);

   /* 2. walk the list of conference members and count the number of modules
         able to automatically switch to wideband and the number of modules
         configured to work in wideband only. */
   pList = conf_listStart;
   while( pList != IFX_NULL )
   {
      /* Search for WB capable FW modules that are set to automatic switching.
         These modules prefer to run in wideband if at least two can be found. */
      if (pList->pModule->sampling_mode == VMMC_CON_SMPL_AUTO)
      {
         wb_pref_cnt++;
      }

      /* Modules which are set fix to wideband force the signalling array
         to work in wideband mode. So the conference is wideband. */
      if (pList->pModule->sampling_mode == VMMC_CON_SMPL_WB)
      {
         wb_only_cnt++;
      }

      pList=pList->next;
   }

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("WB pref cnt = %d   WB only cnt = %d\n",
          wb_pref_cnt, wb_only_cnt));

   /* 3. make decision which sampling mode is required for the signalling array
    * if at least 2 decisive WB capable modules or
    * an active wideband coder channel are/is found
    * configure conference for 16 KHz mode, otherwise for 8KHz
    */
   if( (wb_pref_cnt >= 2) || (wb_only_cnt > 0) )
   {
      sampling_rate = WB_16_KHZ;
   }
   else
   {
      sampling_rate = NB_8_KHZ;
   }

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("Set sampling rate for conference: %s\n",
          sampling_rate == WB_16_KHZ ? "WIDEBAND" : "NARROWBAND"));

   /* 4. Determine if any conference member needs a sampling rate switch.
         This tells us if muting is needed. */
   switching_needed = IFX_FALSE;
   pList = conf_listStart;
   while( pList != IFX_NULL )
   {
      if( vmmc_con_ModSamplingMode( SM_CHECK,
                                    pList->pModule, sampling_rate) == IFX_TRUE )
      {
         switching_needed = IFX_TRUE;
         break;
      }
      pList=pList->next;
   }

   /* 5. mute the coder channels before changing the sampling mode */
   if (switching_needed)
   {
      if(vmmc_con_ConfMemList_MuteCtrlCoderChan (IFX_ENABLE) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("Muting coder channels failed\n"));
         retval = IFX_ERROR;
      }
   }

   /* 6. configure all conference members to the same sampling rate */
   /* This is needed even though we might not switch the sampling rate because
      the coders are configured in this step to use new types. */
   pList = conf_listStart;
   while( pList != IFX_NULL )
   {
      if( vmmc_con_ModSamplingMode( SM_SET, pList->pModule,
                                    sampling_rate ) != IFX_SUCCESS )
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("Sampling Mode Configuration failed for Module 0x%p\n",
               pList->pModule));
         retval = IFX_ERROR;
      }
      pList=pList->next;
   }

   /* 7. unmute the coder channels again */
   /* UnMute coder channels after sampling rate change is done */
   if (switching_needed)
   {
      if( vmmc_con_ConfMemList_MuteCtrlCoderChan( IFX_DISABLE ) != IFX_SUCCESS)
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("Unmuting coder channels failed\n"));
         retval = IFX_ERROR;
      }
   }

   /* 8. delete again the list of conference members */
   vmmc_con_ConfMemListDelete();

   return retval;
}


/**
   Creates list of firmware modules that are connected in a conference.

   \param  pCh             Pointer to VMMC channel structure.
   \param  start_module    Module type to start search.

   \return
   - != IFX_NULL pointer to the first element of the conference member list
   - IFX_NULL if list was not created.
*/
static
CONF_LIST_ELEMENT_t* vmmc_con_ConfMemListCreate( VMMC_CHANNEL *pCh,
                                                 VMMCDSP_MT start_module )
{
   VMMCDSP_MODULE_t    *root;
   CONF_LIST_ELEMENT_t list_elem;

   if( pCh == IFX_NULL )
   {
      return ((CONF_LIST_ELEMENT_t*)IFX_NULL);
   }

  /*
   * set the coder module of the passed channel as root element
   * for conference traversal
   */
   switch(start_module)
   {
      case VMMCDSP_MT_COD:
         root = &pCh->pCON->modCod;
         break;
      case VMMCDSP_MT_PCM:
         root = &pCh->pCON->modPcm;
         break;
      case VMMCDSP_MT_ALM:
         root = &pCh->pCON->modAlm;
         break;
#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
      case VMMCDSP_MT_AUDIO:
         root = &pCh->pCON->modAudio;
         break;
#endif  /* VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO */
#ifdef DECT_SUPPORT
      case VMMCDSP_MT_DECT:
         root = &pCh->pCON->modDect;
         break;
#endif /* DECT_SUPPORT */
#ifdef LIN_SUPPORT
      case VMMCDSP_MT_LIN:
         root = &pCh->pCON->modLin;
         break;
#endif /* LIN_SUPPORT */
      default:
         /* error: not an supported module type */
         /*lint -e(506, 774) */
         VMMC_ASSERT(0);
         return ((CONF_LIST_ELEMENT_t*)IFX_NULL);
   }

   if(!conf_list_created)
   {
      /*
       * copy passed root to list element structure, add element to list and
       * start conference traversal in order to append all conference members
       * to list
       */
      list_elem.pModule = root;
      vmmc_con_ConfMemList_AppendElem( &list_elem );
      vmmc_con_TraverseModules( root );
      conf_list_created = IFX_TRUE;
      /* TRACE(VMMC, DBG_LEVEL_LOW,
         ("Created Conf. Member List with %d elements\n", conf_listMembers)); */
   }

   return conf_listStart;
}


/**
   Deletes list of firmware modules that are connected in a conference.

   \return
   - IFX_SUCCESS if conference member list has been deleted successfully.
   - IFX_ERROR if list or list start pointer does not exist.

   \remarks
   Deletion procedure uses the list element's next pointer to access
   following element.
*/
static IFX_return_t vmmc_con_ConfMemListDelete( IFX_void_t )
{
   CONF_LIST_ELEMENT_t  *pList, *next;

   /* sanity checks */
   if( !conf_list_created )
   {
      return IFX_ERROR;
   }

   if( conf_listStart==IFX_NULL )
   {
      return IFX_ERROR;
   }

   pList = conf_listStart;
   while(pList)
   {
      /* store next list element to be deleted, delete current element and
         increment list pointer */
      next = pList->next;
      VMMC_OS_Free (pList);
      conf_listMembers--;
      pList = next;
   }
   /* sanity check after removal of list elements */
   if( conf_listMembers != 0 )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
           (__FILE__"%d: Invalid number of list members (%d) after deletion\n",
            __LINE__, conf_listMembers));
      return IFX_ERROR;
   }
   conf_listStart = conf_listEnd = IFX_NULL;
   conf_listMembers = 0;
   conf_list_created = IFX_FALSE;
   /* TRACE(VMMC, DBG_LEVEL_LOW,
         ("%s - Conf. Member list removed\n", __FUNCTION__)); */
   return IFX_SUCCESS;
}


/**
   Checks if the passed list element structure is already comprised
   in the conference member list.

\param plist_elem - Handle to list element structure to be checked for
                    list membership.

\return
   - IFX_TRUE if passed element is a list member already.
   - IFX_FALSE if passed element is not a list member.
\remarks
   The list element's adr_id field is used for comparison.
*/
static
IFX_boolean_t vmmc_con_ConfMemList_ElemInList( CONF_LIST_ELEMENT_t* plist_elem )
{
   CONF_LIST_ELEMENT_t* pList = conf_listStart;
   IFX_boolean_t elem_found = IFX_FALSE;

   /* scan list for passed element,
      element identification is done by adr_id field comparison */
   while((pList != IFX_NULL) && (elem_found == IFX_FALSE))
   {
      if( pList->pModule == plist_elem->pModule )
          elem_found = IFX_TRUE;

      pList=pList->next;
   }
   return elem_found;
}


/**
   Append element to conference member list.

\param plist_elem - Handle to list element structure to be checked for
                    list membership.

\return
   - IFX_TRUE if passed element is a list member already.
   - IFX_FALSE if passed element is not a list member.
\remarks
*/
static
IFX_return_t vmmc_con_ConfMemList_AppendElem( CONF_LIST_ELEMENT_t* plist_elem )
{
   CONF_LIST_ELEMENT_t *pNew;  /* pointer to new list element */

   /*
    * allocate memory for new list element
    */
   pNew = (CONF_LIST_ELEMENT_t*)VMMC_OS_Malloc (sizeof(CONF_LIST_ELEMENT_t));
   if( pNew == IFX_NULL )
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
            ("Could not allocate memory for new conf. member list element\n"));
      return IFX_ERROR;
   }

   /* copy the content from the passed pointer */
   *pNew = *plist_elem;

   /* append the new element to the end of the list */
   if( conf_listStart == IFX_NULL )
   {
      /* add first element to list -> set start pointer to first list element */
      conf_listStart = pNew;
   }
   else
   {
      /* add nth element to the list */
         conf_listEnd->next = pNew;
      }
   /* update list end pointer to new element and count added element */
   conf_listEnd = pNew;
   conf_listEnd->next = IFX_NULL;
   conf_listMembers++;

/* TRACE(VMMC, DBG_LEVEL_LOW,
         ("Added Module: Channel=%1d type=%1d ID=%p\n",
          plist_elem->pModule->pCh->nChannel - 1,
          plist_elem->pModule->nModType, plist_elem->pModule)); */

   return IFX_SUCCESS;
}


#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
/**
   Interchanges the Connection of Data Channels 0 and 1 to the Audio Module.
\param pDev      - pointer to VMMC_DEVICE structure
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   In detail the inputs IA and I1 of the audio module are interchanged.
   Additionally the I1 inputs of the Sig Modules 0 and 1 are interchanged.
*/
IFX_return_t VMMC_CON_Interchange_AudioDataChannels( VMMC_DEVICE *pDev )
{
   IFX_uint8_t sig0_I1;
   IFX_uint8_t sig1_I1;
   IFX_uint8_t aud0_I1;
   IFX_uint8_t aud0_IA;
   IFX_return_t ret = IFX_ERROR;

   if( pDev != IFX_NULL )
   {
      /* get I1 and IA of audio module, I1 of Sig 0 and Sig 1 */
      aud0_IA = VMMC_CON_Get_AUDIO_AUX_SignalInput( pDev, 0);
      aud0_I1 = VMMC_CON_Get_AUDIO_SignalInput( pDev, 0, 0);
      sig0_I1 = VMMC_CON_Get_SIG_SignalInput( &pDev->pChannel[0], 0);
      sig1_I1 = VMMC_CON_Get_SIG_SignalInput( &pDev->pChannel[1], 0);

      if( aud0_IA == ECMD_IX_EMPTY ||
          aud0_I1 == ECMD_IX_EMPTY ||
          sig0_I1 == ECMD_IX_EMPTY ||
          sig1_I1 == ECMD_IX_EMPTY )
      {
         /* if at least one of the signals to be interchanged is not connected
            to signaling array, return error */
         TRACE(VMMC, DBG_LEVEL_HIGH, ("%s - Input not connected\n", __FUNCTION__));
         return ret;
      }

      /* change inputs of audio module */
      pDev->pChannel[0].pCON->modAudio.in[0].i    = aud0_IA;
      pDev->pChannel[0].pCON->modAudioAux.in[0].i = aud0_I1;
      /* Change outputs of audio module -> I1 of Sig 0 resp. Sig. 1 */
      pDev->pChannel[0].pCON->modSig.in[0].i = sig1_I1;
      pDev->pChannel[1].pCON->modSig.in[0].i = sig0_I1;
      /* set modified flag for modules to be updated */
      pDev->pChannel[0].pCON->modSig.modified = IFX_TRUE;
      pDev->pChannel[1].pCON->modSig.modified = IFX_TRUE;
      pDev->pChannel[0].pCON->modAudio.modified = IFX_TRUE;
      pDev->pChannel[0].pCON->modAudioAux.modified = IFX_TRUE;
      VMMC_CON_ConnectConfigure(pDev);
      ret = IFX_SUCCESS;
   }
   return ret;
}

#endif   /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
