/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_vmmc_pmc.c
   Implements the interface towards the system power management control unit.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_api.h"
#include "drv_vmmc_pmc.h"
#include "drv_vmmc_fw_commands.h"
#include "drv_vmmc_access.h"                    /* HOST_PROTECT... macros */
#include "drv_vmmc_init.h"                      /* VMMC_GetDevice() */

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */

/* ============================= */
/* Type definitions              */
/* ============================= */

/* ============================= */
/* Function declaration          */
/* ============================= */
static IFX_PMCU_RETURN_t vmmc_pmc_state_change(IFX_PMCU_STATE_t pmcuState);
static IFX_PMCU_RETURN_t vmmc_pmc_state_get(IFX_PMCU_STATE_t *pmcuState);
static IFX_PMCU_RETURN_t vmmc_pmc_prechange(IFX_PMCU_MODULE_t pmcuModule,
                                            IFX_PMCU_STATE_t newState,
                                            IFX_PMCU_STATE_t oldState);
static IFX_PMCU_RETURN_t vmmc_pmc_postchange(IFX_PMCU_MODULE_t pmcuModule,
                                             IFX_PMCU_STATE_t newState,
                                             IFX_PMCU_STATE_t oldState);
static IFX_boolean_t vmmc_pmc_IsFwActive(void);
#if 0
static IFX_boolean_t vmmc_pmc_IsSleepMode(void);
#endif

/* ============================= */
/* Function definitions          */
/* ============================= */
/**
   Initialise Power management control.

   Create data structure in the device and register with the PMCU driver.

   \param  pDev         Pointer to VMMC device structure.

   \return IFX_SUCCESS or error code
*/
IFX_int32_t VMMC_PMC_Init(VMMC_DEVICE *pDev)
{
   IFX_PMCU_REGISTER_t pmcuRegister;
   IFX_return_t ret = IFX_SUCCESS;
   /* Dependency list. */
   /* must be static to let the gcc accept the static initialisation */
   static IFX_PMCU_MODULE_DEP_t depList=
   {
      /* nDepth */
	   1,
      {
         {  /* module id */
            IFX_PMCU_MODULE_CPU,
            /* onState */
            IFX_PMCU_STATE_D0,
            /* standBy */
            IFX_PMCU_STATE_D3,
            /* lpStandBy */
            IFX_PMCU_STATE_D3,
            /* offState */
            IFX_PMCU_STATE_D0D3
         }
      }
   };

   /* Power management control data array */
   pDev->pPMC = VMMC_OS_Malloc (sizeof (VMMC_PMC_CHANNEL_t) * VMMC_MAX_CH_NR);
   if (pDev->pPMC == IFX_NULL)
   {
      return IFX_ERROR;
   }
   memset (pDev->pPMC, 0, sizeof (VMMC_PMC_CHANNEL_t) * VMMC_MAX_CH_NR);
   /* Initialize the power management state. */
   pDev->pmcuState = IFX_PMCU_STATE_D0;

   /* Register with the PMCU driver */
   memset (&pmcuRegister, 0, sizeof(IFX_PMCU_REGISTER_t));
   pmcuRegister.pmcuModule = IFX_PMCU_MODULE_VE;
   /* The device number is stored here for use in the get function. */
   pmcuRegister.pmcuModuleNr = pDev->nDevNr;
   pmcuRegister.pmcuModuleDep = &depList;
   pmcuRegister.ifx_pmcu_state_get = vmmc_pmc_state_get;
   pmcuRegister.ifx_pmcu_state_change = vmmc_pmc_state_change;
   pmcuRegister.pre = vmmc_pmc_prechange;
   pmcuRegister.post = vmmc_pmc_postchange;
   ret = ifx_pmcu_register ( &pmcuRegister );
   if (ret != IFX_PMCU_RETURN_SUCCESS)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
           ("ERROR, VMMC dev %d: Registration to PMCU failed. \n", pDev->nDevNr));
      return IFX_ERROR;
   }

   TRACE(VMMC, DBG_LEVEL_LOW,
        ("VMMC: dev %d successfully registered to the PMCU\n", pDev->nDevNr));

   return IFX_SUCCESS;
}


/**
   Close down Power management control.

   Unregister from the PMCU driver and free data struct in the device.

   \param  pDev         Pointer to VMMC device structure.

   \return IFX_SUCCESS or error code
*/
IFX_void_t VMMC_PMC_Exit(VMMC_DEVICE *pDev)
{
   IFX_PMCU_REGISTER_t pmcuRegister;

   memset (&pmcuRegister, 0, sizeof(pmcuRegister));
   pmcuRegister.pmcuModule = IFX_PMCU_MODULE_VE;
   pmcuRegister.pmcuModuleNr = pDev->nDevNr;
   ifx_pmcu_unregister ( &pmcuRegister );

   /* free Power Management Control data array */
   if (pDev->pPMC != IFX_NULL)
   {
      VMMC_OS_Free (pDev->pPMC);
      pDev->pPMC = IFX_NULL;
   }
}


/**
   Write command to mailbox and send power management reports.

   This function records the status of all relevant algorithms used in the FW.
   It determines if the FW is idle or busy and reports this to PMCU. The PMCU
   can then implement power mangement on the device.

   \param  pDev         Pointer to VMMC device structure.
   \param  pMsg         Pointer to MPS struct with data to write.

   \return
   - IFX_SUCCESS on success.
   - IFX_ERROR on error. Error can occur if command-inbox space is not
                         sufficient or if low level function/macros fail.
   -VMMC_ERR_NO_FIBXMS  Not enough inbox space for writing command.

   \remarks
   The function will always and only write into the command mailbox. It must be
   made sure that only commands are passed to this function.
*/
IFX_int32_t VMMC_PMC_Write(VMMC_DEVICE *pDev, mps_message *pMsg)
{
   /* We only look at the common header of the FW command and the enable field.
      These parts are identical in all commands we are interested in. So we can
      use any of these FW cmd message structures to decode these fields. */
   PCM_CHAN_t *pFwCmd = /*lint --e(826)*/ (PCM_CHAN_t *)pMsg->pData;
   IFX_int32_t err;
   IFX_uint8_t i;
   IFX_return_t ret;
   IFX_boolean_t bActivate = IFX_FALSE;
   IFX_boolean_t bDoPostprocessing = IFX_FALSE;
   IFX_boolean_t bCanRollback = IFX_FALSE;
   union VMMC_PMC_CHANNEL nBackup;

   /* Analyse only write messages and ignore read messages.
      Only messages with payload may contain an enable flag.
      Be paranoid and make sure that the channel field is in range.
      Do not access PMC structs before the device is initialised. */
   if ((pFwCmd->RW == CMDWRITE) && (pFwCmd->LENGTH > 0) &&
       (pFwCmd->CHAN < VMMC_MAX_CH_NR) && (pDev->nDevState & DS_DEV_INIT))
   {
      /* 0) Make a backup of the struct we are about to change so that in the
            case of an error we can do a rollback. */
      nBackup = pDev->pPMC[pFwCmd->CHAN];
      bCanRollback = IFX_TRUE;

      /* 1) Store the EN flag from selected FW command messages in an internal
            structure of the device. */
      switch (pFwCmd->CMD)
      {
      case CMD_EOP:
         /* Decode the relevant EOP commands */
         switch (pFwCmd->MOD)
         {
         case MOD_PCM:
            switch (pFwCmd->ECMD)
            {
            case PCM_CHAN_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.pcm_ch = pFwCmd->EN;
               break;
            case PCM_LEC_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.pcm_lec = pFwCmd->EN;
               break;
            case PCM_ES_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.pcm_es = pFwCmd->EN;
               break;
            case PCM_DCHAN_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.hdlc_ch = pFwCmd->EN;
               break;
            case PCM_SCHAN_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.pcm_lb = pFwCmd->EN;
               break;
            default:
               /* struct was not changed so no need to rollback */
               bCanRollback = IFX_FALSE;
               break;
            }
            break;

         case MOD_ALI:
            switch (pFwCmd->ECMD)
            {
            case ALI_LEC_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.alm_lec = pFwCmd->EN;
               break;
            case ALI_ES_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.alm_es = pFwCmd->EN;
               break;
            default:
               /* struct was not changed so no need to rollback */
               bCanRollback = IFX_FALSE;
               break;
            }
            break;

         case MOD_SIGNALING:
            switch (pFwCmd->ECMD)
            {
            case SIG_CHAN_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_ch = pFwCmd->EN;
               break;
            case SIG_CIDS_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_fskg = pFwCmd->EN;
               break;
            case SIG_CIDR_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_fskd = pFwCmd->EN;
               break;
            case SIG_DTMFATG_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_dtmfg = pFwCmd->EN;
               break;
            case SIG_DTMFR_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_dtmfd = pFwCmd->EN;
               break;
            case SIG_UTG_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_utg1 = pFwCmd->EN;
               break;
            case SIG_UTG2_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_utg2 = pFwCmd->EN;
               break;
            case SIG_MFTD_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_mftd = pFwCmd->EN;
               break;
            case SIG_CPTD_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.sig_cptd = pFwCmd->EN;
               break;
            default:
               /* struct was not changed so no need to rollback */
               bCanRollback = IFX_FALSE;
               break;
            }
            break;

         case MOD_CODER:
            switch (pFwCmd->ECMD)
            {
            case COD_CHAN_SPEECH_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.cod_ch = pFwCmd->EN;
               break;
            case COD_AGC_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.cod_agc = pFwCmd->EN;
               break;
            case COD_FAX_CTRL_ECMD:
               pDev->pPMC[pFwCmd->CHAN].bits.cod_fdp = pFwCmd->EN;
               break;
            default:
               /* struct was not changed so no need to rollback */
               bCanRollback = IFX_FALSE;
               break;
            }
            break;

         /* Ignore all the other modules. */
         default:
            /* struct was not changed so no need to rollback */
            bCanRollback = IFX_FALSE;
            break;
         }
         break;

      case CMD_DECT:
         /* Decode the relevant DECT commands */
         switch (pFwCmd->ECMD)
         {
         case COD_CHAN_SPEECH_ECMD:
            pDev->pPMC[pFwCmd->CHAN].bits.dect_ch = pFwCmd->EN;
            break;
         case DECT_UTG_CTRL_ECMD:
            pDev->pPMC[pFwCmd->CHAN].bits.dect_utg = pFwCmd->EN;
            break;
         default:
            /* struct was not changed so no need to rollback */
            bCanRollback = IFX_FALSE;
            break;
         }
         break;

      default:
         /* struct was not changed so no need to rollback */
         bCanRollback = IFX_FALSE;
         break;
      }

      /* 2) Check if any flag is set in the internal structure of the device.
            This would indicate that the FW is still busy. If all bits are
            cleared the FW is idle.*/
      for (i=0; i < VMMC_MAX_CH_NR ; i++)
      {
         if (pDev->pPMC[i].value != 0)
         {
            bActivate = IFX_TRUE;
            break;
         }
      }

      /* 3) If there is a transistion from idle to active send a notification
            to the PMCU that full performance is needed. This is blocking until
            performance was increased. */
      if ((pDev->bFwActive == IFX_FALSE) && (bActivate == IFX_TRUE))
      {
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("VMMC dev %d: activate FW\n", pDev->nDevNr));

         /* Call PMCU to indicate that FW is about to need full performance.
            This is blocking until the performance was increased. */
         ret = ifx_pmcu_state_req(IFX_PMCU_MODULE_VE,
                                  pDev->nDevNr,
                                  IFX_PMCU_STATE_D0);
         if (ret != IFX_PMCU_RETURN_SUCCESS)
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
                 ("VMMC dev %d: PMCU failed to change state to D0\n",
                  pDev->nDevNr));
         }

         /* FW is active */
         pDev->bFwActive = IFX_TRUE;
         pDev->bSddSleepState = IFX_FALSE;
      }
      else
      {
         /* Remember to check after the write if something needs to be done. */
         bDoPostprocessing = IFX_TRUE;
      }
   }

   /* 4) Send command to FW. */
   /* concurrent access protect by driver, but it should move to here.
       To cease interrupt and to use semaphore here is a good idea.*/
   VMMC_HOST_PROTECT(pDev);
   err = ifx_mps_write_mailbox(command, pMsg);
   VMMC_HOST_RELEASE(pDev);

   /* 5) If the write failed we do a rollback of the status struct. */
   if ((err != IFX_SUCCESS) && (bCanRollback == IFX_TRUE))
   {
      /*lint -e{644} bCanRollback makes sure that nBackup is initialised */
       pDev->pPMC[pFwCmd->CHAN] = nBackup;
       /* NOTE: Currently a performance report done in step 3 is not recalled.
                So PMCU may think we need full performance although activation
                of the algorithm triggering this failed. */
   }

   /* 6) If there is a transistion from active to idle send a notification
         to the PMCU that reduced performance will do. This is blocking until
         performance setting was adapted. */
   if ((bDoPostprocessing == IFX_TRUE) && (err == IFX_SUCCESS) &&
       (pDev->bFwActive == IFX_TRUE) && (bActivate == IFX_FALSE))
   {
      /* Synchronise with the FW to make sure that the command written just
         above has already been processed. For this a read command is written
         and then waited for the response. Because the commands are processed
         in order we can then be sure that as soon as we get the response all
         commands written before this were processed. */
      SYS_VER_t   pCmd;

      /* Do a version read command */
      memset((IFX_void_t *)&pCmd, 0, sizeof(SYS_VER_t));
      pCmd.CMD = CMD_EOP;
      pCmd.MOD = MOD_SYSTEM;
      pCmd.ECMD = ECMD_SYS_VER;
      err = CmdRead(pDev, (IFX_uint32_t *)&pCmd, (IFX_uint32_t *)&pCmd, 4);
      if ( err == IFX_SUCCESS )
      {
         TRACE(VMMC, DBG_LEVEL_LOW,
               ("VMMC dev %d: deactivate FW\n", pDev->nDevNr));

         /* Call PMCU to indicate FW needs only reduced performance.
            This is blocking until the performance was decreased. */
         ret = ifx_pmcu_state_req(IFX_PMCU_MODULE_VE,
                                  pDev->nDevNr,
                                  IFX_PMCU_STATE_D2);
         if (ret != IFX_PMCU_RETURN_SUCCESS)
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("VMMC dev %d: PMCU failed to change state to D2\n",
                   pDev->nDevNr));
         }

         /* Now FW is idle  */
         pDev->bFwActive = IFX_FALSE;
      }
      else
      {
         TRACE(VMMC, DBG_LEVEL_HIGH,
               ("ERROR, VMMC dev %d: PMCU failed to synchronise\n", pDev->nDevNr));
      }
   }

   return err;
}


/**
   Callback used to change module's power state.

   \param  pmcuState    Structure with power management state.

   \return
   - IFX_PMCU_RETURN_SUCCESS Change the Power State successfully.
   - IFX_PMCU_RETURN_ERROR   Error occured during the change of Power State.
*/
static IFX_PMCU_RETURN_t vmmc_pmc_state_change(IFX_PMCU_STATE_t pmcuState)
{
   VMMC_DEVICE  *pDev = IFX_NULL;

   if (VMMC_GetDevice (0, &pDev) != IFX_SUCCESS)
   {
      /* Should not happen. Module 0 should be always available. */
      TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("ERROR, VMMC failed to get device 0.\n"));
      return IFX_PMCU_RETURN_ERROR;
   }
   pDev->pmcuState = pmcuState;
   return IFX_PMCU_RETURN_SUCCESS;
}


/**
   Get the status of the voice FW.

   \param  pmcuState    Pointer to return power state.

   \return
   - IFX_PMCU_RETURN_SUCCESS Get power-state successfully.
   - IFX_PMCU_RETURN_ERROR   Get power-state failed.
*/
static IFX_PMCU_RETURN_t vmmc_pmc_state_get(IFX_PMCU_STATE_t *pmcuState)
{
   VMMC_DEVICE  *pDev = IFX_NULL;

   if (VMMC_GetDevice (0, &pDev) != IFX_SUCCESS)
   {
      /* Should not happen. Module 0 should be always available. */
      TRACE(VMMC, DBG_LEVEL_HIGH,
                  ("ERROR, VMMC failed to get device 0.\n"));
      return IFX_PMCU_RETURN_ERROR;
   }
   *pmcuState = pDev->pmcuState;
   return IFX_PMCU_RETURN_SUCCESS;
}


/**
   Callback to be called before module changes his state to new.

   \param  pmcuModule   Module
   \param  newState     New state
   \param  oldState     Old state

   \return
   - IFX_SUCCESS on success.
   - IFX_ERROR on error.
*/
static IFX_PMCU_RETURN_t vmmc_pmc_prechange(IFX_PMCU_MODULE_t pmcuModule,
                                            IFX_PMCU_STATE_t newState,
                                            IFX_PMCU_STATE_t oldState)
{
   if (pmcuModule != IFX_PMCU_MODULE_VE)
   {
      TRACE(VMMC, DBG_LEVEL_HIGH,
           ("ERROR, VMMC: Invalide module Id (module: %d) (%s, %d\n) ",
            pmcuModule,  __FILE__, __LINE__));
      return IFX_PMCU_RETURN_ERROR;
   }

   if (newState > oldState)
   {
      /* CPU module requests lower clock. Check if FW is active. */
      if (vmmc_pmc_IsFwActive() == IFX_TRUE)
      {
         TRACE(VMMC, DBG_LEVEL_LOW,
         ("VMMC: Deny lower clock - FW is active (module:%d, newState:%d,"
          " oldState:%d)\n", pmcuModule, newState, oldState));
         return IFX_PMCU_RETURN_DENIED;
      }

   }

   return IFX_PMCU_RETURN_SUCCESS;
}


/**
   Callback to be called after module changes his state to new state.

   \param   pmcuModule      Module
   \param   newState        New state
   \param   oldState        Old state

   \return
   - IFX_SUCCESS on success.
   - IFX_ERROR on error.
*/
static IFX_PMCU_RETURN_t vmmc_pmc_postchange(IFX_PMCU_MODULE_t pmcuModule,
                                             IFX_PMCU_STATE_t newState,
                                             IFX_PMCU_STATE_t oldState)
{
   switch (pmcuModule)
   {
      case IFX_PMCU_MODULE_CPU_PS:
         /* check hook status and eventually enable 8kHz clock. */
         break;

      default:
         break;
   }
   return IFX_PMCU_RETURN_SUCCESS;
}


/**
   Check if FW is active.

   \return
   - IFX_TRUE if FW is active.
   - IFX_FALSE if FW is not active.
*/
static IFX_boolean_t vmmc_pmc_IsFwActive(void)
{
   IFX_uint16_t     dev;
   VMMC_DEVICE      *pDev = IFX_NULL;

   for (dev = 0; dev < VMMC_MAX_DEVICES; dev++)
   {
      if (VMMC_GetDevice (dev, &pDev) == IFX_SUCCESS)
      {
         if (pDev->bFwActive == IFX_TRUE)
         {
            /* FW is active. There is no need to check other devices. */
            TRACE(VMMC, DBG_LEVEL_LOW,
               ("VMMC dev %d: FW is active \n", dev));
            return IFX_TRUE;
         }
      }
   }
   return IFX_FALSE;
}


#if 0 /* for future use for core power gating support */
/**
   Check if SDD entered the SLEEP mode.

   \return
   - IFX_TRUE if all devices are in SLEEP mode.
   - IFX_FALSE in other cases.
*/
static IFX_boolean_t vmmc_pmc_IsSleepMode(void)
{
   IFX_uint16_t     dev;
   VMMC_DEVICE      *pDev = IFX_NULL;

   for (dev = 0; dev < VMMC_MAX_DEVICES; dev++)
   {
      if (VMMC_GetDevice (dev, &pDev) == IFX_SUCCESS)
      {
         if (pDev->bSddSleepState == IFX_FALSE)
         {
            /* At least one device is not in SLEEP mode. Return IFX_FALSE. */
            TRACE(VMMC, DBG_LEVEL_LOW,
               ("VMMC dev %d: device not in SLEEP mode\n", dev));
            return IFX_FALSE;
         }
      }
   }
   return IFX_TRUE;
}
#endif
