/******************************************************************************

                               Copyright (c) 2009
                            Infineon Technologies AG
                     Am Campeon 1-12; 81726 Munich, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file lib_tapi_lt_common_cpe.c
   Contains implementation of following user interface functions for GR.909
   line testing:
   - Ifxphone_LT_GR909_Config
   - Ifxphone_LT_GR909_Start
   - Ifxphone_LT_GR909_Stop
   - Ifxphone_LT_GR909_GetResults
*/

/* ============================= */
/* Includes                      */
/* ============================= */
/* system includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>

#include "sys_debug.h"
#include "ifx_types.h"
#include "drv_tapi_io.h"
#include "lib_tapi_lt_gr909.h"

/* ============================= */
/* Defines                       */
/* ============================= */
/* ============================= */
/* Local macros & definitions    */
/* ============================= */
DECLARE_TRACE_GROUP(TAPI);
/* ============================= */
/* Local function declaration    */
/* ============================= */
#if (defined (VMMC) && defined (TAPI_GR909))
extern IFX_void_t VMMC_LT_GR909_CalcResults(IFX_LT_GR909_RESULT_t *p_res,
                                            IFX_TAPI_GR909_RESULT_t *pTapiRes);
extern IFX_LT_GR909_CFG_t vmmc_cfg;
#endif
#if (defined (VIN_2CPE) && defined (TAPI_GR909))
extern IFX_void_t VINCPE_LT_GR909_CalcResults(IFX_LT_GR909_RESULT_t *p_res,
                                            IFX_TAPI_GR909_RESULT_t *pTapiRes);
extern IFX_LT_GR909_CFG_t vincpe_cfg;
#endif
#if (defined (DXT) && defined (TAPI_GR909))
extern IFX_void_t DXT_LT_GR909_CalcResults(IFX_LT_GR909_RESULT_t *p_res,
                                            IFX_TAPI_GR909_RESULT_t *pTapiRes);
extern IFX_LT_GR909_CFG_t dxt_cfg;
#endif

/* ============================= */
/* Local function definition     */
/* ============================= */
/* ============================= */
/* Global function definition    */
/* ============================= */
/**
   Configure system parameters (SLIC) to use for values calculation,
    e.g Voltage divider resistors.

   \param dev   - enum of \ref IFX_TAPI_GR909_DEV_t type
   \param p_cfg - handle to IFX_LT_GR909_CFG_t structure
   \return
      IFX_SUCCESS / IFX_ERROR
   \remark
      Usage of this function is optional. Default parameters are used
      otherwise.
*/
IFX_return_t Ifxphone_LT_GR909_Config (IFX_TAPI_GR909_DEV_t dev,
                                                    IFX_LT_GR909_CFG_t *p_cfg)
{
   IFX_return_t ret = IFX_SUCCESS;
   IFX_LT_GR909_CFG_t *pUserCfg = IFX_NULL;

   /* check config integrity to insure software safety */
   if ((p_cfg->f_R1 == 0.0) || (p_cfg->f_R2 == 0.0) || (p_cfg->f_R3 == 0.0))
      return IFX_ERROR;

   /* use user config */
   switch (dev)
   {
#if (defined (VMMC) && defined (TAPI_GR909))
      case IFX_TAPI_GR909_DEV_VMMC:
         pUserCfg = &vmmc_cfg;
         break;
#endif
#if (defined (VIN_2CPE) && defined (TAPI_GR909))
      case IFX_TAPI_GR909_DEV_VINCPE:
         pUserCfg = &vincpe_cfg;
         break;
#endif
#if (defined (DXT) && defined (TAPI_GR909))
      case IFX_TAPI_GR909_DEV_DXT:
         pUserCfg = &dxt_cfg;
         break;
#endif
      default:
         TRACE(TAPI, DBG_LEVEL_NORMAL,
                 ("\n\rIfxphone_LT_GR909_Config: incorrect device type\n\r"));
   }

   if (pUserCfg != IFX_NULL)
   {
      *pUserCfg = *p_cfg;
   }
   else
   {
      TRACE(TAPI, DBG_LEVEL_NORMAL,
                 ("\n\rIfxphone_LT_GR909_Config: pUserCfg is NULL!\n\r"));
      ret = IFX_ERROR;
   }

   return ret;
}

/**
   Start a GR909 test or test sequence according to measument mask
   set with \ref IFX_LT_GR909_MASK_t.

   \param fd_line   - line file descriptor
   \param b_euLike  - IFX_TRUE  : EU like powerline frequency (50 Hz)
                   IFX_FALSE : US like power line frequency (60 Hz)
   \param meas_mask - Measurement mask set with values out
                   of \ref IFX_LT_GR909_MASK_t
   \return
      IFX_SUCCESS / IFX_ERROR
*/
IFX_return_t Ifxphone_LT_GR909_Start (IFX_int32_t   fd_line,
                                      IFX_boolean_t b_euLike,
                                      IFX_uint32_t  meas_mask)
{
   IFX_return_t             ret;
   IFX_TAPI_GR909_START_t   meas_start;
   IFX_uint32_t             *mask = &meas_start.test_mask;

   memset (&meas_start, 0, sizeof (meas_start));

   /* setup powerline frequency  */
   meas_start.pl_freq = ((b_euLike == IFX_TRUE) ? IFX_TAPI_GR909_EU_50HZ :
                                                       IFX_TAPI_GR909_US_60HZ);
   /* setup driver test mask according to user mask */
   *mask |= ((meas_mask & IFX_LT_GR909_HPT_MASK)  ? IFX_TAPI_GR909_HPT  : 0);
   *mask |= ((meas_mask & IFX_LT_GR909_FEMF_MASK) ? IFX_TAPI_GR909_FEMF : 0);
   *mask |= ((meas_mask & IFX_LT_GR909_RFT_MASK)  ? IFX_TAPI_GR909_RFT  : 0);
   *mask |= ((meas_mask & IFX_LT_GR909_ROH_MASK)  ? IFX_TAPI_GR909_ROH  : 0);
   *mask |= ((meas_mask & IFX_LT_GR909_RIT_MASK)  ? IFX_TAPI_GR909_RIT  : 0);

   /* no test ? */
   if (*mask == 0)
      return IFX_ERROR;

   ret = ioctl(fd_line, IFX_TAPI_GR909_START, (IFX_int32_t) &meas_start);

   return ret;
}

/**
   Stop a GR909 test or test sequence

   \param fd_line   - line file descriptor
   \return
      IFX_SUCCESS / IFX_ERROR
*/
IFX_return_t Ifxphone_LT_GR909_Stop (IFX_int32_t fd_line)
{
   IFX_return_t ret;

   ret = ioctl(fd_line, IFX_TAPI_GR909_STOP, 0);

   return ret;
}

/**
   Gets Gr909 measurement results.

   \param fd_line  - line file descriptor
   \param p_res  - handle to user result structure
   \return
      IFX_SUCCESS / IFX_ERROR
   \remark
      Only evaluate results which are marked as valid.
*/
IFX_return_t Ifxphone_LT_GR909_GetResults (IFX_int32_t            fd_line,
                                           IFX_LT_GR909_RESULT_t *p_res)
{
   IFX_return_t             ret;
   IFX_TAPI_GR909_RESULT_t  meas_res;
   IFX_void_t (*pf_calc)(IFX_LT_GR909_RESULT_t *pRes,
                                 IFX_TAPI_GR909_RESULT_t *pTapiRes) = IFX_NULL;

   /* reset structures */
   memset (&meas_res, 0, sizeof (meas_res));
   memset (p_res, 0, sizeof (*p_res));

   /* read results */
   ret = ioctl(fd_line, IFX_TAPI_GR909_RESULT, (IFX_int32_t)&meas_res);
   if (ret != IFX_SUCCESS)
      return IFX_ERROR;

   switch (meas_res.dev)
   {
#if (defined (VMMC) && defined (TAPI_GR909))
      case IFX_TAPI_GR909_DEV_VMMC:
         pf_calc = VMMC_LT_GR909_CalcResults;
         break;
#endif /* VMMC */

#if (defined (VIN_2CPE) && defined (TAPI_GR909))
      case IFX_TAPI_GR909_DEV_VINCPE:
         pf_calc = VINCPE_LT_GR909_CalcResults;
         break;
#endif /* VIN_2CPE */

#if (defined (DXT) && defined (TAPI_GR909))
      case IFX_TAPI_GR909_DEV_DXT:
         pf_calc = DXT_LT_GR909_CalcResults;
         break;
#endif /* DXT */

      default:
         TRACE(TAPI, DBG_LEVEL_NORMAL,
              ("\n\rIfxphone_LT_GR909_GetResults: incorrect device type\n\r"));
   }

   if (pf_calc != IFX_NULL)
   {
      pf_calc(p_res, &meas_res);
   }
   else
   {
      TRACE(TAPI, DBG_LEVEL_NORMAL,
                 ("\n\rIfxphone_LT_GR909_GetResults: pf_calc is NULL!\n\r"));
      ret = IFX_ERROR;
   }

   return ret;
}

