/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

****************************************************************************
   Module      : drv_mps_vmmc_danube.c
   Description : This file contains the implementation of the Danube specific
                 driver functions.
*******************************************************************************/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_config.h"

#ifdef SYSTEM_DANUBE            /* defined in drv_mps_vmmc_config.h */

#include <linux/autoconf.h>

/* lib_ifxos headers */
#include "ifx_types.h"
#include "ifxos_linux_drv.h"
#include "ifxos_copy_user_space.h"
#include "ifxos_time.h"
#include "ifxos_event.h"
#include "ifxos_lock.h"
#include "ifxos_select.h"
#include "ifxos_interrupt.h"

#include <asm/ifx/ifx_regs.h>
#include <asm/ifx/ifx_gpio.h>
#include <asm/ifx/common_routines.h>

#include "drv_mps_vmmc.h"
#include "drv_mps_vmmc_dbg.h"
#include "drv_mps_vmmc_device.h"

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */

/* ============================= */
/* Global variable definition    */
/* ============================= */
extern mps_comm_dev ifx_mps_dev;
extern mps_comm_dev *pMPSDev;
extern IFXOS_event_t fw_ready_evt;

/* ============================= */
/* Global function declaration   */
/* ============================= */
extern IFX_uint32_t *danube_get_cp1_base (IFX_void_t);
extern IFX_uint32_t ifx_mps_reset_structures (mps_comm_dev * pDev);
extern IFX_int32_t ifx_mps_bufman_close (IFX_void_t);
extern IFX_uint32_t *ifx_get_cp1_base (IFX_void_t);
/* ============================= */
/* Local function declaration    */
/* ============================= */
IFX_void_t ifx_mps_release (IFX_void_t);

/* ============================= */
/* Local variable definition     */
/* ============================= */

/* ============================= */
/* Local function definition     */
/* ============================= */

/******************************************************************************
 * DANUBE Specific Routines
 ******************************************************************************/
/**
 * Firmware download to Voice CPU
 * This function performs a firmware download to the coprocessor.
 *
 * \param   pMBDev    Pointer to mailbox device structure
 * \param   pFWDwnld  Pointer to firmware structure
 * \return  0         IFX_SUCCESS, firmware ready
 * \return  -1        IFX_ERROR, firmware not downloaded.
 * \ingroup Internal
 */
IFX_int32_t ifx_mps_download_firmware (mps_mbx_dev *pMBDev, mps_fw *pFWDwnld)
{
   IFX_uint32_t mem, cksum;
   IFX_uint8_t crc;
   IFX_boolean_t bMemReqNotPresent = IFX_FALSE;

   /* copy FW footer from user space */
   if (IFX_NULL == IFXOS_CpyFromUser(pFW_img_data,
                           pFWDwnld->data+pFWDwnld->length/4-sizeof(*pFW_img_data)/4,
                           sizeof(*pFW_img_data)))
   {
      TRACE (MPS, DBG_LEVEL_HIGH,
                  (KERN_ERR "[%s %s %d]: copy_from_user error\r\n",
                   __FILE__, __func__, __LINE__));
      return IFX_ERROR;
   }

   if(FW_FORMAT_NEW)
   {
      IFX_uint32_t plt = pFW_img_data->fw_vers >> 8 & 0xf;

      /* platform check */
      if (plt != FW_PLT_DANUBE)
      {
         TRACE (MPS, DBG_LEVEL_HIGH,("WRONG FIRMWARE PLATFORM!\n"));
         return IFX_ERROR;
      }
   }

   mem = pFW_img_data->mem;

   /* memory requirement sanity check */
   if ((crc = ~((mem >> 16) + (mem >> 8) + mem)) != (mem >> 24))
   {
      TRACE (MPS, DBG_LEVEL_HIGH,
          ("[%s %s %d]: warning, image does not contain size - assuming 1MB!\n",
           __FILE__, __func__, __LINE__));
      mem = 1 * 1024 * 1024;
      bMemReqNotPresent = IFX_TRUE;
   }
   else
   {
      mem &= 0x00FFFFFF;
   }

   /* check if FW image fits in available memory space */
   if (mem > ifx_get_cp1_size())
   {
      TRACE (MPS, DBG_LEVEL_HIGH,
      ("[%s %s %d]: error, firmware memory exceeds reserved space (%i > %i)!\n",
                 __FILE__, __func__, __LINE__, mem, ifx_get_cp1_size()));
      return IFX_ERROR;
   }

   /* reset the driver */
   ifx_mps_reset ();

   /* call BSP to get cpu1 base address */
   cpu1_base_addr = ifx_get_cp1_base ();

   /* check if CPU1 base address is sane */
   if (cpu1_base_addr == IFX_NULL || !cpu1_base_addr)
   {
      TRACE (MPS, DBG_LEVEL_HIGH,
             (KERN_ERR "IFX_MPS: CPU1 base address is invalid!\r\n"));
      return IFX_ERROR;
   }
   else
   {
      /* check if CPU1 address is 1MB aligned */
      if ((IFX_uint32_t)cpu1_base_addr & 0xfffff)
      {
         TRACE (MPS, DBG_LEVEL_HIGH,
               (KERN_ERR "IFX_MPS: CPU1 base address is not 1MB aligned!\r\n"));
         return IFX_ERROR;
      }
   }

   /* further use uncached value */
   cpu1_base_addr = (IFX_uint32_t *)KSEG1ADDR(cpu1_base_addr);

   /* free all data buffers that might be currently used by FW */
   if (IFX_NULL != ifx_mps_bufman_freeall)
   {
      ifx_mps_bufman_freeall();
   }

   if(FW_FORMAT_NEW)
   {
      /* adjust download length */
      pFWDwnld->length -= (sizeof(*pFW_img_data)-sizeof(IFX_uint32_t));
   }
   else
   {
      pFWDwnld->length -= sizeof(IFX_uint32_t);

      /* handle unlikely case if FW image does not contain memory requirement -
         assumed for old format only */
      if (IFX_TRUE == bMemReqNotPresent)
         pFWDwnld->length += sizeof(IFX_uint32_t);

      /* in case of old FW format always assume that FW is encrypted;
         use compile switch USE_PLAIN_VOICE_FIRMWARE for plain FW */
#ifndef USE_PLAIN_VOICE_FIRMWARE
      pFW_img_data->enc = 1;
#else
#warning Using unencrypted firmware!
      pFW_img_data->enc = 0;
#endif /* USE_PLAIN_VOICE_FIRMWARE */
      /* initializations for the old format */
      pFW_img_data->st_addr_crc = 2*sizeof(IFX_uint32_t) +
                               FW_DANUBE_OLD_FMT_XCPT_AREA_SZ;
      pFW_img_data->en_addr_crc = pFWDwnld->length;
      pFW_img_data->fw_vers = 0;
      pFW_img_data->magic = 0;
   }

   /* copy FW image to base address of CPU1 */
   if (IFX_NULL ==
       IFXOS_CpyFromUser ((IFX_void_t *)cpu1_base_addr,
                          (IFX_void_t *)pFWDwnld->data, pFWDwnld->length))
   {
      TRACE (MPS, DBG_LEVEL_HIGH,
             (KERN_ERR "[%s %s %d]: copy_from_user error\r\n", __FILE__,
              __func__, __LINE__));
      return IFX_ERROR;
   }

   /* process firmware decryption */
   if (pFW_img_data->enc == 1)
   {
      if(FW_FORMAT_NEW)
      {
         /* adjust decryption length (avoid decrypting CRC32 checksum) */
         pFWDwnld->length -= sizeof(IFX_uint32_t);
      }
      /* BootROM actually decrypts n+4 bytes if n bytes were passed for
         decryption. Subtract sizeof(u32) from length to avoid decryption
         of data beyond the FW image code */
      pFWDwnld->length -= sizeof(IFX_uint32_t);
      ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_CFG_STAT = 0x00020000;
   }
   else
   {
      ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_CFG_STAT |= 0x00700000;
   }

   ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_BOOT_SIZE = pFWDwnld->length;
   ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_BOOT_RVEC =
                                                 (IFX_uint32_t)cpu1_base_addr;

   /* start CPU1 */
   ifx_mps_release ();

   /* calculate CRC32 checksum over downloaded image */
   cksum = ifx_mps_fw_crc32(cpu1_base_addr, pFW_img_data);

   /* verify the checksum */
   if(FW_FORMAT_NEW)
   {
      if (cksum != pFW_img_data->crc32)
      {
         TRACE (MPS, DBG_LEVEL_HIGH,
                ("MPS: FW checksum error: img=0x%08x calc=0x%08x\r\n",
                pFW_img_data->crc32, cksum));
         return IFX_ERROR;
      }
   }
   else
   {
      /* just store self-calculated checksum */
      pFW_img_data->crc32 = cksum;
   }

   /* get FW version */
   return ifx_mps_get_fw_version (0);
}


/**
 * Restart CPU1
 * This function restarts CPU1 by accessing the reset request register and
 * reinitializes the mailbox.
 *
 * \return  0        IFX_SUCCESS, successful restart
 * \return  -1       IFX_ERROR, if reset failed
 * \ingroup Internal
 */
IFX_int32_t ifx_mps_restart (IFX_void_t)
{
   /* Disable GPTC Interrupt to CPU1 */
   ifx_mps_shutdown_gpt ();
   /* wait 100 ms */
   IFXOS_MSecSleep (100);
   /* raise reset request for CPU1 and reset driver structures */
   ifx_mps_reset ();
   /* re-configure GPTC */
   ifx_mps_init_gpt ();
   /* update CPU1 boot parameters */
   ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_CP0_STATUS = 0;
   ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_CP0_EPC = 0;
   ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_CP0_EEPC = 0;
   ifx_mps_dev.base_global->MBX_CPU1_BOOT_CFG.MPS_CFG_STAT = 0x00700000;
   /* let CPU1 run */
   ifx_mps_release ();
   TRACE (MPS, DBG_LEVEL_HIGH, ("IFX_MPS: Restarting firmware..."));
   return ifx_mps_get_fw_version (0);
}

/**
 * Shutdown CPU1
 * This function stops CPU1 by clearing the boot ready bit in RCU_RST_REQ.
 *
 * \ingroup Internal
 */
IFX_void_t ifx_mps_shutdown (IFX_void_t)
{
   ifx_mps_shutdown_gpt ();
   *IFX_RCU_RST_REQ |= (IFX_RCU_RST_REQ_CPU1);
}

/******************************************************************************
 *
 ******************************************************************************/

/**
 * Reset CPU1
 * This function causes a reset of CPU1 by clearing the CPU0 boot ready bit
 * in the reset request register RCU_RST_REQ.
 * It does not change the boot configuration registers for CPU0 or CPU1.
 *
 * \return  0        IFX_SUCCESS, cannot fail
 * \ingroup Internal
 */
IFX_void_t ifx_mps_reset (IFX_void_t)
{
   *IFX_RCU_RST_REQ |= (IFX_RCU_RST_REQ_CPU1);
   smp_wmb ();
   /* reset driver */
   ifx_mps_reset_structures (pMPSDev);
   ifx_mps_bufman_close ();
   return;
}

IFX_void_t ifx_mps_release (IFX_void_t)
{
   IFX_int_t ret;
   IFX_int32_t RetCode = 0;

   *IFX_RCU_RST_REQ |= 0x20000000;
   *IFX_RCU_RST_REQ &= (~8);
   smp_wmb ();
   /* sleep 3 seconds until FW is ready */
   ret = IFXOS_EventWait (&fw_ready_evt, 3000, &RetCode);
   if ((ret == IFX_ERROR) && (RetCode == 1))
   {
      /* timeout */
      TRACE (MPS, DBG_LEVEL_HIGH,
             (KERN_ERR "[%s %s %d]: Timeout waiting for firmware ready.\r\n",
              __FILE__, __func__, __LINE__));
   }
   return;
}

/**
 * WDT expiry
 * This function is called in interrupt ctx
 *
 * \return  none
 * \ingroup Internal
 */
IFX_void_t ifx_mps_wdog_expiry()
{
   IFX_uint32_t flags;

   IFXOS_LOCKINT (flags);
   /* recalculate and compare the firmware checksum */
   ifx_mps_fw_crc_compare(cpu1_base_addr, pFW_img_data);

   /* dump exception area on a console */
   ifx_mps_dump_fw_xcpt(cpu1_base_addr, pFW_img_data);
   IFXOS_UNLOCKINT (flags);
}

#endif /* SYSTEM_DANUBE */
