/// <reference types="node" />
import { SessionIdentifier } from "../../types";
import { CameraStreamingDelegate, ResourceRequestReason, StateChangeDelegate } from "../controller";
import type { CameraRTPStreamManagement } from "../definitions";
import { LegacyCameraSource } from "./Camera";
import RTPProxy from "./RTPProxy";
declare const enum StreamingStatus {
    AVAILABLE = 0,
    IN_USE = 1,
    UNAVAILABLE = 2
}
/**
 * @group Camera
 */
export declare const enum VideoCodecType {
    H264 = 0
}
/**
 * @group Camera
 */
export declare const enum H264Profile {
    BASELINE = 0,
    MAIN = 1,
    HIGH = 2
}
/**
 * @group Camera
 */
export declare const enum H264Level {
    LEVEL3_1 = 0,
    LEVEL3_2 = 1,
    LEVEL4_0 = 2
}
/**
 * @group Camera
 */
export declare const enum VideoCodecPacketizationMode {
    NON_INTERLEAVED = 0
}
/**
 * @group Camera
 */
export declare const enum AudioBitrate {
    VARIABLE = 0,
    CONSTANT = 1
}
/**
 * @group Camera
 */
export declare const enum AudioSamplerate {
    KHZ_8 = 0,
    KHZ_16 = 1,
    KHZ_24 = 2
}
/**
 * @group Camera
 */
export declare const enum SRTPCryptoSuites {
    AES_CM_128_HMAC_SHA1_80 = 0,
    AES_CM_256_HMAC_SHA1_80 = 1,
    NONE = 2
}
/**
 * @group Camera
 * @deprecated renamed to {@link CameraStreamingOptions}
 */
export type StreamControllerOptions = CameraStreamingOptions;
/**
 * @group Camera
 */
export type CameraStreamingOptions = CameraStreamingOptionsBase & (CameraStreamingOptionsLegacySRTP | CameraStreamingOptionsSupportedCryptoSuites);
/**
 * @group Camera
 */
export interface CameraStreamingOptionsBase {
    proxy?: boolean;
    disable_audio_proxy?: boolean;
    video: VideoStreamingOptions;
    /**
     * "audio" is optional and only needs to be declared if audio streaming is supported.
     * If defined the Microphone service will be added and Microphone volume control will be made available.
     * If not defined hap-nodejs will expose a default codec in order for the video stream to work
     */
    audio?: AudioStreamingOptions;
}
/**
 * @group Camera
 */
export interface CameraStreamingOptionsLegacySRTP {
    srtp: boolean;
}
/**
 * @group Camera
 */
export interface CameraStreamingOptionsSupportedCryptoSuites {
    supportedCryptoSuites: SRTPCryptoSuites[];
}
/**
 * @group Camera
 */
export type VideoStreamingOptions = {
    codec: H264CodecParameters;
    resolutions: Resolution[];
    cvoId?: number;
};
/**
 * @group Camera
 */
export interface H264CodecParameters {
    levels: H264Level[];
    profiles: H264Profile[];
}
/**
 * @group Camera
 */
export type Resolution = [number, number, number];
/**
 * @group Camera
 */
export type AudioStreamingOptions = {
    codecs: AudioStreamingCodec[];
    twoWayAudio?: boolean;
    comfort_noise?: boolean;
};
/**
 * @group Camera
 */
export type AudioStreamingCodec = {
    type: AudioStreamingCodecType | string;
    audioChannels?: number;
    bitrate?: AudioBitrate;
    samplerate: AudioStreamingSamplerate[] | AudioStreamingSamplerate;
};
/**
 * @group Camera
 */
export declare const enum AudioStreamingCodecType {
    PCMU = "PCMU",
    PCMA = "PCMA",
    AAC_ELD = "AAC-eld",
    OPUS = "OPUS",
    MSBC = "mSBC",
    AMR = "AMR",
    AMR_WB = "AMR-WB"
}
/**
 * @group Camera
 */
export declare const enum AudioStreamingSamplerate {
    KHZ_8 = 8,
    KHZ_16 = 16,
    KHZ_24 = 24
}
/**
 * @group Camera
 */
export type StreamSessionIdentifier = string;
/**
 * @group Camera
 */
export type SnapshotRequest = {
    height: number;
    width: number;
    /**
     * An optional {@link ResourceRequestReason}. The client decides if it wants to send this value. It is typically
     * only sent in the context of HomeKit Secure Video Cameras.
     * This value might be used by a `CameraStreamingDelegate` for informational purposes.
     * When `handleSnapshotRequest` is called, it is already checked if the respective reason is allowed in the current camera configuration.
     */
    reason?: ResourceRequestReason;
};
/**
 * @group Camera
 */
export type PrepareStreamRequest = {
    sessionID: StreamSessionIdentifier;
    targetAddress: string;
    addressVersion: "ipv4" | "ipv6";
    audio: Source;
    video: Source;
};
/**
 * @group Camera
 */
export type Source = {
    port: number;
    srtpCryptoSuite: SRTPCryptoSuites;
    srtp_key: Buffer;
    srtp_salt: Buffer;
    proxy_rtp?: number;
    proxy_rtcp?: number;
};
/**
 * @group Camera
 */
export type PrepareStreamResponse = {
    /**
     * @deprecated The local ip address will be automatically determined by HAP-NodeJS.
     *   Any value set will be ignored. You may only still set a value to support version prior to 0.7.9
     */
    address?: string | Address;
    /**
     * Any value set to this optional property will overwrite the automatically determined local address,
     * which is sent as RTP endpoint to the iOS device.
     */
    addressOverride?: string;
    video: SourceResponse | ProxiedSourceResponse;
    audio?: SourceResponse | ProxiedSourceResponse;
};
/**
 * @group Camera
 * @deprecated just supply the address directly in {@link PrepareStreamRequest}
 */
export type Address = {
    address: string;
    type?: "v4" | "v6";
};
/**
 * @group Camera
 */
export interface SourceResponse {
    port: number;
    ssrc: number;
    srtp_key?: Buffer;
    srtp_salt?: Buffer;
}
/**
 * @group Camera
 */
export interface ProxiedSourceResponse {
    proxy_pt: number;
    proxy_server_address: string;
    proxy_server_rtp: number;
    proxy_server_rtcp: number;
}
/**
 * @group Camera
 */
export declare const enum StreamRequestTypes {
    RECONFIGURE = "reconfigure",
    START = "start",
    STOP = "stop"
}
/**
 * @group Camera
 */
export type StreamingRequest = StartStreamRequest | ReconfigureStreamRequest | StopStreamRequest;
/**
 * @group Camera
 * @deprecated replaced by {@link StreamingRequest}
 */
export type StreamRequest = {
    sessionID: SessionIdentifier;
    type: StreamRequestTypes;
    video?: VideoInfo;
    audio?: AudioInfo;
};
/**
 * @group Camera
 */
export type StartStreamRequest = {
    sessionID: StreamSessionIdentifier;
    type: StreamRequestTypes.START;
    video: VideoInfo;
    audio: AudioInfo;
};
/**
 * @group Camera
 */
export type ReconfigureStreamRequest = {
    sessionID: StreamSessionIdentifier;
    type: StreamRequestTypes.RECONFIGURE;
    video: ReconfiguredVideoInfo;
};
/**
 * @group Camera
 */
export type StopStreamRequest = {
    sessionID: StreamSessionIdentifier;
    type: StreamRequestTypes.STOP;
};
/**
 * @group Camera
 */
export type AudioInfo = {
    codec: AudioStreamingCodecType;
    channel: number;
    bit_rate: number;
    sample_rate: AudioStreamingSamplerate;
    packet_time: number;
    pt: number;
    ssrc: number;
    max_bit_rate: number;
    rtcp_interval: number;
    comfort_pt: number;
    comfortNoiseEnabled: boolean;
};
/**
 * @group Camera
 */
export type VideoInfo = {
    codec: VideoCodecType;
    profile: H264Profile;
    level: H264Level;
    packetizationMode: VideoCodecPacketizationMode;
    cvoId?: number;
    width: number;
    height: number;
    fps: number;
    pt: number;
    ssrc: number;
    max_bit_rate: number;
    rtcp_interval: number;
    mtu: number;
};
/**
 * @group Camera
 */
export type ReconfiguredVideoInfo = {
    width: number;
    height: number;
    fps: number;
    max_bit_rate: number;
    rtcp_interval: number;
};
/**
 * @group Camera
 */
export interface RTPStreamManagementState {
    id: number;
    active: boolean;
}
/**
 * @group Camera
 */
export declare class RTPStreamManagement {
    /**
     * @deprecated Please use the SRTPCryptoSuites const enum above.
     */
    static SRTPCryptoSuites: typeof SRTPCryptoSuites;
    /**
     * @deprecated Please use the H264Profile const enum above.
     */
    static VideoCodecParamProfileIDTypes: typeof H264Profile;
    /**
     * @deprecated won't be updated anymore. Please use the H264Level const enum above.
     */
    static VideoCodecParamLevelTypes: Readonly<{
        TYPE3_1: 0;
        TYPE3_2: 1;
        TYPE4_0: 2;
    }>;
    private readonly id;
    private readonly delegate;
    readonly service: CameraRTPStreamManagement;
    private stateChangeDelegate?;
    requireProxy: boolean;
    disableAudioProxy: boolean;
    supportedCryptoSuites: SRTPCryptoSuites[];
    videoOnly: boolean;
    readonly supportedRTPConfiguration: string;
    readonly supportedVideoStreamConfiguration: string;
    readonly supportedAudioStreamConfiguration: string;
    /**
     * @deprecated
     */
    connectionID?: SessionIdentifier;
    private activeConnection?;
    private readonly activeConnectionClosedListener;
    sessionIdentifier?: StreamSessionIdentifier;
    /**
     * @private private API
     */
    streamStatus: StreamingStatus;
    private ipVersion?;
    selectedConfiguration: string;
    setupEndpointsResponse: string;
    /**
     * @private deprecated API
     */
    audioProxy?: RTPProxy;
    /**
     * @private deprecated API
     */
    videoProxy?: RTPProxy;
    /**
     * A RTPStreamManagement is considered disabled if `HomeKitCameraActive` is set to false.
     * We use a closure based approach to retrieve the value of this characteristic.
     * The characteristic is managed by the RecordingManagement.
     */
    private readonly disabledThroughOperatingMode?;
    constructor(id: number, options: CameraStreamingOptions, delegate: CameraStreamingDelegate, service?: CameraRTPStreamManagement, disabledThroughOperatingMode?: () => boolean);
    forceStop(): void;
    getService(): CameraRTPStreamManagement;
    /**
     * @deprecated
     */
    handleCloseConnection(connectionID: SessionIdentifier): void;
    handleFactoryReset(): void;
    destroy(): void;
    private constructService;
    private setupServiceHandlers;
    private handleSessionClosed;
    private streamingIsDisabled;
    private _handleSelectedStreamConfigurationWrite;
    private _handleStartStream;
    private handleReconfigureStream;
    private _handleStopStream;
    private handleSetupEndpoints;
    private generateSetupEndpointResponse;
    private _updateStreamStatus;
    private static _supportedRTPConfiguration;
    private static _supportedVideoStreamConfiguration;
    private checkForLegacyAudioCodecRepresentation;
    private _supportedAudioStreamConfiguration;
    private resetSetupEndpointsResponse;
    private resetSelectedStreamConfiguration;
    /**
     * @private
     */
    serialize(): RTPStreamManagementState | undefined;
    /**
     * @private
     */
    deserialize(serialized: RTPStreamManagementState): void;
    /**
     * @private
     */
    setupStateChangeDelegate(delegate?: StateChangeDelegate): void;
}
/**
 * @group Camera
 * @deprecated - only there for backwards compatibility, please use {@link RTPStreamManagement} directly
 */
export declare class StreamController extends RTPStreamManagement {
    /**
     *  options get saved so we can still support {@link Accessory.configureCameraSource}
     */
    options: CameraStreamingOptions;
    constructor(id: number, options: CameraStreamingOptions, delegate: LegacyCameraSource, service?: CameraRTPStreamManagement);
}
export {};
//# sourceMappingURL=RTPStreamManagement.d.ts.map