"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NumericCharacteristicMonitor = exports.BinaryConditionCharacteristicMonitor = exports.MappingCharacteristicMonitor = exports.PassthroughCharacteristicMonitor = exports.NestedCharacteristicMonitor = void 0;
class BaseCharacteristicMonitor {
    constructor(key, service, characteristic) {
        this.key = key;
        this.service = service;
        this.characteristic = characteristic;
    }
    callback(state) {
        if (this.key in state) {
            let value = state[this.key];
            if (value !== undefined) {
                value = this.transformValueFromMqtt(value);
                if (value !== undefined) {
                    this.service.updateCharacteristic(this.characteristic, value);
                }
            }
        }
    }
}
class NestedCharacteristicMonitor {
    constructor(key, monitors) {
        this.key = key;
        this.monitors = monitors;
        if (monitors.length === 0) {
            throw new RangeError(`No monitors passed to NestedCharacteristicMonitor for key ${key}.`);
        }
    }
    callback(state) {
        if (this.key in state) {
            const nested_state = state[this.key];
            this.monitors.forEach(m => m.callback(nested_state));
        }
    }
}
exports.NestedCharacteristicMonitor = NestedCharacteristicMonitor;
class PassthroughCharacteristicMonitor extends BaseCharacteristicMonitor {
    constructor(key, service, characteristic) {
        super(key, service, characteristic);
    }
    transformValueFromMqtt(value) {
        return value;
    }
}
exports.PassthroughCharacteristicMonitor = PassthroughCharacteristicMonitor;
class MappingCharacteristicMonitor extends BaseCharacteristicMonitor {
    constructor(key, service, characteristic, mapping) {
        super(key, service, characteristic);
        this.mapping = mapping;
        if (mapping.size === 0) {
            throw new RangeError(`Empty mapping passed to MappingCharacteristicMonitor for key ${key} on service ${this.service.UUID}.`);
        }
    }
    transformValueFromMqtt(value) {
        return this.mapping.get(value);
    }
}
exports.MappingCharacteristicMonitor = MappingCharacteristicMonitor;
class BinaryConditionCharacteristicMonitor extends BaseCharacteristicMonitor {
    constructor(key, service, characteristic, condition, value_true, value_false) {
        super(key, service, characteristic);
        this.condition = condition;
        this.value_true = value_true;
        this.value_false = value_false;
    }
    transformValueFromMqtt(value) {
        return (this.condition(value)) ? this.value_true : this.value_false;
    }
}
exports.BinaryConditionCharacteristicMonitor = BinaryConditionCharacteristicMonitor;
class NumericCharacteristicMonitor extends BaseCharacteristicMonitor {
    constructor(key, service, characteristic, input_min, input_max, output_min, output_max) {
        super(key, service, characteristic);
        this.input_min = input_min;
        this.input_max = input_max;
        this.output_min = output_min;
        this.output_max = output_max;
        if (input_min === input_max) {
            throw new RangeError(`input min/max equal on NumericCharacteristicMonitor for key ${key} on service ${this.service.UUID}.`);
        }
        if (output_min !== undefined && output_min === output_max) {
            throw new RangeError(`output min/max equal on NumericCharacteristicMonitor for key ${key} on service ${this.service.UUID}.`);
        }
    }
    transformValueFromMqtt(value) {
        const input = value;
        let out_minimum;
        let out_maximum;
        const actualCharacteristic = this.service.getCharacteristic(this.characteristic);
        if (this.output_min === undefined) {
            if (actualCharacteristic === undefined || actualCharacteristic.props.minValue === undefined) {
                throw new Error('NumericCharacteristicMonitor initialized without output_min, but it is not provided by characteristic either.');
            }
            out_minimum = actualCharacteristic.props.minValue;
        }
        else {
            out_minimum = this.output_min;
        }
        if (this.output_max === undefined) {
            if (actualCharacteristic === undefined || actualCharacteristic.props.maxValue === undefined) {
                throw new Error('NumericCharacteristicMonitor initialized without output_max, but it is not provided by characteristic either.');
            }
            out_maximum = actualCharacteristic.props.maxValue;
        }
        else {
            out_maximum = this.output_max;
        }
        if (input <= this.input_min) {
            return out_minimum;
        }
        if (input >= this.input_max) {
            return out_maximum;
        }
        const percentage = (input - this.input_min) / (this.input_max - this.input_min);
        return out_minimum + (percentage * (out_maximum - out_minimum));
    }
}
exports.NumericCharacteristicMonitor = NumericCharacteristicMonitor;
//# sourceMappingURL=monitor.js.map